/*
 * Copyright (C) 2001, 2002 Free Software Foundation
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Authors : Carlos Garca Campos <carlosgc@gnome.org>
 */


#include <config.h>

#include <gnome.h>
#include <panel-applet.h>
#include <panel-applet-gconf.h>
#include <glade/glade.h>

#include "cpufreq-applet.h"
#include "cpufreq-prefs.h"
#include "cpufreq.h"

static void     cpufreq_applet_preferences_dialog (BonoboUIComponent *uic, CPUFreqApplet *applet);
static void     cpufreq_applet_help_cb  (BonoboUIComponent *uic, CPUFreqApplet *applet);
static void     cpufreq_applet_about_cb (BonoboUIComponent *uic, CPUFreqApplet *applet);
static gint     cpufreq_applet_get_max_cpu (void);
static void     cpufreq_applet_destroy  (CPUFreqApplet *applet);
static void     cpufreq_setup_widgets   (CPUFreqApplet *applet);
static GtkWidget *cpufreq_applet_new    (CPUFreqApplet *applet);
static gboolean cpufreq_applet_fill     (CPUFreqApplet *applet);
static gboolean cpufreq_applet_factory  (CPUFreqApplet *applet, const gchar *iid, gpointer gdata);

static const BonoboUIVerb cpufreq_menu_verbs [] = {
	   BONOBO_UI_UNSAFE_VERB ("CPUFreqAppletPreferences",
						 cpufreq_applet_preferences_dialog),
	   BONOBO_UI_UNSAFE_VERB ("CPUFreqAppletHelp",
						 cpufreq_applet_help_cb),
	   BONOBO_UI_UNSAFE_VERB ("CPUFreqAppletAbout",
						 cpufreq_applet_about_cb),
	   BONOBO_UI_VERB_END
};

static GType
cpufreq_applet_get_type (void)
{
	   static GType type = 0;

	   if (!type) {
			 static const GTypeInfo info = {
				    sizeof (PanelAppletClass),
				    NULL, NULL, NULL, NULL, NULL,
				    sizeof (CPUFreqApplet),
				    0, NULL, NULL
			 };

			 type = g_type_register_static (
				    PANEL_TYPE_APPLET, "CPUFreqApplet", &info, 0);
	   }

	   return type;
}

void
cpufreq_applet_display_error (const gchar *error_message)
{
	   GtkWidget *dialog;

	   dialog = gtk_message_dialog_new (NULL,
								 GTK_DIALOG_DESTROY_WITH_PARENT,
								 GTK_MESSAGE_ERROR,
								 GTK_BUTTONS_CLOSE,
								 _(error_message));
	   gtk_dialog_run (GTK_DIALOG (dialog));
	   gtk_widget_destroy (dialog);
}

static void
cpufreq_applet_preferences_dialog (BonoboUIComponent *uic, CPUFreqApplet *applet)
{
	   g_return_if_fail (PANEL_IS_APPLET (PANEL_APPLET (applet)));
	   
	   cpufreq_preferences_dialog_run (applet);
}

static void
cpufreq_applet_help_cb (BonoboUIComponent *uic, CPUFreqApplet *applet)
{
	   GError *error;
	   
	   g_return_if_fail (PANEL_IS_APPLET (PANEL_APPLET (applet)));

	   error = NULL;
	   gnome_help_display (PACKAGE, NULL, &error);

	   if (error) {
			 if (error->code == GNOME_HELP_ERROR_INTERNAL) {
				    cpufreq_applet_display_error ("Sorry, a internal error has been ocurred "
										    "with the CPU Frequency Scaling Monitor help");
			 } else {
				    cpufreq_applet_display_error ("Sorry, the document can not be found");
			 }
			 
			 g_error_free (error);
	   }
}

static void
cpufreq_applet_about_cb (BonoboUIComponent *uic, CPUFreqApplet *applet)
{
	   static GtkWidget *about = NULL;
	   GdkPixbuf *pixbuf;
	   const gchar *authors[] = {
			 "Carlos Garcia Campos <carlosgc@gnome.org>",
			 " ",
			 _("Graphic Arts:"),
			 "Pablo Arroyo Loma <zzioma@yahoo.es>",
			 NULL
	   };
	   const gchar *documenters[] = {
			 "Carlos Garcia Campos <carlosgc@gnome.org>",
			 NULL
	   };
	   const gchar *translator_credits = _("translator_credits");

	   g_return_if_fail (PANEL_IS_APPLET (PANEL_APPLET (applet)));
	   
	   if (applet->about_dialog != NULL) {
			 gtk_window_set_screen (GTK_WINDOW (applet->about_dialog),
							    gtk_widget_get_screen (GTK_WIDGET (applet)));

			 gtk_window_present (GTK_WINDOW (applet->about_dialog));
			 return;
	   }

	   pixbuf = gdk_pixbuf_new_from_file (ICONDIR"/gnome-cpufreq-applet/cpufreq-applet.png", NULL);

	   applet->about_dialog = gnome_about_new (
			 _("CPU Frequency Scaling Monitor"),
			 VERSION,
			 _("Copyright (C) 2004 Free Software Foundation. Inc."),
			 _("This utility shows the current CPU Frequency Scaling."),
			 authors,
			 documenters,
			 strcmp (translator_credits, "translator_credits") != 0 ? translator_credits : NULL,
			 pixbuf);

	   g_object_unref (pixbuf);

	   gtk_window_set_screen (GTK_WINDOW (applet->about_dialog),
						 gtk_widget_get_screen (GTK_WIDGET (applet)));

	   g_signal_connect (applet->about_dialog, "destroy",
					 G_CALLBACK (gtk_widget_destroyed),
					 &applet->about_dialog);
	   
	   gtk_widget_show (applet->about_dialog);

	   return;
}

static gint
cpufreq_applet_get_max_cpu ()
{
	   gint mcpu = -1;
	   gchar *file = NULL;

	   do {
			 if (file) g_free (file);
			 mcpu ++;
			 file = g_strdup_printf ("/sys/devices/system/cpu/cpu%d", mcpu);
	   } while (g_file_test (file, G_FILE_TEST_EXISTS));
	   g_free (file);
	   mcpu --;

	   if (mcpu >= 0)
			 return mcpu;

	   do {
			 if (file) g_free (file);
			 mcpu ++;
			 file = g_strdup_printf ("/proc/sys/cpu/%d", mcpu);
	   } while (g_file_test (file, G_FILE_TEST_EXISTS));
	   g_free (file);
	   mcpu --;

	   if (mcpu >= 0)
			 return mcpu;
	   else
			 return 0;
}

void
cpufreq_applet_update (CPUFreqApplet *applet, CPUFreqMode mode)
{
	   gchar *pixmaps[] = {
			 ICONDIR"/gnome-cpufreq-applet/cpufreq-25.png",
			 ICONDIR"/gnome-cpufreq-applet/cpufreq-50.png",
			 ICONDIR"/gnome-cpufreq-applet/cpufreq-75.png",
			 ICONDIR"/gnome-cpufreq-applet/cpufreq-100.png",
			 NULL };
	   gint perc, image;
	   gchar *text_label, *text_tip;

	   g_return_if_fail (PANEL_IS_APPLET (PANEL_APPLET (applet)));
	   
	   switch (applet->show_mode) {
	   case MODE_FREQUENCY:
			 if (applet->show_unit) {
				    text_label = g_strdup_printf ("%s %s", applet->freq, applet->unit);
				    gtk_label_set_text (GTK_LABEL (applet->label), text_label);
				    g_free (text_label);
			 } else {
				    gtk_label_set_text (GTK_LABEL (applet->label), applet->freq);
			 }
			 
			 gtk_widget_show (applet->label);
			 break;
	   case MODE_PERCENTAGE:
			 gtk_label_set_text (GTK_LABEL (applet->label), applet->perc);
			 gtk_widget_show (applet->label);
			 break;
	   case MODE_NONE:
			 gtk_widget_hide (applet->label);
	   }

	   if (mode == POWERSAVE) {
			 text_tip = g_strdup_printf ("CPU %d - Powersave\n%s %s (%s)",
								    applet->cpu, applet->freq,
								    applet->unit, applet->perc);
	   } else if (mode == PERFORMANCE) {
			 text_tip = g_strdup_printf ("CPU %d - Performance\n%s %s (%s)",
								    applet->cpu, applet->freq,
								    applet->unit, applet->perc);
	   } else {
			 text_tip = g_strdup_printf ("CPU %d - Userspace\n%s %s (%s)",
								    applet->cpu, applet->freq,
								    applet->unit, applet->perc);
	   }
	   
	   gtk_tooltips_set_tip (applet->tips, GTK_WIDGET (applet), text_tip, NULL);
	   g_free (text_tip);

	   perc = atoi (g_strndup (applet->perc, strlen (applet->perc) - 1));
	   /* 0-29   -> 25%
	    * 30-69  -> 50%
	    * 70-89  -> 75%
	    * 90-100 -> 100%
	    */
	   if (perc < 30)
			 image = 0;
	   else if ((perc >= 30) && (perc < 70))
			 image = 1;
	   else if ((perc >= 70) && (perc < 90))
			 image = 2;
	   else
			 image = 3;
	   
	   if (applet->pixbufs[image] == NULL) {
			 applet->pixbufs[image] = gdk_pixbuf_new_from_file (pixmaps[image], NULL);
	   }

	   gtk_image_set_from_pixbuf (GTK_IMAGE (applet->pixmap), applet->pixbufs[image]);
	   
	   gtk_widget_show (applet->pixmap);
}

static void
cpufreq_applet_destroy (CPUFreqApplet *applet)
{
	   gint i;
	   
	   g_return_if_fail (PANEL_IS_APPLET (PANEL_APPLET (applet)));

	   if (applet->timeout_handler > 0)
			 g_source_remove (applet->timeout_handler);

	   if (applet->tips)
			 g_object_unref (G_OBJECT (applet->tips));

	   for (i=0; i<=3; i++) {
			 if (applet->pixbufs[i])
				    g_object_unref (G_OBJECT (applet->pixbufs[i]));
	   }

	   if (applet->freq) {
			 g_free (applet->freq);
			 applet->freq = NULL;
	   }
	   
	   if (applet->perc) {
			 g_free (applet->perc);
			 applet->perc = NULL;
	   }
	   
	   if (applet->unit) {
			 g_free (applet->unit);
			 applet->unit = NULL;
	   }

	   if (applet->about_dialog) {
			 gtk_widget_destroy (applet->about_dialog);
			 applet->about_dialog = NULL;
	   }

	   if (applet->prefs) {
			 gtk_widget_destroy (applet->prefs);
			 applet->prefs = NULL;
	   }
}	

static void
cpufreq_setup_widgets (CPUFreqApplet *applet)
{
	   GtkRequisition req;
	   gint total_size = 4;
	   gboolean horizontal = FALSE;
	   gint panel_size;

	   g_return_if_fail (PANEL_IS_APPLET (PANEL_APPLET (applet)));
	   
	   panel_size = panel_applet_get_size (PANEL_APPLET (applet));

	   switch (panel_applet_get_orient (PANEL_APPLET (applet))) {
	   case PANEL_APPLET_ORIENT_LEFT:
	   case PANEL_APPLET_ORIENT_RIGHT:
			 horizontal = FALSE;
			 break;
	   case PANEL_APPLET_ORIENT_UP:
	   case PANEL_APPLET_ORIENT_DOWN:
			 horizontal = TRUE;
			 break;
	   }

	   if (applet->label)
			 gtk_widget_destroy (applet->label);
	   
	   applet->label = gtk_label_new ("---");
	   
	   gtk_widget_size_request (applet->label, &req);
	   if (applet->show_mode != MODE_NONE)
			 gtk_widget_show (applet->label);

	   if (horizontal)
			 total_size += req.height;
	   else
			 total_size += req.width;

	   if (applet->pixmap)
			 gtk_widget_destroy (applet->pixmap);
	   
	   applet->pixmap =
			 gtk_image_new_from_file (ICONDIR"/gnome-cpufreq-applet/cpufreq-na.png");
	   gtk_widget_size_request (applet->pixmap, &req);
	   gtk_widget_show (applet->pixmap);

	   if (horizontal)
			 total_size += req.height;
	   else
			 total_size += req.width;

	   /* FIXME */
	   if (applet->show_unit) {
			 if (horizontal)
				    total_size += 3;
			 else
				    total_size += 3;
	   }
	   
	   if (applet->box)
			 gtk_widget_destroy (applet->box);

	   if (horizontal && (total_size <= panel_size))
			 applet->box = gtk_vbox_new (FALSE, 2);
	   else if (horizontal && (total_size > panel_size))
			 applet->box = gtk_hbox_new (FALSE, 2);
	   else if (!horizontal && (total_size <= panel_size))
			 applet->box = gtk_hbox_new (FALSE, 2);
	   else
			 applet->box = gtk_vbox_new (FALSE, 2);

	   gtk_box_pack_start (GTK_BOX (applet->box), applet->pixmap, TRUE, TRUE, 0);
	   gtk_box_pack_start (GTK_BOX (applet->box), applet->label, TRUE, TRUE, 0);
	   
	   gtk_widget_show (applet->box);

	   gtk_container_add (GTK_CONTAINER (applet), applet->box);
}


static GtkWidget *
cpufreq_applet_new (CPUFreqApplet *applet)
{
	   AtkObject *atk_obj;
	   gint i;
	   gchar *text_tip;
	   GError *error;

	   panel_applet_add_preferences (PANEL_APPLET (applet),
							   "/schemas/apps/gnome-cpufreq-applet/prefs", NULL);
	   
	   panel_applet_set_flags (PANEL_APPLET (applet), PANEL_APPLET_EXPAND_MINOR);

	   /* New applet, default values */
	   applet->cpu = 0;
	   applet->timeout_handler = 0;
	   applet->mcpu = cpufreq_applet_get_max_cpu ();
	   applet->prefs = NULL;
	   applet->about_dialog = NULL;

	   error = NULL;
	   applet->show_mode = panel_applet_gconf_get_int (PANEL_APPLET (applet), "show_mode", &error);
	   
        /* In case anything went wrong with gconf, get back to the default */
	   if (error || applet->show_mode < MODE_FREQUENCY || applet->show_mode > MODE_NONE) {
			 applet->show_mode = MODE_FREQUENCY;
			 if (error)
				    g_error_free (error);
	   }

	   error = NULL;
	   applet->show_unit = panel_applet_gconf_get_bool (PANEL_APPLET (applet), "show_unit", &error);

	   /* In case anything went wrong with gconf, get back to the default */
	   if (error) {
			 applet->show_unit = TRUE;
			 g_error_free (error);
	   }
	   
	   applet->freq = NULL;
	   applet->perc = NULL;
	   applet->unit = NULL;

	   for (i=0; i<=3; i++)
			 applet->pixbufs[i] = NULL;

	   applet->tips = gtk_tooltips_new ();
	   g_object_ref (G_OBJECT (applet->tips));

	   cpufreq_setup_widgets (applet);
	   
	   g_signal_connect (G_OBJECT (applet), "destroy",
					 G_CALLBACK (cpufreq_applet_destroy),
					 NULL);

	   /* Setup the menus */
	   panel_applet_setup_menu_from_file (PANEL_APPLET (applet),
								   NULL,
								   "GNOME_CPUFreqApplet.xml",
								   NULL,
								   cpufreq_menu_verbs,
								   applet);

	   /* Not in 2.4 */
	   /*if (panel_applet_get_locked_down (PANEL_APPLET (applet))) {
			 BonoboUIComponent *popup_component;
			 
			 popup_component = panel_applet_get_popup_component (PANEL_APPLET (applet));
			 
			 bonobo_ui_component_set_prop (popup_component,
									 "/commands/CPUFreqPreferences",
									 "hidden", "1",
									 NULL);
									 }*/
	   

	   if (g_file_test ("/sys/devices/system/cpu/cpu0/cpufreq", G_FILE_TEST_EXISTS)) { /* 2.6 kernel */
			 applet->timeout_handler = g_timeout_add (1000, cpufreq_get_from_sysfs, (gpointer) applet);
	   } else if (g_file_test ("/proc/cpufreq", G_FILE_TEST_EXISTS)) { /* 2.4 kernel */
			 applet->timeout_handler = g_timeout_add (1000, cpufreq_get_from_procfs, (gpointer) applet);
	   } else if (g_file_test ("/proc/cpuinfo", G_FILE_TEST_EXISTS)) {
			 /* If there is no cpufreq support it shows only the cpu frequency,
			  * I think is better than do nothing. I have to notify it to the user, because
			  * he could think that cpufreq is supported but it doesn't work succesfully
			  */
			 cpufreq_applet_display_error ("There is no cpufreq support in your system, needed "
									 "to use CPU Frequency Scaling Monitor.\n"
									 "If your system already supports cpufreq, check whether "
									 "/sys or /proc file systems are mounted.\n"
									 "Now the applet will only show the current "
									 "cpu frequency.");
			 
			 if (cpufreq_get_from_procfs_cpuinfo (applet)) {
				    text_tip = g_strdup_printf ("CPU %d - cpufreq Not Supported\n%s (%s)",
										  applet->cpu, applet->freq, applet->perc);
				    gtk_tooltips_set_tip (applet->tips, GTK_WIDGET (applet), text_tip, NULL);
				    g_free (text_tip);
			 } else {
				    gtk_tooltips_set_tip (applet->tips, GTK_WIDGET (applet),
									 _("cpufreq Not Supported"),
									 NULL);
			 }
			 
			 gtk_widget_show (applet->pixmap);
	   }

	   atk_obj = gtk_widget_get_accessible (GTK_WIDGET (applet));

	   if (GTK_IS_ACCESSIBLE (atk_obj)) {
			 atk_object_set_name (atk_obj, _("CPU Frequency Scaling Monitor"));
			 atk_object_set_description (atk_obj, _("This utility shows the current CPU Frequency"));
	   }

	   g_signal_connect (G_OBJECT (applet), "change_size",
					 G_CALLBACK (cpufreq_setup_widgets), applet);
	   g_signal_connect (G_OBJECT (applet), "change_orient",
					 G_CALLBACK (cpufreq_setup_widgets), applet);

	   return GTK_WIDGET (applet);
}

static gboolean
cpufreq_applet_fill (CPUFreqApplet *applet)
{
	   g_return_val_if_fail (PANEL_IS_APPLET (PANEL_APPLET (applet)), FALSE);
	   
	   gnome_window_icon_set_default_from_file
			 (ICONDIR"/gnome-cpufreq-applet/cpufreq-applet.png");

	   glade_gnome_init ();

	   cpufreq_applet_new (applet);

	   gtk_widget_show (GTK_WIDGET (applet));
	   
	   return TRUE;
}

static gboolean
cpufreq_applet_factory (CPUFreqApplet *applet, const gchar *iid, gpointer gdata)
{
	   gboolean retval = FALSE;

	   if (!strcmp (iid, "OAFIID:GNOME_CPUFreqApplet"))
			 retval = cpufreq_applet_fill (applet);

	   return retval;
}

PANEL_APPLET_BONOBO_FACTORY ("OAFIID:GNOME_CPUFreqApplet_Factory",
					    cpufreq_applet_get_type (),
					    PACKAGE, VERSION,
					    (PanelAppletFactoryCallback) cpufreq_applet_factory,
					    NULL)
