/*
 * Copyright (C) 2001, 2002 Free Software Foundation
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Authors : Carlos Garca Campos <carlosgc@gnome.org>
 */

#include <config.h>

#include <gnome.h>
#include <panel-applet.h>
#include <panel-applet-gconf.h>
#include <glade/glade.h>

#include "cpufreq-applet.h"
#include "cpufreq-prefs.h"

static void cpufreq_prefs_response_cb        (GtkDialog *dialog, gint response, gpointer gdata);
static void cpufreq_prefs_show_freq_toggled  (GtkWidget *show_freq, gpointer *gdata);
static void cpufreq_prefs_show_unit_toggled  (GtkWidget *show_unit, gpointer *gdata);
static void cpufreq_prefs_show_perc_toggled  (GtkWidget *show_perc, gpointer *gdata);
static void cpufreq_prefs_show_none_toggled  (GtkWidget *show_none, gpointer *gdata);
static void cpufreq_prefs_cpu_number_changed (GtkWidget *cpu_number, gpointer *gdata);

static void
cpufreq_prefs_show_freq_toggled (GtkWidget *show_freq, gpointer *gdata)
{
	   GtkWidget *show_unit;
	   CPUFreqApplet *applet;
	   gchar *text_label;

	   applet = (CPUFreqApplet *) gdata;
	   show_unit = g_object_get_data (G_OBJECT (applet->prefs), "prefs-show-unit");

	   if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (show_freq))) {
			 /* Show cpu usage in frequency (Hz) */
			 gtk_widget_set_sensitive (show_unit, TRUE);
			 gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (show_unit), applet->show_unit);
			 applet->show_mode = MODE_FREQUENCY;
			 
			 if (applet->freq) {
				    if (applet->show_unit) {
						  text_label = g_strdup_printf ("%s %s", applet->freq, applet->unit);
						  gtk_label_set_label (GTK_LABEL (applet->label), text_label);
						  g_free (text_label);
				    } else {
						  gtk_label_set_label (GTK_LABEL (applet->label), applet->freq);
				    }
			 }
				    
			 gtk_widget_show (applet->label);
			 panel_applet_gconf_set_int (PANEL_APPLET (applet), "show_mode",
								    applet->show_mode, NULL);
	   } else {
			 gtk_widget_set_sensitive (show_unit, FALSE);
	   }
}

static void
cpufreq_prefs_show_unit_toggled (GtkWidget *show_unit, gpointer *gdata)
{
	   CPUFreqApplet *applet;
	   gchar *text_label;

	   applet = (CPUFreqApplet *) gdata;

	   if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (show_unit))) {
			 if (applet->freq) {
				    text_label = g_strdup_printf ("%s %s", applet->freq, applet->unit);
				    gtk_label_set_label (GTK_LABEL (applet->label), text_label);
				    g_free (text_label);
			 }

			 applet->show_unit = TRUE;
	   } else {
			 if (applet->freq)
				    gtk_label_set_label (GTK_LABEL (applet->label), applet->freq);

			 applet->show_unit = FALSE;
	   }

	   gtk_widget_show (applet->label);
	   panel_applet_gconf_set_bool (PANEL_APPLET (applet), "show_unit",
							  applet->show_unit, NULL);
}

static void
cpufreq_prefs_show_perc_toggled (GtkWidget *show_perc, gpointer *gdata)
{
	   CPUFreqApplet *applet;

	   applet = (CPUFreqApplet *) gdata;

	   if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (show_perc))) {
			 /* Show cpu usage in percentage */
			 applet->show_mode = MODE_PERCENTAGE;
			 if (applet->perc)
				    gtk_label_set_label (GTK_LABEL (applet->label), applet->perc);
			 gtk_widget_show (applet->label);
			 panel_applet_gconf_set_int (PANEL_APPLET (applet), "show_mode",
								    applet->show_mode, NULL);
	   }
}

static void
cpufreq_prefs_show_none_toggled (GtkWidget *show_none, gpointer *gdata)
{
	   CPUFreqApplet *applet;

	   applet = (CPUFreqApplet *) gdata;

	   if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (show_none))) {
			 /* Don't show cpu usage */
			 applet->show_mode = MODE_NONE;
			 gtk_widget_hide (applet->label);
			 panel_applet_gconf_set_int (PANEL_APPLET (applet), "show_mode",
								    applet->show_mode, NULL);
	   }
}

static void
cpufreq_prefs_cpu_number_changed (GtkWidget *cpu_number, gpointer *gdata)
{
	   CPUFreqApplet *applet;

	   applet = (CPUFreqApplet *) gdata;

	   applet->cpu = gtk_option_menu_get_history (GTK_OPTION_MENU (cpu_number));
	   panel_applet_gconf_set_int (PANEL_APPLET (applet), "cpu", applet->cpu, NULL);
}
	   
static void
cpufreq_prefs_response_cb (GtkDialog *dialog, gint response, gpointer gdata)
{
	   GError *error;
	   CPUFreqApplet *applet;

	   applet = (CPUFreqApplet *) gdata;

	   if (response == GTK_RESPONSE_HELP) {
			 error = NULL;
			 gnome_help_display (PACKAGE, "cpufreq-applet-prefs", &error);
			 if (error) {
				    if (error->code == GNOME_HELP_ERROR_INTERNAL) {
						  cpufreq_applet_display_error ("Sorry, a internal error has been ocurred "
												  "with the CPU Frequency Scaling Monitor help");
				    } else {
						  cpufreq_applet_display_error ("Sorry, the document can not be found");
				    }
				    
				    g_error_free (error);
			 }
	   } else {
			 gtk_widget_destroy (applet->prefs);
			 applet->prefs = NULL;
	   }
}


void 
cpufreq_preferences_dialog_run (CPUFreqApplet *applet)
{
	   GladeXML  *xml;
	   GtkWidget *show_freq, *show_unit, *show_perc, *show_none;
	   GtkWidget *cpu_number;
	   GtkWidget *monitored_cpu_box;
	   GtkWidget *menu;
	   GtkWidget *item;
	   gint i;
	   gchar *text_label;

	   g_return_if_fail (PANEL_IS_APPLET (PANEL_APPLET (applet)));

	   if (applet->prefs) { /* the dialog already exist, only show it */
			 gtk_window_present (GTK_WINDOW (applet->prefs));
			 return;
	   }
	   
	   xml = glade_xml_new (DATADIR"/gnome-cpufreq-applet/cpufreq-preferences.glade",
					    "prefs_dialog", NULL);

	   applet->prefs = glade_xml_get_widget (xml, "prefs_dialog");
	   show_freq = glade_xml_get_widget (xml, "prefs_show_freq");
	   show_unit = glade_xml_get_widget (xml, "prefs_show_unit");
	   show_perc = glade_xml_get_widget (xml, "prefs_show_perc");
	   show_none = glade_xml_get_widget (xml, "prefs_show_none");
	   cpu_number = glade_xml_get_widget (xml, "prefs_cpu_number");
	   monitored_cpu_box = glade_xml_get_widget (xml, "monitored_cpu_box");

	   g_object_unref (G_OBJECT (xml));

	   gtk_window_set_screen (GTK_WINDOW (applet->prefs),
						 gtk_widget_get_screen (GTK_WIDGET (applet)));

	   gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (show_unit), applet->show_unit);

	   if (applet->show_mode == MODE_FREQUENCY) {
			 gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (show_freq), TRUE);
			 gtk_widget_set_sensitive (show_unit, TRUE);
	   } else if (applet->show_mode == MODE_PERCENTAGE) {
			 gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (show_perc), TRUE);
			 gtk_widget_set_sensitive (show_unit, FALSE);
	   } else {
			 gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (show_none), TRUE);
			 gtk_widget_set_sensitive (show_unit, FALSE);
	   }

	   /* Fill the option menu */
	   menu = gtk_menu_new ();

	   for (i=0; i <= applet->mcpu; i++) {
			 text_label = g_strdup_printf ("CPU %d", i);
			 item = gtk_menu_item_new_with_label (text_label);
			 g_free (text_label);
			 gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
			 gtk_widget_show (item);
	   }

	   gtk_option_menu_set_menu (GTK_OPTION_MENU (cpu_number), menu);
	   gtk_option_menu_set_history (GTK_OPTION_MENU (cpu_number), applet->cpu);

	   if (applet->mcpu > 0)
			 gtk_widget_show (monitored_cpu_box);
	   else
			 gtk_widget_hide (monitored_cpu_box);

	   g_object_set_data (G_OBJECT (applet->prefs), "prefs-show-unit", show_unit);
	   g_signal_connect (G_OBJECT (applet->prefs), "response",
					 G_CALLBACK (cpufreq_prefs_response_cb),
					 (gpointer) applet);
	   g_signal_connect (G_OBJECT (applet->prefs), "destroy",
					 G_CALLBACK (gtk_widget_destroy),
					 NULL);
	   g_signal_connect (G_OBJECT (show_freq), "toggled",
					 G_CALLBACK (cpufreq_prefs_show_freq_toggled),
					 (gpointer) applet);
	   g_signal_connect (G_OBJECT (show_unit), "toggled",
					 G_CALLBACK (cpufreq_prefs_show_unit_toggled),
					 (gpointer) applet);
	   g_signal_connect (G_OBJECT (show_perc), "toggled",
					 G_CALLBACK (cpufreq_prefs_show_perc_toggled),
					 (gpointer) applet);
	   g_signal_connect (G_OBJECT (show_none), "toggled",
					 G_CALLBACK (cpufreq_prefs_show_none_toggled),
					 (gpointer) applet);
	   g_signal_connect (G_OBJECT (cpu_number), "changed",
					 G_CALLBACK (cpufreq_prefs_cpu_number_changed),
					 (gpointer) applet);

	   gtk_widget_show (applet->prefs);
}
