/* GNOME Volume Control
 * Copyright (C) 2003-2004 Ronald Bultje <rbultje@ronald.bitfreak.net>
 *
 * main.c: intialization, window setup
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <getopt.h>
#include <glib.h>
#include <gnome.h>
#include <gst/gst.h>
#include <gst/audio/mixerutils.h>

#include "keys.h"
#include "stock.h"
#include "window.h"

/*
 * Probe for mixer elements. Set up GList * with elements,
 * where each element has a GObject data node set of the
 * name "gnome-volume-control-name" with the value being
 * the human-readable name of the element.
 *
 * All elements in the returned GList * are in state
 * GST_STATE_NULL.
 */

static gboolean
mixer_filter_func (GstMixer * mixer, gpointer user_data)
{
  GstElementFactory *factory;
  const gchar *long_name;
  gchar *devname = NULL;
  gchar *name;
  gint *p_count = (gint *) user_data;

  /* fetch name */
  if (g_object_class_find_property (G_OBJECT_GET_CLASS (G_OBJECT (mixer)),
                                    "device-name")) {
    g_object_get (mixer, "device-name", &devname, NULL);
    GST_DEBUG ("device name: %s", GST_STR_NULL (devname));
  } else {
    devname = NULL;
    GST_DEBUG ("device name unknown, no 'device-name' property");
  }
    
  factory = gst_element_get_factory (GST_ELEMENT (mixer));
  long_name = gst_element_factory_get_longname (factory);

  if (devname) {
    name = g_strdup_printf ("%s (%s)", devname, long_name);
    g_free (devname);
  } else {
    gchar *title;

    *p_count += 1;

    title = g_strdup_printf (_("Unknown Volume Control %d"),  *p_count);
    name = g_strdup_printf ("%s (%s)", title, long_name);
    g_free (title);
  }

  g_object_set_data_full (G_OBJECT (mixer),
                          "gnome-volume-control-name",
                          name,
                          (GDestroyNotify) g_free);

  GST_DEBUG ("Adding '%s' to list of available mixers", name);

  gst_element_set_state (GST_ELEMENT (mixer), GST_STATE_NULL);

  return TRUE; /* add mixer to list */
}

static GList *
create_mixer_collection (void)
{
  GList *mixer_list;
  gint counter = 0;

  mixer_list = gst_audio_default_registry_mixer_filter (mixer_filter_func,
                                                        FALSE,
                                                        &counter);

  return mixer_list;
}

static void
register_stock_icons (void)
{
  GtkIconFactory *icon_factory;
  struct {
    gchar *filename, *stock_id;
  } list[] = {
    { "3dsound.png",      GNOME_VOLUME_CONTROL_STOCK_3DSOUND     },
    { "headphones.png",   GNOME_VOLUME_CONTROL_STOCK_HEADPHONES  },
    { "mixer.png",        GNOME_VOLUME_CONTROL_STOCK_MIXER       },
    { "noplay.png",       GNOME_VOLUME_CONTROL_STOCK_NOPLAY      },
    { "norecord.png",     GNOME_VOLUME_CONTROL_STOCK_NORECORD    },
    { "phone.png",        GNOME_VOLUME_CONTROL_STOCK_PHONE       },
    { "play.png",         GNOME_VOLUME_CONTROL_STOCK_PLAY        },
    { "record.png",       GNOME_VOLUME_CONTROL_STOCK_RECORD      },
    { "tone.png",         GNOME_VOLUME_CONTROL_STOCK_TONE        },
    { "video.png",        GNOME_VOLUME_CONTROL_STOCK_VIDEO       },
    { NULL, NULL }
  };
  gint num;

  icon_factory = gtk_icon_factory_new ();
  gtk_icon_factory_add_default (icon_factory);

  for (num = 0; list[num].filename != NULL; num++) {
    gchar *filename =
	gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_PIXMAP,
				   list[num].filename, TRUE, NULL);

    if (filename) {
      GdkPixbuf *pixbuf = gdk_pixbuf_new_from_file (filename, NULL);
      GtkIconSet *icon_set = gtk_icon_set_new_from_pixbuf (pixbuf);

      gtk_icon_factory_add (icon_factory, list[num].stock_id, icon_set);
      g_free (filename);
    }
  }
}

static void
cb_destroy (GtkWidget *widget,
	    gpointer   data)
{
  gtk_main_quit ();
}

void
cb_check_resize (GtkContainer    *container,
      		  gpointer         user_data)
{
  GConfClient *client;
  gint width, height;

  client = gconf_client_get_default();
  gtk_window_get_size (GTK_WINDOW (container), &width, &height);
  gconf_client_set_int (client, PREF_UI_WINDOW_WIDTH, width, NULL);
  gconf_client_set_int (client, PREF_UI_WINDOW_HEIGHT, height, NULL);
}

gint
main (gint   argc,
      gchar *argv[])
{
  GOptionContext *ctx;
  GOptionGroup *group;
  gchar *appfile;
  GtkWidget *win;
  GList *elements;

  /* i18n */
  bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);

  ctx = g_option_context_new ("gnome-volume-control");
  g_option_context_add_group (ctx, gst_init_get_option_group ());

  gnome_program_init ("gnome-volume-control", VERSION,
                      LIBGNOMEUI_MODULE, argc, argv,
                      GNOME_PARAM_GOPTION_CONTEXT, ctx,
                      GNOME_PARAM_APP_DATADIR, DATA_DIR,
                      NULL);

  /* init ourselves */
  register_stock_icons ();

  /* add appicon image */
  appfile = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_PIXMAP,
				       "mixer.png", TRUE,
				       NULL);
  if (appfile) {
    gnome_window_icon_set_default_from_file (appfile);
    g_free (appfile);
  }

  if (!(elements = create_mixer_collection ())) {
    win = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_ERROR,
				  GTK_BUTTONS_CLOSE,
				  _("No volume control GStreamer plugins and/or devices found."));
    gtk_widget_show (win);
    gtk_dialog_run (GTK_DIALOG (win));
    gtk_widget_destroy (win);
    return -1;
  }

  /* window contains everything automagically */
  win = gnome_volume_control_window_new (elements);
  g_signal_connect (win, "destroy", G_CALLBACK (cb_destroy), NULL);
  g_signal_connect (win, "check_resize", G_CALLBACK (cb_check_resize), NULL);

  gtk_widget_show (win);
  gtk_main ();

  return 0;
}
