/*
 *  PGColorTextField.m
 *  graphviz
 *
 *  Created by Glen Low on Sat May 01 2004.
 *  Copyright (c) 2004, Pixelglow Software. All rights reserved.
 *  http://www.pixelglow.com/graphviz/
 *  graphviz@pixelglow.com
 *
 *  Redistribution and use in source and binary forms, with or without modification, are permitted
 *  provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright notice, this list of conditions
 *    and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
 *    and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *  * Neither the name of Pixelglow Software nor the names of its contributors may be used to endorse or
 *    promote products derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


#import "PGColorTextFieldCell.h"
#import "PGNilColor.h"

@interface PGColorTextFormatter: NSFormatter
	{
		NSColorList* colors_;
	}
	
- (void) setColors: (NSColorList*) colors;

@end

@implementation PGColorTextFieldCell

- (id) initTextCell: (NSString*) string
	{
		if ((self = [super initTextCell: string]))
			[self setFormatter: [[[PGColorTextFormatter alloc] init] autorelease]];				
			
			
		return self;
	}
	
- (id) copyWithZone: (NSZone*) zone
	{
		PGColorTextFieldCell* copy = [super copyWithZone: zone];
		[copy->colors_ retain];
		
		NSFormatter* formatter = [copy formatter];
		if ([formatter respondsToSelector: @selector (setColors:)])
			[formatter performSelector: @selector (setColors:) withObject: copy->colors_];
		
		return copy;
	}
				
- (NSColorList*) colors
	{
		return colors_;
	}
	
- (void) setColors: (NSColorList*) colors
	{
		if (colors_ != colors)
			{
				[colors_ release];
				colors_ = [colors retain];
			
				NSFormatter* formatter = [self formatter];
				if ([formatter respondsToSelector: @selector (setColors:)])
					[formatter performSelector: @selector (setColors:) withObject: colors];
			}
	}

- (void) drawWithFrame: (NSRect) cellFrame inView: (NSView*) controlView
	{
		NSRect textFrame = NSMakeRect (
			cellFrame.origin.x,
			cellFrame.origin.y,
			cellFrame.size.width - cellFrame.size.height,
			cellFrame.size.height);
		[super drawWithFrame: textFrame inView: controlView];
		
		NSRect swatchFrame = NSMakeRect (
			cellFrame.origin.x + cellFrame.size.width - cellFrame.size.height + 4.0,
			cellFrame.origin.y,
			cellFrame.size.height,
			cellFrame.size.height);
			
		if (!NSIsEmptyRect (swatchFrame))
			{
				id objectValue = [self objectValue];
				if (!objectValue || ![objectValue isKindOfClass: [NSColor class]])
					objectValue = [PGNilColor nilColor];
					
				[[objectValue shadowWithLevel: 0.2] set];
				[NSBezierPath fillRect: NSInsetRect (swatchFrame, 3.0, 3.0)];
				[objectValue drawSwatchInRect: NSInsetRect (swatchFrame, 4.0, 4.0)];
			}
	}
	
- (BOOL) trackMouse: (NSEvent*) event inRect: (NSRect) cellFrame ofView: (NSView*) controlView untilMouseUp: (BOOL) untilMouseUp
	{
		if ([controlView convertPoint: [event locationInWindow] fromView: nil].x < cellFrame.origin.x + cellFrame.size.width - cellFrame.size.height)
			return [super trackMouse: event inRect: cellFrame ofView: controlView untilMouseUp: untilMouseUp];
		else
			{
				event = [[controlView window] nextEventMatchingMask: NSLeftMouseUpMask];
				if ([controlView
					mouse: [controlView convertPoint: [event locationInWindow] fromView: nil]
					inRect: cellFrame])
					{
						[NSApp orderFrontColorPanel: self];
						return YES;
					}
				else
					return NO;				
			}
	}

- (void) editWithFrame: (NSRect) rect inView: (NSView*) controlView editor: (NSText*) text delegate: (id) object event: (NSEvent*) event
	{
		rect.size.width -= rect.size.height + 3.0;
		[super editWithFrame: rect inView: controlView editor: text delegate: object event: event];
	}

- (void) selectWithFrame: (NSRect) rect inView: (NSView*) controlView editor: (NSText*) text delegate: (id) object start: (int) selStart length: (int) selLength
	{
		rect.size.width -= rect.size.height + 3.0;
		[super selectWithFrame: rect inView: controlView editor: text delegate: object start: selStart length: selLength];
	}
	
- (void) dealloc
	{
		[colors_ release];
		colors_ = nil;
		
		[super dealloc];
	}
	
@end

@implementation PGColorTextFormatter

- (id) init
	{
		if ((self = [super init]))
			colors_ = nil;
		return self;
	}

- (void) setColors: (NSColorList*) colors
	{
		if (colors_ != colors)
			{
				[colors_ release];
				colors_ = [colors retain];
			}
	}
	
- (NSString*) stringForObjectValue: (id) object
	{
		if (!object || [object isKindOfClass: [PGNilColor class]])
			return @"";
		else if ([object isKindOfClass: [NSColor class]])
			{
				NSEnumerator* eachColorKey = [[colors_ allKeys] objectEnumerator];
				NSString* nextColorKey;
				while ((nextColorKey = [eachColorKey nextObject]))
					if ([[colors_ colorWithKey: nextColorKey] isEqual: object])
						return nextColorKey;
				
				NSColor* rgb = [object colorUsingColorSpaceName: NSCalibratedRGBColorSpace];
				int red = [rgb redComponent] * 255.0;
				int green = [rgb greenComponent] * 255.0;
				int blue = [rgb blueComponent] * 255.0;
				int alpha = [rgb alphaComponent] * 255.0;
				return alpha == 255 ?
					[NSString stringWithFormat: @"#%02x%02x%02x", red, green, blue] :
					[NSString stringWithFormat: @"#%02x%02x%02x%02x", red, green, blue, alpha];
			}
		else
			return [object description];
	}

- (BOOL) getObjectValue: (id*) object forString: (NSString*) string errorDescription: (NSString**) error
	{
		if (!string || [string isEqualToString: @""])
			*object = [PGNilColor nilColor];
		else
			{
				NSColor* color = [colors_ colorWithKey: string];
				if (color)
					*object = color;
				else
					{
						unsigned int red;
						unsigned int green;
						unsigned int blue;
						unsigned int alpha;
						
						switch (sscanf ([string UTF8String], "#%2x%2x%2x%2x", &red, &green, &blue, &alpha))
							{
								case 3:
									alpha = 255;
								case 4:
									*object = [NSColor
										colorWithCalibratedRed: red / 255.0
										green: green / 255.0
										blue: blue / 255.0
										alpha: alpha / 255.0];
									break;
								default:
									*object = string;
							}
					}
			}
		return YES;
	}
	
- (BOOL) isPartialStringValid: (NSString**) partialStringPtr proposedSelectedRange: (NSRangePointer) proposedSelRangePtr originalString: (NSString*) origString originalSelectedRange: (NSRange) origSelRange errorDescription: (NSString**) error
	{
		int partialStringLength = [*partialStringPtr length];
		if (origSelRange.location + origSelRange.length == [origString length]
			&& proposedSelRangePtr->length == 0
			&& proposedSelRangePtr->location == partialStringLength
			&& partialStringLength > origSelRange.location)
			{
				NSArray* allColorKeys = [colors_ allKeys];
				
				int select = 0;
				int start = 0;
				int finish = [allColorKeys count];
				
				while (start < finish)
					{
						select = (start + finish - 1) / 2;
						
						switch ([*partialStringPtr compare: [allColorKeys objectAtIndex: select]])
							{
								case NSOrderedAscending:
									finish = select;
									break;
								case NSOrderedDescending:
									start = select + 1;
									break;
								case NSOrderedSame:
									return YES;
									break;
							}
					}
					
				if (start < [allColorKeys count])
					{
						NSString* selectedColor = [allColorKeys objectAtIndex: start];
						if ([selectedColor hasPrefix: *partialStringPtr])
							{
								*partialStringPtr = selectedColor;
								*proposedSelRangePtr = NSMakeRange (partialStringLength, [selectedColor length] - partialStringLength);
								return NO;
							}
					}
			}
		
		return YES;
	}
	
- (void) dealloc
	{
		[colors_ release];
		[super dealloc];
	}

@end
