/*
 *  PGGraphvizSchema.m
 *  graphviz
 *
 *  Created by Glen Low on Mon May 03 2004.
 *  Copyright (c) 2004, Pixelglow Software. All rights reserved.
 *  http://www.pixelglow.com/graphviz/
 *  graphviz@pixelglow.com
 *
 *  Redistribution and use in source and binary forms, with or without modification, are permitted
 *  provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright notice, this list of conditions
 *    and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
 *    and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *  * Neither the name of Pixelglow Software nor the names of its contributors may be used to endorse or
 *    promote products derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


#include <QuickTime/QuickTime.h>

#import "PGAttribute.h"
#import "PGDescriptor.h"
#import "PGGraphvizSchema.h"
#import "PGOption.h"
#import "PGFontTextFieldCell.h"
#import "PGColorTextFieldCell.h"
#import "PGPopUpTextFieldCell.h"

@implementation PGGraphvizSchema

+ (PGGraphvizSchema*) mainSchema
	{
		static PGGraphvizSchema* schemaInstance = nil;
		if (!schemaInstance)
			schemaInstance = [[PGGraphvizSchema alloc] init];
			
		return schemaInstance;
	}
	
+ (NSColorList*) colors
	{
		static NSColorList* colorInstance = nil;
		if (!colorInstance)
			colorInstance = [[NSColorList alloc] initWithName: @"Graphviz" fromFile: [[NSBundle mainBundle] pathForResource: @"Graphviz" ofType: @"clr"]];
		return colorInstance;
	}

- (id) init
	{
		if ((self = [super init]))
			{
				choices_ = [[NSMutableDictionary alloc] init];
				
				// grab the Choices.plist
				NSDictionary* rawChoices = [[NSDictionary alloc]
					initWithContentsOfFile: [[NSBundle mainBundle] pathForResource: @"Choices" ofType: @"plist"]];
					
				// go through each key in the Choices.plist (each key corresponds to a type)
				NSEnumerator* eachRawChoiceKey = [rawChoices keyEnumerator];
				NSString* nextRawChoiceKey;
				while ((nextRawChoiceKey = [eachRawChoiceKey nextObject]))
					{
						NSArray* nextRawChoices = [rawChoices objectForKey: nextRawChoiceKey];
						NSMutableArray* nextChoices = [[NSMutableArray alloc] init];
						
						// go through each object in the value of the Choice.plist (each object corresponds to a choice of the type)
						NSEnumerator* eachRawChoiceObject = [nextRawChoices objectEnumerator];
						NSString* nextRawChoiceObject;
						while ((nextRawChoiceObject = [eachRawChoiceObject nextObject]))
							{
								PGDescriptor* nextChoiceObject = [[PGDescriptor alloc]
									initWithScope: nextRawChoiceKey
									name: nextRawChoiceObject];
								[nextChoices addObject: nextChoiceObject];
								[nextChoiceObject release];
							}

						// add any further choices that can only be determined at runtime e.g. fonts
						SEL customAdder = NSSelectorFromString ([NSString stringWithFormat: @"%@s:", nextRawChoiceKey]);
						if ([self respondsToSelector: customAdder])
							[self performSelector: customAdder withObject: nextChoices];
					
						// sort by description and remember a menu representing the choices
						if ([nextChoices count])
							{
								[nextChoices sortUsingSelector: @selector (compare:)];
								
								NSMenu* nextMenu = [[NSMenu alloc] init];
								NSEnumerator* eachChoice = [nextChoices objectEnumerator];
								id nextChoice;
								while ((nextChoice = [eachChoice nextObject]))
									{
										NSMenuItem* nextItem = [[NSMenuItem alloc] initWithTitle:
											[nextChoice description] action: nil keyEquivalent: @""];
										[nextItem setRepresentedObject: nextChoice];
										if ([nextItem respondsToSelector: @selector (setToolTip:)] &&
											[nextChoice respondsToSelector: @selector (tip)])
											[nextItem performSelector: @selector (setToolTip:)
												withObject: [nextChoice performSelector: @selector (tip)]];
										[nextMenu addItem: nextItem];
										[nextItem release];
									}							
								
								[choices_ setObject: nextMenu forKey: nextRawChoiceKey];
								[nextMenu release];
							}
							
						[nextChoices release];
					}
					
				[rawChoices release];
				
				NSNumberFormatter* intFormatter = [[NSNumberFormatter alloc] init];
				[intFormatter setFormat: @"#,##0;0;-#,##0"];
				NSNumberFormatter* doubleFormatter = [[NSNumberFormatter alloc] init];
				[doubleFormatter setFormat: @"#,##0.00;0.00;-#,##0.00"];
				
				formatters_ = [[NSDictionary alloc] initWithObjectsAndKeys:
					doubleFormatter, @"double",
					intFormatter, @"int",
					nil];
				
				[intFormatter release];
				[doubleFormatter release];
				
				attributes_ = [[NSMutableArray alloc] init];
				
				NSArray* rawAttributes = [[NSArray alloc]
					initWithContentsOfFile: [[NSBundle mainBundle] pathForResource: @"Attributes" ofType: @"plist"]];
				
				NSEnumerator* eachRawAttribute = [rawAttributes objectEnumerator];
				NSDictionary* nextRawAttribute;
				while ((nextRawAttribute = [eachRawAttribute nextObject]))
					[attributes_ addObject: [PGAttribute attributeWithDictionary: nextRawAttribute]];

				[rawAttributes release];
				
			}
		return self;
	}

- (NSArray*) optionsForComponent: (NSString*) component
	{
		NSMutableArray* options = [[NSMutableArray alloc] init];
				
		NSEnumerator* eachAttribute = [attributes_ objectEnumerator];
		PGAttribute* nextAttribute;
		while ((nextAttribute = [eachAttribute nextObject]))
			if ([[nextAttribute usedBy] containsObject: component])
				{
					PGOption* nextOption = [[PGOption alloc] initWithSchema: self prefix:
						[[component substringWithRange: NSMakeRange (0, 1)] uppercaseString]
						attribute: nextAttribute];
					[options addObject: nextOption];
					[nextOption release];
				}
						
		[options sortUsingSelector: @selector (compare:)];
		return [options autorelease];
	}
	
- (NSCell*) cellForType: (NSString*) type
	{
		NSMenu* choices = [choices_ objectForKey: type];
		NSFormatter* formatter = [formatters_ objectForKey: type];
		
		if ([type isEqualToString: @"font"])
			{
				PGFontTextFieldCell* fontField = [[PGFontTextFieldCell alloc] init];
				
				[fontField setControlSize: NSSmallControlSize];
				[fontField setEditable: YES];
				[fontField setTag: PGFontTag];
				[fontField setWraps: NO];
				if ([fontField respondsToSelector: @selector (setPlaceholderString:)])
					[fontField performSelector: @selector (setPlaceholderString:) withObject: @"unchanged"];
			
				return [fontField autorelease];
			}
		if ([type isEqualToString: @"color"])
			{
				PGColorTextFieldCell* colorField = [[PGColorTextFieldCell alloc] init];
				
				[colorField setColors: [PGGraphvizSchema colors]];
				[colorField setControlSize: NSSmallControlSize];
				[colorField setEditable: YES];
				[colorField setTag: PGColorTag];
				[colorField setWraps: NO];
				if ([colorField respondsToSelector: @selector (setPlaceholderString:)])
					[colorField performSelector: @selector (setPlaceholderString:) withObject: @"unchanged"];
			
				return [colorField autorelease];
			}
		else if (choices)
			{
				PGPopUpTextFieldCell* popupField = [[PGPopUpTextFieldCell alloc] init];
				[popupField setControlSize: NSSmallControlSize];
				[popupField setEditable: YES];
				[popupField setMenu: choices];
				[popupField setTag: PGChoicesTag];
				[popupField setWraps: NO];
				if ([popupField respondsToSelector: @selector (setPlaceholderString:)])
					[popupField performSelector: @selector (setPlaceholderString:) withObject: @"unchanged"];
			
				return [popupField autorelease];
			}
		else 		 
			{
				NSTextFieldCell* textField = [[NSTextFieldCell alloc] init];
				[textField setControlSize: NSSmallControlSize];
				[textField setEditable: YES];
				[textField setFormatter: formatter];
				[textField setTag: PGTextTag];
				[textField setWraps: NO];
				if ([textField respondsToSelector: @selector (setPlaceholderString:)])
					[textField performSelector: @selector (setPlaceholderString:) withObject: @"unchanged"];
					
				return [textField autorelease];							
			}
	}

- (void) formats: (NSMutableArray*) choices
	{
		// load bitmap formats from Quicktime
		Component nextComponent = 0;

		ComponentDescription criteria;
		criteria.componentType = GraphicsExporterComponentType;
		criteria.componentSubType = 0;
		criteria.componentManufacturer = 0;
		criteria.componentFlagsMask = graphicsExporterIsBaseExporter;

		Handle componentName = NewHandle (0);
		while ((nextComponent = FindNextComponent (nextComponent, &criteria)))
			{
				ComponentDescription componentDescription;
				GetComponentInfo (nextComponent, &componentDescription, componentName, 0, 0);
				HLock (componentName);
				CFStringRef nameString = CFStringCreateWithPascalString (kCFAllocatorDefault, *componentName, kCFStringEncodingUTF8);
				HUnlock (componentName);

				OSType extension;
				GraphicsExportGetDefaultFileNameExtension ((GraphicsExportComponent) nextComponent, &extension);
				NSString* extensionString =
					[[[NSString stringWithCString: (char*) &extension length: sizeof (extension)]
						stringByTrimmingCharactersInSet: [NSCharacterSet whitespaceCharacterSet]]
						lowercaseString];
				
				PGDescriptor* nextFormat = [[PGDescriptor alloc] initWithScope: @"format" name: extensionString tip: nil description: (NSString*) nameString];
				
				[choices addObject: nextFormat];
				[nextFormat release];
					
				CFRelease (nameString);
			}
		DisposeHandle (componentName);	
	}
	
- (void) dealloc
	{
		[choices_ release];
		[formatters_ release];
		[attributes_ release];
		
		[super dealloc];
	}

@end

