/*
 * Copyright(C) 2010 Neil Jagdish Patel
 * Copyright(C) 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 * Authored by Ken VaDine <ken.vandine@canonical.com>
 */

[DBus (name = "com.Gwibber.Service")]
private interface ServiceInterface : GLib.Object {
        public abstract void Start () throws GLib.IOError;
        public abstract void Quit () throws GLib.IOError;
        public abstract void Refresh () throws GLib.IOError;
        public abstract void SendMessage (string msg) throws GLib.IOError;
        public abstract void Send (string msg) throws GLib.IOError;
        public abstract void Like (string mid, string account) throws GLib.IOError;
        public abstract void Retweet (string mid, string account) throws GLib.IOError;
        public abstract void PerformOp (string op) throws GLib.IOError;
        public abstract void UpdateIndicators (string stream) throws GLib.IOError;
        public abstract string GetAvatarPath (string url) throws GLib.IOError;
        public abstract string GetFeatures () throws GLib.IOError;
        public abstract string GetServices () throws GLib.IOError;
        public abstract string GetVersion () throws GLib.IOError;
        public signal void LoadingStarted ();
        public signal void LoadingComplete ();
        public signal void Error (string error_str);
}

namespace Gwibber
{
    public class Service : Object
    {
        private const string service_name  = "com.Gwibber.Service";
        private const string service_path  = "/com/gwibber/Service";

        private ServiceInterface service;
        private HashTable<string,HashTable> services_table;
        private HashTable<string,List> service_table;
        private HashTable<string,HashTable> features_table;
        private HashTable<string,Value?> feature_table;
	private Gwibber.Utils utils;
	private Gwibber.Messages messages;

        /** 
            Service::is_available:
            @arg0: The current state

            Emitted when com.Gwibber.Service availability state changes
        */
        public signal void is_available(bool is_up);

        /**
            Service::loading_started:

            Emitted when an operations starts
        */

        public signal void loading_started();
        /**
            Service::loading_started:

            Emitted when an operations is complete
        */
        public signal void loading_complete();

        /**
            Service::error:
            @arg0: String containing the account id
            @arg1: String containing the error type
            @arg2: String containing the error message

            Emitted when an error is received from the service
        */
        public signal void error(string id, string type, string message);

        construct
        {
            try
            {
                service = Bus.get_proxy_sync(BusType.SESSION,
                                                  service_name,
                                                  service_path);
                

                utils = new Gwibber.Utils();
                utils.setup(service_name);
                utils.available.connect(service_available);
                service.LoadingStarted.connect(on_loading_started);
                service.LoadingComplete.connect(on_loading_complete);
                        service.Error.connect((source) => {
                                on_error (source);});


            }
            catch (GLib.IOError e)
            {
                warning ("Unable to get Gwibber service: "+e.message);
            }

            messages = new Gwibber.Messages ();
        }

	public void service_available(bool is_up)
	{
		is_available(is_up);
	}

	public void on_loading_started()
	{
		loading_started();
	}

	public void on_loading_complete()
	{
		loading_complete();
	}

        public void on_error(string error_str)
        {
            
            try {
                var parser = new Json.Parser();
                parser.load_from_data(error_str, -1);
                Json.Object root_object = parser.get_root().get_object();
                var error_object = root_object.get_member("error").get_object();
                var account_object = error_object.get_member("account").get_object();
                string id = account_object.get_member("id").get_string();
                string type = error_object.get_member("type").get_string();
                string message = error_object.get_member("message").get_string();
                error(id, type, message);
            } catch(GLib.Error e) {
                warning (e.message);
            }
        }

        /**
            start:
        */
        public void start()
        {
            try {
                service.Start();
            } catch (GLib.IOError e) {
                warning (e.message);
            }
        }

        /**
            quit:

            Tells the gwibber service to shutdown.
        */
        public void quit()
        {
            try {
                service.Quit();
            } catch (GLib.IOError e) {
                warning (e.message);
            }
        }

        /**
            refresh:

            Tells the gwibber service to perform a refresh.
        */
        public void refresh()
        {
            try {
                service.Refresh();
            } catch (GLib.IOError e) {
                warning (e.message);
            }
        }

        /**
            like:
            @id: the gwibber message id
            @account: the account id
           
            returns a bool
        */
        public bool like (string id, string account)
        {
            string mid = "";
            if (id != null)
            {
              try {
                var msg = messages.get_message (id);
                if (msg.length < 1)
                  return false;

                var parser = new Json.Parser();
                parser.load_from_data(msg);

                mid = parser.get_root().get_object().get_string_member ("mid");
                if (mid.length <1)
                  return false;
                try {
                  service.Like(mid, account);
                  return true;
                } catch (GLib.IOError e) {
                  warning (e.message);
                  return false;
                }
                return false;
              } catch (GLib.IOError e) {
                  warning (e.message);
                  return false;
              }
            }
            return false;
        }



        /**
            retweet:
            @id: the gwibber message id
            @account: the account id
           
            returns a bool
        */
        public bool retweet (string id, string account)
        {
            string mid = "";
            if (id != null)
            {
              try {
                var msg = messages.get_message (id);
                if (msg.length < 1)
                  return false;

                var parser = new Json.Parser();
                parser.load_from_data(msg);

                mid = parser.get_root().get_object().get_string_member ("mid");
                if (mid.length <1)
                  return false;
                try {
                  service.Retweet(mid, account);
                  return true;
                } catch (GLib.IOError e) {
                  warning (e.message);
                  return false;
                }
                return false;
              } catch (GLib.IOError e) {
                  warning (e.message);
                  return false;
              }
            }
            return false;
        }


        /**
            send_message:
            @message: The message to post to Gwibber
            @id: The gwibber message id or null
            @action: The action or null
            @account: The account to post from  or null

            Posts a message
        */
        public bool send_message(string? message, string? id, string? action, string[]? accounts)
        {
            size_t length;
            string json;
        
            if ((id != null) && (action != null))
            {
              try {
                var msg = messages.get_message (id);
                if (msg.length < 1)
                  return false;

                var msg_gen = new Json.Generator();
                var root = new Json.Node(Json.NodeType.OBJECT);
                var object = new Json.Object();
                root.set_object(object);
                msg_gen.set_root(root);
                
                var parser = new Json.Parser();
                parser.load_from_data(msg);

                object.set_string_member("message", message);
                if (action == "reply")
                  object.set_object_member("target",  parser.get_root().get_object());
                if (action == "private")
                  object.set_object_member("private",  parser.get_root().get_object());
                object.set_string_member("action", action);
                if (accounts.length > 0)
                {
                  var acct_array = new Json.Array ();
                  foreach (var a in accounts)
                    acct_array.add_string_element (a);
                  object.set_array_member("accounts", acct_array);
                }
                json = msg_gen.to_data(out length);

                service.Send((string)json);
                return true;
              } catch (GLib.IOError e) {
                warning (e.message);
                return false;
              }
           } else 
           {
             try {
                service.SendMessage(message);
                return true;
             } catch (GLib.IOError e) {
                 warning (e.message);
                 return false;
             }
           }
           return false;
        }


        public GLib.HashTable? services()
        {
            services_table = new HashTable<string,HashTable>.full(str_hash, str_equal, g_free, GLib.HashTable.remove_all);
            try {
                string services = service.GetServices();
                var parser = new Json.Parser();
                try {
                    parser.load_from_data(services);
                } catch (GLib.Error e) {
                    warning (e.message);
                    return null;
                }
                Json.Object root_object = parser.get_root().get_object();
                foreach(var service in root_object.get_members()) {
                    service_table = new HashTable<string,List?>.full(str_hash, str_equal, g_free, g_free);
                    Json.Object services_object = root_object.get_member(service).get_object();
                    foreach(var member in services_object.get_members()) {
                        if (services_object.get_member(member).type_name() == "JsonArray") {
                            Json.Array service_array = services_object.get_array_member(member);
                            var service_attr_list = new List<string> ();
                            for(int i = 0; i < service_array.get_length(); i++) {
                                string f = service_array.get_element(i).get_string ();
                                service_attr_list.append(f);
                            }

                            service_table.insert(member, service_attr_list.copy ());
                            continue;
                        }
                    }
                    services_table.insert(service, service_table);
                }
                return services_table;
            } catch(GLib.Error e) {
                warning(e.message);
                return null;
            }
        }

        public GLib.HashTable? features()
        {
            features_table = new HashTable<string,HashTable>.full(str_hash, str_equal, g_free, GLib.HashTable.remove_all);
            try {
                string features = service.GetFeatures();
                var parser = new Json.Parser();
                try {
                    parser.load_from_data(features, -1);
                } catch (GLib.Error e) {
                    warning (e.message);
                    return null;
                }
                var root_object = parser.get_root().get_object();
                foreach(var feature in root_object.get_members()) {
                    feature_table = new HashTable<string,Value?>.full(str_hash, str_equal, g_free, g_free);
                    var features_object = root_object.get_member(feature).get_object();
                    foreach(var member in features_object.get_members()) {
                        GLib.Value? value = null;
                        if(features_object.get_member(member).is_null()) {
                            continue;
                        } else if(features_object.get_member(member).type_name() == "gboolean") {
                            value = features_object.get_member(member).get_boolean();
                        } else { 
                            features_object.get_member(member).get_value(out value);
                        }
                        feature_table.insert(member, value);
                    }
                    features_table.insert(feature, feature_table);
                }
                return features_table;
            } catch(GLib.IOError e) {
                warning(e.message);
                return null;
            }
        }
    

        /**
            version:

            Return value: the version of the Gwibber service that is running.
        */
        public string version()
        {
            try {
                return service.GetVersion();
            } catch(GLib.IOError e) {
                warning(e.message);
            }
            return "";
        }

        /**
            avatar_path:
            @url: url to the avatar to cache

            Return value: path to the cached avatar as a string
        */
        public string? avatar_path(string url) throws GLib.Error
        {
            string _avatar_cache_image = Path.build_path(Path.DIR_SEPARATOR_S, Environment.get_user_cache_dir(), "gwibber/avatars", url.replace("/",""));
            var file = File.new_for_path (_avatar_cache_image);
            if (file.query_exists ())
              return _avatar_cache_image;
            else
              _avatar_cache_image = service.GetAvatarPath(url);
            return _avatar_cache_image;
        }

        public void perform (string id) throws GLib.Error
        {
          try {
            service.PerformOp("{\"id\": \"" + id + "\"}");
          } catch(GLib.IOError e) {
            warning(e.message);
          }
        }


        public void follow (string account, string screen_name, bool follow) throws GLib.Error
        {
          string action = "unfollow";
          if (follow)
            action = "follow";

          try {
            service.PerformOp("{\"account\": \"" + account + "\", \"operation\": \"" + action + "\", \"args\": {\"screen_name\": \"" + screen_name + "\"}}");
          } catch(GLib.IOError e) {
            warning(e.message);
          }
        }

        public void update_indicators(string? stream)
        {
           if (stream == null)
             stream = "home";
           try {
             service.UpdateIndicators(stream);
           } catch (GLib.IOError e) {
             warning (e.message);
           }
        }

}
}
