/*
 * Copyright(C) 2010 Neil Jagdish Patel
 * Copyright(C) 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 * Authored by Ken VaDine <ken.vandine@canonical.com>
 */

[DBus (name = "com.Gwibber.Accounts")]
private interface AccountsInterface : Object {
	public signal void Updated (string account);
	public signal void Created (string account);
	public signal void Deleted (string account);
	public abstract string List () throws GLib.IOError;
	public abstract string Get (string id) throws GLib.IOError;
	public abstract void SendEnabled (string id) throws GLib.IOError;
}

namespace Gwibber
{


    public class Accounts : Object
    {
        private const string accounts_name  = "com.Gwibber.Accounts";
        private const string accounts_path  = "/com/gwibber/Accounts";

        private AccountsInterface accounts_service;
	private Gwibber.Utils utils;

	/** 
            Accounts::is_available:
            @arg0: The current state

            Emitted when com.Gwibber.Accounts availability state changes
	*/
	public signal void is_available(bool is_up);

        /** 
            Accounts::updated:
            @arg0: HashTable of an account

            Emitted when an account has changed
        */
        public signal void updated(Gwibber.Account account);

        /** 
            Accounts::created:
            @arg0: HashTable of an account

            Emitted when an account has changed
        */
        public signal void created(Gwibber.Account account);

        /** 
            Accounts::deleted:
            @arg0: account id as a string

            Emitted when an account has changed
        */
        public signal void deleted(string id);

    public Accounts()
    {
            try
            {
                accounts_service = Bus.get_proxy_sync(BusType.SESSION,
                                                      accounts_name,
                                                      accounts_path);

                utils = new Gwibber.Utils();
                utils.setup(accounts_name);
                utils.available.connect(accounts_available);

                accounts_service.Updated.connect((source) => {
                    account_updated (source);
                });

                accounts_service.Created.connect((source) => {
                    account_created (source);
                });

                accounts_service.Deleted.connect((source) => {
                    account_deleted (source);
                });
            }
            catch(GLib.IOError e)
            {
                warning ("Unable to get Gwibber accounts: "+e.message);
            }
        }

        public GLib.List<Gwibber.Account> list()
        {
            GLib.List<Gwibber.Account> accounts = null;
            try {
                string accounts_string = accounts_service.List();
                var parser = new Json.Parser();
                try {
                    parser.load_from_data(accounts_string, -1);
                    Json.Array nodeArray = parser.get_root().get_array();
                    for(int i = 0; i < nodeArray.get_length(); i++) {
                        string id = nodeArray.get_element(i).get_object().get_string_member("id");
                        var account = new Gwibber.Account (id);
                        accounts.append(account);
                    }
                } catch (GLib.Error e) {
                    warning (e.message);
                }
            } catch (GLib.IOError e) {
                warning (e.message);
            }
            return accounts;
        }

        public string lookup_str_by_id (string id)
        {
           try {
               string account = accounts_service.Get(id);
               return account;
           } catch (GLib.IOError e) {
               warning (e.message);
               return "";
           }
        }

        public Gwibber.Account lookup_by_id (string id)
        {
            var acct = new Gwibber.Account (id);
            return acct;
        }


        /**
            send_enabled:

            Toggle the value of send_enabled for an account.
        */
        public void send_enabled(string id)
        {
            try {
                accounts_service.SendEnabled(id);
            } catch(GLib.IOError e) {
                warning (e.message);
            }
        }

	public void account_updated(string account_in)
	{
            string account_string = (string)account_in;
            try {
                var parser = new Json.Parser();
                parser.load_from_data(account_string, -1);
                var root_object = parser.get_root().get_object();
                string id = root_object.get_string_member("id");
                debug ("ID is %s", id);
                var account = new Gwibber.Account (id);
                updated(account);
            } catch(GLib.Error e) {
                warning (e.message);
            }
        }

        public void account_created(string account_in)
        {
            string account_string = (string)account_in;
            try {
                var parser = new Json.Parser();
                parser.load_from_data(account_string, -1);
                var root_object = parser.get_root().get_object();
                string id = root_object.get_string_member("id");
                debug ("ID is %s", id);
                var account = new Gwibber.Account (id);
                created(account);
            } catch(GLib.Error e) {
                warning (e.message);
            }
        }
        public void account_deleted(string account_in)

        {
            string account_string = (string)account_in;
            try {
                var parser = new Json.Parser();
                parser.load_from_data(account_string, -1);
                var root_object = parser.get_root().get_object();
                string id = root_object.get_string_member("id");
                debug ("ID is %s", id);
                deleted(id);
            } catch(GLib.Error e) {
                warning (e.message);
            }
        }


	public void accounts_available(bool is_up)
	{
		is_available(is_up);
	}

    }
}
