//
// Filename: xojpanel.cpp
//
// Implementation of the XojPanel widget.
// xojpanel (the app) displays the LCD contents of many
// Hewlett-Packard all-in-one printers.

/* LICENSE
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.

   In addition, as a special exception, Joe Piolunek
   gives permission to link the code of this program with the
   OpenSSL library (or with modified versions of OpenSSL that use the
   same license as OpenSSL), and distribute linked combinations including
   the two.  You must obey the GNU General Public License in all
   respects for all of the code used other than OpenSSL.  If you modify
   this file, you may extend this exception to your version of the
   file, but you are not obligated to do so.  If you do not wish to
   do so, delete this exception statement from your version.
*/

/*
  Copyright (C) 2000-2002 Joe Piolunek

  This is an expanded and nearly completely re-written version
  of the original xojpanel application released by Andreas Fester.

  Thanks to David Paschal for contributing a patch to allow xojpanel to exit
  gracefully if the peripheral does not support LCD message requests.
  David Paschal also did the ojlib->PTAL conversion.
*/

//
#include "xojpanel.h"
#include "hpoj_mini.xpm"
#include "hpojlcd.xpm"
#include "hpoj_lcdmon.xpm"
//
#include <qapplication.h>
#include <qfontmetrics.h>
#include <qtimer.h>
#include <qpainter.h>
#include <qlabel.h>
#include <qpixmap.h>
#include <qstring.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
//
/* SET LCD FONT ************************************/
// If you want to change the default font
// (as appears in LCD), do it here.
// Uncomment ( remove the "//" ) one line for
// your choice of font and one line for your preferred
// fontsize. Choosing a fixed-width font will usually
// give better results. You can specify a font not listed
// here. If you comment out all of the font choices below,
// Qt will attempt to use your local system's font setting.
//
// Note: earlier versions of Qt may not like the font
// you specify. You may need to experiment a little to get
// the results you want.
//
// Fixed-width fonts
//
#define DEFAULT_FONT "Courier" //
// #define DEFAULT_FONT "Adobe-courier"
// #define DEFAULT_FONT "clean"
// #define DEFAULT_FONT "dec-terminal"
// #define DEFAULT_FONT "misc-fixed"
// #define DEFAULT_FONT "quicktype mono"
// #define DEFAULT_FONT "Courier new"
// #define DEFAULT_FONT "LCD"
// #define FONTSIZE 12
// #define FONTSIZE 14
#define FONTSIZE 16 //
// #define FONTSIZE 20
//
// Prints the ASCII decimal codes of "raw" LCD messages to stdout.
// Use if needed to identify 'special' characters used by your model.
// #define SHOW_LCD_MESSAGE_ASCII_CODES
//
// Prints the contents of the character map to stdout.
// #define DISPLAY_CHARACTER_MAP
//
// In a case where you are using KDE2 and you built xojpanel
// with qt-2.0.x, you may see a message like this:
// "XError: BadValue(...)2". If so, try uncommenting the next line.
// #define HAVE_QT_20
//
// Scroll speed and delay options, LCD text color.
const unsigned int PRINTER_POLL_REPEAT_INTERVAL = 500;	// in ms. (val incr. = speed decr.)
const unsigned int SCROLL_INTERVAL = 333;		// in ms. (val incr. = speed decr.)
const unsigned int SCROLL_START_DELAY = 3;		// Number of  'scrolls' skipped.
const unsigned int SCROLL_END_DELAY = 2;		// Number of  'scrolls' skipped.
const unsigned int MAX_LCD_CHARS = 20;			// Number of characters in LCD.
const QColor *LCD_TEXT_COLOR = new QColor(0, 0, 0);	// Color of LCD text.
const QColor XOJPANEL_BKGR_COLOR = QColor( 190, 190, 190 ); // 190, 190, 190
const QString ALT_TITLEBAR_TEXT = "Device status";
/**************************************************************************************/
//
//
/**
 * XojPanel class constructor.
 * Creates the XojPanel widget. xojpanel is a simple graphical app
 * that shows the HP printer's current LCD window contents, scrolling
 * them if necessary.
 */
XojPanel::XojPanel(int argc, char **argv, QWidget *parent, const char *name )
			: QWidget(parent,name)
{
	// printf( "XojPanel class constructor called." );

	caption = "";
	useCaption = false;
	ptalDeviceName = "";
	bool deviceNameInTitle = true;
	bool displayHelp = false;
	bool isTranslationEnabled = true;
	bool deviceNameSpecified = false;
	bool possibleDeviceNameFound = false;


	// Parse command line
	// Loops through once for each argument presented.
	// For arguments with identifiers,
	// count is advanced an extra step.
	for (int i = 1; i < argc; i++ ) {
		if(displayHelp) break;
		
		else if( qstrcmp(argv[i], "-help" ) == 0) {
			displayHelp = true;
			break;
		}

		else if( qstrcmp(argv[i], "-notrans" ) == 0)
			isTranslationEnabled = false;

		else if( qstrcmp(argv[i], "-hidedevname") == 0)
			deviceNameInTitle = false;

		else if(qstrcmp(argv[i], "-caption") == 0) {
			caption = argv[i+1];
			useCaption = true;
			i++;		
		}
		
		else if( (!possibleDeviceNameFound) && argv[i][0] != '-' ) {
			// Assume first unidentified argument is ptalDeviceName
			// printf("\nUsing %s as deviceName.\n", argv[i]);
			possibleDeviceNameFound = true;
			deviceNameSpecified = true;
			ptalDeviceName = argv[i];
		}
		
		else if( possibleDeviceNameFound ) {
			// A second unidentified argument was found.
			printf("Unrecognized option \"%s\".\n",argv[i]);
			displayHelp = true;
			break;
		
		} else { // No more possibilities
			printf("Unrecognized option \"%s\".\n",argv[i]);
			displayHelp = true;
			break;
		}
	}

	if( displayHelp ) { // Output this text and exit.
		/*#ifdef RELEASE_VERSION // Probably won't be used
			printf( "\n%s\n", (const char*)RELEASE_VERSION );
		#endif*/
		printf("\nSyntax: %s [options]\n"

			"\nOptions:\n\n"
			

			"\t-help\t\tPrint this information and exit.\n"

			"\t-caption <text> Optional text to be displayed in titlebar.\n"

			"\t-notrans\tDisables 'special character' translation.\n"

			"\t-hidedevname\tPrevents PTAL device name from\n"
			"\t\t\tbeing displayed in the titlebar.\n\n"

			"\tThe first option *not* prefixed with a dash ('-')\n"
			"\twill be be used as the PTAL device name. If a default\n"
			"\tdevice was specified in the hpoj setup process\n"
			"\t( 'ptal-init setup' ), then that device name will be used,\n"
			"\twithout needing to specify it here.\n\n"

			"\tCopyright (C) 2000-2002 Joe Piolunek\n\n"
			"\tThis software is distributed under the terms of the GNU General Public\n"
			"\tLicense (GPL); see the source distribution for copying conditions.\n\n"
			"\tTHERE IS NO WARRANTY OF ANY KIND, WHETHER EXPRESSED OR IMPLIED;\n"
			"\tNOT EVEN FOR MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.\n\n"
			,*argv);
		exit (0);
	}

	// Set up PTAL.
	if (ptalInit()==PTAL_ERROR) {
		printf("Failed to initialize PTAL!\n");		
		exit(1);
	}

	if (! deviceNameSpecified) {
		dev=ptalDeviceOpen(NULL);
		if (dev) ptalDeviceName = ptalDeviceGetName(dev);
		// printf("Trying to connect to device  %s\n\n",ptalDeviceName);
	}else{
		dev=ptalDeviceOpen(ptalDeviceName);
	}


	if (!dev) {
		printf("Failed to open device \"%s\"!\n", ptalDeviceName);
		printf("\nTry using \'-help\'.\n\n");
		exit(1);
	}
	if (ptalPmlOpen(dev)==PTAL_ERROR) {
		printf("Failed to open PML on device \"%s\"!\n", ptalDeviceName);
		exit(1);
	}
	/* TODO: Define these OIDs in a common include file. */
	objLine1=objHpLine1=ptalPmlAllocateID(dev,"\x1\x1\x2\x14\x2\x1\x1");
	objLine2=objHpLine2=ptalPmlAllocateID(dev,"\x1\x1\x2\x14\x2\x2\x1");
	objSpmLine1=ptalPmlAllocateID(dev,"\x2\x10\x5\x1\x2\x1\x1");
	objSpmLine2=ptalPmlAllocateID(dev,"\x2\x10\x5\x1\x2\x1\x2");

	// buildCharacterMap( specialTranslationModel, isTranslationEnabled );
	buildCharacterMap( isTranslationEnabled );

	getPrinterLCDMessages(); // Makes the messages available at start.
	createInterface( ptalDeviceName, deviceNameInTitle );
	createTimers(); // Creates any QTimers we need.
}


/**
 * Enables immediate *current* text display on restoring/uncovering
 * window. Re-implements QWidget::paintEvent()
 * @param event Pointer to the paintEvent
 */
void
XojPanel::paintEvent( QPaintEvent *event )
{
	// printf( " XojPanel::paintEvent( QPaintEvent * )\n" );

	lcdGraphic->setPixmap( *pm );
}


/**
 * Closes the xojpanel application when a window 'exit' or 'close'
 * button is pushed. Reimplemented from QWidget::closeEvent()
 * @param closeEvent Pointer to the xojpanel object's closeEvent
 */
void
XojPanel::closeEvent( QCloseEvent *closeEvent )
{
	// printf( "void XojPanel::closeEvent()\n\n" );

	ptalDone();
	closeEvent->accept(); // This is the main widget, so app will be terminated.
}


/**
 * Builds a persistent character map on startup.
 * @param isTransEnabled If TRUE, enables 'special character' translation.
 */
void
XojPanel::buildCharacterMap( bool isTransEnabled )
// XojPanel::buildCharacterMap( char* specialModel, bool isTransEnabled )
{
	// printf("\nvoid buildCharacterMap( char* type, char* model )\n");
	// printf("type = %s, model = %s\n", type, model );

	static unsigned char characterMap[256] = { 0 };

	// Create persistent character translation map
	int i;
	for ( i = 0x00; i <= 0xFF; i++ ) characterMap[i] = i;    // Identity-map everything initially
	for ( i = 0x00; i <= 0x1F; i++ ) characterMap[i] = 0x20; // Fill with spaces
	for ( i = 0x7F; i <= 0xA0; i++ ) characterMap[i] = 0x20; // Fill with spaces
	// characterMap[0xFF] = 0x20; // Set to a space.

	if(isTransEnabled) { // Modify characterMap[]
		// These models supposedly use the same character maps.
		// - G series (G55, G55xi, G85, G85xi, G95)
		// - K series (K60, K60xi, K80, K80xi)
		// - V series (V40, V40xi, V45)
		// - PSC 700 series (750, 750xi)
		// - PSC 900 series (950, 950xi)
		//
		// Hopefully this set of common translations won't cause
		// conflicts. They should be irrelevant to models that
		// don't require character translation.

		// Position in map:	// Conversion used:
		characterMap[0x10] = 0xab; // Left arrow to '<<'
		characterMap[0x11] = 0xbb; // Right arrow to '>>'
		characterMap[0x12] = 0xa3; // Large pound(?) to small pound
		characterMap[0x13] = 0xbb; // Right hollow triangle to '>>'
		characterMap[0x80] = 0xab; // Left hollow triangle to '<<'
		characterMap[0x81] = 0xbb; // Right black triangle to '>>'
		characterMap[0x82] = 0x2a; // Hollow circle to '*'
		characterMap[0x83] = 0x2a; // Black circle to '*'
		characterMap[0x85] = 0x2a; // Black vertical oval to '*'
		characterMap[0x9f] = 0x20; // Black vertical box to 'space'
		characterMap[0xa0] = 0xab; // Left black triangle to '<<'
	}

	characterTranslationMap = characterMap; // Set pointer

	#ifdef DISPLAY_CHARACTER_MAP
		printf("\nContents of character map:\n\n");
		for(int i = 0; i < 256; i++) {
			printf("characterMap[%d] = >%c<\n",i,characterMap[i]);
		}
		printf("<end of map>\n\n");
	#endif
}


/**
 * Sets up xojpanel's graphical interface.
 * @param deviceName Identifies device in titlebar.
 */
void
XojPanel::createInterface( QString deviceName, bool deviceNameInTitle )
{
	// printf("void XojPanel::createInterface( void )\n\n" );

	// Configure the titlebar text based on how the
	// application is called from the command-line.
	QString titlebarText = "status";
	if ( deviceNameInTitle ) titlebarText.prepend( deviceName + " - " );
	else if (!useCaption) titlebarText = ALT_TITLEBAR_TEXT;	
	if (useCaption) titlebarText = (QString)caption;
	this->setCaption(titlebarText);
		
	this->setIcon(QPixmap(( const char**) hpoj_mini_xpm));

	#ifdef DEFAULT_FONT
		#ifdef FONTSIZE
			QFont f = QFont( DEFAULT_FONT, FONTSIZE );
			this->setFont( f );
			f.setFixedPitch( true );
			f.setStyleHint( QFont::AnyStyle );
		#endif
	#endif

	this->setBackgroundColor( XOJPANEL_BKGR_COLOR );
	this->setMinimumSize( 330, 150 ); // 330, 150
	this->setMaximumSize( 330, 150 ); // 330, 150

	createLCD();
	createCoverGraphic();
}


/**
 * Places a 254x40 pixel "LCD" background pixmap on window.
 */
void
XojPanel::createLCD( void )
{
	// printf( "XojPanel::createLCD( void )\n" );

	lcdPixmap = new QPixmap((const char**)hpojlcd_xpm);
	pm = new QPixmap((const char**)hpojlcd_xpm);

	// Create the LCD Graphic.
	lcdGraphic = new QLabel( this, "lcdGraphic",(uint)0 );
	lcdGraphic->setPixmap( *lcdPixmap );

	// actual size of graphic - 254x40
	lcdGraphic->setGeometry( 38, 90, 254, 40 ); // 38, 90 254, 40

	updateLCDPixmap(); // get text from printer and display it.
}


/**
 * Places a 290x70 pixel decorative label on the window.
 */
void
XojPanel::createCoverGraphic( void )
{
	// printf( "bool createCoverGraphic( void )\n\n");

	QLabel *coverGraphic = new QLabel( this, "label", (uint)0);
	coverGraphic->setPixmap(QPixmap((const char**)hpoj_lcdmon_xpm ));
	coverGraphic->setGeometry( 20, 8, 290, 70 ); // 20, 7, 290, 70
	coverGraphic->setBackgroundColor( XOJPANEL_BKGR_COLOR );
}


/**
 * Creates timers for such things as retrieving
 * printer LCD messages and scroll speed.
 */
void
XojPanel::createTimers( void ) // Timer settings at top of file.
{
	// printf( "XojPanel::createTimers( void )\n" );

	// Create a timer for polling printer for new lcd messages.
	QTimer *getLCDMessagesTimer = new QTimer( this );
	connect( getLCDMessagesTimer, SIGNAL(timeout()), SLOT(getPrinterLCDMessages()) );
	getLCDMessagesTimer->start( PRINTER_POLL_REPEAT_INTERVAL ); // in milliseconds.

	// Create a timer for lcd repainting.
	// Controls scroll speed.
	QTimer *scrollTimer = new QTimer( this );
	connect( scrollTimer, SIGNAL(timeout()), SLOT(updateLCDPixmap()) );
	scrollTimer->start( SCROLL_INTERVAL ); // in milliseconds.
}


/**
 * Retrieves printer's LCD messages, then immediately sends
 * them to adjustMessageContents() to have any whitespace
 * removed. Timer controlled SLOT.
 */
void
XojPanel::getPrinterLCDMessages( void )
{
	// printf( "void XojPanel::getPrinterLCDMessages( void )\n" );

	int r1,r2;
	char buffer[PTAL_PML_MAX_VALUE_LEN];

	// read the current contents of the printer's LCD
retry:
	r1=ptalPmlRequestGet(objLine1,0);
	r2=ptalPmlRequestGet(objLine2,0);
	if (r1==PTAL_ERROR || r2==PTAL_ERROR) {
		if (objLine1!=objSpmLine1) {
			objLine1=objSpmLine1;
			objLine2=objSpmLine2;
			goto retry;
		}

		r1=ptalPmlGetStatus(objLine1);
		r2=ptalPmlGetStatus(objLine2);
		printf("Errors while executing PML requests: "
			"0x%2.2X, 0x%2.2X\n",r1,r2);
		if (r1==PTAL_PML_ERROR_UNKNOWN_OBJECT_IDENTIFIER) {
			printf("This model doesn't seem to support "
				"LCD readback.\n");
		}
		exit(1);
	}
	// Get the LCD message for line 1 from peripheral
	if (objLine1==objSpmLine1) {
		ptalPmlGetValue(objLine1,0,buffer,PTAL_PML_MAX_VALUE_LEN);
	} else {
		ptalPmlGetStringValue(objLine1,0,buffer,PTAL_PML_MAX_VALUE_LEN);
	}
	// Modify text if needed and assign to Line1
	adjustMessageContents( buffer, 1 );

	// Get the LCD message for line 2 from peripheral
	if (objLine2==objSpmLine2) {
		ptalPmlGetValue(objLine2,0,buffer,PTAL_PML_MAX_VALUE_LEN);
	} else {
		ptalPmlGetStringValue(objLine2,0,buffer,PTAL_PML_MAX_VALUE_LEN);
	}
	// Modify text if needed and assign to Line2
	adjustMessageContents( buffer, 2 );

	// Creates a fake LCD message.
	// The array needs to be null-terminated.
	/*
	char asciiStringArray[] = { ' ',0x83,' ', 'U','s','e',' ',0x81,
			' ','A','n','d',' ','p','u','s','h',' ',
			'E','n','t','e','r','.',' ',' ',' ','\0'} ; // 0xbb = ">>"
	adjustMessageContents(asciiStringArray, 2);
	*/
	// Tests handling of leading and trailing whitespace
	/*
	char asciiStringArray[] = {"  starts and ends with two spaces  "};
	adjustMessageContents(asciiStringArray, 2);
	*/

	// A little fun
	// Line1 = "Open the pod bay door please, HAL. {8-:( )";
	// Line2 = "   I'm sorry Dave, I can't do that. :-(";
	//
	// Line1 = "Fourscore and seven years ago, our forefathers had no MTV.";

	// Shows strings with whitespace removed. Printed to stdout.
	// printf( "Strings with whitespace removed--:\n" );
	// printf("new Line1 -->%s<--end\n",(const char*)Line1);
	// printf("new Line2 -->%s<--end\n",(const char*)Line2 );
}


/**
 * When the function receives a string array, the array is parsed
 * through, with each element being assigned its corresponding
 * element from the charmap. Any spaces at the trailing end of
 * the string are then removed. Finally, the adjusted string is
 * assigned to Line1 or Line2 as appropriate.
 * @param string[] The "raw" string array as it comes from the peripheral
 * @param lineNum The line number ( 1, 2 ) of the LCD message
*/
void
XojPanel::adjustMessageContents( char string[], int lineNum )
{
	// printf("XojPanel::adjustMessageContents(char string[], int lineNum )\n");

	int length = strlen(string);
	int i = 0;
	int spaces = 0;
	QString tmpString;

	#ifdef SHOW_LCD_MESSAGE_ASCII_CODES
		// Show contents of original string as sent from device.
		static bool beenHere = false;
		if ( ! beenHere ) printf("\nPrinting 'raw message' characters\n"
				"and their decimal ASCII codes.\n\n");
		beenHere = true;

		for ( int i=0; i<length; i++){
			printf("string[%d] =\t->%c<-\t(ASCII %d)\n",
			i, (unsigned char)string[i], (unsigned char)string[i] );
		}
		if ( string[0] != 0 ) printf("\nNew message:\n");
	#endif

	// Convert each character in string[] to corresponding
	// character in characterTranslationMap[]
	for ( int i = 0; i < length; i++ ) {
		string[i] = characterTranslationMap[ (unsigned char)string[i] ];
	}

	spaces = 0;
	i = length - 1 ;

	// Count spaces on the trailing end of the string.
	while( string[i] == 0x20 ) {
		// printf("i is now= %d\n", i);
		spaces++;
		// printf("spaces= %d\n\n", spaces);

		// Don't cycle through again if the entire string
		// is counted for removal. It could result in a segfault.
		if( spaces >= length ) break;
		i--;
	}

	tmpString = string;
	// printf("tmpString= %s\n",(const char*)tmpString );

	// Remove trailing spaces and assign text to appropriate message line.
	if ( lineNum == 1 ){
		// printf("tmpString Line1 = *%s*\n", (const char*)tmpString );
		Line1 = tmpString.remove( length - spaces, spaces ); // index, #spaces.
		// printf("Line1 = *%s*\n", (const char*)Line1 );
	}else{
		if ( lineNum == 2 ){
			// printf("tmpString Line2 = *%s*\n", (const char*)tmpString );
			Line2 = tmpString.remove( length - spaces, spaces );
			// printf("Line2 = *%s*\n", (const char*)Line2 );
		}
	}
}


/**
 * Checks current LCD messages, looks to see if the contents
 * are new, then sends the information to displayLCDMessage().
 * Timer controlled SLOT.
 */
void
XojPanel::updateLCDPixmap( void )
{
	// printf( "void XojPanel::updateLCDPixmap( void )\n" );

	static bool newLine1Message = true;
	static bool newLine2Message = true;

	// For local copies of the printer's LCD text strings.
	static QString *Line1Copy = new QString;
	static QString *Line2Copy = new QString;

	if(qstrcmp( Line1, *Line1Copy ) == 0 ) { // No new message from printer.
		newLine1Message = false;
	}else{ // New message from printer.
		newLine1Message = true;
		*Line1Copy = Line1; // We're using a local copy of Line1 and Line2.
	}

	if(qstrcmp( Line2, *Line2Copy ) == 0 ) {
		newLine2Message = false;
	}else{
		newLine2Message = true;
		*Line2Copy = Line2; // We're using a local copy of Line1 and Line2.
	}

	displayLCDMessage( Line1Copy, Line2Copy, newLine1Message, newLine2Message );
}


/**
 * Displays the printer's LCD message text
 * on the pixmap, scrolling if necessary.
 * If the contents of the peripheral's LCD line
 * are new, the corresponding string is updated
 * and scrolled from the beginning.
 * @param firstLine Pointer to whitespace-stripped
 *      version of LCD Line1 contents
 * @param secondLine Pointer to whitespace-stripped
 *      version of LCD Line2 contents
 * @param newLine1 True if Line1 contains new contents; else false.
 * @param newLine2 True if Line2 contains new contents; else false.
 */
void
XojPanel::displayLCDMessage( QString * firstLine, QString * secondLine,
					bool newLine1, bool newLine2 )
{
	/*
	printf("XojPanel::displayLCDMessage( QString * firstLine, QString * secondLine,"
						"bool newLine1, bool newLine2");
	*/

	// Places text on pixmap. Scrolls if necessary.
	// Called by updateLCDPixmap (SLOT).
	static QString *line_1; line_1 = firstLine;  // Stripped of whitespace already.
	static QString *line_2; line_2 = secondLine; // Stripped of whitespace already.

	static bool newLine1Message; newLine1Message = newLine1;
	static bool newLine2Message; newLine2Message = newLine2;

	bool line1Painted = false;
	bool line2Painted = false;
	static unsigned int line1ScrollStartDelay = 0;
	static unsigned int line1ScrollEndDelay = 0;
	static unsigned int line2ScrollStartDelay = 0;
	static unsigned int line2ScrollEndDelay = 0;
	static unsigned int beginAtLine1Char = 0; // For scrolling line 1
	static unsigned int beginAtLine2Char = 0; // For scrolling line 2

	QString *line1Segment = new QString; // These seem to need deleting when no longer
	QString *line2Segment = new QString; // needed. Memory leaks appear if not done.

	*pm = *lcdPixmap; // Create a text-free copy of the LCD background pixmap

	QPainter* p = new QPainter;	// Draw the LCD text contents into the text-free pixmap.
	p->begin( pm );			// Delete p after drawing is done.
	p->setPen(*LCD_TEXT_COLOR);	// Color of LCD text.
	p->setFont( font() ); // QApplication::font() returns default font for this app.

	const unsigned int x = 10;		// Initial horizontal coordinate for text drawing.

	static const int y_Line1 = findCoordY( 1 );	// Get Line1 y-coordinate.
	static const int y_Line2 = findCoordY( 2 );	// Get Line2 y-coordinate.


	// Paint LCD line 1 text.

	// Paints the whole line if short.
	if( line_1->length() < (MAX_LCD_CHARS + 1) ) { // It's a short line.
		drawLCDTextLine( p, line_1, x, y_Line1 );
		line1Painted = true;
	}

	// It's a long line, so we do something a little more complicated.
	if( line_1->length() > MAX_LCD_CHARS ) { // It's a long line. Advance text one char position.
		if( newLine1 ) {
			line1ScrollStartDelay = SCROLL_START_DELAY;
			beginAtLine1Char = 0; // Since it's a new line, start at the start.
		}

		line1Painted = false;

		if( ( line1ScrollStartDelay == 0 ) && ( line1ScrollEndDelay == 0 ) ) { // no delays
			*line1Segment = line_1->mid( beginAtLine1Char , MAX_LCD_CHARS ); // Select midstring.
			drawLCDTextLine( p, line1Segment, x, y_Line1 ); // Paint it.
			line1Painted = true;
			beginAtLine1Char++;

			if( ( beginAtLine1Char + MAX_LCD_CHARS) > line_1->length() ) { // End of line. Delays needed.
			/*
			printf( " beginAtLine1Char = %d\n", beginAtLine1Char );
			printf( "line_1->length() = %d\n", line_1->length() );
			*/

				line1ScrollEndDelay = SCROLL_END_DELAY; // Wait at the end of the line.
				// printf("line1ScrollEndDelay-- = %d\n", line1ScrollEndDelay );

				beginAtLine1Char = 0;	// For rescroll later.
				// printf("line1ScrollEndDelay-- = %d\n", line1ScrollEndDelay );
			}
		}

		if( !line1Painted && (line1ScrollEndDelay > 0) ) { // Delay at end of line.
			line1ScrollStartDelay = 0;

			/*
			printf( "line1ScrollEndDelay = %d\n", line1ScrollEndDelay ); // test
			printf( "line1ScrollStartDelay = %d\n", line1ScrollStartDelay );
			*/

			*line1Segment = line_1->mid( (line_1->length() - MAX_LCD_CHARS), line_1->length() );
			drawLCDTextLine( p, line1Segment, x, y_Line1 ); // End-of-line.
			line1Painted = true;
			line1ScrollEndDelay--;

			if( line1ScrollEndDelay < 1 ) { // after the '--' above.
				line1ScrollStartDelay = SCROLL_START_DELAY;
			}

			/*
			printf( "line1ScrollEndDelay = %d\n", line1ScrollEndDelay );
			printf( "line1ScrollStartDelay = %d\n", line1ScrollStartDelay );
			*/
		}

		if( !line1Painted && (line1ScrollStartDelay > 0)) { // Delay at beginning of line.

			// printf( "line1ScrollEndDelay = %d\n", line1ScrollEndDelay );

			line1ScrollStartDelay--;
			line1ScrollEndDelay = 0;

			/*
			printf( "line1ScrollStartDelay = %d\n", line1ScrollStartDelay );
			printf( "line1ScrollEndDelay = %d\n", line1ScrollEndDelay );
			*/

			*line1Segment = line_1->mid( 0, MAX_LCD_CHARS ); // First 16 characters of message.
			drawLCDTextLine( p, line1Segment, x, y_Line1 ); // Draw start of line.

			/*
			printf( "line1ScrollStartDelay = %d\n", line1ScrollStartDelay );
			printf( "line1ScrollEndDelay = %d\n", line1ScrollEndDelay );
			*/

			line1Painted = true;
		}
	}

	if( !line1Painted ) printf( "Error: Line1 has not been painted.\n" );

	// Paint LCD line 2 text.

	// For LCD line 2. Paints the whole line if short.
	if( line_2->length() < (MAX_LCD_CHARS + 1) ) {	  // It's a short line
		drawLCDTextLine( p, line_2, x, y_Line2 ); // Paint it.
		line2Painted = true;
	}

	// It's a long line, so we do something a little more complicated.
	  if( line_2->length() > MAX_LCD_CHARS ) { // It's a long line. Advance text one char. position.

		if( newLine2 ) {
			line2ScrollStartDelay = SCROLL_START_DELAY;
			beginAtLine2Char = 0;
		}

		line2Painted = false;

		if( ( line2ScrollStartDelay == 0 ) && ( line2ScrollEndDelay == 0 ) ) {  // no delays
			*line2Segment = line_2->mid( beginAtLine2Char , MAX_LCD_CHARS ); // Select midstring.
			drawLCDTextLine( p, line2Segment, x, y_Line2 ); // Paint it.
			line2Painted = true;
			beginAtLine2Char++;

			if( ( beginAtLine2Char + MAX_LCD_CHARS) > line_2->length() ) { // end of line. Delays needed.

				/*
				printf( " beginAtLine2Char = %d\n", beginAtLine2Char );
				printf( "line_2->length() = %d\n", line_2->length() );
				*/

				beginAtLine2Char = 0; // Set midstring to beginning of line.

				/*
				printf("line2ScrollEndDelay-- = %d\n", line2ScrollEndDelay );
				*/

				line2ScrollEndDelay = SCROLL_END_DELAY;

				/*
				printf("line2ScrollEndDelay-- = %d\n", line2ScrollEndDelay );
				*/
			}
		}

		if( !line2Painted && (line2ScrollEndDelay > 0) ) { // Delay at end of line.
			line2ScrollStartDelay = 0;

			/*
			printf( "line2ScrollEndDelay = %d\n", line2ScrollEndDelay );
			printf( "line2ScrollStartDelay = %d\n", line2ScrollStartDelay );
			*/

			*line2Segment = line_2->mid( (line_2->length() - MAX_LCD_CHARS), line_2->length() );
			drawLCDTextLine( p, line2Segment, x, y_Line2 ); // End-of-line.
			line2Painted = true;
			line2ScrollEndDelay--;

			if( line2ScrollEndDelay < 1 ) { // due to the '--' above.
				line2ScrollStartDelay = SCROLL_START_DELAY;
			}

			/*
			printf( "line2ScrollEndDelay = %d\n", line2ScrollEndDelay );
			printf( "line2ScrollStartDelay = %d\n", line2ScrollStartDelay );
			*/
		}

		if( !line2Painted && (line2ScrollStartDelay > 0)) { // Delay at beginning of line.

			/*
			printf( "line2ScrollEndDelay = %d\n", line2ScrollEndDelay );
			*/

			line2ScrollStartDelay--;
			line2ScrollEndDelay = 0;

			/*
			printf( "line2ScrollStartDelay = %d\n", line2ScrollStartDelay );
			printf( "line2ScrollEndDelay = %d\n", line2ScrollEndDelay );
			*/

		        *line2Segment = line_2->mid( 0, MAX_LCD_CHARS ); // First 16 characters of message.
			drawLCDTextLine( p, line2Segment, x, y_Line2 ); // Draw start of line.

			/*
			printf( "line2ScrollStartDelay = %d\n", line2ScrollStartDelay );
			printf( "line2ScrollEndDelay = %d\n", line2ScrollEndDelay );
			*/

			line2Painted = true;
		}
	}

	if( !line2Painted ) printf( "Error: Line2 has not been painted.\n" );

	delete line1Segment; // plugs memory leak.
	delete line2Segment; // plugs memory leak.

	p->end(); // We're finished painting.
	delete p; // plugs a memory leak

	// Copy the updated pixmap to the LCD.
	bitBlt( lcdGraphic, 0, 0, pm );
}


/**
 * Paints line of text onto pixmap. The QPainter
 * for this is created in displayLCDMessages().
 * @param p Pointer to the QPainter that redraws the LCD background
 * @param line Pointer to the QString that needs to be redrawn
 * @param x Horizontal offset of the first character
 * @param y Vertical offset of the first character
 */
void
XojPanel::drawLCDTextLine( QPainter * p, QString * line, int x, int y )
{
	// printf( "void drawLCDTextLine( QPainter * p, QString * line, int x, int y )\n" );

	unsigned int i;

	if( line->length() < (MAX_LCD_CHARS + 1) ) { // It's short enough to paint all at once.
		for (i = 0;  i < line->length();  i++, x += 12)
			p->drawText( x, y, line->mid(i, 1), 1);
	}
	else {
		printf( "Error: in \"void drawLCDTextLine(...)\": Line length exceeds ");
		printf( "%i characters.\n", MAX_LCD_CHARS );
		printf( "line = ->%s<-end\n", (const char*)line );
	}
}


/**
* Determines the y-coordinate for LCD text drawing.
* Vertically centers the text line in its space.
* @param lineNum The LCD text line-number (1 or 2).
*/
int
XojPanel::findCoordY( int lineNum )
{
	// printf("XojPanel::findCoordY( int lineNum )\n");
	if(lineNum == 1) return 17;
	else return 33;
}


/**
 * Declares xojpanel object and begins execution.
 */
int main( int argc, char **argv )
{
	#ifdef HAVE_QT_20
		QApplication::setDesktopSettingsAware(FALSE);
	#endif

	QApplication app(argc,argv);

	XojPanel *xojpanel = new XojPanel(argc,argv);

	app.setMainWidget( xojpanel );
	xojpanel->show();

	int ret = app.exec();
	return ret;
}

// End of file
