// ExMlcCommandChannel.cpp

/* Copyright (C) 2000-2003 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Original author: David Paschal */

#include <stdio.h>
#ifndef EX_TRANSPORT_UNIX_PORT
#include <bp/ex/ExFileNum.h>
#define cFileNum CFILENUM(cModExio,cFileNum_transport_ExMlcCommandChannel)
#include <lm/LoggingManager.h>
#endif

#include <ExMlcTransport.h>

/*****************************************************************************\
|* class ExMlcCommandChannel
\*****************************************************************************/

/*---------------------------------------------------------------------------*\
|* static const member definitions:
\*---------------------------------------------------------------------------*/

const int ExMlcCommandChannel::COMMAND_SOCKET;
const int ExMlcCommandChannel::COMMAND_REPLY_TIMEOUT;
const int ExMlcCommandChannel::MAX_COMMAND_PACKET_LENGTH;
const int ExMlcCommandChannel::MAX_SERVICE_NAME_LENGTH;
const int ExMlcCommandChannel::COMMAND_MAX_OUTSTANDING_CREDIT;
const int ExMlcCommandChannel::CMD_INIT;
const int ExMlcCommandChannel::CMD_INIT_REPLY;
const int ExMlcCommandChannel::CMD_OPEN_CHANNEL;
const int ExMlcCommandChannel::CMD_OPEN_CHANNEL_REPLY;
const int ExMlcCommandChannel::CMD_CLOSE_CHANNEL;
const int ExMlcCommandChannel::CMD_CLOSE_CHANNEL_REPLY;
const int ExMlcCommandChannel::CMD_CREDIT;
const int ExMlcCommandChannel::CMD_CREDIT_REPLY;
const int ExMlcCommandChannel::CMD_CREDIT_REQUEST;
const int ExMlcCommandChannel::CMD_CREDIT_REQUEST_REPLY;
const int ExMlcCommandChannel::CMD_CONFIG_SOCKET;
const int ExMlcCommandChannel::CMD_CONFIG_SOCKET_REPLY;
const int ExMlcCommandChannel::CMD_EXIT;
const int ExMlcCommandChannel::CMD_EXIT_REPLY;
const int ExMlcCommandChannel::CMD_GET_SOCKET_ID;
const int ExMlcCommandChannel::CMD_GET_SOCKET_ID_REPLY;
const int ExMlcCommandChannel::CMD_GET_SERVICE_NAME;
const int ExMlcCommandChannel::CMD_GET_SERVICE_NAME_REPLY;
const int ExMlcCommandChannel::CMD_ERROR;
const int ExMlcCommandChannel::REPLY_BIT;
const int ExMlcCommandChannel::LEN_REQUEST;
const int ExMlcCommandChannel::LEN_REPLY;
const int ExMlcCommandChannel::LEN_INIT;
const int ExMlcCommandChannel::LEN_INIT_REPLY;
const int ExMlcCommandChannel::MLC_LEN_OPEN_CHANNEL;
const int ExMlcCommandChannel::DOT4_LEN_OPEN_CHANNEL;
const int ExMlcCommandChannel::MLC_LEN_OPEN_CHANNEL_REPLY;
const int ExMlcCommandChannel::DOT4_LEN_OPEN_CHANNEL_REPLY;
const int ExMlcCommandChannel::LEN_CLOSE_CHANNEL;
const int ExMlcCommandChannel::MLC_LEN_CLOSE_CHANNEL_REPLY;
const int ExMlcCommandChannel::DOT4_LEN_CLOSE_CHANNEL_REPLY;
const int ExMlcCommandChannel::LEN_CREDIT;
const int ExMlcCommandChannel::MLC_LEN_CREDIT_REPLY;
const int ExMlcCommandChannel::DOT4_LEN_CREDIT_REPLY;
const int ExMlcCommandChannel::LEN_CREDIT_REQUEST;
const int ExMlcCommandChannel::MLC_LEN_CREDIT_REQUEST_REPLY;
const int ExMlcCommandChannel::DOT4_LEN_CREDIT_REQUEST_REPLY;
const int ExMlcCommandChannel::LEN_CONFIG_SOCKET;
const int ExMlcCommandChannel::LEN_CONFIG_SOCKET_REPLY;
const int ExMlcCommandChannel::LEN_EXIT;
const int ExMlcCommandChannel::LEN_EXIT_REPLY;
const int ExMlcCommandChannel::LEN_GET_SOCKET_ID;
const int ExMlcCommandChannel::LEN_GET_SOCKET_ID_REPLY;
const int ExMlcCommandChannel::LEN_GET_SERVICE_NAME;
const int ExMlcCommandChannel::LEN_GET_SERVICE_NAME_REPLY;
const int ExMlcCommandChannel::MLC_LEN_ERROR;
const int ExMlcCommandChannel::DOT4_LEN_ERROR;

/*---------------------------------------------------------------------------*\
|* Constructor, destructor, reset, dump, handleMsg:
\*---------------------------------------------------------------------------*/

ExMlcCommandChannel::ExMlcCommandChannel(ExMlcTransport *pMlcTransport,
    int channel,ExMgr *pMgr,ExPhysicalPort *pPhysicalPort,int channelCount,
    int final):
      ExMlcTransportChannel(pMlcTransport,channel,COMMAND_SOCKET,
       pMgr,pPhysicalPort,this,0) {
	disableCreditCommands=1;

	/* Allocate buffer pool for forward commands. */
	pForwardCommandPool=new ExBufferPool((void *)this,
		sizeof(CommandUnion),
		calculateForwardCommandPoolSize(channelCount),
		pMgr->getPortNumber(),pMgr->getBufferPoolMgr());
	pForwardNonconsumingQueue=new ExBdrQueue("forwardNonconsumingQueue");
	pForwardReplyQueue=new ExBdrQueue("forwardReplyQueue");
	pForwardRequestQueue=new ExBdrQueue("forwardRequestQueue");

	pCommandReplyTimeoutMsg=pMgr->getFreeMsg();
	pCommandReplyTimeoutMsg->setType(
		(ExMsgType)MSG_COMMAND_REPLY_TIMEOUT);
	pCommandReplyTimer=new ExCountingWatchdogTimer(this,
		pCommandReplyTimeoutMsg,
		DEBUG_STRING("pCommandReplyTimer"));

	if (final) {
		reset(RESET_STARTUP);
		registerStuff("ExMlcCommandChannel");
	}
}

ExMlcCommandChannel::~ExMlcCommandChannel(void) {
	delete pForwardCommandPool;

	pCommandReplyTimer->removeMsg();
	delete pCommandReplyTimer;
	pMgr->returnMsg(pCommandReplyTimeoutMsg);
}

void ExMlcCommandChannel::flushQueue(ExBdrQueue *pQueue) {
	ExBdr *pBdr;

	while (42) {
		pBdr=pQueue->Pop();
		if (!pBdr) break;
		pForwardCommandPool->returnBuffer(pBdr);
	}
}

void ExMlcCommandChannel::reset(int rtype) {
	int timeout;

	ExMlcTransportChannel::reset(rtype);

	flushQueue(pForwardNonconsumingQueue);
	flushQueue(pForwardReplyQueue);
	flushQueue(pForwardRequestQueue);
	pCommandReplyTimer->reset();

	if (rtype!=RESET_CLOSE &&
	    rtype!=RESET_START_DEACTIVATE &&
	    rtype!=RESET_FINISH_DEACTIVATE) {
		timeout=COMMAND_REPLY_TIMEOUT;
		tknobGetWorkingValue(port,EX_KNOB_MLC_COMMAND_REPLY_TIMEOUT,
			&timeout);
		pCommandReplyTimer->setDelay(timeout);
		forwardRequestCredit.reset();

		allowErrorPackets=1;
		tknobGetWorkingValue(port,EX_KNOB_MLC_SEND_ERROR_PACKETS,
			&allowErrorPackets);
		lastRequest=ERROR;
		lastPsid=ERROR;
		lastSsid=ERROR;

		uncreditedBuffers=bufferCount=3;

		remoteSocket=COMMAND_SOCKET;

		maxForwardPacketSize=MAX_COMMAND_PACKET_LENGTH;
		maxReversePacketSize=MAX_COMMAND_PACKET_LENGTH;

		forwardMaxOutstandingCredit=COMMAND_MAX_OUTSTANDING_CREDIT;
		reverseMaxOutstandingCredit=COMMAND_MAX_OUTSTANDING_CREDIT;
	}
}

void ExMlcCommandChannel::registerStuff(char *title) {
	ExMlcTransportChannel::registerStuff(title);

	nettestRegister(NETTEST_TYPE_INT32,
		pCommandReplyTimer->pointToDelay(),
		"%s[%d,%d].commandReplyTimer.delay",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		pCommandReplyTimer->pointToCount(),
		"%s[%d,%d].commandReplyTimer.count",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		forwardRequestCredit.pointToCurrent(),
		"%s[%d,%d].forwardRequestCredit",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&allowErrorPackets,
		"%s[%d,%d].allowErrorPackets",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&lastRequest,
		"%s[%d,%d].lastRequest",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&lastPsid,
		"%s[%d,%d].lastPsid",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&lastSsid,
		"%s[%d,%d].lastSsid",
		title,port,channel);
}

#ifdef JD_DEBUGLITE
void ExMlcCommandChannel::dump(void) {
	ExMlcTransportChannel::dump();
	printf("----------------\n");
	printf("pForwardCommandPool=0x%8.8X\n",
		(int)pForwardCommandPool);
	printf("pForwardNonconsumingQueue=0x%8.8X\n",
		(int)pForwardNonconsumingQueue);
	printf("pForwardReplyQueue=0x%8.8X\n",
		(int)pForwardReplyQueue);
	printf("pForwardRequestQueue=0x%8.8X\n",
		(int)pForwardRequestQueue);
	printf("forwardRequestCredit: ");
		forwardRequestCredit.dump();
	printf("pCommandReplyTimer=0x%8.8X (count=%d)\n",
		(int)pCommandReplyTimer,pCommandReplyTimer->getCount());
	printf("pCommandReplyTimeoutMsg=0x%8.8X\n",
		(int)pCommandReplyTimeoutMsg);
	printf("allowErrorPackets=%d\n",
		allowErrorPackets);
	printf("lastRequest=0x%2.2X\n",
		lastRequest);
	printf("lastPsid=%d\n",
		lastPsid);
	printf("lastSsid=%d\n",
		lastSsid);
}
#endif

void ExMlcCommandChannel::handleMsg(ExMsg *pMsg) {
	switch (pMsg->getType()) {
	   case MSG_COMMAND_REPLY_TIMEOUT:
		if (!pCommandReplyTimer->isCancelled()) {
			LOG_ERROR(cEXTP,0,cCausePeriphError,
				"Command reply timeout on port=%d, count=%d, "
				"lastRequest=0x%2.2X!\n",port,
				pCommandReplyTimer->getCount(),lastRequest);
			pMgr->exClose(ExMlcTransport::REASON_REPLY_TIMEOUT);
		}
		pCommandReplyTimer->setMsg(pCommandReplyTimeoutMsg);
		break;

	   default:
		ExMlcTransportChannel::handleMsg(pMsg);
	}
}

/*---------------------------------------------------------------------------*\
|* Allocate channel:
\*---------------------------------------------------------------------------*/

void ExMlcCommandChannel::allocate(ExService *pService,SCD scd,
    int forwardDataPriority,int minBuffersRequired,int benefitOfMoreBuffers,
    int reverseDataBufferSize) {
	ExMlcTransportChannel::allocate(pService,scd,forwardDataPriority,
		minBuffersRequired,benefitOfMoreBuffers,reverseDataBufferSize);

	setOpen();

	setDatalensFromPacketSizes();
}

/*---------------------------------------------------------------------------*\
|* Forward data flow:
\*---------------------------------------------------------------------------*/

void ExMlcCommandChannel::getEmptyBuffer(ExBdr **ppBdr,CommandUnion **pData) {
	LOGD_ASSERT(ppBdr && pData,cEXTP,0,cCauseBadParm,"");
	if (isClosing()) {
		*ppBdr=0;
		LOGD_ERROR(cEXTP,0,cCauseBadState,
			"getEmptyBuffer(port=%d): closing!\n",
			port);
		return;
	}
	*ppBdr=pForwardCommandPool->getBuffer();
	LOG_ASSERT(*ppBdr,cEXTP,0,cCauseNoMem,"");
	*pData=bdrGetCommandUnion(*ppBdr);

	(*pData)->options.consumeCredit=1;
	(*pData)->options.grantCredit=1;
	(*pData)->options.psid=ERROR;
	(*pData)->options.ssid=ERROR;
	(*pData)->options.closeAfterSend=0;
}

void ExMlcCommandChannel::putFullBuffer(ExBdr *pBdr,CommandUnion *data) {
	ExBdrQueue *pQueue;

	/* Put packet onto right queue. */
	if (!data->options.consumeCredit) {
		pQueue=pForwardNonconsumingQueue;
	} else if (data->request.command&REPLY_BIT) {
		pQueue=pForwardReplyQueue;
	} else {
		pQueue=pForwardRequestQueue;
	}
	pQueue->Add(pBdr);

	/* Start reply timer for requests (except Error). */
	int command=data->request.command;
	if (!(command&REPLY_BIT) && command!=CMD_ERROR) {
		pCommandReplyTimer->start();
	}

	/* Try to send it. */
	forwardDataAvailable();
}

ExBdr *ExMlcCommandChannel::servicePullForwardData(void) {
	/* First try to pull a packet from the credit-nonconsuming queue. */
	ExBdr *pBdr=pForwardNonconsumingQueue->Pop();

	/* Else try to pull a packet from the reply queue. */
	if (!pBdr) {
		pBdr=pForwardReplyQueue->Pop();

		/* Else try to pull a packet from the request queue,
		 * but only if there's request credit available! */
		if (!pBdr) {
			ExBdr *_pBdr=pForwardRequestQueue->Peek();
			if (_pBdr && forwardRequestCredit.grab()!=ERROR) {
				pBdr=pForwardRequestQueue->Pop();
				LOGD_ASSERT(pBdr==_pBdr,
					cEXTP,0,cCauseBadParm,"");
			}
		}
	}
	if (pBdr) {
		/* If we pulled a request, then remember any
		 * socket IDs in the command, because MLC
		 * replies don't provide them. */
		CommandUnion *data=bdrGetCommandUnion(pBdr);
		if (!(data->request.command&REPLY_BIT)) {
			lastRequest=data->request.command;
			lastPsid=data->options.psid;
			lastSsid=data->options.ssid;
		}
	}

	return pBdr;
}

int ExMlcCommandChannel::grabForwardCredit(ExBdr *pBdr) {
	CommandUnion *data=bdrGetCommandUnion(pBdr);

	/* Init, InitReply, and Error don't consume credit. */
	if (!data->options.consumeCredit) return 0;
	return ExMlcTransportChannel::grabForwardCredit(pBdr);
}

int ExMlcCommandChannel::handlePiggybackCreditPart1(int credit,
    unsigned char *_data,int datalen,int cmdinfo) {
	CommandUnion *data=(CommandUnion *)_data;
	int expectedCredit=0,requestCredit=0,resetCredit=0;

	if (datalen>=LEN_REQUEST) {
		int command=data->request.command;

		if (command&REPLY_BIT) {
			expectedCredit=1;
			requestCredit=1;

			/* InitReply sets, not adds, forward credit. */
			if (command==CMD_INIT_REPLY) {
				resetCredit=1;
			}

		} else if (command!=CMD_ERROR) {
			expectedCredit=1;

			/* Init sets, not adds, forward credit. */
			/* Enable and test this if we someday decide
			 * to allow peripheral-initiated Inits. */
			// if (command==CMD_INIT) {
			//	resetCredit=1;
			// }
		}
	}

	/* Convert MLC implicit credit to explicit credit. */
	if (pMlcTransport->revisionIsMlc()) {
		/* Complain if explicit piggyback credit was provided. */
		if (credit) {
			LOGD_ERROR(cEXTP,0,cCausePeriphError,
				"handlePiggybackCreditPart1(port=%d,"
				"channel=%d): "
				"incorrect MLC implicit credit=%d!\n",
				port,channel,credit);
		}

		credit=expectedCredit;

	/* For 1284.4, complain if peripheral granted credit when
	 * it shouldn't have granted any.  Other than that, we don't
	 * care if it granted more than we expected, except that we
	 * won't use the excess for forward request credit. */
	} else if (credit && !expectedCredit) {
		LOGD_ERROR(cEXTP,0,cCausePeriphError,
			"handlePiggybackCreditPart1(port=%d,channel=%d): "
			"unexpected 1284.4 credit=%d!\n",
			port,channel,credit);
		credit=expectedCredit;
	}

	/* Reset credit if necessary. */
	if (resetCredit) {
		if (requestCredit) {
			forwardRequestCredit.reset();
			forwardRequestCredit.preset(credit);
		}

		forwardCredit.reset();
		forwardCredit.preset(credit);
	}

	return ExMlcTransportChannel::handlePiggybackCreditPart1(
		credit,_data,datalen,requestCredit);
}

int ExMlcCommandChannel::handleForwardCreditPart1(int credit,
    int requestCredit) {
	if (forwardRequestCredit.increment(requestCredit)==ERROR) {
		LOGD_ERROR(cEXTP,0,cCausePeriphError,
			"handleForwardCreditPart1(port=%d,channel=%d,"
			"credit=%d,requestCredit=%d): "
			"request credit overflow!\n",
			port,channel,credit,requestCredit);
		return ERROR;
	}

	return ExMlcTransportChannel::handleForwardCreditPart1(credit,0);
}

void ExMlcCommandChannel::serviceForwardDataResponse(ExBdr *pBdr,int status) {
#if 0
	if (status!=OK) {
		/* TODO: How do we handle this? */
	}
#endif

	CommandUnion *data=bdrGetCommandUnion(pBdr);
	int closeAfterSend=data->options.closeAfterSend;

	pForwardCommandPool->returnBuffer(pBdr);

	if (closeAfterSend) {
		pMgr->exClose(closeAfterSend);
	}
}

/*---------------------------------------------------------------------------*\
|* Reverse data flow:
\*---------------------------------------------------------------------------*/

int ExMlcCommandChannel::grabReverseCredit(unsigned char *_data,int datalen) {
	CommandUnion *data=(CommandUnion *)_data;

	data->options.consumeCredit=1;
	if (datalen>=LEN_REQUEST) {
		int command=data->request.command;

		/* Init, InitReply, and Error don't consume credit. */
		if (command==CMD_INIT || command==CMD_INIT_REPLY ||
		    command==CMD_ERROR) {
			data->options.consumeCredit=0;

			return 0;
		}
	}

	return ExMlcTransportChannel::grabReverseCredit(_data,datalen);
}

int ExMlcCommandChannel::grantReverseCredit(int max,
    unsigned char *_data,int datalen) {
	CommandUnion *data=(CommandUnion *)_data;

	/* Don't check datalen, because that'll break on empty packets. */
	if (!data) {
		LOGD_ERROR(cEXTP,0,cCausePeriphError,
			"grantReverseCredit(port=%d,channel=%d): "
			"only piggyback credit allowed!\n",
			port,channel);
		return 0;
	}

	if (max>data->options.grantCredit) {
		max=data->options.grantCredit;
	}

	int credit=ExMlcTransportChannel::grantReverseCredit(max,_data,datalen);
#ifdef JD_DEBUGLITE
	if (credit!=data->options.grantCredit) {
		LOGD_ERROR(cEXTP,0,cCausePeriphError,
			"grantReverseCredit(port=%d,channel=%d): "
			"granted %d credits, expected=%d!\n",
			port,channel,credit,data->options.grantCredit);
	}
#endif

	return credit;
}

int ExMlcCommandChannel::prepareToSendPiggybackCredit(int credit) {
	/* For MLC (not 1284.4), convert explicit credit to implicit credit. */
	if (pMlcTransport->revisionIsMlc()) {
		credit=0;
	}

	return credit;
}

/* TODO: For non-channelized replies, error out if the corresponding
 * request isn't pending. */
void ExMlcCommandChannel::serviceReverseDataReceived(ExBdr *pBdr,int status,
    unsigned char *_data,int datalen) {
#if 0
	if (status!=OK) {
		/* TODO: How do we handle this? */
	}
#endif

#if 0
	if (datalen<LEN_REQUEST) {
		/* TODO: What should we do about empty packets? */
	}
#endif

	CommandUnion *data=(CommandUnion *)_data;
	int command=data->request.command;
	int dontStopReplyTimer=0;

	if (!pMlcTransport->revisionIsSet() &&
	    command!=CMD_INIT && command!=CMD_INIT_REPLY &&
	    command!=CMD_ERROR) {
		sendError(getPrimarySocket(),getSecondarySocket(),
			ExMlcTransport::MLC_ERROR_NOT_INITIALIZED,
			ExMlcTransport::DOT4_ERROR_UNOPENED_CHANNEL_DATA);

	} else switch (command) {
	   case CMD_INIT:
		handleInit(data,datalen);
		break;

	   case CMD_INIT_REPLY:
		if (handleInitReply(data,datalen)==ERROR) {
			dontStopReplyTimer=1;
		}
		break;

	   case CMD_OPEN_CHANNEL:
		handleOpenChannel(data,datalen);
		break;

	   case CMD_OPEN_CHANNEL_REPLY:
		handleOpenChannelReply(data,datalen);
		break;

	   case CMD_CLOSE_CHANNEL:
		handleCloseChannel(data,datalen);
		break;

	   case CMD_CLOSE_CHANNEL_REPLY:
		handleCloseChannelReply(data,datalen);
		break;

	   case CMD_CREDIT:
		handleCredit(data,datalen);
		break;

	   case CMD_CREDIT_REPLY:
		handleCreditReply(data,datalen);
		break;

	   case CMD_CREDIT_REQUEST:
		handleCreditRequest(data,datalen);
		break;

	   case CMD_CREDIT_REQUEST_REPLY:
		handleCreditRequestReply(data,datalen);
		break;

	   case CMD_CONFIG_SOCKET:
		if (!pMlcTransport->revisionIsMlc()) goto abort;
		handleConfigSocket(data,datalen);
		break;

	   case CMD_CONFIG_SOCKET_REPLY:
		if (!pMlcTransport->revisionIsMlc()) goto abort;
		handleConfigSocketReply(data,datalen);
		break;

	   case CMD_EXIT:
		handleExit(data,datalen);
		break;

	   case CMD_EXIT_REPLY:
		handleExitReply(data,datalen);
		break;

	   case CMD_GET_SOCKET_ID:
		if (pMlcTransport->revisionIsMlc()) goto abort;
		handleGetSocketID(data,datalen);
		break;

	   case CMD_GET_SOCKET_ID_REPLY:
		if (pMlcTransport->revisionIsMlc()) goto abort;
		handleGetSocketIDReply(data,datalen);
		break;

	   case CMD_GET_SERVICE_NAME:
		if (pMlcTransport->revisionIsMlc()) goto abort;
		handleGetServiceName(data,datalen);
		break;

	   case CMD_GET_SERVICE_NAME_REPLY:
		if (pMlcTransport->revisionIsMlc()) goto abort;
		handleGetServiceNameReply(data,datalen);
		break;

	   case CMD_ERROR:
		handleError(data,datalen);
		break;

	   default:
	   abort:
		if (pMlcTransport->revisionIsMlc()) {
			sendGenericReply(command|REPLY_BIT,
				ExMlcTransport::MLC_RESULT_UNKNOWN_COMMAND);

		} else /* if (pMlcTransport->revisionIsDot4()) */ {
			dot4SendError(getPrimarySocket(),getSecondarySocket(),
				ExMlcTransport::DOT4_ERROR_UNKNOWN_COMMAND);
		}
	}

	if (command&REPLY_BIT && !dontStopReplyTimer) {
		pCommandReplyTimer->stop();
	}

	/* If this packet didn't consume credit, then skip
	 * returnBufferNotification so we don't grant credit. */
	if (!data->options.consumeCredit) {
		ExTransport::unmarkBDR(pBdr);
#ifdef JD_DEBUGLITE
		countReverseBufferReturns.increment();
#endif
	}

	/* TODO: Assert that this isn't a chain? */
	pBdr->setNext(0);
	pBdr->returnBuffer();
}

STATUS ExMlcCommandChannel::checkDatalen(CommandUnion *data,int datalen,
    int expectedDatalen) {
	if (datalen==expectedDatalen ||
	    (pMlcTransport->revisionIsMlc() &&
	     data->request.command&REPLY_BIT &&
	     data->reply.result==ExMlcTransport::MLC_RESULT_UNKNOWN_COMMAND)) {
		return OK;
	}

	LOGD_ERROR(cEXTP,0,cCausePeriphError,
		"checkDatalen(port=%d,command=0x%2.2X): "
		"datalen=%d, expected=%d!\n",
		port,data->request.command,datalen,expectedDatalen);

	if (datalen>expectedDatalen) return OK;

	sendError(getPrimarySocket(),getSecondarySocket(),
		ExMlcTransport::MLC_ERROR_BAD_COMMAND_PACKET_LENGTH,
		ExMlcTransport::DOT4_ERROR_MALFORMED_PACKET);

	return ERROR;
}

/*---------------------------------------------------------------------------*\
|* Macros:
\*---------------------------------------------------------------------------*/

#define DECLARE_VARIABLES \
	ExBdr *pBdr; \
	CommandUnion *data;

#define GET_EMPTY_BUFFER(cmd) \
	do { \
		getEmptyBuffer(&pBdr,&data); \
		if (!pBdr) return; \
		data->request.command=cmd; \
	} while (0)

#define PUT_FULL_BUFFER (putFullBuffer(pBdr,data))

#define CHECK_DATALEN(expectedDatalen) \
	do { \
		if (checkDatalen(data,datalen,expectedDatalen)==ERROR) \
			return; \
	} while (0)

#define CHECK_DATALEN_RETURN_ERROR(expectedDatalen) \
	do { \
		if (checkDatalen(data,datalen,expectedDatalen)==ERROR) \
			return ERROR; \
	} while (0)

#define SET_DATALEN(datalen) (pBdr->setDataLength(datalen))
#define GET_RESULT (result=data->reply.result)
#define SET_RESULT (data->reply.result=result)

#define REQUEST_LOOKUP_TCD(functionName,mlcError,dot4ReplyCall) \
	if (!(tcd=pMlcTransport->lookupChannel(psid,ssid))) { \
		LOGD_ERROR(cEXTP,0,cCausePeriphError, \
			functionName "(port=%d): " \
			"invalid psid=%d, ssid=%d!\n", \
			port,psid,ssid); \
		if (pMlcTransport->revisionIsMlc()) { \
			mlcSendError(mlcError); \
		} else /* if (pMlcTransport->revisionIsDot4()) */ { \
			dot4ReplyCall; \
		} \
		return; \
	}

#define REPLY_LOOKUP_TCD(functionName) \
	if (!(tcd=pMlcTransport->lookupChannel(psid,ssid))) { \
		LOGD_ERROR(cEXTP,0,cCausePeriphError, \
			functionName "(port=%d): " \
			"invalid psid=%d, ssid=%d!\n", \
			port,psid,ssid); \
		sendError(psid,ssid, \
			ExMlcTransport::MLC_ERROR_REPLY_WITHOUT_REQUEST, \
			ExMlcTransport::DOT4_ERROR_REPLY_WITHOUT_REQUEST); \
	}

#define SETUP_LEN_SERVICE_NAME \
	int lenServiceName=strlen(serviceName); \
	if (lenServiceName>MAX_SERVICE_NAME_LENGTH) { \
		lenServiceName=MAX_SERVICE_NAME_LENGTH; \
	}

#define SETUP_LEN_SERVICE_NAME_REPLY \
	SETUP_LEN_SERVICE_NAME; \
	if (result!=ExMlcTransport::RESULT_SUCCESS) { \
		lenServiceName=0; \
	}

#define SET_DATALEN_PLUS_SERVICE_NAME(cmd) \
	SET_DATALEN(cmd+lenServiceName)

#define SET_SERVICE_NAME(cmdpkt) \
	do { \
		strncpy((cmdpkt).serviceName,serviceName,lenServiceName); \
		(cmdpkt).serviceName[lenServiceName]=0; \
	} while (0)

#define GET_SERVICE_NAME(cmdpkt) \
	do { \
		serviceName=(cmdpkt).serviceName; \
		((unsigned char *)(data))[datalen]=0; \
	} while (0)

/*---------------------------------------------------------------------------*\
|* Routines to pack and enqueue forward commands:
|* Routines to unpack and route reverse commands:
\*---------------------------------------------------------------------------*/

void ExMlcCommandChannel::sendInit(int revision) {
	DECLARE_VARIABLES;

	GET_EMPTY_BUFFER(CMD_INIT);
	data->options.consumeCredit=0;
	ungrantReverseCredit();
	reverseCredit.preset(data->options.grantCredit);

		SET_DATALEN(LEN_INIT);
		data->init.revision=revision;

		LOG_INFO(cEXTP,0,
			"sendInit(port=%d): revision=0x%2.2X.\n",
			port,revision);

	pCommandReplyTimer->reset();
	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::handleInit(CommandUnion *data,int datalen) {
	int revision;

		revision=data->init.revision;

		LOG_INFO(cEXTP,0,
			"handleInit(port=%d): revision=0x%2.2X.\n",
			port,revision);

		CHECK_DATALEN(LEN_INIT);

	pMlcTransport->handleInit(revision);
}

void ExMlcCommandChannel::sendInitReply(int result,int revision) {
	DECLARE_VARIABLES;

	GET_EMPTY_BUFFER(CMD_INIT_REPLY);
	data->options.consumeCredit=0;
	SET_RESULT;

		SET_DATALEN(LEN_INIT_REPLY);
		data->initReply.revision=revision;

		LOG_INFO(cEXTP,0,
			"sendInitReply(port=%d): "
			"result=%d, revision=0x%2.2X.\n",
			port,result,revision);

	PUT_FULL_BUFFER;
}

STATUS ExMlcCommandChannel::handleInitReply(CommandUnion *data,int datalen) {
	int result,revision;

	GET_RESULT;

		revision=data->initReply.revision;

		LOG_INFO(cEXTP,0,
			"handleInitReply(port=%d): "
			"result=%d, revision=0x%2.2X.\n",
			port,result,revision);

		CHECK_DATALEN_RETURN_ERROR(LEN_INIT_REPLY);

	return pMlcTransport->handleInitReply(result,revision);
}

void ExMlcCommandChannel::sendOpenChannel(int psid,int ssid,
    int maxPriToSecPacketSize,int maxSecToPriPacketSize,
    int maxOutstandingCredit,int credit) {
	DECLARE_VARIABLES;

	GET_EMPTY_BUFFER(CMD_OPEN_CHANNEL);

		data->openChannel.psid=data->options.psid=psid;
		data->openChannel.ssid=data->options.ssid=ssid;

	if (pMlcTransport->revisionIsMlc()) {
		SET_DATALEN(MLC_LEN_OPEN_CHANNEL);
		BEND_SET_WORD(data->openChannel.mlc.credit,credit);

		LOG_INFO(cEXTP,0,
			"sendOpenChannel(port=%d,MLC): psid=%d, ssid=%d, "
			"credit=%d.\n",
			port,psid,ssid,credit);

	} else /* if (pMlcTransport->revisionIsDot4()) */ {
		SET_DATALEN(DOT4_LEN_OPEN_CHANNEL);
		BEND_SET_WORD(data->openChannel.dot4.maxPriToSecPacketSize,
			maxPriToSecPacketSize);
		BEND_SET_WORD(data->openChannel.dot4.maxSecToPriPacketSize,
			maxSecToPriPacketSize);
		BEND_SET_WORD(data->openChannel.dot4.maxOutstandingCredit,
			maxOutstandingCredit);

		LOG_INFO(cEXTP,0,
			"sendOpenChannel(port=%d,1284.4): psid=%d, ssid=%d, "
			"maxPriToSecPacketSize=%d, maxSecToPriPacketSize=%d, "
			"maxOutstandingCredit=%d.\n",
			port,psid,ssid,
			maxPriToSecPacketSize,maxSecToPriPacketSize,
			maxOutstandingCredit);
	}

	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::handleOpenChannel(CommandUnion *data,int datalen) {
	int psid,ssid;
	int maxPriToSecPacketSize=0,maxSecToPriPacketSize=0;
	int maxOutstandingCredit=0,credit=0;

		psid=data->openChannel.psid;
		ssid=data->openChannel.ssid;

	if (pMlcTransport->revisionIsMlc()) {
		credit=BEND_GET_WORD(data->openChannel.mlc.credit);

		LOG_INFO(cEXTP,0,
			"handleOpenChannel(port=%d,MLC): psid=%d, ssid=%d, "
			"credit=%d.\n",
			port,psid,ssid,credit);

		CHECK_DATALEN(MLC_LEN_OPEN_CHANNEL);

	} else /* if (pMlcTransport->revisionIsDot4()) */ {
		maxPriToSecPacketSize=BEND_GET_WORD(
			data->openChannel.dot4.maxPriToSecPacketSize);
		maxSecToPriPacketSize=BEND_GET_WORD(
			data->openChannel.dot4.maxSecToPriPacketSize);
		maxOutstandingCredit=BEND_GET_WORD(
			data->openChannel.dot4.maxOutstandingCredit);

		LOG_INFO(cEXTP,0,
			"handleOpenChannel(port=%d,1284.4): psid=%d, ssid=%d, "
			"maxPriToSecPacketSize=%d, maxSecToPriPacketSize=%d, "
			"maxOutstandingCredit=%d.\n",
			port,psid,ssid,
			maxPriToSecPacketSize,maxSecToPriPacketSize,
			maxOutstandingCredit);

		CHECK_DATALEN(DOT4_LEN_OPEN_CHANNEL);
	}

	pMlcTransport->handleOpenChannel(psid,ssid,
		maxPriToSecPacketSize,maxSecToPriPacketSize,
		maxOutstandingCredit,credit);
}

void ExMlcCommandChannel::sendOpenChannelReply(int result,int psid,int ssid,
    int maxPriToSecPacketSize,int maxSecToPriPacketSize,
    int maxOutstandingCredit,int credit) {
	DECLARE_VARIABLES;

	GET_EMPTY_BUFFER(CMD_OPEN_CHANNEL_REPLY);
	SET_RESULT;

	if (pMlcTransport->revisionIsMlc()) {
		SET_DATALEN(MLC_LEN_OPEN_CHANNEL_REPLY);
		BEND_SET_WORD(data->openChannelReply.mlc.credit,credit);

		LOG_INFO(cEXTP,0,
			"sendOpenChannelReply(port=%d,MLC): result=%d, "
			"(psid=%d, ssid=%d), credit=%d.\n",
			port,result,psid,ssid,credit);

	} else /* if (pMlcTransport->revisionIsDot4()) */ {
		SET_DATALEN(DOT4_LEN_OPEN_CHANNEL_REPLY);
		data->openChannelReply.dot4.psid=psid;
		data->openChannelReply.dot4.ssid=ssid;
		BEND_SET_WORD(data->openChannelReply.dot4.maxPriToSecPacketSize,
			maxPriToSecPacketSize);
		BEND_SET_WORD(data->openChannelReply.dot4.maxSecToPriPacketSize,
			maxSecToPriPacketSize);
		BEND_SET_WORD(data->openChannelReply.dot4.maxOutstandingCredit,
			maxOutstandingCredit);
		BEND_SET_WORD(data->openChannelReply.dot4.credit,
			credit);

		LOG_INFO(cEXTP,0,
			"sendOpenChannelReply(port=%d,1284.4): result=%d, "
			"psid=%d, ssid=%d, "
			"maxPriToSecPacketSize=%d, maxSecToPriPacketSize=%d, "
			"maxOutstandingCredit=%d, credit=%d.\n",
			port,result,psid,ssid,
			maxPriToSecPacketSize,maxSecToPriPacketSize,
			maxOutstandingCredit,credit);
	}

	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::handleOpenChannelReply(CommandUnion *data,
    int datalen) {
	int result,psid=lastPsid,ssid=lastSsid;
	int maxPriToSecPacketSize=0,maxSecToPriPacketSize=0;
	int maxOutstandingCredit=0,credit;
	ExMlcTransportChannel *tcd;

	GET_RESULT;

	if (pMlcTransport->revisionIsMlc()) {
		credit=BEND_GET_WORD(data->openChannelReply.mlc.credit);

		LOG_INFO(cEXTP,0,
			"handleOpenChannelReply(port=%d,MLC): result=%d, "
			"(psid=%d, ssid=%d), credit=%d.\n",
			port,result,psid,ssid,credit);

		CHECK_DATALEN(MLC_LEN_OPEN_CHANNEL_REPLY);

	} else /* if (pMlcTransport->revisionIsDot4()) */ {
		psid=data->openChannelReply.dot4.psid;
		ssid=data->openChannelReply.dot4.ssid;
		maxPriToSecPacketSize=BEND_GET_WORD(
			data->openChannelReply.dot4.maxPriToSecPacketSize);
		maxSecToPriPacketSize=BEND_GET_WORD(
			data->openChannelReply.dot4.maxSecToPriPacketSize);
		maxOutstandingCredit=BEND_GET_WORD(
			data->openChannelReply.dot4.maxOutstandingCredit);
		credit=BEND_GET_WORD(data->openChannelReply.dot4.credit);

		LOG_INFO(cEXTP,0,
			"handleOpenChannelReply(port=%d,1284.4): result=%d, "
			"psid=%d, ssid=%d, "
			"maxPriToSecPacketSize=%d, maxSecToPriPacketSize=%d, "
			"maxOutstandingCredit=%d, credit=%d.\n",
			port,result,psid,ssid,
			maxPriToSecPacketSize,maxSecToPriPacketSize,
			maxOutstandingCredit,credit);

		CHECK_DATALEN(DOT4_LEN_OPEN_CHANNEL_REPLY);
	}

	REPLY_LOOKUP_TCD("handleOpenChannelReply");

	tcd->handleOpenChannelReply(result,psid,ssid,
		maxPriToSecPacketSize,maxSecToPriPacketSize,
		maxOutstandingCredit,credit);
}

void ExMlcCommandChannel::sendCloseChannel(int psid,int ssid) {
	DECLARE_VARIABLES;

	GET_EMPTY_BUFFER(CMD_CLOSE_CHANNEL);

		data->closeChannel.psid=data->options.psid=psid;
		data->closeChannel.ssid=data->options.ssid=ssid;

		SET_DATALEN(LEN_CLOSE_CHANNEL);

		LOG_INFO(cEXTP,0,
			"sendCloseChannel(port=%d): psid=%d, ssid=%d.\n",
			port,psid,ssid);

	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::handleCloseChannel(CommandUnion *data,int datalen) {
	int psid,ssid;
	ExMlcTransportChannel *tcd;

		psid=data->closeChannel.psid;
		ssid=data->closeChannel.ssid;

		LOG_INFO(cEXTP,0,
			"handleCloseChannel(port=%d): psid=%d, ssid=%d.\n",
			port,psid,ssid);

		CHECK_DATALEN(LEN_CLOSE_CHANNEL);

	REQUEST_LOOKUP_TCD("handleCloseChannel",
		ExMlcTransport::MLC_ERROR_UNOPENED_CHANNEL_CLOSE,
		sendCloseChannelReply(
			ExMlcTransport::DOT4_RESULT_CHANNEL_NOT_OPEN,
			psid,ssid));

	/* Don't allow the command channel to be closed. */
	if (tcd==this) {
		LOGD_ERROR(cEXTP,0,cCausePeriphError,
			"handleCloseChannel(port=%d): "
			"peripheral tried to close command channel!\n",
			port);

		if (pMlcTransport->revisionIsMlc()) {
			/* MLC needs an error code about closing the
			 * command channel. */
			mlcSendError(
			    ExMlcTransport::MLC_ERROR_UNOPENED_CHANNEL_CLOSE);

		} else /* if (pMlcTransport->revisionIsDot4()) */ {
			sendCloseChannelReply(
			    ExMlcTransport::DOT4_RESULT_COMMAND_CHANNEL_CLOSE,
			    psid,ssid);
		}

		return;
	}

	tcd->handleCloseChannel(psid,ssid);
}

void ExMlcCommandChannel::sendCloseChannelReply(int result,int psid,int ssid) {
	DECLARE_VARIABLES;

	GET_EMPTY_BUFFER(CMD_CLOSE_CHANNEL_REPLY);
	SET_RESULT;

	if (pMlcTransport->revisionIsMlc()) {
		SET_DATALEN(MLC_LEN_CLOSE_CHANNEL_REPLY);

		LOG_INFO(cEXTP,0,
			"sendCloseChannelReply(port=%d,MLC): result=%d, "
			"(psid=%d, ssid=%d).\n",
			port,result,psid,ssid);

	} else /* if (pMlcTransport->revisionIsDot4()) */ {
		SET_DATALEN(DOT4_LEN_CLOSE_CHANNEL_REPLY);
		data->closeChannelReply.dot4.psid=psid;
		data->closeChannelReply.dot4.ssid=ssid;

		LOG_INFO(cEXTP,0,
			"sendCloseChannelReply(port=%d,1284.4): result=%d, "
			"psid=%d, ssid=%d.\n",
			port,result,psid,ssid);
	}

	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::handleCloseChannelReply(CommandUnion *data,
    int datalen) {
	int result,psid=lastPsid,ssid=lastSsid;
	ExMlcTransportChannel *tcd;

	GET_RESULT;

	if (pMlcTransport->revisionIsMlc()) {
		LOG_INFO(cEXTP,0,
			"handleCloseChannelReply(port=%d,MLC): result=%d, "
			"(psid=%d, ssid=%d).\n",
			port,result,psid,ssid);

		CHECK_DATALEN(MLC_LEN_CLOSE_CHANNEL_REPLY);

	} else /* if (pMlcTransport->revisionIsDot4()) */ {
		psid=data->closeChannelReply.dot4.psid;
		ssid=data->closeChannelReply.dot4.ssid;

		LOG_INFO(cEXTP,0,
			"handleCloseChannelReply(port=%d,1284.4): result=%d, "
			"psid=%d, ssid=%d.\n",
			port,result,psid,ssid);

		CHECK_DATALEN(DOT4_LEN_CLOSE_CHANNEL_REPLY);
	}

	REPLY_LOOKUP_TCD("handleCloseChannelReply");

	tcd->handleCloseChannelReply(result,psid,ssid);
}

void ExMlcCommandChannel::sendCredit(int psid,int ssid,int credit) {
	DECLARE_VARIABLES;

	GET_EMPTY_BUFFER(CMD_CREDIT);

		data->credit.psid=data->options.psid=psid;
		data->credit.ssid=data->options.ssid=ssid;

		SET_DATALEN(LEN_CREDIT);
		BEND_SET_WORD(data->credit.credit,credit);

		LOG_INFO(cEXTP,0,
			"sendCredit(port=%d): "
			"psid=%d, ssid=%d, credit=%d.\n",
			port,psid,ssid,credit);

	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::handleCredit(CommandUnion *data,int datalen) {
	int psid,ssid,credit;
	ExMlcTransportChannel *tcd;

		psid=data->credit.psid;
		ssid=data->credit.ssid;
		credit=BEND_GET_WORD(data->credit.credit);

		LOG_INFO(cEXTP,0,
			"handleCredit(port=%d): "
			"psid=%d, ssid=%d, credit=%d.\n",
			port,psid,ssid,credit);

		CHECK_DATALEN(LEN_CREDIT);

	REQUEST_LOOKUP_TCD("handleCredit",
		ExMlcTransport::MLC_ERROR_UNOPENED_CHANNEL_CREDIT,
		sendCreditReply(
			ExMlcTransport::DOT4_RESULT_CHANNEL_NOT_OPEN,
			psid,ssid));

	tcd->handleCredit(psid,ssid,credit,0);
}

void ExMlcCommandChannel::sendCreditReply(int result,int psid,int ssid) {
	DECLARE_VARIABLES;

	GET_EMPTY_BUFFER(CMD_CREDIT_REPLY);
	SET_RESULT;

	if (pMlcTransport->revisionIsMlc()) {
		SET_DATALEN(MLC_LEN_CREDIT_REPLY);

		LOG_INFO(cEXTP,0,
			"sendCreditReply(port=%d,MLC): result=%d, "
			"(psid=%d, ssid=%d).\n",
			port,result,psid,ssid);

	} else /* if (pMlcTransport->revisionIsDot4()) */ {
		SET_DATALEN(DOT4_LEN_CREDIT_REPLY);
		data->creditReply.dot4.psid=psid;
		data->creditReply.dot4.ssid=ssid;

		LOG_INFO(cEXTP,0,
			"sendCreditReply(port=%d,1284.4): result=%d, "
			"psid=%d, ssid=%d.\n",
			port,result,psid,ssid);
	}

	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::handleCreditReply(CommandUnion *data,int datalen) {
	int result,psid=lastPsid,ssid=lastSsid;
	ExMlcTransportChannel *tcd;

	GET_RESULT;

	if (pMlcTransport->revisionIsMlc()) {
		LOG_INFO(cEXTP,0,
			"handleCreditReply(port=%d,MLC): result=%d, "
			"(psid=%d, ssid=%d).\n",
			port,result,psid,ssid);

		CHECK_DATALEN(MLC_LEN_CREDIT_REPLY);

	} else /* if (pMlcTransport->revisionIsDot4()) */ {
		psid=data->creditReply.dot4.psid;
		ssid=data->creditReply.dot4.ssid;

		LOG_INFO(cEXTP,0,
			"handleCreditReply(port=%d,1284.4): result=%d, "
			"psid=%d, ssid=%d.\n",
			port,result,psid,ssid);

		CHECK_DATALEN(DOT4_LEN_CREDIT_REPLY);
	}

	REPLY_LOOKUP_TCD("handleCreditReply");

	tcd->handleCreditReply(result,psid,ssid);
}

void ExMlcCommandChannel::sendCreditRequest(int psid,int ssid,
    int creditRequested,int maxOutstandingCredit) {
	DECLARE_VARIABLES;

	GET_EMPTY_BUFFER(CMD_CREDIT_REQUEST);

		SET_DATALEN(LEN_CREDIT_REQUEST);
		data->creditRequest.psid=data->options.psid=psid;
		data->creditRequest.ssid=data->options.ssid=ssid;

	if (pMlcTransport->revisionIsMlc()) {
		BEND_SET_WORD(data->creditRequest.mlc.creditRequested,
			creditRequested);

		LOG_INFO(cEXTP,0,
			"sendCreditRequest(port=%d,MLC): "
			"psid=%d, ssid=%d, creditRequested=%d.\n",
			port,psid,ssid,creditRequested);

	} else /* if (pMlcTransport->revisionIsDot4()) */ {
		BEND_SET_WORD(data->creditRequest.dot4.maxOutstandingCredit,
			maxOutstandingCredit);

		LOG_INFO(cEXTP,0,
			"sendCreditRequest(port=%d,1284.4): "
			"psid=%d, ssid=%d, maxOutstandingCredit=%d.\n",
			port,psid,ssid,maxOutstandingCredit);
	}

	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::handleCreditRequest(CommandUnion *data,int datalen) {
	int psid,ssid;
	int creditRequested=0,maxOutstandingCredit=0;
	ExMlcTransportChannel *tcd;

		psid=data->creditRequest.psid;
		ssid=data->creditRequest.ssid;

	if (pMlcTransport->revisionIsMlc()) {
		creditRequested=BEND_GET_WORD(
			data->creditRequest.mlc.creditRequested);

		LOG_INFO(cEXTP,0,
			"handleCreditRequest(port=%d,MLC): "
			"psid=%d, ssid=%d, creditRequested=%d.\n",
			port,psid,ssid,creditRequested);

	} else /* if (pMlcTransport->revisionIsDot4()) */ {
		maxOutstandingCredit=BEND_GET_WORD(
			data->creditRequest.dot4.maxOutstandingCredit);

		LOG_INFO(cEXTP,0,
			"handleCreditRequest(port=%d,1284.4): "
			"psid=%d, ssid=%d, maxOutstandingCredit=%d.\n",
			port,psid,ssid,maxOutstandingCredit);

	}
		CHECK_DATALEN(LEN_CREDIT_REQUEST);

	REQUEST_LOOKUP_TCD("handleCreditRequest",
		ExMlcTransport::MLC_ERROR_UNOPENED_CHANNEL_CREDIT,
		sendCreditRequestReply(
			ExMlcTransport::DOT4_RESULT_CHANNEL_NOT_OPEN,
			psid,ssid,0));

	tcd->handleCreditRequest(psid,ssid,
		creditRequested,maxOutstandingCredit);
}

void ExMlcCommandChannel::sendCreditRequestReply(int result,int psid,int ssid,
    int credit) {
	DECLARE_VARIABLES;

	GET_EMPTY_BUFFER(CMD_CREDIT_REQUEST_REPLY);
	SET_RESULT;

	if (pMlcTransport->revisionIsMlc()) {
		SET_DATALEN(MLC_LEN_CREDIT_REQUEST_REPLY);
		BEND_SET_WORD(data->creditRequestReply.mlc.credit,credit);

		LOG_INFO(cEXTP,0,
			"sendCreditRequestReply(port=%d,MLC): result=%d, "
			"(psid=%d, ssid=%d), credit=%d.\n",
			port,result,psid,ssid,credit);

	} else /* if (pMlcTransport->revisionIsDot4()) */ {
		SET_DATALEN(DOT4_LEN_CREDIT_REQUEST_REPLY);
		data->creditRequestReply.dot4.psid=psid;
		data->creditRequestReply.dot4.ssid=ssid;
		BEND_SET_WORD(data->creditRequestReply.dot4.credit,credit);

		LOG_INFO(cEXTP,0,
			"sendCreditRequestReply(port=%d,1284.4): result=%d, "
			"psid=%d, ssid=%d, credit=%d.\n",
			port,result,psid,ssid,credit);
	}

	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::handleCreditRequestReply(CommandUnion *data,
    int datalen) {
	int result,psid=lastPsid,ssid=lastSsid;
	int credit;
	ExMlcTransportChannel *tcd;

	GET_RESULT;

	if (pMlcTransport->revisionIsMlc()) {
		credit=BEND_GET_WORD(data->creditRequestReply.mlc.credit);

		LOG_INFO(cEXTP,0,
			"handleCreditRequestReply(port=%d,MLC): result=%d, "
			"(psid=%d, ssid=%d), credit=%d.\n",
			port,result,psid,ssid,credit);

		CHECK_DATALEN(MLC_LEN_CREDIT_REQUEST_REPLY);

	} else /* if (pMlcTransport->revisionIsDot4()) */ {
		psid=data->creditRequestReply.dot4.psid;
		ssid=data->creditRequestReply.dot4.ssid;
		credit=BEND_GET_WORD(data->creditRequestReply.dot4.credit);

		LOG_INFO(cEXTP,0,
			"handleCreditRequestReply(port=%d,1284.4): result=%d, "
			"psid=%d, ssid=%d, credit=%d.\n",
			port,result,psid,ssid,credit);

		CHECK_DATALEN(DOT4_LEN_CREDIT_REQUEST_REPLY);
	}

	REPLY_LOOKUP_TCD("handleCreditRequestReply");

	tcd->handleCreditRequestReply(result,psid,ssid,credit,0);
}

void ExMlcCommandChannel::sendConfigSocket(int socketID,
    int maxPriToSecPacketSize,int maxSecToPriPacketSize,
    int statusLevel) {
	DECLARE_VARIABLES;

	GET_EMPTY_BUFFER(CMD_CONFIG_SOCKET);
	data->options.psid=data->options.ssid=socketID;

		SET_DATALEN(LEN_CONFIG_SOCKET);
		data->configSocket.socketID=socketID;
		BEND_SET_WORD(data->configSocket.maxPriToSecPacketSize,
			maxPriToSecPacketSize);
		BEND_SET_WORD(data->configSocket.maxSecToPriPacketSize,
			maxSecToPriPacketSize);
		data->configSocket.statusLevel=statusLevel;

		LOG_INFO(cEXTP,0,
			"sendConfigSocket(port=%d): socketID=%d, "
			"maxPriToSecPacketSize=%d, maxSecToPriPacketSize=%d, "
			"statusLevel=%d.\n",
			port,socketID,
			maxPriToSecPacketSize,maxSecToPriPacketSize,
			statusLevel);

	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::handleConfigSocket(CommandUnion *data,int datalen) {
	int socketID;
	int maxPriToSecPacketSize,maxSecToPriPacketSize;
	int statusLevel;

		socketID=data->configSocket.socketID;
		maxPriToSecPacketSize=BEND_GET_WORD(
			data->configSocket.maxPriToSecPacketSize);
		maxSecToPriPacketSize=BEND_GET_WORD(
			data->configSocket.maxSecToPriPacketSize);
		statusLevel=data->configSocket.statusLevel;

		LOG_INFO(cEXTP,0,
			"handleConfigSocket(port=%d): socketID=%d, "
			"maxPriToSecPacketSize=%d, maxSecToPriPacketSize=%d, "
			"statusLevel=%d.\n",
			port,socketID,
			maxPriToSecPacketSize,maxSecToPriPacketSize,
			statusLevel);

		CHECK_DATALEN(LEN_CONFIG_SOCKET);

	pMlcTransport->handleConfigSocket(socketID,
		maxPriToSecPacketSize,maxSecToPriPacketSize,
		statusLevel);
}

void ExMlcCommandChannel::sendConfigSocketReply(int result,int socketID,
    int maxPriToSecPacketSize,int maxSecToPriPacketSize,
    int statusLevel) {
	DECLARE_VARIABLES;

	GET_EMPTY_BUFFER(CMD_CONFIG_SOCKET_REPLY);
	SET_RESULT;

		SET_DATALEN(LEN_CONFIG_SOCKET_REPLY);
		BEND_SET_WORD(data->configSocketReply.maxPriToSecPacketSize,
			maxPriToSecPacketSize);
		BEND_SET_WORD(data->configSocketReply.maxSecToPriPacketSize,
			maxSecToPriPacketSize);
		data->configSocketReply.statusLevel=statusLevel;

		LOG_INFO(cEXTP,0,
			"sendConfigSocketReply(port=%d): result=%d, "
			"(socketID=%d), "
			"maxPriToSecPacketSize=%d, maxSecToPriPacketSize=%d, "
			"statusLevel=%d.\n",
			port,result,socketID,
			maxPriToSecPacketSize,maxSecToPriPacketSize,
			statusLevel);

	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::handleConfigSocketReply(CommandUnion *data,
    int datalen) {
	int result,socketID=lastSsid;
	int maxPriToSecPacketSize,maxSecToPriPacketSize;
	int statusLevel;

	GET_RESULT;

		maxPriToSecPacketSize=BEND_GET_WORD(
			data->configSocketReply.maxPriToSecPacketSize);
		maxSecToPriPacketSize=BEND_GET_WORD(
			data->configSocketReply.maxSecToPriPacketSize);
		statusLevel=data->configSocketReply.statusLevel;

		LOG_INFO(cEXTP,0,
			"handleConfigSocketReply(port=%d): result=%d, "
			"(socketID=%d), "
			"maxPriToSecPacketSize=%d, maxSecToPriPacketSize=%d, "
			"statusLevel=%d.\n",
			port,result,socketID,
			maxPriToSecPacketSize,maxSecToPriPacketSize,
			statusLevel);

		CHECK_DATALEN(LEN_CONFIG_SOCKET_REPLY);

	pMlcTransport->handleConfigSocketReply(result,socketID,
		maxPriToSecPacketSize,maxSecToPriPacketSize,
		statusLevel);
}

void ExMlcCommandChannel::sendExit(void) {
	DECLARE_VARIABLES;

	GET_EMPTY_BUFFER(CMD_EXIT);

		SET_DATALEN(LEN_EXIT);

		LOG_INFO(cEXTP,0,
			"sendExit(port=%d).\n",
			port);

	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::handleExit(CommandUnion *data,int datalen) {
		CHECK_DATALEN(LEN_EXIT);

		LOG_INFO(cEXTP,0,
			"handleExit(port=%d).\n",
			port);

	pMlcTransport->handleExit();
}

void ExMlcCommandChannel::sendExitReply(int result) {
	DECLARE_VARIABLES;

	GET_EMPTY_BUFFER(CMD_EXIT_REPLY);
	data->options.closeAfterSend=ExMlcTransport::REASON_SENT_EXIT_REPLY;
	SET_RESULT;

		SET_DATALEN(LEN_EXIT_REPLY);

		LOG_INFO(cEXTP,0,
			"sendExitReply(port=%d): result=%d.\n",
			port,result);

	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::handleExitReply(CommandUnion *data,int datalen) {
	int result;

	GET_RESULT;

		LOG_INFO(cEXTP,0,
			"handleExitReply(port=%d): result=%d.\n",
			port,result);

		CHECK_DATALEN(LEN_EXIT_REPLY);

	pMlcTransport->handleExitReply(result);
}

void ExMlcCommandChannel::sendGetSocketID(char *serviceName) {
	DECLARE_VARIABLES;
	SETUP_LEN_SERVICE_NAME;

	GET_EMPTY_BUFFER(CMD_GET_SOCKET_ID);

		SET_DATALEN_PLUS_SERVICE_NAME(LEN_GET_SOCKET_ID);
		SET_SERVICE_NAME(data->getSocketID);

		LOG_INFO(cEXTP,0,
			"sendGetSocketID(port=%d): "
			"serviceName=<%s>.\n",
			port,serviceName);

	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::handleGetSocketID(CommandUnion *data,int datalen) {
	char *serviceName;

		GET_SERVICE_NAME(data->getSocketID);

		LOG_INFO(cEXTP,0,
			"handleGetSocketID(port=%d): "
			"serviceName=<%s>.\n",
			port,serviceName);

		if (datalen<LEN_GET_SOCKET_ID)
			CHECK_DATALEN(LEN_GET_SOCKET_ID);

	pMlcTransport->handleGetSocketID(serviceName);
}

void ExMlcCommandChannel::sendGetSocketIDReply(int result,
    int socketID,char *serviceName) {
	DECLARE_VARIABLES;
	SETUP_LEN_SERVICE_NAME_REPLY;

	GET_EMPTY_BUFFER(CMD_GET_SOCKET_ID_REPLY);
	SET_RESULT;

		SET_DATALEN_PLUS_SERVICE_NAME(LEN_GET_SOCKET_ID_REPLY);
		data->getSocketIDReply.socketID=socketID;
		SET_SERVICE_NAME(data->getSocketIDReply);

		LOG_INFO(cEXTP,0,
			"sendGetSocketIDReply(port=%d): result=%d, "
			"socketID=%d, serviceName=<%s>.\n",
			port,result,socketID,serviceName);

	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::handleGetSocketIDReply(CommandUnion *data,
    int datalen) {
	int result,socketID;
	char *serviceName;

	GET_RESULT;

		socketID=data->getSocketIDReply.socketID;
		GET_SERVICE_NAME(data->getSocketIDReply);

		LOG_INFO(cEXTP,0,
			"handleGetSocketIDReply(port=%d): result=%d, "
			"socketID=%d, serviceName=<%s>.\n",
			port,result,socketID,serviceName);

		if (datalen<LEN_GET_SOCKET_ID_REPLY)
			CHECK_DATALEN(LEN_GET_SOCKET_ID_REPLY);

	pMlcTransport->handleGetSocketIDReply(result,socketID,serviceName);
}

void ExMlcCommandChannel::sendGetServiceName(int socketID) {
	DECLARE_VARIABLES;

	GET_EMPTY_BUFFER(CMD_GET_SERVICE_NAME);

		SET_DATALEN(LEN_GET_SERVICE_NAME);
		data->getServiceName.socketID=socketID;

		LOG_INFO(cEXTP,0,
			"sendGetServiceName(port=%d): "
			"socketID=%d.\n",
			port,socketID);

	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::handleGetServiceName(CommandUnion *data,
    int datalen) {
	int socketID;

		socketID=data->getServiceName.socketID;

		LOG_INFO(cEXTP,0,
			"handleGetServiceName(port=%d): "
			"socketID=%d.\n",
			port,socketID);

		CHECK_DATALEN(LEN_GET_SERVICE_NAME);

	pMlcTransport->handleGetServiceName(socketID);
}

void ExMlcCommandChannel::sendGetServiceNameReply(int result,
    int socketID,char *serviceName) {
	DECLARE_VARIABLES;
	SETUP_LEN_SERVICE_NAME_REPLY;

	GET_EMPTY_BUFFER(CMD_GET_SERVICE_NAME_REPLY);
	SET_RESULT;

		SET_DATALEN_PLUS_SERVICE_NAME(LEN_GET_SERVICE_NAME_REPLY);
		data->getServiceNameReply.socketID=socketID;
		SET_SERVICE_NAME(data->getServiceNameReply);

		LOG_INFO(cEXTP,0,
			"sendGetServiceNameReply(port=%d): result=%d, "
			"socketID=%d, serviceName=<%s>.\n",
			port,result,socketID,serviceName);

	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::handleGetServiceNameReply(CommandUnion *data,
    int datalen) {
	int result,socketID;
	char *serviceName;

	GET_RESULT;

		socketID=data->getServiceNameReply.socketID;
		GET_SERVICE_NAME(data->getServiceNameReply);

		LOG_INFO(cEXTP,0,
			"handleGetServiceNameReply(port=%d): result=%d, "
			"socketID=%d, serviceName=<%s>.\n",
			port,result,socketID,serviceName);

		if (datalen<LEN_GET_SERVICE_NAME_REPLY)
			CHECK_DATALEN(LEN_GET_SERVICE_NAME_REPLY);

	pMlcTransport->handleGetServiceNameReply(result,socketID,serviceName);
}

void ExMlcCommandChannel::mlcSendError(int error) {
	DECLARE_VARIABLES;

	LOGD_ERROR(cEXTP,0,cCausePeriphError,
		"sendError(port=%d,MLC): "
		"error=0x%2.2X.\n",
		port,error);

	if (!allowErrorPackets) {
		LOG_INFO(cEXTP,0,
			"Error packets disabled on port=%d.\n",port);
		pMgr->exClose(ExMlcTransport::REASON_SENT_ERROR);
		return;
	}

	pMlcTransport->stopReverseData();

	GET_EMPTY_BUFFER(CMD_ERROR);
	data->options.consumeCredit=0;
	data->options.grantCredit=0;
	data->options.closeAfterSend=ExMlcTransport::REASON_SENT_ERROR;

		SET_DATALEN(MLC_LEN_ERROR);
		data->error.mlc.error=error;

	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::dot4SendError(int psid,int ssid,int error) {
	DECLARE_VARIABLES;

	LOGD_ERROR(cEXTP,0,cCausePeriphError,
		"sendError(port=%d,1284.4): "
		"psid=%d, ssid=%d, error=0x%2.2X.\n",
		port,psid,ssid,error);

	if (!allowErrorPackets) {
		LOG_INFO(cEXTP,0,
			"Error packets disabled on port=%d.\n",port);
		pMgr->exClose(ExMlcTransport::REASON_SENT_ERROR);
		return;
	}

	pMlcTransport->stopReverseData();

	GET_EMPTY_BUFFER(CMD_ERROR);
	data->options.consumeCredit=0;
	data->options.grantCredit=0;
	data->options.closeAfterSend=ExMlcTransport::REASON_SENT_ERROR;

		SET_DATALEN(DOT4_LEN_ERROR);
		data->error.dot4.psid=psid;
		data->error.dot4.ssid=ssid;
		data->error.dot4.error=error;

	PUT_FULL_BUFFER;
}

void ExMlcCommandChannel::sendError(int psid,int ssid,
    int mlcError,int dot4Error) {
	if (pMlcTransport->revisionIsMlc()) {
		mlcSendError(mlcError);
	} else /* if (pMlcTransport->revisionIsDot4()) */ {
		dot4SendError(psid,ssid,dot4Error);
	}
}

void ExMlcCommandChannel::handleError(CommandUnion *data,int datalen) {
	int psid=0,ssid=0,error;

	if (!pMlcTransport->revisionIsSet()) {
		if (datalen<DOT4_LEN_ERROR) goto mlcError;
		goto dot4Error;
	}

	if (pMlcTransport->revisionIsMlc()) {
mlcError:
		error=data->error.mlc.error;

		LOGD_ERROR(cEXTP,0,cCausePeriphError,
			"handleError(port=%d,MLC): "
			"error=0x%2.2X.\n",
			port,error);

		CHECK_DATALEN(MLC_LEN_ERROR);

	} else /* if (pMlcTransport->revisionIsDot4()) */ {
dot4Error:
		psid=data->error.dot4.psid;
		ssid=data->error.dot4.ssid;
		error=data->error.dot4.error;

		LOGD_ERROR(cEXTP,0,cCausePeriphError,
			"handleError(port=%d,1284.4): "
			"psid=%d, ssid=%d, error=0x%2.2X.\n",
			port,psid,ssid,error);

		CHECK_DATALEN(DOT4_LEN_ERROR);
	}

	pMlcTransport->handleError(psid,ssid,error);
}

void ExMlcCommandChannel::sendGenericReply(int command,int result) {
	DECLARE_VARIABLES;

	GET_EMPTY_BUFFER(command|REPLY_BIT);
	SET_RESULT;

		SET_DATALEN(LEN_REPLY);

		LOGD_ERROR(cEXTP,0,cCausePeriphError,
			"sendGenericReply(port=%d): "
			"command=0x%2.2X, result=%d.\n",
			port,command,result);

	PUT_FULL_BUFFER;
}
