/******************************************************************************

  Copyright(c) 2003 - 2006 Intel Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify it
  under the terms of version 2 of the GNU General Public License as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin Street, Fifth Floor, Boston, MA 02110, USA

  The full GNU General Public License is included in this distribution in the
  file called LICENSE.

  Contact Information:
  James P. Ketrenos <ipw2100-admin@linux.intel.com>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

******************************************************************************/

#ifndef __iwlwifi_h__
#define __iwlwifi_h__

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/version.h>
#include <linux/pci.h>
#include <linux/netdevice.h>
#include <linux/ethtool.h>
#include <linux/skbuff.h>
#include <linux/etherdevice.h>
#include <linux/delay.h>
#include <linux/random.h>

#include <linux/firmware.h>
#include <linux/wireless.h>
#include <asm/io.h>
#include <linux/workqueue.h>

#include <net/mac80211.h>
#include <linux/wireless.h>

#include <../net/mac80211/ieee80211_i.h>
#include <../net/mac80211/ieee80211_rate.h>

#include "iwlwifi_hw.h"

#define DRV_NAME	"iwlwifi"

/* Debug and printf string expansion helpers for printing bitfields */
#define BIT_FMT8 "%c%c%c%c-%c%c%c%c"
#define BIT_FMT16 BIT_FMT8 ":" BIT_FMT8
#define BIT_FMT32 BIT_FMT16 " " BIT_FMT16

#define BITC(x,y) (((x>>y)&1)?'1':'0')
#define BIT_ARG8(x) \
BITC(x,7),BITC(x,6),BITC(x,5),BITC(x,4),\
BITC(x,3),BITC(x,2),BITC(x,1),BITC(x,0)

#define BIT_ARG16(x) \
BITC(x,15),BITC(x,14),BITC(x,13),BITC(x,12),\
BITC(x,11),BITC(x,10),BITC(x,9),BITC(x,8),\
BIT_ARG8(x)

#define BIT_ARG32(x) \
BITC(x,31),BITC(x,30),BITC(x,29),BITC(x,28),\
BITC(x,27),BITC(x,26),BITC(x,25),BITC(x,24),\
BITC(x,23),BITC(x,22),BITC(x,21),BITC(x,20),\
BITC(x,19),BITC(x,18),BITC(x,17),BITC(x,16),\
BIT_ARG16(x)

#define WLAN_FC_GET_TYPE(fc)    (((fc) & IEEE80211_FCTL_FTYPE))
#define WLAN_FC_GET_STYPE(fc)   (((fc) & IEEE80211_FCTL_STYPE))
#define WLAN_GET_SEQ_FRAG(seq)  ((seq) & 0x000f)
#define WLAN_GET_SEQ_SEQ(seq)   ((seq) >> 4)

/*DSCAPE move to mac80211.h*/
#define IEEE80211_DATA_LEN              2304
#define IEEE80211_1ADDR_LEN 10
#define IEEE80211_2ADDR_LEN 16
#define IEEE80211_3ADDR_LEN 24
#define IEEE80211_4ADDR_LEN 30
#define IEEE80211_FCS_LEN    4
#define IEEE80211_HLEN                  (IEEE80211_4ADDR_LEN)
#define IEEE80211_FRAME_LEN             (IEEE80211_DATA_LEN + IEEE80211_HLEN)
#define IEEE80211_24GHZ_MIN_CHANNEL 1
#define IEEE80211_24GHZ_MAX_CHANNEL 14
#define IEEE80211_24GHZ_CHANNELS    14

#define IEEE80211_52GHZ_MIN_CHANNEL 34
#define IEEE80211_52GHZ_MAX_CHANNEL 165
#define IEEE80211_52GHZ_CHANNELS    131
//#define MAC_FMT "%02x:%02x:%02x:%02x:%02x:%02x"
//#define MAC_ARG(x) ((u8*)(x))[0],((u8*)(x))[1],((u8*)(x))[2],((u8*)(x))[3],((u8*)(x))[4],((u8*)(x))[5]

#define ieee80211chan2mhz(x) \
        (((x) <= 14) ? \
        (((x) == 14) ? 2484 : ((x) * 5) + 2407) : \
        ((x) + 1000) * 5)

#define QOS_CONTROL_LEN 2

/*DSCAPE-END*/

#ifdef CONFIG_IWLWIFI_DEBUG
#define IPW_DEBUG(level, fmt, args...) \
do { if (ipw_debug_level & (level)) \
  printk(KERN_DEBUG DRV_NAME": %c %s " fmt, \
         in_interrupt() ? 'I' : 'U', __FUNCTION__ , ## args); } while (0)
#else
#define IPW_DEBUG(level, fmt, args...) do {} while (0)
#endif				/* CONFIG_IWLWIFI_DEBUG */

/*
 * To use the debug system;
 *
 * If you are defining a new debug classification, simply add it to the #define
 * list here in the form of:
 *
 * #define IPW_DL_xxxx VALUE
 *
 * shifting value to the left one bit from the previous entry.  xxxx should be
 * the name of the classification (for example, WEP)
 *
 * You then need to either add a IPW_xxxx_DEBUG() macro definition for your
 * classification, or use IPW_DEBUG(IPW_DL_xxxx, ...) whenever you want
 * to send output to that classification.
 *
 * To add your debug level to the list of levels seen when you perform
 *
 * % cat /proc/net/ipw/debug_level
 *
 * you simply need to add your entry to the ipw_debug_levels array.
 *
 * If you do not see debug_level in /proc/net/ipw then you do not have
 * CONFIG_IWLWIFI_DEBUG defined in your kernel configuration
 *
 */

#define IPW_DL_INFO          (1<<2)
#define IPW_DL_WX            (1<<3)
#define IPW_DL_MAC80211        (1<<4)
#define IPW_DL_HOST_COMMAND  (1<<5)
#define IPW_DL_STATE         (1<<6)

#define IPW_DL_RADIO         (1<<7)
#define IPW_DL_POWER         (1<<8)

#define IPW_DL_NOTIF         (1<<10)
#define IPW_DL_SCAN          (1<<11)
#define IPW_DL_ASSOC         (1<<12)
#define IPW_DL_DROP          (1<<13)

#define IPW_DL_FW            (1<<16)
#define IPW_DL_RF_KILL       (1<<17)
#define IPW_DL_FW_ERRORS     (1<<18)

#define IPW_DL_LED           (1<<19)

#define IPW_DL_RATE          (1<<20)

#define IPW_DL_FRAG          (1<<21)
#define IPW_DL_WEP           (1<<22)
#define IPW_DL_TX            (1<<23)
#define IPW_DL_RX            (1<<24)
#define IPW_DL_ISR           (1<<25)
#define IPW_DL_IO            (1<<27)
#define IPW_DL_11H           (1<<28)

#define IPW_DL_STATS         (1<<29)
#define IPW_DL_MERGE         (1<<30)
#define IPW_DL_QOS           (1<<31)

#define IPW_ERROR(f, a...) printk(KERN_ERR DRV_NAME ": " f, ## a)
#define IPW_WARNING(f, a...) printk(KERN_ERR DRV_NAME ": " f, ## a)
#define IPW_DEBUG_INFO(f, a...)    IPW_DEBUG(IPW_DL_INFO, f, ## a)

#define IPW_DEBUG_MAC80211(f, a...)     IPW_DEBUG(IPW_DL_MAC80211, f, ## a)
#define IPW_DEBUG_WX(f, a...)     IPW_DEBUG(IPW_DL_WX, f, ## a)
#define IPW_DEBUG_SCAN(f, a...)   IPW_DEBUG(IPW_DL_SCAN, f, ## a)
#define IPW_DEBUG_RX(f, a...)     IPW_DEBUG(IPW_DL_RX, f, ## a)
#define IPW_DEBUG_TX(f, a...)     IPW_DEBUG(IPW_DL_TX, f, ## a)
#define IPW_DEBUG_ISR(f, a...)    IPW_DEBUG(IPW_DL_ISR, f, ## a)
#define IPW_DEBUG_LED(f, a...) IPW_DEBUG(IPW_DL_LED, f, ## a)
#define IPW_DEBUG_WEP(f, a...)    IPW_DEBUG(IPW_DL_WEP, f, ## a)
#define IPW_DEBUG_HC(f, a...) IPW_DEBUG(IPW_DL_HOST_COMMAND, f, ## a)
#define IPW_DEBUG_FRAG(f, a...) IPW_DEBUG(IPW_DL_FRAG, f, ## a)
#define IPW_DEBUG_FW(f, a...) IPW_DEBUG(IPW_DL_FW, f, ## a)
#define IPW_DEBUG_RF_KILL(f, a...) IPW_DEBUG(IPW_DL_RF_KILL, f, ## a)
#define IPW_DEBUG_DROP(f, a...) IPW_DEBUG(IPW_DL_DROP, f, ## a)
#define IPW_DEBUG_IO(f, a...) IPW_DEBUG(IPW_DL_IO, f, ## a)
#define IPW_DEBUG_RATE(f, a...) IPW_DEBUG(IPW_DL_RATE, f, ## a)
#define IPW_DEBUG_NOTIF(f, a...) IPW_DEBUG(IPW_DL_NOTIF, f, ## a)
#define IPW_DEBUG_ASSOC(f, a...) IPW_DEBUG(IPW_DL_ASSOC | IPW_DL_INFO, f, ## a)
#define IPW_DEBUG_STATS(f, a...) IPW_DEBUG(IPW_DL_STATS, f, ## a)
#define IPW_DEBUG_MERGE(f, a...) IPW_DEBUG(IPW_DL_MERGE, f, ## a)
#define IPW_DEBUG_QOS(f, a...)   IPW_DEBUG(IPW_DL_QOS, f, ## a)
#define IPW_DEBUG_RADIO(f, a...)  IPW_DEBUG(IPW_DL_RADIO, f, ## a)
#define IPW_DEBUG_POWER(f, a...)  IPW_DEBUG(IPW_DL_POWER, f, ## a)
#define IPW_DEBUG_11H(f, a...)  IPW_DEBUG(IPW_DL_11H, f, ## a)

/* Amount of time after sending an association request before we assume
 * the request has failed in ms (default 750ms)*/
#define ASSOCIATE_TIMEOUT 750

#define IPW_MIN_CHANNEL_24  1
#define IPW_MAX_CHANNEL_24  14
#define IPW_MIN_CHANNEL_52  34
#define IPW_MAX_CHANNEL_52  165
#define IPW_MAX_CHANNEL_MIX (IPW_MAX_CHANNEL_24 + IPW_MAX_CHANNEL_52)

enum ipw_antenna_type {
	ANT_NONE,
	ANT_MAIN,
	ANT_AUX,
	ANT_BOTH,
};

#define SEQ_TO_FIFO(x)  ((x >> 8) & 0xbf)
#define FIFO_TO_SEQ(x)  ((x & 0xbf) << 8)
#define SEQ_TO_INDEX(x) (x & 0xff)
#define INDEX_TO_SEQ(x) (x & 0xff)
#define SEQ_HUGE_FRAME  (0x4000)
#define SEQ_RX_FRAME    (0x8000)

enum {
	/* CMD_SIZE_NORMAL = 0, */
	CMD_SIZE_HUGE = (1 << 0),
	/* CMD_SYNC = 0, */
	CMD_ASYNC = (1 << 1),
	/* CMD_NO_SKB = 0, */
	CMD_WANT_SKB = (1 << 2),
	/* CMD_LOCK = 0, */
	CMD_NO_LOCK = (1 << 4),
};

struct ipw_cmd;
struct ipw_priv;

typedef int (*IPW_CALLBACK_FUNC) (struct ipw_priv * priv,
				  struct ipw_cmd * cmd, struct sk_buff * skb);

struct ipw_cmd_meta {
	union {
		struct ipw_cmd_meta *source;
		struct sk_buff *skb;
		IPW_CALLBACK_FUNC callback;
	} __attribute__ ((packed)) u;

	u16 len;

	/* The CMD_SIZE_HUGE flag bit indicates that the command
	 * structure is stored at the end of the shared queue memory. */
	u8 flags;

	u8 token;
} __attribute__ ((packed));

struct ipw_cmd {
	struct ipw_cmd_meta meta;
	struct ipw_cmd_header hdr;
	union {
		struct ipw_addsta_cmd addsta;
		struct ipw_led_cmd led;
		u32 flags;
		u8 val8;
		u16 val16;
		u32 val32;
		struct ipw_bt_cmd bt;
		struct ipw_rxon_time_cmd rxon_time;
		struct ipw_powertable_cmd powertable;
		struct ipw_qosparam_cmd qosparam;
		struct ipw_tx_cmd tx;
		struct ipw_key_cmd key;
		struct ipw_tx_beacon_cmd tx_beacon;
		struct ipw_rxon_assoc_cmd rxon_assoc;
		struct ipw_rate_scaling_cmd_specifics rate_scale;
		u8 *indirect;
		u8 payload[360];
	} __attribute__ ((packed)) cmd;
} __attribute__ ((packed));

struct ipw_host_cmd {
	u8 id;
	u16 len;
	struct ipw_cmd_meta meta;
	void *data;
};

#define MAX_NUM_OF_TBS          (20)
#define TFD_MAX_PAYLOAD_SIZE (sizeof(struct ipw_cmd) - \
                              sizeof(struct ipw_cmd_meta))
struct ipw_tx_info {
	struct ieee80211_tx_status status;
	struct sk_buff *skb[MAX_NUM_OF_TBS];
};

/**
 * Tx Queue for DMA. Queue consists of circular buffer of
 * BD's and required locking structures.
 */
struct ipw_queue {
	int n_bd;		       /**< number of BDs in this queue */
	int first_empty;	       /**< 1-st empty entry (index) */
	int last_used;		       /**< last used entry (index) */
	dma_addr_t dma_addr;		/**< physical addr for BD's */
	int n_window;
	u32 id;
	u32 element_size;
	int low_mark;		       /**< low watermark, resume queue if free space more than this */
	int high_mark;		       /**< high watermark, stop queue if free space less than this */
} __attribute__ ((packed));

struct ipw_tx_queue {
	struct ipw_queue q;
	u8 *bd;
	struct ipw_cmd *cmd;
	dma_addr_t dma_addr_cmd;
	struct ipw_tx_info *txb;
	int need_update;	/* flag to indicate we need to update read/write index */
};

/*
 * RX related structures and functions
 */
#define RX_FREE_BUFFERS 64
#define RX_LOW_WATERMARK 8

#define SUP_RATE_11A_MAX_NUM_CHANNELS  8
#define SUP_RATE_11B_MAX_NUM_CHANNELS  4
#define SUP_RATE_11G_MAX_NUM_CHANNELS  12

/* statistics command response */
#define IPW_TX_QUEUE_1        1
#define IPW_TX_QUEUE_2        2
#define IPW_TX_QUEUE_3        3
#define IPW_TX_QUEUE_4        4
#define IPW_MAX_NUM_QUEUES   16

#define IPW_RX_FRAME_SIZE        (4 + sizeof(struct ipw_rx_frame))

struct ipw_rx_mem_buffer {
	dma_addr_t dma_addr;
	struct sk_buff *skb;
	struct list_head list;
};

struct ipw_rx_queue {
	void *bd;
	dma_addr_t dma_addr;
	struct ipw_rx_mem_buffer pool[RX_QUEUE_SIZE + RX_FREE_BUFFERS];
	struct ipw_rx_mem_buffer *queue[RX_QUEUE_SIZE];
	u32 processed;		/* Internal index to last handled Rx packet */
	u32 read;		/* Shared index to newest available Rx buffer */
	u32 write;		/* Shared index to oldest written Rx packet */
	u32 free_count;		/* Number of pre-allocated buffers in rx_free */
	/* Each of these lists is used as a FIFO for ipw_rx_mem_buffers */
	struct list_head rx_free;	/* Own an SKBs */
	struct list_head rx_used;	/* No SKB allocated */
	int need_update;	/* flag to indicate we need to update read/write index */
	spinlock_t lock;
};

#define IPW_SUPPORTED_RATES_IE_LEN         8

#define SCAN_INTERVAL 100

#define MAX_A_CHANNELS  252
#define MIN_A_CHANNELS  7

#define MAX_B_CHANNELS  14
#define MIN_B_CHANNELS  1

#define STATUS_HCMD_ACTIVE      (1<<0)	/**< host command in progress */

#define STATUS_INT_ENABLED      (1<<1)
#define STATUS_RF_KILL_HW       (1<<2)
#define STATUS_RF_KILL_SW       (1<<3)
#define STATUS_RF_KILL_MASK     (STATUS_RF_KILL_HW | STATUS_RF_KILL_SW)

#define STATUS_INIT             (1<<4)
#define STATUS_ALIVE            (1<<5)
#define STATUS_READY            (1<<6)
#define STATUS_GEO_CONFIGURED   (1<<8)
#define STATUS_EXIT_PENDING     (1<<9)
#define STATUS_IN_SUSPEND       (1<<10)

#define STATUS_AUTH             (1<<13)

#define STATUS_DISASSOCIATING   (1<<15)

#define STATUS_ROAMING           (1<<16)
#define STATUS_SCANNING          (1<<17)
#define STATUS_SCAN_ABORTING     (1<<19)
#define STATUS_SCAN_PENDING      (1<<20)
#define STATUS_SCAN_HW           (1<<21)

#define STATUS_POWER_PMI        (1<<24)
#define STATUS_RESTRICTED       (1<<26)
#define STATUS_FW_ERROR         (1<<27)

#define STATUS_TX_MEASURE       (1<<28)

#define STATUS_SECURITY_UPDATED (1<<31)	/* Security sync needed */

#define CFG_STATIC_CHANNEL      (1<<0)	/* Restrict assoc. to single channel */
#define CFG_STATIC_ESSID        (1<<1)	/* Restrict assoc. to single SSID */
#define CFG_STATIC_BSSID        (1<<2)	/* Restrict assoc. to single BSSID */
#define CFG_CUSTOM_MAC          (1<<3)
#define CFG_PREAMBLE_LONG       (1<<4)
#define CFG_ADHOC_PERSIST       (1<<5)
#define CFG_ASSOCIATE           (1<<6)
#define CFG_FIXED_RATE          (1<<7)
#define CFG_ADHOC_CREATE        (1<<8)
#define CFG_NO_LED              (1<<9)
#define CFG_BACKGROUND_SCAN     (1<<10)
#define CFG_TXPOWER_LIMIT       (1<<11)
#define CFG_NO_ROAMING          (1<<12)
#define CFG_SHORT_SLOT          (1<<13)

#define CAP_SHARED_KEY          (1<<0)	/* Off = OPEN */
#define CAP_PRIVACY_ON          (1<<1)	/* Off = No privacy */
#define CAP_RF_HW_KILL          (1<<2)	/* Off = no HW rf kill support */
#define CAP_RF_SW_KILL          (1<<3)	/* Off = no HW rf kill support */

//todoG need to support adding adhoc station MAX_STATION should be 25
#define IPW_INVALID_STATION     (0xff)

#define MAX_TID_COUNT        6

#define IPW_INVALID_RATE     0xFF
#define IPW_INVALID_VALUE    -1
struct ipw_tid_data {
	u16 seq_number;
};

struct ipw_hw_key {
	ieee80211_key_alg alg;
	int keylen;
	u8 key[32];
};

union ipw_ht_rate_supp {
	u16 rates;
	struct {
		u8 siso_rate;
		u8 mimo_rate;
	};
};

struct ipw_station_entry {
	struct ipw_addsta_cmd sta;
	struct ipw_tid_data tid[MAX_TID_COUNT];
	union {
		struct {
			u8 rate;
			u8 flags;
		} s;
		u16 rate_n_flags;
	} current_rate;
	u8 used;
	struct ipw_hw_key keyinfo;
};

struct ipw_rate_info {
	u8 rate_plcp;
	u8 rate_ieee;
	s32 rate_scale_index;
	u32 bps;		/* Bits per symbol, only OFDM */
	u32 dur_ack;
	u32 dur_rts;
	u32 dur_cts;
	u32 dur_back;
};

#define AVG_ENTRIES 8
struct average {
	long entries[AVG_ENTRIES];
	long sum;
	u8 pos;
	u8 init;
};

#define IPW_LED_ACTIVITY                (1<<0)
#define IPW_LED_LINK                    (1<<1)

struct ipw_led {
	u8 time_on;		/* ON time in interval units - 0 == OFF */
	u8 time_off;		/* OFF time in interval units - 0 == always ON if
				 * time_on != 0 */
};

struct ipw_led_info {
	u32 interval;		/* uSec length of "interval" */
	struct ipw_led activity;
	struct ipw_led link;
	struct ipw_led tech;
};

struct ipw_shared_t {
	volatile u32 tx_base_ptr[8];
	volatile u32 rx_read_ptr[3];
};

struct fw_image_desc {
	void *v_addr;
	dma_addr_t p_addr;
	u32 len;
};

struct ipw_tpt_entry {
	s32 min_rssi;
	u32 no_protection_tpt;
	u32 cts_rts_tpt;
	u32 cts_to_self_tpt;
	s32 rate_scale_index;
};

struct ipw_rate_scale_data {
	u64 data;
	s32 success_counter;
	s32 success_ratio;
	s32 counter;
	s32 average_tpt;
	unsigned long stamp;
};

struct ipw_rate_scale_priv {
	struct ipw_rate_scale_data win[IPW_MAX_RATES];
};

struct ipw_lq_mngr {
	spinlock_t lock;
	s32 max_window_size;
	struct ipw_rate_scaling_cmd_specifics scale_rate_cmd;
	s32 *expected_tpt;
	u8 *next_higher_rate;
	u8 *next_lower_rate;
	unsigned long stamp;
	unsigned long stamp_last;
	u32 flush_time;
	u32 tx_packets;
};

#define IPW_IBSS_MAC_HASH_SIZE 31

struct ipw_ibss_seq {
	u8 mac[ETH_ALEN];
	u16 seq_num;
	u16 frag_num;
	unsigned long packet_time;
	struct list_head list;
};

/* Power management (not Tx power) structures */

struct ipw_power_vec_entry {
	struct ipw_powertable_cmd cmd;
	u8 no_dtim;
};
#define IPW_POWER_RANGE_0  (0)
#define IPW_POWER_RANGE_1  (1)

#define IPW_POWER_MODE_CAM           0x00	//(always on)
#define IPW_POWER_INDEX_3            0x03
#define IPW_POWER_INDEX_5            0x05
#define IPW_POWER_AC                 0x06
#define IPW_POWER_BATTERY            0x07
#define IPW_POWER_LIMIT              0x07
#define IPW_POWER_MASK               0x0F
#define IPW_POWER_ENABLED            0x10
#define IPW_POWER_LEVEL(x)           ((x) & IPW_POWER_MASK)

struct ipw_power_mgr {
	spinlock_t lock;
	struct ipw_power_vec_entry pwr_range_0[IPW_POWER_AC];
	struct ipw_power_vec_entry pwr_range_1[IPW_POWER_AC];
	u8 active_index;
	u32 dtim_val;
};

/* The LED interval is expressed in uSec and is the time
 * unit by which all other LED command are represented
 *
 * A value of '1000' for makes each unit 1ms.
 */

#define IPW_LED_INTERVAL 1000

#define DEFAULT_POWER_SAVE_ON       LED_SOLID_ON
#define DEFAULT_POWER_SAVE_OFF      0
#define DEFAULT_POWER_SAVE_INTERVAL 1000

struct ipw_activity_blink {
	u16 throughput;		/* threshold in Mbs */
	u8 off;			/* OFF time in interval units - 0 == always ON if
				 * time_on != 0 */
	u8 on;			/* ON time in interval units - 0 == OFF */
};

enum {
	IPW_LED_LINK_UNINIT = 0,
	IPW_LED_LINK_RADIOOFF,
	IPW_LED_LINK_UNASSOCIATED,
	IPW_LED_LINK_SCANNING,
	IPW_LED_LINK_ASSOCIATED,
	IPW_LED_LINK_ROAMING,
};

struct ipw_link_blink {
	u16 interval;		/* Number of interval units per second */
	u8 off;			/* OFF time in interval units - 0 == always ON if
				 * time_on != 0 */
	u8 on;			/* ON time in interval units - 0 == OFF */
};

struct ipw_frame {
	union {
		struct ieee80211_hdr frame;
		u8 raw[IEEE80211_FRAME_LEN];
		u8 cmd[360];
	} u;
	struct list_head list;
};

enum ipw_prom_filter {
	IPW_PROM_CTL_HEADER_ONLY = (1 << 0),
	IPW_PROM_MGMT_HEADER_ONLY = (1 << 1),
	IPW_PROM_DATA_HEADER_ONLY = (1 << 2),
	IPW_PROM_ALL_HEADER_ONLY = 0xf,	/* bits 0..3 */
	IPW_PROM_NO_TX = (1 << 4),
	IPW_PROM_NO_RX = (1 << 5),
	IPW_PROM_NO_CTL = (1 << 6),
	IPW_PROM_NO_MGMT = (1 << 7),
	IPW_PROM_NO_DATA = (1 << 8),
};

struct ipw_priv;
struct ipw_prom_priv {
	struct ipw_priv *priv;
	struct ieee80211_device *ieee;
	enum ipw_prom_filter filter;
	int tx_packets;
	int rx_packets;
};

/* Magic struct that slots into the radiotap header -- no reason
 * to build this manually element by element, we can write it much
 * more efficiently than we can parse it. ORDER MATTERS HERE
 *
 * When sent to us via the simulated Rx interface in sysfs, the entire
 * structure is provided regardless of any bits unset.
 */
struct ieee80211_radiotap_header {
	u8 it_version;		/* Version 0. Only increases
				 * for drastic changes,
				 * introduction of compatible
				 * new fields does not count.
				 */
	u8 it_pad;
	u16 it_len;		/* length of the whole
				 * header in bytes, including
				 * it_version, it_pad,
				 * it_len, and data fields.
				 */
	u32 it_present;		/* A bitmap telling which
				 * fields are present. Set bit 31
				 * (0x80000000) to extend the
				 * bitmap by another 32 bits.
				 * Additional extensions are made
				 * by setting bit 31.
				 */
};

struct ipw_rt_hdr {
	struct ieee80211_radiotap_header rt_hdr;
	u64 rt_tsf;		/* TSF */
	u8 rt_flags;		/* radiotap packet flags */
	u8 rt_rate;		/* rate in 500kb/s */
	u16 rt_channel;		/* channel in mhz */
	u16 rt_chbitmask;	/* channel bitfield */
	s8 rt_dbmsignal;	/* signal in dbM, kluged to signed */
	s8 rt_dbmnoise;
	u8 rt_antenna;		/* antenna number */
	u8 payload[0];		/* payload... */
} __attribute__ ((packed));

/* channel tx power info for normal tx and scan */

/* spectrum and channel data structures */
#define IPW_NUM_SCAN_RATES         (2)

#define IPW_SCAN_FLAG_24GHZ  (1<<0)
#define IPW_SCAN_FLAG_52GHZ  (1<<1)
#define IPW_SCAN_FLAG_ACTIVE (1<<2)
#define IPW_SCAN_FLAG_DIRECT (1<<3)

#define IPW_MAX_CMD_SIZE 1024

struct ipw_channel_tgd_info {
	u8 type;
	s8 max_power;
};

struct ipw_channel_tgh_info {
	s64 last_radar_time;
};

/* current Tx power values to use, one for each rate for each channel.
 * requested power is limited by:
 * -- regulatory EEPROM limits for this channel
 * -- hardware capabilities (clip-powers)
 * -- spectrum management
 * -- user preference (e.g. iwconfig)
 * when requested power is set, base power index must also be set. */
struct ipw_channel_power_info {
	struct ipw_tx_power tpc;	/* actual radio and DSP gain settings */
	s8 power_table_index;	/* actual (temp-comp'd) index into gain table */
	s8 base_power_index;	/* gain index for req. power at factory temp. */
	s8 requested_power;	/* power (dBm) requested for this chnl/rate */
};

/* current scan Tx power values to use, one for each scan rate for each channel.
 */
struct ipw_scan_power_info {
	struct ipw_tx_power tpc;	/* actual radio and DSP gain settings */
	s8 power_table_index;	/* actual (temp-comp'd) index into gain table */
	s8 requested_power;	/* scan pwr (dBm) requested for chnl/rate */
};

/* Channel unlock period is 15 seconds. If no beacon or probe response
 * has been received within 15 seconds on a locked channel then the channel
 * remains locked. */
#define TX_UNLOCK_PERIOD 15

/* CSA lock period is 15 seconds.  If a CSA has been received on a channel in
 * the last 15 seconds, the channel is locked */
#define CSA_LOCK_PERIOD 15
/*
 * One for each channel, holds all channel setup data
 * Some of the fields (e.g. eeprom and flags/max_power_avg) are redundant
 *     with one another!
 */
struct ipw_channel_info {
	struct ipw_channel_tgd_info tgd;
	struct ipw_channel_tgh_info tgh;
	struct ipw_eeprom_channel eeprom;	/* EEPROM regulatory limit */

	u8 channel;		/* channel number */
	u8 flags;		/* flags copied from EEPROM */
	s8 max_power_avg;	/* (dBm) regul. eeprom, normal Tx, any rate */
	s8 curr_txpow;		/* (dBm) regulatory/spectrum/user (not h/w) */
	s8 min_power;		/* always 0 */
	s8 scan_power;		/* (dBm) regul. eeprom, direct scans, any rate */

	u8 group_index;		/* 0-4, maps channel to group1/2/3/4/5 */
	u8 band_index;		/* 0-4, maps channel to band1/2/3/4/5 */
	u8 phymode;		/* MODE_IEEE80211{A,B,G} */

	u8 tx_locked;		/* 0 - Tx allowed.  1 - Tx disabled.
				 * Any channel requiring RADAR DETECT or PASSIVE ONLY
				 * is Tx locked until a beacon or probe response is
				 * received (for PASSIVE ONLY) */
	unsigned long rx_unlock;	/* For channels that are not tx_unlocked,
					 * this is the time (in seconds) of the last
					 * frame that will unlock this channel.  If
					 * more than 15s have passed then the channel
					 * is not unlocked. */
	unsigned long csa_received;	/* Time that the last CSA was received on
					 * this channel, or 0 if never.  A channel
					 * can only be scanned or used if no CSA
					 * has been received in the past 15s */

	/* Radio/DSP gain settings for each "normal" data Tx rate.
	 * These include, in addition to RF and DSP gain, a few fields for
	 *   remembering/modifying gain settings (indexes). */
	struct ipw_channel_power_info power_info[IPW_MAX_RATES];

	/* Radio/DSP gain settings for each scan rate, for directed scans. */
	struct ipw_scan_power_info scan_pwr_info[IPW_NUM_SCAN_RATES];
};

struct ipw_clip_group {
	/* maximum power level to prevent clipping for each rate, derived by
	 *   us from this band's saturation power in EEPROM */
	const s8 clip_powers[IPW_MAX_RATES];
};

struct ipw_driver_hw_info {

	u32 eeprom_size;
	u16 max_queue_number;
	u16 max_num_rate;
	u16 number_of_stations;
	u16 broadcast_id;
	u16 ac_queue_count;
	u8 cmd_queue_no;
	u8 start_cmd_queue;
	u32 rx_buffer_size;
	u32 max_inst_size;
	u32 max_data_size;
	u16 tx_cmd_len;
	int statistics_size;
	u16 max_rxq_size;
	u16 max_rxq_log;
	u16 rate_scale_size;
	u16 add_station_size;
	u32 cck_flag;
	void *shared_virt;
	dma_addr_t shared_phys;

	int (*nic_stop_master) (struct ipw_priv * priv);
	int (*nic_start) (struct ipw_priv * priv);
	int (*nic_init) (struct ipw_priv * priv);
	int (*nic_reset) (struct ipw_priv * priv);

	int (*queue_init) (struct ipw_priv * priv,
			   struct ipw_queue * q, int count, int size, u32 id);
	int (*attach_buffer_to_tfd_frame) (void *ptr, dma_addr_t addr, u16 len);
	int (*queue_tx_free_tfd) (struct ipw_priv * priv,
				  struct ipw_tx_queue * txq);

	int (*tx_queue_update_wr_ptr) (struct ipw_priv * priv,
				       struct ipw_tx_queue * txq,
				       int tx_id, u16 byte_cnt);
	int (*stop_tx_queue) (struct ipw_priv * priv);
	int (*tx_reset) (struct ipw_priv * priv);
	int (*queue_tx_init) (struct ipw_priv * priv,
			      struct ipw_tx_queue * q, int count, u32 id);

	int (*queue_tx_free) (struct ipw_priv * priv,
			      struct ipw_tx_queue * txq);

	int (*rx_queue_update_wr_ptr) (struct ipw_priv * priv,
				       struct ipw_rx_queue * q);
	int (*rxq_stop) (struct ipw_priv * priv);
	int (*rx_init) (struct ipw_priv * priv, struct ipw_rx_queue * rxq);

	int (*get_rx_read) (struct ipw_priv * priv);

	int (*alive_notify) (struct ipw_priv * priv);

	int (*download_ucode) (struct ipw_priv * priv,
			       struct fw_image_desc * desc,
			       u32 mem_size, dma_addr_t dst_addr);

	int (*init_rate_scale) (struct ipw_priv * priv,
				struct ipw_rate_scaling_info * table);

	int (*get_fw) (struct ipw_priv * priv, const struct firmware ** fw);

	int (*rx_cmd) (struct ipw_priv * priv,
		       struct ipw_rx_mem_buffer * rxb, int cmd);

	int (*download_ucode_base) (struct ipw_priv * priv,
				    u8 * image, u32 len);

	int (*handle_reply_tx) (struct ipw_priv * priv, void *ptr,
				struct ieee80211_tx_status * status);

	int (*get_beacon_cmd) (struct ipw_priv * priv,
			       struct ipw_frame * frame, u16 rate);

	int (*tx_cmd) (struct ipw_priv * priv, struct ipw_cmd * out_cmd,
		       u8 sta_id, dma_addr_t scratch_phys,
		       struct ieee80211_hdr * hdr, u8 hdr_len,
		       struct ieee80211_tx_control * ctrl);

	int (*get_temperature) (struct ipw_priv * priv);

	int (*init_hw_rates) (struct ipw_priv * priv,
			      struct ieee80211_rate * rates);

	int (*send_rxon_assoc) (struct ipw_priv * priv);
	int (*send_power_mode) (struct ipw_priv * priv, u32 mode);
	void (*irq_tasklet) (struct ipw_priv * priv);
};

#define IPW_CHANNEL_WIDTH_20MHZ   0
#define IPW_CHANNEL_WIDTH_40MHZ   1

#define IPW_MIMO_PS_AUTO          0
#define IPW_MIMO_PS_NONE          1
#define IPW_MIMO_PS_DYNAMIC       2
#define IPW_MIMO_PS_ALWAYS        3

#define IPW_OPERATION_MODE_AUTO     0
#define IPW_OPERATION_MODE_HT_ONLY  1
#define IPW_OPERATION_MODE_MIXED    2
#define IPW_OPERATION_MODE_20MHZ    3

#define IPW_EXT_CHANNEL_OFFSET_AUTO   0
#define IPW_EXT_CHANNEL_OFFSET_ABOVE  1
#define IPW_EXT_CHANNEL_OFFSET_       2
#define IPW_EXT_CHANNEL_OFFSET_BELOW  3
#define IPW_EXT_CHANNEL_OFFSET_MAX    4

#define STA_FLG_RTS_MIMO_PROT_POS	(17)
#define STA_FLG_RTS_MIMO_PROT_MSK	(1 << STA_FLG_RTS_MIMO_PROT_POS)
#define STA_FLG_AGG_MPDU_8US_POS	(18)
#define STA_FLG_AGG_MPDU_8US_MSK	(1 << STA_FLG_AGG_MPDU_8US_POS)
#define STA_FLG_MAX_AGG_SIZE_POS	(19)
#define STA_FLG_MAX_AGG_SIZE_MSK	(3 << STA_FLG_MAX_AGG_SIZE_POS)
#define STA_FLG_FAT_EN_POS		(21)
#define STA_FLG_FAT_EN_MSK		(1 << STA_FLG_FAT_EN_POS)
#define STA_FLG_MIMO_DIS_POS		(22)
#define STA_FLG_MIMO_DIS_MSK		(1 << STA_FLG_MIMO_DIS_POS)
#define STA_FLG_AGG_MPDU_DENSITY_POS	(23)
#define STA_FLG_AGG_MPDU_DENSITY_MSK	(7 << STA_FLG_AGG_MPDU_DENSITY_POS)
#define HT_SHORT_GI_20MHZ_ONLY          (1 << 0)
#define HT_SHORT_GI_40MHZ_ONLY          (1 << 1)

/* KDRN */
#define RATE_MCS_CODE_MSK 0x7
#define RATE_MCS_MIMO_POS 3
#define RATE_MCS_MIMO_MSK 0x8
#define RATE_MCS_HT_DUP_POS 5
#define RATE_MCS_HT_DUP_MSK 0x20
#define RATE_MCS_FLAGS_POS 8
#define RATE_MCS_HT_POS 8
#define RATE_MCS_HT_MSK 0x100
#define RATE_MCS_CCK_POS 9
#define RATE_MCS_CCK_MSK 0x200
#define RATE_MCS_GF_POS 10
#define RATE_MCS_GF_MSK 0x400

#define RATE_MCS_FAT_POS 11
#define RATE_MCS_FAT_MSK 0x800
#define RATE_MCS_DUP_POS 12
#define RATE_MCS_DUP_MSK 0x1000
#define RATE_MCS_SGI_POS 13
#define RATE_MCS_SGI_MSK 0x2000

#define IPW_QOS_NONE      0
#define IPW_QOS_11H       1
#define IPW_QOS_WMM       2

union ipw_qos_capability {
	struct {
		u8 edca_count:4;	/* bit 0-3 */
		u8 q_ack:1;	/* bit 4     */
		u8 queue_request:1;	/* bit 5 */
		u8 txop_request:1;	/* bit 6 */
		u8 reserved:1;	/* bit 7 */
	} q_AP;
	struct {
		u8 acvo_APSD:1;	/* bit 0     */
		u8 acvi_APSD:1;	/* bit 1     */
		u8 ac_bk_APSD:1;	/* bit 2     */
		u8 ac_be_APSD:1;	/* bit 3     */
		u8 q_ack:1;	/* bit 4     */
		u8 max_len:2;	/* bit 5-6   */
		u8 more_data_ack:1;	/* bit 7 */
	} q_STA;
	u8 val;
};

/* QoS structures */

struct ipw_qos_info {
	int qos_enable;
	int qos_active;
	union ipw_qos_capability qos_cap;
	struct ipw_qosparam_cmd def_qos_parm;
};

struct ipw_priv {
	/* ieee device used by generic ieee processing code */
	struct ieee80211_hw *ieee;
	struct ieee80211_channel *ieee_channels;
	struct ieee80211_rate *ieee_rates;
	/* temporary frame storage list */
	struct list_head free_frames;
	int frames_count;

	struct rate_control_ops rate_control;

	struct ieee80211_conf active_conf;

	const struct ieee80211_hw_mode *modes;

	/* spectrum measurement report caching */
	struct ipw_spectrum_notification measure_report;

	/* we allocate array of ipw_channel_info for NIC's valid channels.
	 *    Access via channel # using indirect index array */
	struct ipw_channel_info *channel_info;	/* channel info array */
	u8 channel_count;	/* # of channels */

	/* each calibration channel group in the EEPROM has a derived
	 * clip setting for each rate. */
	const struct ipw_clip_group clip_groups[5];

	/* thermal calibration */
	int curr_temperature;
	int last_temperature;

	/* Scan related variables */
	u8 scan_band;
	unsigned long last_scan_jiffies;
	unsigned long scan_start;
	unsigned long scan_pass_start;
	unsigned long scan_start_tsf;

	int scan_bands;

	int one_direct_scan;
	u8 direct_ssid_len;
	u8 direct_ssid[IW_ESSID_MAX_SIZE];
	struct ipw_scan_cmd *scan;

	u8 only_active_channel;
	/* spinlock */
	spinlock_t lock;
	struct mutex mutex;

	/* basic pci-network driver stuff */
	struct pci_dev *pci_dev;
	struct net_device *net_dev;

	/* Promiscuous mode */
	struct ipw_prom_priv *prom_priv;
	struct net_device *prom_net_dev;

	/* pci hardware address support */
	void __iomem *hw_base;
	unsigned long hw_len;

	struct fw_image_desc ucode_boot;
	struct fw_image_desc ucode_boot_data;
	struct fw_image_desc ucode_code;
	struct fw_image_desc ucode_data;

	struct ipw_rxon_time_cmd rxon_timing;

	struct ipw_rxon_cmd active_rxon;
	struct ipw_rxon_cmd staging_rxon;

	struct ipw_alive_resp card_alive;

	/* LED related variables */
	struct ipw_activity_blink activity;
	unsigned long led_packets;
	int led_state;
	u16 active_rate;
	u16 active_rate_basic;

	/* HT variables */
	u8 is_dup;
	u8 is_ht_enabled;
	u8 channel_width;	/* 0=20MHZ, 1=40MHZ */
	u8 operation_mode;
	u8 mimo_ps_mode;
	u8 ext_channel_offset;
	u8 active_rate_ht[2];

	u8 last_phy_res[100];

	/* Rate scaling data */
	struct ipw_lq_mngr lq_mngr;
	s8 data_retry_limit;
	u8 retry_rate;

	wait_queue_head_t wait_command_queue;

	int activity_timer_active;

	/* Rx and Tx DMA processing queues */
	struct ipw_rx_queue *rxq;
	struct ipw_tx_queue txq[IPW_MAX_NUM_QUEUES];
	u32 status;
	u32 config;
	u32 capability;

	u32 port_type;
	u32 missed_beacon_threshold;
	u32 roaming_threshold;

	struct ipw_power_mgr power_data;

	union {
		struct ipw_notif_statistics ipw3945;
		u8 payload[500];
	} statistics;

	unsigned long last_statistics_time;
	/* context information */
	u8 essid[IW_ESSID_MAX_SIZE];
	u8 essid_len;
	u8 nick[IW_ESSID_MAX_SIZE];
	u16 rates_mask;

	u32 power_mode;
	u32 antenna;
	u8 bssid[ETH_ALEN];
	u16 rts_threshold;
	u8 mac_addr[ETH_ALEN];
	u32 curr_channel_flag;

	/*station table variables */
	spinlock_t sta_lock;
	u8 num_stations;
	struct ipw_station_entry stations[IPW3945_NUM_OF_STATIONS];

	u8 netdev_registered;
	int is_abg;

	u32 notif_missed_beacons;

	/* Wireless statistics */
	unsigned long last_rx_jiffies;
	u32 last_beacon_time;
	u64 last_tsf;
	s32 last_rx_rssi;
	u16 last_noise;
	struct average average_missed_beacons;
	struct average average_rssi;
	struct average average_noise;

	/* Statistics and counters normalized with each association */
	u32 last_missed_beacons;
	u32 last_tx_packets;
	u32 last_rx_packets;
	u32 last_tx_failures;
	u32 last_rx_err;
	u32 last_rate;

	u32 missed_beacons;
	unsigned long rx_packets;
	unsigned long tx_packets;
	unsigned long long rx_bytes;
	unsigned long long tx_bytes;
	u32 quality;

	/* Duplicate packet detection */
	u16 last_seq_num;
	u16 last_frag_num;
	unsigned long last_packet_time;
	struct list_head ibss_mac_hash[IPW_IBSS_MAC_HASH_SIZE];

	/* eeprom */
	struct ipw_eeprom eeprom;

	struct iw_statistics wstats;

	int freq_band;		/* 2.4Ghz, 5.2Ghz, Mixed */
	int iw_mode;
	int perfect_rssi;
	int worst_rssi;
	struct sk_buff *ibss_beacon;
	unsigned long last_scan;
	u32 timestamp0;
	u32 timestamp1;

	struct ipw_driver_hw_info hw_setting;
	u8 is_3945;
	int interface_id;
	u16 assoc_id;
	u16 assoc_capability;

	/*DSCAPE-END */
	/* Driver and iwconfig driven work queue */
	struct workqueue_struct *workqueue;

	struct work_struct up;
	struct work_struct down;
	struct work_struct calibrated_work;
	struct work_struct scan_completed;
	struct work_struct rx_replenish;
	struct work_struct rf_kill;
	struct work_struct abort_scan;
	struct work_struct link_up;
	struct work_struct update_link_led;
	struct work_struct link_down;
	struct work_struct associate_timeout;
	struct work_struct auth_work;
	struct work_struct post_associate;
	struct work_struct report_work;

	struct timer_list rate_scale_flush;

	struct tasklet_struct irq_tasklet;

	struct delayed_work alive_start;
	struct delayed_work activity_timer;
	struct delayed_work request_scan;
	struct delayed_work thermal_periodic;
	struct delayed_work gather_stats;
	struct delayed_work scan_check;
	struct delayed_work resume_work;

#define IPW_DEFAULT_TX_POWER 0x0F
	s8 user_txpower_limit;
	s8 max_channel_txpower_limit;

#ifdef CONFIG_PM
	u32 pm_state[16];
#endif

	/* Used to pass the current INTA value from ISR to Tasklet */
	u32 isr_inta;

	/* QoS */
	struct ipw_qos_info qos_data;
	struct work_struct qos_activate;
	/*********************************/

	/* debugging info */
	u32 framecnt_to_us;
};				/*ipw_priv */

/* debug macros */

#include <linux/ctype.h>

/*
* Register bit definitions
*/

/* NIC type as found in the one byte EEPROM_NIC_TYPE  offset*/

/* Defines a single bit in a by bit number (0-31) */

/* Interrupts masks */
enum {
	IPW_FW_ERROR_OK = 0,
	IPW_FW_ERROR_FAIL,
	IPW_FW_ERROR_MEMORY_UNDERFLOW,
	IPW_FW_ERROR_MEMORY_OVERFLOW,
	IPW_FW_ERROR_BAD_PARAM,
	IPW_FW_ERROR_BAD_CHECKSUM,
	IPW_FW_ERROR_NMI_INTERRUPT,
	IPW_FW_ERROR_BAD_DATABASE,
	IPW_FW_ERROR_ALLOC_FAIL,
	IPW_FW_ERROR_DMA_UNDERRUN,
	IPW_FW_ERROR_DMA_STATUS,
	IPW_FW_ERROR_DINO_ERROR,
	IPW_FW_ERROR_EEPROM_ERROR,
	IPW_FW_ERROR_SYSASSERT,
	IPW_FW_ERROR_FATAL_ERROR
};

#define AUTH_OPEN       0
#define AUTH_SHARED_KEY 1

#define HC_ASSOCIATE      0
#define HC_REASSOCIATE    1
#define HC_DISASSOCIATE   2
#define HC_IBSS_START     3
#define HC_DISASSOC_QUIET 5

#define HC_QOS_SUPPORT_ASSOC  0x01

#define IPW_RATE_CAPABILITIES 1
#define IPW_RATE_CONNECT      0

/*
 * Rate values and masks
 */
#define IPW_TX_RATE_1MB  0x0A
#define IPW_TX_RATE_2MB  0x14
#define IPW_TX_RATE_5MB  0x37
#define IPW_TX_RATE_6MB  0x0D
#define IPW_TX_RATE_9MB  0x0F
#define IPW_TX_RATE_11MB 0x6E
#define IPW_TX_RATE_12MB 0x05
#define IPW_TX_RATE_18MB 0x07
#define IPW_TX_RATE_24MB 0x09
#define IPW_TX_RATE_36MB 0x0B
#define IPW_TX_RATE_48MB 0x01
#define IPW_TX_RATE_54MB 0x03

#define IPW_MIN_RSSI_VAL                 -100
#define IPW_MAX_RSSI_VAL                    0
#define IPW_RSSI_OFFSET                    95
#define IPW_RATE_SCALE_FLUSH          (3*HZ/10)	//300 milli
#define IPW_RATE_SCALE_WIN_FLUSH      (HZ/2)	//500 milli
#define IPW_RATE_SCALE_HIGH_TH          11520
#define IPW_RATE_SCALE_MIN_FAILURE_TH       8
#define IPW_RATE_SCALE_MIN_SUCCESS_TH       8
#define IPW_RATE_SCALE_DECREASE_TH       1920

#define IEEE_A            (1<<0)
#define IEEE_B            (1<<1)
#define IEEE_G            (1<<2)
#define IEEE_MODE_MASK    (IEEE_A|IEEE_B|IEEE_G)

#define IEEE80211_24GHZ_BAND     (1<<0)
#define IEEE80211_52GHZ_BAND     (1<<1)

#define IPW3945_RX_BUF_SIZE 3000

#endif				/* __iwlwifi_h__ */
