/*
	Copyright (C) 2004 - 2007 rt2x00 SourceForge Project
	<http://rt2x00.serialmonkey.com>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
	Module: rt73usb
	Abstract: rt73usb device specific routines.
	Supported chipsets: RT2573.
 */

/*
 * Set enviroment defines for rt2x00.h
 */
#define DRV_NAME "rt73usb"

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/usb.h>
#include <linux/delay.h>
#include <linux/etherdevice.h>
#include <linux/firmware.h>
#include <linux/crc-itu-t.h>

#include "rt2x00.h"
#include "rt2x00lib.h"
#include "rt2x00usb.h"
#include "rt73usb.h"

/*
 * Register access.
 * All access to the CSR registers will go through the methods
 * rt2x00_register_read and rt2x00_register_write.
 * BBP and RF register require indirect register access,
 * and use the CSR registers BBPCSR and RFCSR to achieve this.
 * These indirect registers work with busy bits,
 * and we will try maximal REGISTER_BUSY_COUNT times to access
 * the register while taking a REGISTER_BUSY_DELAY us delay
 * between each attampt. When the busy bit is still set at that time,
 * the access attempt is considered to have failed,
 * and we will print an error.
 */
static int rt2x00_vendor_request(const struct rt2x00_dev *rt2x00dev,
	const u8 request, const u8 type, const u16 offset,
	u32 value, void *buffer, const u16 buffer_length, const u16 timeout)
{
	struct usb_device *usb_dev = interface_to_usbdev(
		rt2x00dev_usb(rt2x00dev));
	int status;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		status = usb_control_msg(
			usb_dev,
			(type == USB_VENDOR_REQUEST_IN) ?
				usb_rcvctrlpipe(usb_dev, 0) :
				usb_sndctrlpipe(usb_dev, 0),
			request, type, value, offset, buffer, buffer_length,
			timeout);
		if (status >= 0)
			return 0;
	}

	ERROR("vendor request error. Request 0x%02x failed "
		"for offset 0x%04x with error %d.\n", request, offset, status);

	return status;
}

static inline void rt2x00_register_read(
	const struct rt2x00_dev *rt2x00dev,
	const u16 offset, u32 *value)
{
	__le32 reg;
	rt2x00_vendor_request(
		rt2x00dev, USB_MULTI_READ, USB_VENDOR_REQUEST_IN,
		offset, 0x00, &reg, 4, REGISTER_TIMEOUT);
	*value = le32_to_cpu(reg);
}

static inline void rt2x00_register_multiread(
	const struct rt2x00_dev *rt2x00dev,
	const u16 offset, u32 *value, const u32 length)
{
	rt2x00_vendor_request(
		rt2x00dev, USB_MULTI_READ, USB_VENDOR_REQUEST_IN,
		offset, 0x00, value, length,
		REGISTER_TIMEOUT * (length / sizeof(u32)));
}

static inline void rt2x00_register_write(
	const struct rt2x00_dev *rt2x00dev,
	const u16 offset, u32 value)
{
	__le32 reg = cpu_to_le32(value);
	rt2x00_vendor_request(
		rt2x00dev, USB_MULTI_WRITE, USB_VENDOR_REQUEST_OUT,
		offset, 0x00, &reg, 4, REGISTER_TIMEOUT);
}

static inline void rt2x00_register_multiwrite(
	const struct rt2x00_dev *rt2x00dev,
	const u16 offset, u32 *value, const u32 length)
{
	rt2x00_vendor_request(
		rt2x00dev, USB_MULTI_WRITE, USB_VENDOR_REQUEST_OUT,
		offset, 0x00, value, length,
		REGISTER_TIMEOUT * (length / sizeof(u32)));
}

static u32 rt2x00_bbp_check(const struct rt2x00_dev *rt2x00dev)
{
	u32 reg;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00dev, PHY_CSR3, &reg);
		if (!rt2x00_get_field32(reg, PHY_CSR3_BUSY))
			return reg;
		udelay(REGISTER_BUSY_DELAY);
	}

	return 0xffff;
}

static void rt2x00_bbp_write(const struct rt2x00_dev *rt2x00dev,
	const u8 reg_id, const u8 value)
{
	u32 reg;

	/*
	 *  Wait until the BBP becomes ready.
	 */
	if (rt2x00_bbp_check(rt2x00dev) == 0xffff) {
		ERROR("PHY_CSR3 register busy. Write failed.\n");
		return;
	}

	/*
	 * Write the data into the BBP.
	 */
	reg = 0;
	rt2x00_set_field32(&reg, PHY_CSR3_VALUE, value);
	rt2x00_set_field32(&reg, PHY_CSR3_REGNUM, reg_id);
	rt2x00_set_field32(&reg, PHY_CSR3_BUSY, 1);
	rt2x00_set_field32(&reg, PHY_CSR3_READ_CONTROL, 0);

	rt2x00_register_write(rt2x00dev, PHY_CSR3, reg);
}

static void rt2x00_bbp_read(const struct rt2x00_dev *rt2x00dev,
	const u8 reg_id, u8 *value)
{
	u32 reg;

	/*
	 *  Wait until the BBP becomes ready.
	 */
	if (rt2x00_bbp_check(rt2x00dev) == 0xffff) {
		ERROR("PHY_CSR3 register busy. Read failed.\n");
		return;
	}

	/*
	 * Write the request into the BBP.
	 */
	reg =0;
	rt2x00_set_field32(&reg, PHY_CSR3_REGNUM, reg_id);
	rt2x00_set_field32(&reg, PHY_CSR3_BUSY, 1);
	rt2x00_set_field32(&reg, PHY_CSR3_READ_CONTROL, 1);

	rt2x00_register_write(rt2x00dev, PHY_CSR3, reg);

	/*
	 *  Wait until the BBP becomes ready.
	 */
	reg = rt2x00_bbp_check(rt2x00dev);
	if (reg == 0xffff)
		ERROR("PHY_CSR3 register busy. Read failed.\n");

	*value = rt2x00_get_field32(reg, PHY_CSR3_VALUE);
}

static void rt2x00_rf_write(const struct rt2x00_dev *rt2x00dev,
	const u32 value)
{
	u32 reg;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00dev, PHY_CSR4, &reg);
		if (!rt2x00_get_field32(reg, PHY_CSR4_BUSY))
			goto rf_write;
		udelay(REGISTER_BUSY_DELAY);
	}

	ERROR("PHY_CSR4 register busy. Write failed.\n");
	return;

rf_write:
	reg = 0;
	rt2x00_set_field32(&reg, PHY_CSR4_VALUE, value);
	rt2x00_set_field32(&reg, PHY_CSR4_NUMBER_OF_BITS, 20);
	rt2x00_set_field32(&reg, PHY_CSR4_IF_SELECT, 0);
	rt2x00_set_field32(&reg, PHY_CSR4_BUSY, 1);

	rt2x00_register_write(rt2x00dev, PHY_CSR4, reg);
}

#if defined(CONFIG_RT2X00_DEBUGFS) || defined(CONFIG_RT2X00_DEBUGFS_MODULE)
#include "rt2x00debug.h"

#define CSR_OFFSET(__word)	( CSR_REG_BASE + ((__word) * sizeof(u32)) )

static void rt73usb_read_csr(struct rt2x00_dev *rt2x00dev,
	const unsigned long word, void *data)
{
	rt2x00_register_read(rt2x00dev, CSR_OFFSET(word), data);
}

static void rt73usb_write_csr(struct rt2x00_dev *rt2x00dev,
	const unsigned long word, void *data)
{
	rt2x00_register_write(rt2x00dev, CSR_OFFSET(word), *((u32*)data));
}

static void rt73usb_read_eeprom(struct rt2x00_dev *rt2x00dev,
	const unsigned long word, void *data)
{
	rt2x00_eeprom_read(rt2x00dev, word, (u16*)data);
}

static void rt73usb_write_eeprom(struct rt2x00_dev *rt2x00dev,
	const unsigned long word, void *data)
{
	rt2x00_eeprom_write(rt2x00dev, word, *(u16*)data);
}

static void rt73usb_read_bbp(struct rt2x00_dev *rt2x00dev,
	const unsigned long word, void *data)
{
	rt2x00_bbp_read(rt2x00dev, word, ((u8*)data));
}

static void rt73usb_write_bbp(struct rt2x00_dev *rt2x00dev,
	const unsigned long word, void *data)
{
	rt2x00_bbp_write(rt2x00dev, word, *((u8*)data));
}

static struct rt2x00debug rt73usb_rt2x00debug = {
	.owner 		= THIS_MODULE,
	.mod_name	= DRV_NAME,
	.mod_version	= DRV_VERSION,
	.reg_csr	= {
		.read		= rt73usb_read_csr,
		.write		= rt73usb_write_csr,
		.word_size	= sizeof(u32),
		.length		= CSR_REG_SIZE,
	},
	.reg_eeprom	= {
		.read		= rt73usb_read_eeprom,
		.write		= rt73usb_write_eeprom,
		.word_size	= sizeof(u16),
		.length		= EEPROM_SIZE,
	},
	.reg_bbp	= {
		.read		= rt73usb_read_bbp,
		.write		= rt73usb_write_bbp,
		.word_size	= sizeof(u8),
		.length		= BBP_SIZE,
	},
};

static void rt73usb_open_debugfs(struct rt2x00_dev *rt2x00dev)
{
	if (!rt2x00debug_register(&rt73usb_rt2x00debug,
		rt2x00dev->hw->wiphy, rt2x00dev))
		ERROR("Failed to register debug handler.\n");
}

static void rt73usb_close_debugfs(struct rt2x00_dev *rt2x00dev)
{
	rt2x00debug_deregister(&rt2x00dev->debugfs_data);
}
#else /* CONFIG_RT2X00_DEBUGFS */
static inline void rt73usb_open_debugfs(struct rt2x00_dev *rt2x00dev){}
static inline void rt73usb_close_debugfs(struct rt2x00_dev *rt2x00dev){}
#endif /* CONFIG_RT2X00_DEBUGFS */

/*
 * Configuration handlers.
 */
static void rt73usb_config_bssid(struct rt2x00_dev *rt2x00dev, u8 *bssid)
{
	u32 reg;

	/*
	 * The BSSID is passed to us as an array of bytes,
	 * that array is little endian, so no need for byte ordering.
	 * We only need to set the BSS ID MASK at the correct offset.
	 */
	rt2x00_register_multiwrite(rt2x00dev, MAC_CSR4, (u32*)bssid, ETH_ALEN);

	rt2x00_register_read(rt2x00dev, MAC_CSR5, &reg);
	rt2x00_set_field32(&reg, MAC_CSR5_BSS_ID_MASK, 3);
	rt2x00_register_write(rt2x00dev, MAC_CSR5, reg);
}

static void rt73usb_config_promisc(struct rt2x00_dev *rt2x00dev, int promisc)
{
	u32 reg;

	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);

	if (promisc) {
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_NOT_TO_ME, 0);
		SET_FLAG(rt2x00dev, INTERFACE_ENABLED_PROMISC);
	} else {
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_NOT_TO_ME, 1);
		CLEAR_FLAG(rt2x00dev, INTERFACE_ENABLED_PROMISC);
	}

	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);
}

static void rt73usb_config_type(struct rt2x00_dev *rt2x00dev, int type)
{
	u32 reg;

	/*
	 * Only continue when there is something to be done.
	 */
	if (!(GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED) ^
	      GET_FLAG(rt2x00dev, INTERFACE_ENABLED)) &&
	    !(GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) ^
	      GET_FLAG(rt2x00dev, INTERFACE_ENABLED_MONITOR)))
		return;

	rt2x00_register_write(rt2x00dev, TXRX_CSR9, 0);

	/*
	 * Apply hardware packet filter.
	 */
	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);

	if (!GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) &&
	    (type == IEEE80211_IF_TYPE_IBSS || type == IEEE80211_IF_TYPE_STA))
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_TO_DS, 1);
	else
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_TO_DS, 0);

	rt2x00_set_field32(&reg, TXRX_CSR0_DROP_CRC, 1);
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR)) {
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_PHYSICAL, 0);
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_CONTROL, 0);
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_VERSION_ERROR, 0);
	} else {
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_PHYSICAL, 1);
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_CONTROL, 1);
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_VERSION_ERROR, 1);
	}

	rt2x00_set_field32(&reg, TXRX_CSR0_DROP_MULTICAST, 0);
	rt2x00_set_field32(&reg, TXRX_CSR0_DROP_BORADCAST, 0);

	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);

	/*
	 * Enable promisc mode when in monitor mode.
	 */
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR))
		rt73usb_config_promisc(rt2x00dev, 1);

	/*
	 * Enable synchronisation.
	 */
	rt2x00_register_read(rt2x00dev, TXRX_CSR9, &reg);
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED)) {
		rt2x00_set_field32(&reg, TXRX_CSR9_BEACON_INTERVAL, 100 * 16);
		rt2x00_set_field32(&reg, TXRX_CSR9_TSF_TICKING, 1);
		rt2x00_set_field32(&reg, TXRX_CSR9_TBTT_ENABLE, 1);
	}

	rt2x00_set_field32(&reg, TXRX_CSR9_BEACON_GEN, 0);
	if (type == IEEE80211_IF_TYPE_IBSS || type == IEEE80211_IF_TYPE_AP)
		rt2x00_set_field32(&reg, TXRX_CSR9_TSF_SYNC, 2);
	else if (type == IEEE80211_IF_TYPE_STA)
		rt2x00_set_field32(&reg, TXRX_CSR9_TSF_SYNC, 1);
	else if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) &&
		 !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
		rt2x00_set_field32(&reg, TXRX_CSR9_TSF_SYNC, 0);

	rt2x00_register_write(rt2x00dev, TXRX_CSR9, reg);

	/*
	 * Change flags of enabled interfaces.
	 */
	if (type != IEEE80211_IF_TYPE_MNTR) {
		if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
			SET_FLAG(rt2x00dev, INTERFACE_ENABLED);
		else
			CLEAR_FLAG(rt2x00dev, INTERFACE_ENABLED);
	} else {
		if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR))
			SET_FLAG(rt2x00dev, INTERFACE_ENABLED_MONITOR);
		else
			CLEAR_FLAG(rt2x00dev, INTERFACE_ENABLED_MONITOR);
	}
}

static void rt73usb_config_channel(struct rt2x00_dev *rt2x00dev,
	int rf2, int channel, int freq, int txpower)
{
	u8 reg = 0;
	u32 rf1 = rt2x00dev->rf1;
	u32 rf3 = rt2x00dev->rf3;
	u32 rf4 = 0;

	/*
	 * Only continue when there is something to be done.
	 */
	if (channel == rt2x00dev->rx_status.channel)
		return;

	if (txpower == 0xff)
		txpower = rt2x00dev->tx_power;
	else
		txpower = TXPOWER_TO_DEV(txpower);

	if (rt2x00_rf(&rt2x00dev->chip, RF5225)) {
		if (channel <= 14)
			rf3 = 0x00068455;
		else if (channel >= 36 && channel <= 48)
			rf3 = 0x0009be55;
		else if (channel >= 52 && channel <= 64)
			rf3 = 0x0009ae55;
		else if (channel >= 100 && channel <= 112)
			rf3 = 0x000bae55;
		else
			rf3 = 0x000bbe55;
	}

	if (channel < 14) {
		if (channel & 0x01)
			rf4 = 0x000fea0b;
		else
			rf4 = 0x000fea1f;
	} else if (channel == 14) {
		rf4 = 0x000fea13;
	} else {
		switch (channel) {
			case 36:
			case 56:
			case 116:
			case 136:
				rf4 = 0x000fea23;
				break;
			case 40:
			case 60:
			case 100:
			case 120:
			case 140:
				rf4 = 0x000fea03;
				break;
			case 44:
			case 64:
			case 104:
			case 124:
				rf4 = 0x000fea0b;
				break;
			case 48:
			case 108:
			case 128:
				rf4 = 0x000fea13;
				break;
			case 52:
			case 112:
			case 132:
				rf4 = 0x000fea1b;
				break;
			case 149:
				rf4 = 0x000fea1f;
				break;
			case 153:
				rf4 = 0x000fea27;
				break;
			case 157:
				rf4 = 0x000fea07;
				break;
			case 161:
				rf4 = 0x000fea0f;
				break;
			case 165:
				rf4 = 0x000fea17;
				break;
		}
	}

	if (rt2x00_rf(&rt2x00dev->chip, RF2527) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5225))
		rf4 |= 0x00010000;

	/*
	 * Set TXpower.
	 */
	rt2x00_set_field32(&rf3, RF3_TXPOWER, txpower);

	INFO("Switching channel. RF1: 0x%08x, RF2: 0x%08x, RF3: 0x%08x, "
		"RF4: 0x%08x.\n", rf1, rf2, rf3, rf4);

	/*
	 * Set Frequency offset.
	 */
	rt2x00_set_field32(&rf4, RF4_FREQ_OFFSET, rt2x00dev->freq_offset);

	rt2x00_bbp_read(rt2x00dev, 3, &reg);
	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF2527))
		reg &= ~0x01;
	else
		reg |= 0x01;
	rt2x00_bbp_write(rt2x00dev, 3, reg);

	rt2x00_rf_write(rt2x00dev, rf1);
	rt2x00_rf_write(rt2x00dev, rf2);
	rt2x00_rf_write(rt2x00dev, rf3 & ~0x00000004);
	rt2x00_rf_write(rt2x00dev, rf4);

	rt2x00_rf_write(rt2x00dev, rf1);
	rt2x00_rf_write(rt2x00dev, rf2);
	rt2x00_rf_write(rt2x00dev, rf3 | 0x00000004);
	rt2x00_rf_write(rt2x00dev, rf4);

	rt2x00_rf_write(rt2x00dev, rf1);
	rt2x00_rf_write(rt2x00dev, rf2);
	rt2x00_rf_write(rt2x00dev, rf3 & ~0x00000004);
	rt2x00_rf_write(rt2x00dev, rf4);

	msleep(1);

	/*
	 * Update active info for RX.
	 */
	rt2x00dev->rx_status.freq = freq;
	rt2x00dev->rx_status.channel = channel;

	/*
	 * Update rf fields
	 */
	rt2x00dev->rf1 = rf1;
	rt2x00dev->rf2 = rf2;
	rt2x00dev->rf3 = rf3;
	rt2x00dev->rf4 = rf4;

	rt2x00dev->tx_power = txpower;
}

static void rt73usb_config_txpower(struct rt2x00_dev *rt2x00dev, int txpower)
{
	txpower = TXPOWER_TO_DEV(txpower);

	/*
	 * Only continue when there is something to be done.
	 */
	if (txpower == rt2x00dev->tx_power)
		return;

	rt2x00_set_field32(&rt2x00dev->rf3, RF3_TXPOWER, txpower);

	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf1);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf2);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf3 & ~0x00000004);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf4);

	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf1);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf2);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf3 | 0x00000004);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf4);

	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf1);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf2);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf3 & ~0x00000004);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf4);

	rt2x00dev->tx_power = txpower;
}

static void rt73usb_config_antenna(struct rt2x00_dev *rt2x00dev,
	int antenna_tx, int antenna_rx)
{
	u32 reg;
	u8 reg_r3;
	u8 reg_r4;
	u8 reg_r77;
	u8 frame_type;

	/*
	 * Only continue when there is something to be done.
	 */
	if (rt2x00dev->rx_status.antenna == antenna_rx)
		return;

	rt2x00_register_read(rt2x00dev, PHY_CSR0, &reg);

	if (rt2x00dev->curr_hwmode == HWMODE_A) {
		if (GET_FLAG(rt2x00dev, CONFIG_EXTERNAL_LNA)) {
			rt2x00_bbp_write(rt2x00dev, 17, 0x38);
			rt2x00_bbp_write(rt2x00dev, 96, 0x78);
			rt2x00_bbp_write(rt2x00dev, 104, 0x48);
			rt2x00_bbp_write(rt2x00dev, 75, 0x80);
			rt2x00_bbp_write(rt2x00dev, 86, 0x80);
			rt2x00_bbp_write(rt2x00dev, 88, 0x80);
		} else {
			rt2x00_bbp_write(rt2x00dev, 17, 0x28);
			rt2x00_bbp_write(rt2x00dev, 96, 0x58);
			rt2x00_bbp_write(rt2x00dev, 104, 0x38);
			rt2x00_bbp_write(rt2x00dev, 75, 0xfe);
			rt2x00_bbp_write(rt2x00dev, 86, 0xfe);
			rt2x00_bbp_write(rt2x00dev, 88, 0xfe);
		}
		rt2x00_bbp_write(rt2x00dev, 35, 0x60);
		rt2x00_bbp_write(rt2x00dev, 97, 0x58);
		rt2x00_bbp_write(rt2x00dev, 98, 0x58);

		rt2x00_set_field32(&reg, PHY_CSR0_PA_PE_BG, 0);
		rt2x00_set_field32(&reg, PHY_CSR0_PA_PE_A, 1);
	} else {
		if (GET_FLAG(rt2x00dev, CONFIG_EXTERNAL_LNA)) {
			rt2x00_bbp_write(rt2x00dev, 17, 0x30);
			rt2x00_bbp_write(rt2x00dev, 96, 0x68);
			rt2x00_bbp_write(rt2x00dev, 104, 0x3c);
			rt2x00_bbp_write(rt2x00dev, 75, 0x80);
			rt2x00_bbp_write(rt2x00dev, 86, 0x80);
			rt2x00_bbp_write(rt2x00dev, 88, 0x80);
		} else {
			rt2x00_bbp_write(rt2x00dev, 17, 0x20);
			rt2x00_bbp_write(rt2x00dev, 96, 0x48);
			rt2x00_bbp_write(rt2x00dev, 104, 0x2c);
			rt2x00_bbp_write(rt2x00dev, 75, 0xfe);
			rt2x00_bbp_write(rt2x00dev, 86, 0xfe);
			rt2x00_bbp_write(rt2x00dev, 88, 0xfe);
		}
		rt2x00_bbp_write(rt2x00dev, 35, 0x50);
		rt2x00_bbp_write(rt2x00dev, 97, 0x48);
		rt2x00_bbp_write(rt2x00dev, 98, 0x48);

		rt2x00_set_field32(&reg, PHY_CSR0_PA_PE_BG, 1);
		rt2x00_set_field32(&reg, PHY_CSR0_PA_PE_A, 0);
	}

	rt2x00_register_write(rt2x00dev, PHY_CSR0, reg);

	rt2x00_bbp_read(rt2x00dev, 3, &reg_r3);
	rt2x00_bbp_read(rt2x00dev, 4, &reg_r4);
	rt2x00_bbp_read(rt2x00dev, 77, &reg_r77);

	reg_r3 &= ~0x01;
	reg_r4 &= ~0x23;
	frame_type = ~(GET_FLAG(rt2x00dev, CONFIG_FRAME_TYPE) << 5);

	if (rt2x00_rf(&rt2x00dev->chip, RF5226) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5225)) {
		if (antenna_rx == 0) {		/* Diversity. */
			reg_r4 |= 0x02;
			if (rt2x00dev->curr_hwmode != HWMODE_A)
				reg_r4 |= 0x20;
			reg_r4 &= frame_type;
		} else if (antenna_rx == 1) {	/* RX: Antenna A */
			reg_r4 |= 0x01;
			reg_r4 &= frame_type;
			if (rt2x00dev->curr_hwmode == HWMODE_A)
				reg_r77 &= ~0x03;
			else
				reg_r77 |= 0x03;
			rt2x00_bbp_write(rt2x00dev, 77, reg_r77);
		} else if (antenna_rx == 2) {	/* RX: Antenna B */
			reg_r4 |= 0x01;
			reg_r4 &= frame_type;
			if (rt2x00dev->curr_hwmode == HWMODE_A)
				reg_r77 |= 0x03;
			else
				reg_r77 &= ~0x03;
			rt2x00_bbp_write(rt2x00dev, 77, reg_r77);
		}
	} else if (rt2x00_rf(&rt2x00dev->chip, RF2528) ||
		   rt2x00_rf(&rt2x00dev->chip, RF2527)) {
		if (antenna_rx == 0) {		/* Diversity. */
			reg_r4 |= 0x22;
			reg_r4 &= frame_type;
		} else if (antenna_rx == 1) {	/* RX: Antenna A */
			reg_r4 |= 0x21;
			reg_r4 &= frame_type;
			reg_r77 |= 0x03;
			rt2x00_bbp_write(rt2x00dev, 77, reg_r77);
		} else if (antenna_rx == 2) {	/* RX: Antenna B */
			reg_r4 |= 0x21;
			reg_r4 &= frame_type;
			reg_r77 &= ~0x03;
			rt2x00_bbp_write(rt2x00dev, 77, reg_r77);
		}
	}

	rt2x00_bbp_write(rt2x00dev, 3, reg_r3);
	rt2x00_bbp_write(rt2x00dev, 4, reg_r4);

	/*
	 * Update active info for RX.
	 */
	rt2x00dev->rx_status.antenna = antenna_rx;
}

static void rt73usb_config_duration(struct rt2x00_dev *rt2x00dev,
	int short_slot_time)
{
	u32 reg;

	short_slot_time = short_slot_time ? SHORT_SLOT_TIME : SLOT_TIME;

	rt2x00_register_read(rt2x00dev, MAC_CSR9, &reg);
	rt2x00_set_field32(&reg, MAC_CSR9_SLOT_TIME, short_slot_time);
	rt2x00_register_write(rt2x00dev, MAC_CSR9, reg);

	rt2x00_register_read(rt2x00dev, MAC_CSR8, &reg);
	rt2x00_set_field32(&reg, MAC_CSR8_SIFS, SIFS);
	rt2x00_set_field32(&reg, MAC_CSR8_SIFS_AFTER_RX_OFDM, 3);
	rt2x00_set_field32(&reg, MAC_CSR8_EIFS, EIFS);
	rt2x00_register_write(rt2x00dev, MAC_CSR8, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR0_TSF_OFFSET, IEEE80211_HEADER);
	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR4, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR4_AUTORESPOND_ENABLE, 1);
	rt2x00_register_write(rt2x00dev, TXRX_CSR4, reg);
}

static void rt73usb_config_rate(struct rt2x00_dev *rt2x00dev, const int rate)
{
	struct ieee80211_conf *conf = &rt2x00dev->hw->conf;
	u32 reg;
	u32 value;
	u32 preamble;

	preamble = DEVICE_GET_RATE_FIELD(rate, PREAMBLE)
		? SHORT_PREAMBLE : PREAMBLE;

	reg = DEVICE_GET_RATE_FIELD(rate, RATEMASK) & DEV_BASIC_RATE;

	rt2x00_register_write(rt2x00dev, TXRX_CSR5, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
	value = ((conf->flags & IEEE80211_CONF_SHORT_SLOT_TIME) ?
		 SHORT_DIFS :  DIFS) +
		PLCP + preamble + get_duration(ACK_SIZE, 10);
	rt2x00_set_field32(&reg, TXRX_CSR0_RX_ACK_TIMEOUT, value);
	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR4, &reg);
	if (preamble == SHORT_PREAMBLE)
		rt2x00_set_field32(&reg, TXRX_CSR4_AUTORESPOND_PREAMBLE, 1);
	else
		rt2x00_set_field32(&reg, TXRX_CSR4_AUTORESPOND_PREAMBLE, 0);
	rt2x00_register_write(rt2x00dev, TXRX_CSR4, reg);
}

static void rt73usb_config_phymode(struct rt2x00_dev *rt2x00dev,
	const int phymode)
{
	struct ieee80211_hw_mode *mode;
	struct ieee80211_rate *rate;

	/*
	 * Only continue when there is something to be done.
	 */
	if (rt2x00dev->rx_status.phymode == phymode)
		return;

	if (phymode == MODE_IEEE80211A)
		rt2x00dev->curr_hwmode = HWMODE_A;
	else if (phymode == MODE_IEEE80211B)
		rt2x00dev->curr_hwmode = HWMODE_B;
	else
		rt2x00dev->curr_hwmode = HWMODE_G;

	mode = &rt2x00dev->hwmodes[rt2x00dev->curr_hwmode];
	rate = &mode->rates[mode->num_rates - 1];

	rt73usb_config_rate(rt2x00dev, rate->val2);

	/*
	 * Update physical mode for rx ring.
	 */
	rt2x00dev->rx_status.phymode = phymode;
}

static void rt73usb_config_mac_addr(struct rt2x00_dev *rt2x00dev, u8 *addr)
{
	u32 reg;

	/*
	 * The MAC address is passed to us as an array of bytes,
	 * that array is little endian, so no need for byte ordering.
	 * We only need to set the MAC_CSR3_UNICAST_TO_ME_MASK
	 * at the correct offset.
	 */
	rt2x00_register_multiwrite(rt2x00dev, MAC_CSR2, (u32*)addr, ETH_ALEN);

	rt2x00_register_read(rt2x00dev, MAC_CSR3, &reg);
	rt2x00_set_field32(&reg, MAC_CSR3_UNICAST_TO_ME_MASK, 0xff);
	rt2x00_register_write(rt2x00dev, MAC_CSR3, reg);
}

/*
 * Link tuning
 */
static void rt73usb_link_tuner(struct work_struct *work)
{
	struct rt2x00_dev *rt2x00dev =
		container_of(work, struct rt2x00_dev, link.work.work);
	u32 reg;
	u32 rssi;
	u8 reg_r17;
	u8 up_bound;
	u8 low_bound;

	/*
	 * Retrieve link quality.
	 */
	rssi = rt2x00_get_link_rssi(&rt2x00dev->link);
	if (!rssi)
		goto exit;

	/*
	 * Update LED.
	 */
	rt73usb_activity_led(rt2x00dev, rssi);

	/*
	 * Determine upper and lower limits for BBP17 register.
	 */
	if (rt2x00dev->rx_status.phymode == MODE_IEEE80211A) {
		up_bound = 0x48;
		low_bound = 0x28;
	} else {
		if (rssi > 38) {
			up_bound = 0x40;
			low_bound = 0x1c;
		} else if (rssi > 36) {
			up_bound = 0x20;
			low_bound = 0x1c;
		} else {
			up_bound = 0x1c;
			low_bound = 0x1c;
		}

		if (GET_FLAG(rt2x00dev, CONFIG_EXTERNAL_LNA)) {
			up_bound += 0x10;
			low_bound += 0x14;
		}
	}

	rt2x00_bbp_read(rt2x00dev, 17, &reg_r17);

	if (rssi >= 85) {
		if (reg_r17 != 0x60)
			rt2x00_bbp_write(rt2x00dev, 17, 0x60);
		goto exit;
	} else if (rssi >= 62) {
		if (reg_r17 != up_bound)
			rt2x00_bbp_write(rt2x00dev, 17, up_bound);
		goto exit;
	} else if (rssi >= 54) {
		low_bound += 0x10;
		if (reg_r17 != low_bound)
			rt2x00_bbp_write(rt2x00dev, 17, low_bound);
		goto exit;
	} else if (rssi >= 46) {
		low_bound += 0x08;
		if (reg_r17 != low_bound)
			rt2x00_bbp_write(rt2x00dev, 17, low_bound);
		goto exit;
	} else {
		up_bound -= 2 * (46 - rssi);
		if (up_bound < low_bound)
			up_bound =  low_bound;

		if (reg_r17 > up_bound) {
			rt2x00_bbp_write(rt2x00dev, 17, up_bound);
			goto exit;
		}
	}

	rt2x00_register_read(rt2x00dev, STA_CSR1, &reg);
	reg = rt2x00_get_field32(reg, STA_CSR1_FALSE_CCA_ERROR);

	if (reg > 512 && reg_r17 < up_bound)
		rt2x00_bbp_write(rt2x00dev, 17, ++reg_r17);
	else if (reg < 100 && reg_r17 > low_bound)
		rt2x00_bbp_write(rt2x00dev, 17, --reg_r17);

exit:
	if (reg_r17)
		rt2x00_update_link_noise(&rt2x00dev->link, reg_r17);

	queue_delayed_work(rt2x00dev->workqueue, &rt2x00dev->link.work,
		LINK_TUNE_INTERVAL);
}

/*
 * LED functions.
 */
static void rt73usb_enable_led(struct rt2x00_dev *rt2x00dev)
{
	u32 reg;

	rt2x00_register_read(rt2x00dev, MAC_CSR14, &reg);
	rt2x00_set_field32(&reg, MAC_CSR14_ON_PERIOD, 70);
	rt2x00_set_field32(&reg, MAC_CSR14_OFF_PERIOD, 30);
	rt2x00_register_write(rt2x00dev, MAC_CSR14, reg);

	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_RADIO_STATUS, 1);
	if (rt2x00dev->rx_status.phymode == MODE_IEEE80211A)
		rt2x00_set_field16(
			&rt2x00dev->led_reg, MCU_LEDCS_LINK_A_STATUS, 1);
	else
		rt2x00_set_field16(
			&rt2x00dev->led_reg, MCU_LEDCS_LINK_BG_STATUS, 1);

	rt2x00_vendor_request(
		rt2x00dev, USB_LED_CONTROL, USB_VENDOR_REQUEST_OUT,
		0x00, rt2x00dev->led_reg, NULL, 0, REGISTER_TIMEOUT);
}

static void rt73usb_disable_led(struct rt2x00_dev *rt2x00dev)
{
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_RADIO_STATUS, 0);
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_LINK_BG_STATUS, 0);
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_LINK_A_STATUS, 0);

	rt2x00_vendor_request(
		rt2x00dev, USB_LED_CONTROL, USB_VENDOR_REQUEST_OUT,
		0x00, rt2x00dev->led_reg, NULL, 0, REGISTER_TIMEOUT);
}

static void rt73usb_activity_led(struct rt2x00_dev *rt2x00dev, char rssi)
{
	u32 led;

	if (rt2x00dev->led_mode != LED_MODE_SIGNAL_STRENGTH)
		return;

	if (rssi <= 30)
		led = 0;
	else if (rssi <= 39)
		led = 1;
	else if (rssi <= 49)
		led = 2;
	else if (rssi <= 53)
		led = 3;
	else if (rssi <= 63)
		led = 4;
	else
		led = 5;

	rt2x00_vendor_request(
		rt2x00dev, USB_LED_CONTROL, USB_VENDOR_REQUEST_OUT,
		led, rt2x00dev->led_reg, NULL, 0, REGISTER_TIMEOUT);
}

/*
 * Device state switch.
 * This will put the device to sleep, or awake it.
 */
static int rt73usb_set_state(struct rt2x00_dev *rt2x00dev,
	enum dev_state state)
{
	u32 reg;
	unsigned int i;
	char put_to_sleep;
	char current_state;

	put_to_sleep = (state != STATE_AWAKE);

	if (!put_to_sleep)
		rt2x00_vendor_request(rt2x00dev,
			USB_DEVICE_MODE, USB_VENDOR_REQUEST_OUT,
			0x00, USB_MODE_WAKEUP, NULL, 0, REGISTER_TIMEOUT);

	rt2x00_register_read(rt2x00dev, MAC_CSR12, &reg);
	rt2x00_set_field32(&reg, MAC_CSR12_FORCE_WAKEUP, !put_to_sleep);
	rt2x00_set_field32(&reg, MAC_CSR12_PUT_TO_SLEEP, put_to_sleep);
	rt2x00_register_write(rt2x00dev, MAC_CSR12, reg);

	if (put_to_sleep)
		rt2x00_vendor_request(rt2x00dev,
			USB_DEVICE_MODE, USB_VENDOR_REQUEST_OUT,
			0x00, USB_MODE_SLEEP, NULL, 0, REGISTER_TIMEOUT);

	/*
	 * Device is not guaranteed to be in the requested state yet.
	 * We must wait until the register indicates that the
	 * device has entered the correct state.
	 */
	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00dev, MAC_CSR12, &reg);
		current_state = rt2x00_get_field32(reg,
			MAC_CSR12_BBP_CURRENT_STATE);
		if (current_state == !put_to_sleep)
			return 0;
		msleep(10);
	}

	NOTICE("Device failed to enter state %d, "
		"current device state %d.\n", !put_to_sleep, current_state);

	return -EBUSY;
}

/*
 * Initialization functions.
 */
static int rt73usb_alloc_dma_ring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type, const u16 max_entries,
	const u16 data_size, const u16 desc_size)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	unsigned int i;

	ring->stats.limit = max_entries;
	ring->data_size = data_size;
	ring->desc_size = desc_size;

	/*
	 * Allocate all ring entries.
	 */
	ring->entry = kzalloc(ring->stats.limit * sizeof(struct data_entry),
		GFP_KERNEL);
	if (!ring->entry)
		return -ENOMEM;

	/*
	 * Initialize all ring entries to contain valid
	 * addresses.
	 */
	for (i = 0; i < ring->stats.limit; i++) {
		ring->entry[i].flags = 0;
		ring->entry[i].ring = ring;
		ring->entry[i].priv = usb_alloc_urb(0, GFP_KERNEL);
		if (!ring->entry[i].priv)
			return -ENOMEM;

		if (ring_type == RING_RX) {
			ring->entry[i].skb = dev_alloc_skb(NET_IP_ALIGN +
				ring->data_size + ring->desc_size);
			if (!ring->entry[i].skb)
				return -ENOMEM;

			skb_reserve(ring->entry[i].skb, NET_IP_ALIGN);
			skb_put(ring->entry[i].skb,
				ring->data_size + ring->desc_size);
		}
	}

	return 0;
}

static void rt73usb_free_ring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	unsigned int i;

	if (!ring->entry)
		return;

	for (i = 0; i < ring->stats.limit; i++) {
		usb_kill_urb(ring->entry[i].priv);
		usb_free_urb(ring->entry[i].priv);
		if (ring_type == RING_RX)
			kfree_skb(ring->entry[i].skb);
	}

	kfree(ring->entry);
	ring->entry = NULL;
}

static int rt73usb_allocate_dma_rings(struct rt2x00_dev *rt2x00dev)
{
	if (rt73usb_alloc_dma_ring(rt2x00dev, RING_RX,
		RX_ENTRIES, DATA_FRAME_SIZE, RXD_DESC_SIZE) ||
	    rt73usb_alloc_dma_ring(rt2x00dev, RING_AC_VO,
		TX_ENTRIES, DATA_FRAME_SIZE, TXD_DESC_SIZE) ||
	    rt73usb_alloc_dma_ring(rt2x00dev, RING_AC_VI,
		TX_ENTRIES, DATA_FRAME_SIZE, TXD_DESC_SIZE) ||
	    rt73usb_alloc_dma_ring(rt2x00dev, RING_AC_BE,
		TX_ENTRIES, DATA_FRAME_SIZE, TXD_DESC_SIZE) ||
	    rt73usb_alloc_dma_ring(rt2x00dev, RING_AC_BK,
		TX_ENTRIES, DATA_FRAME_SIZE, TXD_DESC_SIZE) ||
	    rt73usb_alloc_dma_ring(rt2x00dev, RING_PRIO,
		TX_ENTRIES, DATA_FRAME_SIZE, TXD_DESC_SIZE) ||
	    rt73usb_alloc_dma_ring(rt2x00dev, RING_BEACON,
		BEACON_ENTRIES, MGMT_FRAME_SIZE, TXD_DESC_SIZE)) {
		return -ENOMEM;
	}

	return 0;
}

static void rt73usb_free_rings(struct rt2x00_dev *rt2x00dev)
{
	rt73usb_free_ring(rt2x00dev, RING_RX);
	rt73usb_free_ring(rt2x00dev, RING_AC_VO);
	rt73usb_free_ring(rt2x00dev, RING_AC_VI);
	rt73usb_free_ring(rt2x00dev, RING_AC_BE);
	rt73usb_free_ring(rt2x00dev, RING_AC_BK);
	rt73usb_free_ring(rt2x00dev, RING_PRIO);
	rt73usb_free_ring(rt2x00dev, RING_BEACON);
}

static void rt73usb_init_rxring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	struct usb_device *usb_dev =
		interface_to_usbdev(rt2x00dev_usb(rt2x00dev));
	unsigned int i;

	ring->type = ring_type;

	for (i = 0; i < ring->stats.limit; i++) {
		usb_fill_bulk_urb(
			ring->entry[i].priv,
			usb_dev,
			usb_rcvbulkpipe(usb_dev, 1),
			ring->entry[i].skb->data,
			ring->entry[i].skb->len,
			rt73usb_interrupt_rxdone,
			&ring->entry[i]);
	}

	rt2x00_ring_index_clear(ring);
}

static void rt73usb_init_txring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	unsigned int i;

	ring->type = ring_type;

	for (i = 0; i < ring->stats.limit; i++)
		CLEAR_FLAGS(&ring->entry[i]);

	rt2x00_ring_index_clear(ring);
}

static int rt73usb_init_rings(struct rt2x00_dev *rt2x00dev)
{
	rt73usb_init_rxring(rt2x00dev, RING_RX);
	rt73usb_init_txring(rt2x00dev, RING_AC_VO);
	rt73usb_init_txring(rt2x00dev, RING_AC_VI);
	rt73usb_init_txring(rt2x00dev, RING_AC_BE);
	rt73usb_init_txring(rt2x00dev, RING_AC_BK);
	rt73usb_init_txring(rt2x00dev, RING_PRIO);
	rt73usb_init_txring(rt2x00dev, RING_BEACON);

	return 0;
}

static int rt73usb_init_registers(struct rt2x00_dev *rt2x00dev)
{
	u32 reg;

	if (rt73usb_set_state(rt2x00dev, STATE_AWAKE))
		return -EBUSY;

	rt2x00_register_write(rt2x00dev, MAC_CSR10, 0x00000718);

	rt2x00_register_write(rt2x00dev, TXRX_CSR0, 0x025eb032);

	rt2x00_register_write(rt2x00dev, TXRX_CSR1, 0x9eaa9eaf);
	rt2x00_register_write(rt2x00dev, TXRX_CSR2, 0x8a8b8c8d);
	rt2x00_register_write(rt2x00dev, TXRX_CSR3, 0x00858687);

	rt2x00_register_write(rt2x00dev, TXRX_CSR7, 0x2e31353b);
	rt2x00_register_write(rt2x00dev, TXRX_CSR8, 0x2a2a2a2c);

	rt2x00_register_write(rt2x00dev, TXRX_CSR15, 0x0000000f);

	rt2x00_register_write(rt2x00dev, MAC_CSR6, 0x00000fff);

	rt2x00_register_write(rt2x00dev, MAC_CSR13, 0x00007f00);

	rt2x00_register_write(rt2x00dev, SEC_CSR0, 0x00000000);
	rt2x00_register_write(rt2x00dev, SEC_CSR1, 0x00000000);
	rt2x00_register_write(rt2x00dev, SEC_CSR5, 0x00000000);

	rt2x00_register_write(rt2x00dev, PHY_CSR1, 0x000023b0);
	rt2x00_register_write(rt2x00dev, PHY_CSR5, 0x00040a06);
	rt2x00_register_write(rt2x00dev, PHY_CSR6, 0x00080606);
	rt2x00_register_write(rt2x00dev, PHY_CSR7, 0x00000408);

	rt2x00_register_read(rt2x00dev, AC_TXOP_CSR0, &reg);
	rt2x00_set_field32(&reg, AC_TXOP_CSR0_AC0_TX_OP, 0);
	rt2x00_set_field32(&reg, AC_TXOP_CSR0_AC1_TX_OP, 0);
	rt2x00_register_write(rt2x00dev, AC_TXOP_CSR0, reg);

	rt2x00_register_read(rt2x00dev, AC_TXOP_CSR1, &reg);
	rt2x00_set_field32(&reg, AC_TXOP_CSR1_AC2_TX_OP, 192);
	rt2x00_set_field32(&reg, AC_TXOP_CSR1_AC3_TX_OP, 48);
	rt2x00_register_write(rt2x00dev, AC_TXOP_CSR1, reg);

	rt2x00_register_read(rt2x00dev, MAC_CSR9, &reg);
	rt2x00_set_field32(&reg, MAC_CSR9_CW_SELECT, 0);
	rt2x00_register_write(rt2x00dev, MAC_CSR9, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR0_AUTO_TX_SEQ, 1);
	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);

	/*
	 * We must clear the error counters.
	 * These registers are cleared on read,
	 * so we may pass a useless variable to store the value.
	 */
	rt2x00_register_read(rt2x00dev, STA_CSR0, &reg);
	rt2x00_register_read(rt2x00dev, STA_CSR1, &reg);
	rt2x00_register_read(rt2x00dev, STA_CSR2, &reg);

	/*
	 * Reset MAC and BBP registers.
	 */
	reg = 0;
	rt2x00_set_field32(&reg, MAC_CSR1_SOFT_RESET, 1);
	rt2x00_set_field32(&reg, MAC_CSR1_BBP_RESET, 1);
	rt2x00_register_write(rt2x00dev, MAC_CSR1, reg);

	rt2x00_register_read(rt2x00dev, MAC_CSR1, &reg);
	rt2x00_set_field32(&reg, MAC_CSR1_SOFT_RESET, 0);
	rt2x00_set_field32(&reg, MAC_CSR1_BBP_RESET, 0);
	rt2x00_register_write(rt2x00dev, MAC_CSR1, reg);

	rt2x00_register_read(rt2x00dev, MAC_CSR1, &reg);
	rt2x00_set_field32(&reg, MAC_CSR1_HOST_READY, 1);
	rt2x00_register_write(rt2x00dev, MAC_CSR1, reg);

	return 0;
}

static int rt73usb_init_bbp(struct rt2x00_dev *rt2x00dev)
{
	unsigned int i;
	u16 eeprom;
	u8 reg_id;
	u8 value;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_bbp_read(rt2x00dev, 0, &value);
		if ((value != 0xff) && (value != 0x00))
			goto continue_csr_init;
		NOTICE("Waiting for BBP register.\n");
		udelay(REGISTER_BUSY_DELAY);
	}

	ERROR("BBP register access failed, aborting.\n");
	return -EACCES;

continue_csr_init:
	rt2x00_bbp_write(rt2x00dev, 3, 0x80);
	rt2x00_bbp_write(rt2x00dev, 15, 0x30);
	rt2x00_bbp_write(rt2x00dev, 17, 0x20);
	rt2x00_bbp_write(rt2x00dev, 21, 0xc8);
	rt2x00_bbp_write(rt2x00dev, 22, 0x38);
	rt2x00_bbp_write(rt2x00dev, 23, 0x06);
	rt2x00_bbp_write(rt2x00dev, 24, 0xfe);
	rt2x00_bbp_write(rt2x00dev, 25, 0x0a);
	rt2x00_bbp_write(rt2x00dev, 26, 0x0d);
	rt2x00_bbp_write(rt2x00dev, 32, 0x0b);
	rt2x00_bbp_write(rt2x00dev, 34, 0x12);
	rt2x00_bbp_write(rt2x00dev, 37, 0x07);
	rt2x00_bbp_write(rt2x00dev, 39, 0xf8);
	rt2x00_bbp_write(rt2x00dev, 41, 0x60);
	rt2x00_bbp_write(rt2x00dev, 53, 0x10);
	rt2x00_bbp_write(rt2x00dev, 54, 0x18);
	rt2x00_bbp_write(rt2x00dev, 60, 0x10);
	rt2x00_bbp_write(rt2x00dev, 61, 0x04);
	rt2x00_bbp_write(rt2x00dev, 62, 0x04);
	rt2x00_bbp_write(rt2x00dev, 75, 0xfe);
	rt2x00_bbp_write(rt2x00dev, 86, 0xfe);
	rt2x00_bbp_write(rt2x00dev, 88, 0xfe);
	rt2x00_bbp_write(rt2x00dev, 90, 0x0f);
	rt2x00_bbp_write(rt2x00dev, 99, 0x00);
	rt2x00_bbp_write(rt2x00dev, 102, 0x16);
	rt2x00_bbp_write(rt2x00dev, 107, 0x04);

	DEBUG("Start initialization from EEPROM...\n");
	for (i = 0; i < EEPROM_BBP_SIZE; i++) {
		rt2x00_eeprom_read(rt2x00dev, EEPROM_BBP_START + i, &eeprom);

		if (eeprom != 0xffff && eeprom != 0x0000) {
			reg_id = rt2x00_get_field16(eeprom, EEPROM_BBP_REG_ID);
			value = rt2x00_get_field16(eeprom, EEPROM_BBP_VALUE);
			DEBUG("BBP: 0x%02x, value: 0x%02x.\n", reg_id, value);
			rt2x00_bbp_write(rt2x00dev, reg_id, value);
		}
	}
	DEBUG("...End initialization from EEPROM.\n");

	return 0;
}

/*
 * Device initialization functions.
 */
static int rt73usb_initialize(struct rt2x00_dev *rt2x00dev)
{
	if (GET_FLAG(rt2x00dev, DEVICE_INITIALIZED))
		return 0;

	/*
	 * We must wait on the firmware before
	 * we can safely continue.
	 */
	if (rt2x00lib_load_firmware_wait(rt2x00dev))
		return -ENODEV;

	/*
	 * Allocate all data rings.
	 */
	if (rt73usb_allocate_dma_rings(rt2x00dev)) {
		ERROR("DMA allocation failed.\n");
		goto exit_fail;
	}

	/*
	 * Reset the channel_change_time value
	 * to make sure it will be correctly initialized
	 * after the radio has been enabled.
	 */
	rt2x00dev->hw->channel_change_time = 0;

	SET_FLAG(rt2x00dev, DEVICE_INITIALIZED);

	return 0;

exit_fail:
	rt73usb_free_rings(rt2x00dev);

	return -EIO;
}

static void rt73usb_uninitialize(struct rt2x00_dev *rt2x00dev)
{
	if (!GET_FLAG(rt2x00dev, DEVICE_INITIALIZED))
		return;

	/*
	 * Cancel scanning.
	 */
	if (rt2x00dev->scan)
		rt2x00_signal_scan(rt2x00dev->scan, SCANNING_CANCELLED);

	/*
	 * Flush out all pending work.
	 */
	flush_workqueue(rt2x00dev->workqueue);

	/*
	 * Free DMA rings.
	 */
	rt73usb_free_rings(rt2x00dev);

	CLEAR_FLAG(rt2x00dev, DEVICE_INITIALIZED);
}

/*
 * Radio control functions.
 */
static void rt73usb_toggle_rx(struct rt2x00_dev *rt2x00dev, int enable)
{
	u32 reg;

	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR0_DISABLE_RX, !enable);
	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);
}

static int rt73usb_enable_radio(struct rt2x00_dev *rt2x00dev)
{
	struct data_ring *ring;
	unsigned int i;

	/*
	 * Don't enable the radio twice.
	 */
	if (GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO))
		return 0;

	/*
	 * Initialize all registers.
	 */
	if (rt73usb_init_rings(rt2x00dev) ||
	    rt73usb_init_registers(rt2x00dev) ||
	    rt73usb_init_bbp(rt2x00dev)) {
		ERROR("Register initialization failed.\n");
		goto exit_fail;
	}

	/*
	 * Determine channel change time.
	 */
	if (rt2x00lib_detect_channel_time(rt2x00dev))
		goto exit_fail;

	SET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO);

	/*
	 * Enable RX.
	 */
	rt73usb_toggle_rx(rt2x00dev, 1);

	ring = &rt2x00dev->ring[RING_RX];
	for (i = 0; i < ring->stats.limit; i++) {
		SET_FLAG(&ring->entry[i], ENTRY_OWNER_NIC);
		usb_submit_urb(ring->entry[i].priv, GFP_ATOMIC);
	}

	/*
	 * Enable LED
	 */
	rt73usb_enable_led(rt2x00dev);

	ieee80211_start_queues(rt2x00dev->hw);
	ieee80211_netif_oper(rt2x00dev->hw, NETIF_WAKE);

	return 0;

exit_fail:
	rt73usb_uninitialize(rt2x00dev);
	return -EIO;
}

static void rt73usb_disable_radio(struct rt2x00_dev *rt2x00dev)
{
	struct data_ring *ring;
	unsigned int i;

	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO))
		return;

	ieee80211_netif_oper(rt2x00dev->hw, NETIF_STOP);
	ieee80211_stop_queues(rt2x00dev->hw);

	/*
	 * Disable LED
	 */
	rt73usb_disable_led(rt2x00dev);

	CLEAR_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO);

	rt2x00_register_write(rt2x00dev, MAC_CSR10, 0x00001818);

	/*
	 * Disable synchronisation.
	 */
	rt2x00_register_write(rt2x00dev, TXRX_CSR9, 0);

	/*
	 * Cancel RX and TX.
	 */
	rt73usb_toggle_rx(rt2x00dev, 0);

	rt2x00_vendor_request(rt2x00dev, USB_RX_CONTROL,
		USB_VENDOR_REQUEST_OUT, 0x00, 0x00, NULL, 0, REGISTER_TIMEOUT);

	ring = &rt2x00dev->ring[RING_RX];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(ring->entry[i].priv);

	ring = &rt2x00dev->ring[RING_AC_VO];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(ring->entry[i].priv);

	ring = &rt2x00dev->ring[RING_AC_VI];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(ring->entry[i].priv);

	ring = &rt2x00dev->ring[RING_AC_BE];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(ring->entry[i].priv);

	ring = &rt2x00dev->ring[RING_AC_BK];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(ring->entry[i].priv);

	ring = &rt2x00dev->ring[RING_PRIO];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(ring->entry[i].priv);

	ring = &rt2x00dev->ring[RING_BEACON];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(ring->entry[i].priv);
}

/*
 * TX descriptor initialization
 */
static void rt73usb_write_tx_desc(struct rt2x00_dev *rt2x00dev,
	struct data_desc *txd, struct ieee80211_hdr *ieee80211hdr,
	unsigned int length, struct ieee80211_tx_control *control)
{
	struct data_ring *ring;
	int tx_rate;
	u32 word;
	u32 duration;
	u32 residual;
	u16 length_high;
	u16 length_low;
	u16 frame_control;
	u16 seq_ctrl;
	char rts_frame;
	char ofdm_rate;
	char req_timestamp;
	char more_frag;
	char ifs;
	char queue;
	u8 signal;
	u8 service;
	u8 bitrate;

	/*
	 * We require the ring structure this packet is being send to.
	 */
	ring = rt2x00_get_ring(rt2x00dev, control->queue);
	if (unlikely(!ring))
		return;

	/*
	 * Read required fields from ieee80211 header.
	 */
	frame_control = le16_to_cpu(ieee80211hdr->frame_control);
	seq_ctrl = le16_to_cpu(ieee80211hdr->seq_ctrl);

	/*
	 * Check if this frame is a RTS frame.
	 */
	rts_frame = is_rts_frame(frame_control);

	/*
	 * Check which rate should be used for this frame.
	 */
	if (rts_frame && control->rts_cts_rate)
		tx_rate = control->rts_cts_rate;
	else
		tx_rate = control->tx_rate;

	/*
	 * Are we working with OFDM rates.
	 */
	ofdm_rate = !!(DEVICE_GET_RATE_FIELD(tx_rate, RATEMASK) &
		       DEV_OFDM_RATE);

	/*
	 * Check if more fragments will follow this frame.
	 */
	more_frag = !!(ieee80211_get_morefrag(ieee80211hdr));

	/*
	 * Beacons and probe responses require the tsf timestamp
	 * to be inserted into the frame.
	 */
	req_timestamp = !!(control->queue == IEEE80211_TX_QUEUE_BEACON ||
			   is_probe_resp(frame_control));

	/*
	 * Determine with what IFS priority this frame should be send.
	 * Set ifs to IFS_SIFS when the this is not the first fragment,
	 * or this fragment came after RTS/CTS.
	 */
	if (((seq_ctrl & IEEE80211_SCTL_FRAG) > 0) || rts_frame)
		ifs = IFS_SIFS;
	else
		ifs = IFS_BACKOFF;

	/*
	 * Determine queue identification number.
	 */
	if (control->queue < rt2x00dev->hw->queues)
		queue = control->queue;
	else
		queue = 15;

	/*
	 * How the length should be processed depends
	 * on if we are working with OFDM rates or not.
	 */
	if (ofdm_rate) {
		residual = 0;
		length_high = ((length + FCS_LEN) >> 6) & 0x3f;
		length_low = ((length + FCS_LEN) & 0x3f);

	} else {
		bitrate = DEVICE_GET_RATE_FIELD(tx_rate, RATE);

		/*
		 * Convert length to microseconds.
		 */
		residual = get_duration_res(length + FCS_LEN, bitrate);
		duration = get_duration(length + FCS_LEN, bitrate);

		if (residual != 0)
			duration++;

		length_high = duration >> 8;
		length_low = duration & 0xff;
	}

	/*
	 * Create the signal and service values.
	 */
	signal = DEVICE_GET_RATE_FIELD(tx_rate, PLCP);
	if (DEVICE_GET_RATE_FIELD(tx_rate, PREAMBLE))
		signal |= 0x08;

	service = 0x04;
	if (residual <= (8 % 11))
		service |= 0x80;

	/*
	 * Start writing the descriptor words.
	 */
	rt2x00_desc_read(txd, 1, &word);
	rt2x00_set_field32(&word, TXD_W1_HOST_Q_ID, queue);
	rt2x00_set_field32(&word, TXD_W1_AIFSN, ring->tx_params.aifs);
	rt2x00_set_field32(&word, TXD_W1_CWMIN, ring->tx_params.cw_min);
	rt2x00_set_field32(&word, TXD_W1_CWMAX, ring->tx_params.cw_max);
	rt2x00_set_field32(&word, TXD_W1_IV_OFFSET, IEEE80211_HEADER);
	rt2x00_set_field32(&word, TXD_W1_HW_SEQUENCE, 1);
	rt2x00_desc_write(txd, 1, word);

	rt2x00_desc_read(txd, 2, &word);
	rt2x00_set_field32(&word, TXD_W2_PLCP_SIGNAL, signal);
	rt2x00_set_field32(&word, TXD_W2_PLCP_SERVICE, service);
	rt2x00_set_field32(&word, TXD_W2_PLCP_LENGTH_LOW, length_low);
	rt2x00_set_field32(&word, TXD_W2_PLCP_LENGTH_HIGH, length_high);
	rt2x00_desc_write(txd, 2, word);

	rt2x00_desc_read(txd, 5, &word);
	rt2x00_set_field32(&word, TXD_W5_TX_POWER,
		TXPOWER_TO_DEV(control->power_level));
	rt2x00_set_field32(&word, TXD_W5_WAITING_DMA_DONE_INT, 1);
	rt2x00_desc_write(txd, 5, word);

	rt2x00_desc_read(txd, 0, &word);
	rt2x00_set_field32(&word, TXD_W0_VALID, 1);
	rt2x00_set_field32(&word, TXD_W0_MORE_FRAG, more_frag);
	rt2x00_set_field32(&word, TXD_W0_ACK,
		!(control->flags & IEEE80211_TXCTL_NO_ACK));
	rt2x00_set_field32(&word, TXD_W0_TIMESTAMP, req_timestamp);
	rt2x00_set_field32(&word, TXD_W0_OFDM, ofdm_rate);
	rt2x00_set_field32(&word, TXD_W0_IFS, ifs);
	rt2x00_set_field32(&word, TXD_W0_RETRY_MODE, 0);
	rt2x00_set_field32(&word, TXD_W0_TKIP_MIC, 0);
	rt2x00_set_field32(&word, TXD_W0_DATABYTE_COUNT, length);
	rt2x00_set_field32(&word, TXD_W0_CIPHER_ALG, CIPHER_NONE);
	rt2x00_desc_write(txd, 0, word);
}

/*
 * TX data initialization
 */
static int rt73usb_write_tx_data(struct rt2x00_dev *rt2x00dev,
	struct data_ring *ring, struct sk_buff *skb,
	struct ieee80211_tx_control *control)
{
	struct usb_device *usb_dev =
		interface_to_usbdev(rt2x00dev_usb(rt2x00dev));
	struct ieee80211_hdr *ieee80211hdr = (struct ieee80211_hdr*)skb->data;
	struct data_entry *entry = rt2x00_get_data_entry(ring);
	struct data_desc *txd;
	u32 length = skb->len;
	u16 fc;

	if (rt2x00_ring_full(ring)) {
		ieee80211_stop_queue(rt2x00dev->hw, control->queue);
		return -EINVAL;
	}

	if (GET_FLAG(entry, ENTRY_OWNER_NIC)) {
		ERROR("Arrived at non-free entry in the non-full queue %d.\n"
			"Please file bug report to %s.\n",
			control->queue, DRV_PROJECT);
		ieee80211_stop_queue( rt2x00dev->hw, control->queue);
		return -EINVAL;
	}

	skb_push(skb, rt2x00dev->hw->extra_tx_headroom);
	txd = (struct data_desc*)skb->data;
	rt73usb_write_tx_desc(rt2x00dev, txd, ieee80211hdr, skb->len, control);
	memcpy(&entry->tx_status.control, control, sizeof(*control));
	entry->skb = skb;

	fc = le16_to_cpu(ieee80211hdr->frame_control);
	if (is_cts_frame(fc) || is_rts_frame(fc))
		SET_FLAG(entry, ENTRY_RTS_CTS_FRAME);

	/*
	 * Length passed to usb_fill_urb cannot be an odd number,
	 * so add 1 byte to make it even.
	 */
	length += rt2x00dev->hw->extra_tx_headroom;
	if (length % 2)
		length++;

	SET_FLAG(entry, ENTRY_OWNER_NIC);
	usb_fill_bulk_urb(
		entry->priv,
		usb_dev,
		usb_sndbulkpipe(usb_dev, 1),
		skb->data,
		length,
		rt73usb_interrupt_txdone,
		entry);
	usb_submit_urb(entry->priv, GFP_ATOMIC);

	rt2x00_ring_index_inc(ring);

	if (rt2x00_ring_full(ring))
		ieee80211_stop_queue(rt2x00dev->hw, control->queue);

	return 0;
}

/*
 * Interrupt functions.
 */
static void rt73usb_interrupt_beacondone(struct urb *urb)
{
	struct data_entry *entry = (struct data_entry*)urb->context;
	struct data_ring *ring = entry->ring;

	if (!GET_FLAG(ring->rt2x00dev, DEVICE_ENABLED_RADIO))
		return;

	/*
	 * Check if this was the guardian beacon,
	 * if that was the case we need to send the real beacon now.
	 * Otherwise we should free the sk_buffer, the device
	 * should be doing the rest of the work now.
	 */
	if (ring->index == 1) {
		rt2x00_ring_index_done_inc(ring);
		entry = rt2x00_get_data_entry(ring);
		usb_submit_urb(entry->priv, GFP_ATOMIC);
		rt2x00_ring_index_inc(ring);
	} else if (ring->index_done == 1) {
		entry = rt2x00_get_data_entry_done(ring);
		if (entry->skb) {
			dev_kfree_skb(entry->skb);
			entry->skb = NULL;
		}
		rt2x00_ring_index_done_inc(ring);
	}
}

static void rt73usb_interrupt_rxdone(struct urb *urb)
{
	struct data_entry *entry = (struct data_entry*)urb->context;
	struct data_ring *ring = entry->ring;
	struct rt2x00_dev *rt2x00dev = ring->rt2x00dev;
	struct data_desc *rxd = (struct data_desc*)entry->skb->data;
	struct sk_buff *skb;
	u32 word0;
	u32 word1;
	int signal;
	int rssi;
	int ofdm;
	u16 size;

	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO) ||
	    !GET_FLAG(entry, ENTRY_OWNER_NIC))
		return;

	CLEAR_FLAG(entry, ENTRY_OWNER_NIC);

	/*
	 * Check if the received data is simply too small
	 * to be actually valid, or if the urb is signaling
	 * a problem.
	 */
	if (urb->actual_length < entry->ring->desc_size || urb->status)
		goto skip_entry;

	rt2x00_desc_read(rxd, 0, &word0);
	rt2x00_desc_read(rxd, 1, &word1);

	/*
	 * TODO: Don't we need to keep statistics
	 * updated about events like CRC and physical errors?
	 */
	if (rt2x00_get_field32(word0, RXD_W0_CRC) ||
	    rt2x00_get_field32(word0, RXD_W0_CIPHER_ERROR))
		goto skip_entry;

	/*
	 * Allocate a new sk_buffer to replace the sk_buffer
	 * that has been filled in by the device.
	 */
	skb = dev_alloc_skb(NET_IP_ALIGN + ring->data_size + ring->desc_size);
	if (!skb)
		return;

	skb_reserve(skb, NET_IP_ALIGN);
	skb_put(skb, ring->data_size + ring->desc_size);

	urb->transfer_buffer = skb->data;
	urb->transfer_buffer_length = skb->len;

	signal = rt2x00_get_field32(word1, RXD_W1_SIGNAL);
	rssi = rt2x00_get_field32(word1, RXD_W1_RSSI);
	ofdm = rt2x00_get_field32(word0, RXD_W0_OFDM);
	rt2x00lib_update_rx_stats(rt2x00dev, signal, rssi, ofdm);

	/*
	 * Trim the skb_buffer to only contain the valid
	 * frame data (so ignore the device's descriptor).
	 */
	size = rt2x00_get_field32(word0, RXD_W0_DATABYTE_COUNT);
	skb_pull(entry->skb, ring->desc_size);
	skb_trim(entry->skb, size);

	/*
	 * Send frame to stack, and set the new sk_buffer
	 * in its place to be able to receive new frames
	 * in the new buffer.
	 */
	ieee80211_rx_irqsafe(rt2x00dev->hw, entry->skb, &rt2x00dev->rx_status);
	entry->skb = skb;

	/*
	 * Update link statistics
	 */
	rt2x00_update_link_rssi(&rt2x00dev->link, rt2x00dev->rx_status.ssi);

skip_entry:
	if (!GET_FLAG(ring->rt2x00dev, DEVICE_ENABLED_RADIO))
		return;

	SET_FLAG(entry, ENTRY_OWNER_NIC);
	usb_submit_urb(urb, GFP_ATOMIC);
	rt2x00_ring_index_inc(ring);
}

static void rt73usb_interrupt_txdone(struct urb *urb)
{
	struct data_entry *entry = (struct data_entry*)urb->context;
	struct data_ring *ring = entry->ring;
	struct rt2x00_dev *rt2x00dev = ring->rt2x00dev;
	struct data_desc *txd = (struct data_desc *)entry->skb->data;
	u32 word;
	int tx_status;
	int ack;

	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO) ||
	    !GET_FLAG(entry, ENTRY_OWNER_NIC))
		return;

	CLEAR_FLAG(entry, ENTRY_OWNER_NIC);

	rt2x00_desc_read(txd, 0, &word);

	ack = rt2x00_get_field32(word, TXD_W0_ACK);
	tx_status = !urb->status ? TX_SUCCESS : TX_FAIL_RETRY;
	rt2x00lib_update_tx_stats(entry, tx_status, ack, 0);

	skb_pull(entry->skb, ring->desc_size);

	/*
	 * If this is not an RTS frame send the tx_status to mac80211,
	 * that method also cleans up the skb structure. When this
	 * is a RTS frame, that it is our job to clean this structure up.
	 */
	if (!GET_FLAG(entry, ENTRY_RTS_CTS_FRAME))
		ieee80211_tx_status_irqsafe(rt2x00dev->hw,
			entry->skb, &entry->tx_status);
	else
		dev_kfree_skb(entry->skb);

	entry->skb = NULL;

	CLEAR_FLAG(entry, ENTRY_RTS_CTS_FRAME);

	rt2x00_ring_index_done_inc(entry->ring);

	/*
	 * Check if we are waiting on an empty queue
	 * to start scanning.
	 */
	if (rt2x00dev->scan &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_AC_VO]) &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_AC_VI]) &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_AC_BE]) &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_AC_BK]) &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_PRIO]))
		rt2x00_signal_scan(rt2x00dev->scan, SCANNING_READY);

	/*
	 * If the data ring was full before the txdone handler
	 * we must make sure the packet queue in the mac80211 stack
	 * is reenabled when the txdone handler has finished.
	 */
	if (!rt2x00_ring_full(ring))
		ieee80211_wake_queue(rt2x00dev->hw,
			entry->tx_status.control.queue);
}

/*
 * IEEE80211 stack callback functions.
 */
static int rt73usb_get_stats(struct ieee80211_hw *hw,
	struct ieee80211_low_level_stats *stats)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	u32 reg;

	/*
	 * Update FCS error count from register.
	 * The dot11ACKFailureCount, dot11RTSFailureCount and
	 * dot11RTSSuccessCount are updated in interrupt time.
	 */
	rt2x00_register_read(rt2x00dev, STA_CSR0, &reg);
	rt2x00dev->low_level_stats.dot11FCSErrorCount +=
		rt2x00_get_field32(reg, STA_CSR0_FCS_ERROR);

	memcpy(stats, &rt2x00dev->low_level_stats, sizeof(*stats));

	return 0;
}

static int rt73usb_set_retry_limit(struct ieee80211_hw *hw,
	u32 short_retry, u32 long_retry)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	u32 reg;

	rt2x00_register_read(rt2x00dev, TXRX_CSR4, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR4_LONG_RETRY_LIMIT, long_retry);
	rt2x00_set_field32(&reg, TXRX_CSR4_SHORT_RETRY_LIMIT, short_retry);
	rt2x00_register_write(rt2x00dev, TXRX_CSR4, reg);

	return 0;
}

static u64 rt73usb_get_tsf(struct ieee80211_hw *hw)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	u64 tsf;
	u32 reg;

	rt2x00_register_read(rt2x00dev, TXRX_CSR13, &reg);
	tsf = (u64)rt2x00_get_field32(reg, TXRX_CSR13_HIGH_TSFTIMER) << 32;
	rt2x00_register_read(rt2x00dev, TXRX_CSR12, &reg);
	tsf |= rt2x00_get_field32(reg, TXRX_CSR12_LOW_TSFTIMER);

	return tsf;
}

static void rt73usb_reset_tsf(struct ieee80211_hw *hw)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	rt2x00_register_write(rt2x00dev, TXRX_CSR12, 0);
	rt2x00_register_write(rt2x00dev, TXRX_CSR13, 0);
}

static int rt73usb_beacon_update(struct ieee80211_hw *hw,
	struct sk_buff *skb, struct ieee80211_tx_control *control)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	struct usb_device *usb_dev =
		interface_to_usbdev(rt2x00dev_usb(rt2x00dev));
	struct data_ring *ring = &rt2x00dev->ring[RING_BEACON];
	struct data_entry *beacon;
	struct data_entry *guardian;
	int length;
	u32 reg;

	/*
	 * Just in case the ieee80211 doesn't set this,
	 * but we need this queue set for the descriptor
	 * initialization.
	 */
	control->queue = IEEE80211_TX_QUEUE_BEACON;

	/*
	 * Obtain 2 entries, one for the guardian byte,
	 * the second for the actual beacon.
	 */
	guardian = rt2x00_get_data_entry(ring);
	rt2x00_ring_index_inc(ring);
	beacon = rt2x00_get_data_entry(ring);

	/*
	 * First we create the beacon.
	 */
	skb_push(skb, ring->desc_size);
	rt73usb_write_tx_desc(rt2x00dev,
		(struct data_desc*)skb->data,
		(struct ieee80211_hdr*)(skb->data + ring->desc_size),
		skb->len - ring->desc_size,
		control);

	/*
	 * Length passed to usb_fill_urb cannot be an odd number,
	 * so add 1 byte to make it even.
	 */
	length = skb->len;
	if (length % 2)
		length++;

	usb_fill_bulk_urb(
		beacon->priv,
		usb_dev,
		usb_sndbulkpipe(usb_dev, 1),
		skb->data,
		length,
		rt73usb_interrupt_beacondone,
		beacon);

	beacon->skb = skb;

	/*
	 * Second we need to create the guardian byte.
	 * We only need a single byte, so lets recycle
	 * the 'flags' field we are not using for beacons.
	 */
	guardian->reg = 0;
	usb_fill_bulk_urb(
		guardian->priv,
		usb_dev,
		usb_sndbulkpipe(usb_dev, 1),
		&guardian->reg,
		1,
		rt73usb_interrupt_beacondone,
		guardian);

	/*
	 * Send out the guardian byte.
	 */
	usb_submit_urb(guardian->priv, GFP_ATOMIC);

	/*
	 * Enable beacon generation.
	 */
	rt2x00_register_read(rt2x00dev, TXRX_CSR9, &reg);
	if (!rt2x00_get_field32(reg, TXRX_CSR9_BEACON_GEN)) {
		rt2x00_set_field32(&reg, TXRX_CSR9_BEACON_GEN, 1);
		rt2x00_register_write(rt2x00dev, TXRX_CSR9, reg);
	}

	return 0;
}

static const struct ieee80211_ops rt73usb_mac80211_ops = {
	.tx			= rt2x00lib_tx,
	.reset			= rt2x00lib_reset,
	.add_interface		= rt2x00lib_add_interface,
	.remove_interface	= rt2x00lib_remove_interface,
	.config			= rt2x00lib_config,
	.config_interface	= rt2x00lib_config_interface,
	.set_multicast_list	= rt2x00lib_set_multicast_list,
	.passive_scan		= rt2x00lib_passive_scan,
	.get_stats		= rt73usb_get_stats,
	.set_retry_limit	= rt73usb_set_retry_limit,
	.conf_tx		= rt2x00lib_conf_tx,
	.get_tx_stats		= rt2x00lib_get_tx_stats,
	.get_tsf		= rt73usb_get_tsf,
	.reset_tsf		= rt73usb_reset_tsf,
	.beacon_update		= rt73usb_beacon_update,
};

static const struct rt2x00lib_ops rt73usb_rt2x00_ops = {
	.initialize		= rt73usb_initialize,
	.uninitialize		= rt73usb_uninitialize,
	.enable_radio		= rt73usb_enable_radio,
	.disable_radio		= rt73usb_disable_radio,
	.toggle_rx		= rt73usb_toggle_rx,
	.write_tx_data		= rt73usb_write_tx_data,
	.config_type		= rt73usb_config_type,
	.config_phymode		= rt73usb_config_phymode,
	.config_channel		= rt73usb_config_channel,
	.config_mac_addr	= rt73usb_config_mac_addr,
	.config_bssid		= rt73usb_config_bssid,
	.config_txpower		= rt73usb_config_txpower,
	.config_antenna		= rt73usb_config_antenna,
	.config_duration	= rt73usb_config_duration,
};

/*
 * Device initialization functions.
 */
static int rt73usb_alloc_eeprom(struct rt2x00_dev *rt2x00dev)
{
	/*
	 * Allocate the eeprom memory, check the eeprom width
	 * and copy the entire eeprom into this allocated memory.
	 */
	rt2x00dev->eeprom = kzalloc(EEPROM_SIZE, GFP_KERNEL);
	if (!rt2x00dev->eeprom)
		return -ENOMEM;

	rt2x00_vendor_request(
		rt2x00dev, USB_EEPROM_READ, USB_VENDOR_REQUEST_IN,
		EEPROM_BASE, 0x00, rt2x00dev->eeprom, EEPROM_SIZE,
		REGISTER_TIMEOUT * (EEPROM_SIZE / sizeof(u16)));

	return 0;
}

static int rt73usb_alloc_rings(struct rt2x00_dev *rt2x00dev)
{
	unsigned int i;

	rt2x00dev->ring = kzalloc(
		sizeof(struct data_ring) * RING_NUM, GFP_KERNEL);
	if (!rt2x00dev->ring) {
		ERROR("Ring allocation failed.\n");
		return -ENOMEM;
	}

	for (i = 0; i < RING_NUM; i++) {
		rt2x00dev->ring[i].rt2x00dev = rt2x00dev;

		/*
		 * Initialize ring parameters.
		 * cw_min: 2^5 = 32.
		 * cw_max: 2^10 = 1024.
		 */
		rt2x00dev->ring[i].tx_params.aifs = 2;
		rt2x00dev->ring[i].tx_params.cw_min = 5;
		rt2x00dev->ring[i].tx_params.cw_max = 10;
	}

	return 0;
}

static int rt73usb_init_eeprom(struct rt2x00_dev *rt2x00dev)
{
	u32 reg;
	u16 value;
	u16 eeprom;

	/*
	 * Read EEPROM word for configuration.
	 */
	rt2x00_eeprom_read(rt2x00dev, EEPROM_ANTENNA, &eeprom);

	/*
	 * Identify RF chipset.
	 */
	value = rt2x00_get_field16(eeprom, EEPROM_ANTENNA_RF_TYPE);
	rt2x00_register_read(rt2x00dev, MAC_CSR0, &reg);
	rt2x00_set_chip(&rt2x00dev->chip, RT73, value, reg);

	if (!rt2x00_rf(&rt2x00dev->chip, RF5226) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF2528) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF5225) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF2527)) {
		ERROR("Invalid RF chipset detected.");
		return -ENODEV;
	}

	/*
	 * Identify default antenna configuration.
	 */
	rt2x00dev->hw->conf.antenna_sel_tx = rt2x00_get_field16(eeprom,
		EEPROM_ANTENNA_TX_DEFAULT);
	rt2x00dev->hw->conf.antenna_sel_rx = rt2x00_get_field16(eeprom,
		EEPROM_ANTENNA_RX_DEFAULT);

	/*
	 * Read the Frame type.
	 */
	if (rt2x00_get_field16(eeprom, EEPROM_ANTENNA_FRAME_TYPE))
		SET_FLAG(rt2x00dev, CONFIG_FRAME_TYPE);

	/*
	 * Read frequency offset.
	 */
	rt2x00_eeprom_read(rt2x00dev, EEPROM_FREQ, &eeprom);
	rt2x00dev->freq_offset = rt2x00_get_field16(eeprom,
					EEPROM_FREQ_OFFSET);
	if (rt2x00dev->freq_offset == 0xff)
		rt2x00dev->freq_offset = 0;

	/*
	 * Read external LNA informations.
	 */
	rt2x00_eeprom_read(rt2x00dev, EEPROM_NIC, &eeprom);
	if (eeprom == 0xffff)
		eeprom = 0;
	if (rt2x00_get_field16(eeprom, EEPROM_NIC_EXTERNAL_LNA))
		SET_FLAG(rt2x00dev, CONFIG_EXTERNAL_LNA);

	/*
	 * Store led settings, for correct led behaviour.
	 */
	rt2x00_eeprom_read(rt2x00dev, EEPROM_LED, &eeprom);

	/*
	 * If the eeprom value is invalid,
	 * switch to default led mode, and clear all bits.
	 */
	if (eeprom == 0xffff) {
		rt2x00dev->led_mode = LED_MODE_DEFAULT;
		eeprom = 0x0000;
	} else
		rt2x00dev->led_mode = rt2x00_get_field16(eeprom,
			EEPROM_LED_LED_MODE);

	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_LED_MODE,
		rt2x00dev->led_mode);
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_GPIO_0,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_0));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_GPIO_1,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_1));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_GPIO_2,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_2));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_GPIO_3,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_3));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_GPIO_4,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_4));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_ACT,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_ACT));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_READY_BG,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_RDY_G));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_READY_A,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_RDY_A));

	return 0;
}

static int rt73usb_init_hw_mac(struct rt2x00_dev *rt2x00dev)
{
	u8 *addr;

	/*
	 * Get the pointer to the MAC address in the EEPROM.
	 */
	addr = rt2x00_eeprom_addr(rt2x00dev, EEPROM_MAC_ADDR_0);

	/*
	 * Check if a valid MAC address is present.
	 */
	if (!is_valid_ether_addr(addr)) {
		ERROR("Invalid MAC address: " MAC_FMT ".\n", MAC_ARG(addr));
		return -EINVAL;
	}

	/*
	 * Write MAC address to register.
	 */
	rt73usb_config_mac_addr(rt2x00dev, addr);

	/*
	 * Copy MAC address to the hw structure.
	 */
	SET_IEEE80211_PERM_ADDR(rt2x00dev->hw, addr);

	return 0;
}

static void rt73usb_init_hw_channels(struct rt2x00_dev *rt2x00dev,
	struct ieee80211_channel *channels)
{
	unsigned int i;
	u32 rf2_base;
	u16 eeprom;
	static const struct {
		unsigned int chip;
		u32 val[3];
	} rf[] = {
		{ RF5226,	{ 0x00002c0c, 0x00068255 } },
		{ RF2528,	{ 0x00002c0c, 0x00068255 } },
		{ RF5225,	{ 0x00002ccc, 0x00000000 } },
		{ RF2527,	{ 0x00002ccc, 0x00068455 } },
	};
	static const u32 vals[] = {
		0x00000786, 0x00000786, 0x0000078a, 0x0000078a,
		0x0000078e, 0x0000078e, 0x00000792, 0x00000792,
		0x00000796, 0x00000796, 0x0000079a, 0x0000079a,
		0x0000079e, 0x000007a2
	};

	/*
	 * Channel initialization.
	 * First we set the basic variables.
	 */
	for (i = 0; i < 13; i++) {
		channels[i].chan = i + 1;
		channels[i].freq = 2407 + ((i + 1) * 5);
		channels[i].flag = IEEE80211_CHAN_W_IBSS |
			IEEE80211_CHAN_W_ACTIVE_SCAN | IEEE80211_CHAN_W_SCAN;
		channels[i].antenna_max = 0xff;
	}

	channels[13].chan = 14;
	channels[13].freq = 2484;
	channels[13].flag = IEEE80211_CHAN_W_IBSS |
		IEEE80211_CHAN_W_ACTIVE_SCAN | IEEE80211_CHAN_W_SCAN;
	channels[13].antenna_max = 0xff;

	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5226)) {
		for (i = 14; i < 38; i++) {
			if (i < 22)
				channels[i].chan = 36;
			else if (i < 33)
				channels[i].chan = 100;
			else
				channels[i].chan = 149;
			channels[i].chan += ((i - 14) * 4);
			channels[i].freq = ((i - 13) + 1000) * 5;
			channels[i].flag = IEEE80211_CHAN_W_IBSS |
				IEEE80211_CHAN_W_ACTIVE_SCAN |
				IEEE80211_CHAN_W_SCAN;
			channels[i].antenna_max = 0xff;
		}
	}

	/*
	 * Set device specific value.
	 */
	rf2_base = 0;
	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF2527))
		rf2_base = 0x00004000;

	for (i = 0; i < ARRAY_SIZE(vals); i++)
		channels[i].val = vals[i] | rf2_base;

	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5226)) {
		static const u32 vals_a[] = {
			0x0000099a, 0x000009a2, 0x000009a6, 0x000009aa,
			0x000009ae, 0x000009b2, 0x000009ba, 0x000009be,
			0x00000a2a, 0x00000a2e, 0x00000a32, 0x00000a36,
			0x00000a3a, 0x00000a82, 0x00000a86, 0x00000a8a,
			0x00000a8e, 0x00000a92, 0x00000a9a, 0x00000aa2,
			0x00000aa6, 0x00000aae, 0x00000ab2, 0x00000ab6
		};

		struct ieee80211_channel *chan = channels + 14;

		for (i = 0; i < ARRAY_SIZE(vals_a); i++)
			(chan++)->val = vals_a[i];
	}

	/*
	 * Set TX power, each EEPROM TXpower entry
	 * contains the TXpower value for 2 channels.
	 */
	for (i = 0; i < EEPROM_TXPOWER_G_SIZE; i++) {
		rt2x00_eeprom_read(rt2x00dev,
			EEPROM_TXPOWER_G_START + i, &eeprom);

		channels[(i * 2)].power_level = TXPOWER_FROM_DEV(
			rt2x00_get_field16(eeprom, EEPROM_TXPOWER_G_1));

		channels[(i * 2) + 1].power_level = TXPOWER_FROM_DEV(
			rt2x00_get_field16(eeprom, EEPROM_TXPOWER_G_2));
	}

	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5226)) {
		for (i = 0; i < EEPROM_TXPOWER_A_SIZE; i++) {
			rt2x00_eeprom_read(rt2x00dev,
				EEPROM_TXPOWER_A_START + i, &eeprom);

			channels[(i * 2)].power_level = TXPOWER_FROM_DEV(
				rt2x00_get_field16(eeprom,
					EEPROM_TXPOWER_A_1));

			channels[(i * 2) + 1].power_level = TXPOWER_FROM_DEV(
				rt2x00_get_field16(eeprom,
					EEPROM_TXPOWER_A_2));
		}
	}

	/*
	 * Set device specific, but channel independent RF values.
	 */
	for (i = 0; i < ARRAY_SIZE(rf); i++) {
		if (rt2x00_rf(&rt2x00dev->chip, rf[i].chip)) {
			rt2x00dev->rf1 = rf[i].val[0];
			rt2x00dev->rf3 = rf[i].val[1];
		}
	}
}

static void rt73usb_init_hw_rates(struct rt2x00_dev *rt2x00dev,
	struct ieee80211_rate *rates)
{
	/*
	 * Rates initialization.
	 */
	device_rate_entry(&rates[0], 10, 0x001, 0x00, IEEE80211_RATE_CCK);
	device_rate_entry(&rates[1], 20, 0x003, 0x01, IEEE80211_RATE_CCK_2);
	device_rate_entry(&rates[2], 55, 0x007, 0x02, IEEE80211_RATE_CCK_2);
	device_rate_entry(&rates[3], 110, 0x00f, 0x03, IEEE80211_RATE_CCK_2);
	device_rate_entry(&rates[4], 60, 0x01f, 0x0b, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[5], 90, 0x03f, 0x0f, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[6], 120, 0x07f, 0x0a, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[7], 180, 0x0ff, 0x0e, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[8], 240, 0x1ff, 0x09, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[9], 360, 0x3ff, 0x0d, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[10], 480, 0x7ff, 0x08, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[11], 540, 0xfff, 0x0c, IEEE80211_RATE_OFDM);
}

static int rt73usb_init_hw_modes(struct rt2x00_dev *rt2x00dev)
{
	struct ieee80211_channel *channels;
	struct ieee80211_rate *rates;
	int status = -ENOMEM;
	int num_modes;
	int num_channels;

	/*
	 * RF2527 and RF2528 only supports 802.11b & 802.11g,
	 * so we should allocate 14 OFDM channels, 4 CCK rates
	 * and 8 OFDM rates.
	 * RF5225 and RF5226 also supports 802.11a, so allocate an
	 * additional 24 5.2GHz channels.
	 */
	num_modes = 2;
	num_channels = 14;
	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5226)) {
		num_modes = 3;
		num_channels = 38;
	}

	rt2x00dev->hwmodes =
		kzalloc(sizeof(struct ieee80211_hw_mode) * num_modes,
			GFP_KERNEL);
	if (!rt2x00dev->hwmodes)
		goto exit;

	channels = kzalloc(sizeof(struct ieee80211_channel) * num_channels,
		GFP_KERNEL);
	if (!channels)
		goto exit_free_modes;

	rates = kzalloc(sizeof(struct ieee80211_rate) * 12, GFP_KERNEL);
	if (!rates)
		goto exit_free_channels;

	/*
	 * Initialize channels and rate arrays.
	 */
	rt73usb_init_hw_channels(rt2x00dev, channels);
	rt73usb_init_hw_rates(rt2x00dev, rates);

	/*
	 * Intitialize 802.11b
	 * Rates: CCK.
	 * Channels: OFDM.
	 */
	rt2x00dev->hwmodes[HWMODE_B].mode = MODE_IEEE80211B;
	rt2x00dev->hwmodes[HWMODE_B].num_channels = 14;
	rt2x00dev->hwmodes[HWMODE_B].num_rates = 4;
	rt2x00dev->hwmodes[HWMODE_B].channels = channels;
	rt2x00dev->hwmodes[HWMODE_B].rates = rates;

	/*
	 * Intitialize 802.11g
	 * Rates: CCK, OFDM.
	 * Channels: OFDM.
	 */
	rt2x00dev->hwmodes[HWMODE_G].mode = MODE_IEEE80211G;
	rt2x00dev->hwmodes[HWMODE_G].num_channels = 14;
	rt2x00dev->hwmodes[HWMODE_G].num_rates = 12;
	rt2x00dev->hwmodes[HWMODE_G].channels = channels;
	rt2x00dev->hwmodes[HWMODE_G].rates = rates;

	/*
	 * Intitialize 802.11a
	 * Rates: OFDM.
	 * Channels: OFDM, UNII, HiperLAN2.
	 */
	if (num_channels == 3) {
		rt2x00dev->hwmodes[HWMODE_A].mode = MODE_IEEE80211A;
		rt2x00dev->hwmodes[HWMODE_A].num_channels = 24;
		rt2x00dev->hwmodes[HWMODE_A].num_rates = 8;
		rt2x00dev->hwmodes[HWMODE_A].channels = &channels[14];
		rt2x00dev->hwmodes[HWMODE_A].rates = &rates[4];
	}

	/*
	 * Register the working modes.
	 */
	status = ieee80211_register_hwmode(rt2x00dev->hw,
		&rt2x00dev->hwmodes[HWMODE_G]);
	if (status)
		goto exit_free_rates;

	status = ieee80211_register_hwmode(rt2x00dev->hw,
		&rt2x00dev->hwmodes[HWMODE_B]);
	if (status)
		goto exit_free_rates;

	if (num_modes == 3) {
		status = ieee80211_register_hwmode(rt2x00dev->hw,
			&rt2x00dev->hwmodes[HWMODE_A]);
		if (status)
			goto exit_free_rates;
	}

	return 0;

exit_free_rates:
	kfree(rates);

exit_free_channels:
	kfree(channels);

exit_free_modes:
	kfree(rt2x00dev->hwmodes);
	rt2x00dev->hwmodes = NULL;

exit:
	ERROR("Allocation ieee80211 modes failed.\n");
	return status;
}

static int rt73usb_init_hw(struct rt2x00_dev *rt2x00dev)
{
	int status;

	if (GET_FLAG(rt2x00dev, DEVICE_INITIALIZED_HW))
		return 0;

	SET_IEEE80211_DEV(rt2x00dev->hw, &rt2x00dev_usb(rt2x00dev)->dev);

	/*
	 * Read MAC address from EEPROM.
	 */
	status = rt73usb_init_hw_mac(rt2x00dev);
	if (status)
		return status;

	/*
	 * Initialize all hw fields.
	 */
	rt2x00dev->hw->flags = IEEE80211_HW_HOST_GEN_BEACON_TEMPLATE |
		IEEE80211_HW_HOST_BROADCAST_PS_BUFFERING |
		IEEE80211_HW_WEP_INCLUDE_IV |
		IEEE80211_HW_DATA_NULLFUNC_ACK |
		IEEE80211_HW_NO_TKIP_WMM_HWACCEL |
		IEEE80211_HW_MONITOR_DURING_OPER;
	rt2x00dev->hw->extra_tx_headroom = TXD_DESC_SIZE;
	rt2x00dev->hw->max_rssi = MAX_RX_SSI;
	rt2x00dev->hw->max_noise = MAX_RX_NOISE;
	rt2x00dev->hw->queues = RING_NUM_TX;

	if (ieee80211_register_hw(rt2x00dev->hw))
		return -EIO;

	status = rt73usb_init_hw_modes(rt2x00dev);
	if (status) {
		ieee80211_unregister_hw(rt2x00dev->hw);
		return status;
	}

	SET_FLAG(rt2x00dev, DEVICE_INITIALIZED_HW);

	return 0;
}

static void rt73usb_load_firmware(const struct firmware *fw, void *context)
{
	struct rt2x00_dev *rt2x00dev = context;
	unsigned int i;
	int status;
	u32 reg;
	u16 crc;

	if (!fw || !fw->size || !fw->data) {
		ERROR("Failed to load Firmware.\n");
		goto exit;
	}

	/*
	 * Wait for stable hardware.
	 */
	for (i = 0; i < 100; i++) {
		rt2x00_register_read(rt2x00dev, MAC_CSR0, &reg);
		if (reg)
			break;
		msleep(1);
	}

	if (!reg) {
		ERROR("Unstable hardware.\n");
		goto exit;
	}

	/*
	 * Validate the firmware using 16 bit CRC.
	 * The last 2 bytes of the firmware are the CRC
	 * so substract those 2 bytes from the CRC checksum,
	 * and set those 2 bytes to 0 when calculating CRC.
	 */
	reg = 0;
	crc = crc_itu_t(0, fw->data, fw->size - 2);
	crc = crc_itu_t(crc, (u8*)&reg, 2);

	if (crc != (fw->data[fw->size - 2] << 8 | fw->data[fw->size - 1])) {
		ERROR("Firmware CRC error.\n");
		goto exit;
	}

	rt2x00_set_chip_fw(&rt2x00dev->chip,
		fw->data[fw->size - 4], fw->data[fw->size - 3]);

	/*
	 * Write firmware to device.
	 */
	for (i = 0; i < fw->size; i += sizeof(u32)) {
		reg = *((u32*)&fw->data[i]);
		rt2x00_register_multiwrite(rt2x00dev, FIRMWARE_IMAGE_BASE + i,
			&reg, sizeof(u32));
	}

	/*
	 * Send firmware request to device to load firmware,
	 * we need to specify a long timeout time.
	 */
	status = rt2x00_vendor_request(rt2x00dev, USB_DEVICE_MODE,
		USB_VENDOR_REQUEST_OUT, 0x00, USB_MODE_FIRMWARE,
		NULL, 0, REGISTER_TIMEOUT_FIRMWARE);
	if (status  < 0) {
		ERROR("Failed to write Firmware to device.\n");
		goto exit;
	}

	SET_FLAG(rt2x00dev, FIRMWARE_LOADED);

	/*
	 * Initialize the remaining bits of the device
	 * that had to wait until the firmware was loaded.
	 */
	if (rt73usb_init_hw(rt2x00dev)) {
		ERROR("Failed to initialize device.\n");
		return;
	}

	rt73usb_disable_led(rt2x00dev);
	rt73usb_open_debugfs(rt2x00dev);

	return;

exit:
	SET_FLAG(rt2x00dev, FIRMWARE_FAILED);
}

static void rt73usb_free_dev(struct rt2x00_dev *rt2x00dev)
{
	/*
	 * Close debugfs entry.
	 */
	rt73usb_close_debugfs(rt2x00dev);

	/*
	 * Free workqueue.
	 */
	if (likely(rt2x00dev->workqueue)) {
		destroy_workqueue(rt2x00dev->workqueue);
		rt2x00dev->workqueue = NULL;
	}

	/*
	 * Free ring structures.
	 */
	kfree(rt2x00dev->ring);
	rt2x00dev->ring = NULL;

	/*
	 * Free EEPROM memory.
	 */
	kfree(rt2x00dev->eeprom);

	/*
	 * Free ieee80211_hw memory.
	 */
	if (likely(rt2x00dev->hwmodes)) {
		kfree(rt2x00dev->hwmodes->channels);
		kfree(rt2x00dev->hwmodes->rates);
		kfree(rt2x00dev->hwmodes);
		rt2x00dev->hwmodes = NULL;
	}
}

static int rt73usb_alloc_dev(struct usb_interface *usb_intf,
	struct ieee80211_hw *hw)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	rt2x00dev->dev = usb_intf;
	rt2x00dev->lib_ops = &rt73usb_rt2x00_ops;
	rt2x00dev->hw_ops = &rt73usb_mac80211_ops;
	rt2x00dev->hw = hw;

	/*
	 * Allocate eeprom data.
	 */
	if (rt73usb_alloc_eeprom(rt2x00dev))
		goto exit;

	/*
	 * Create workqueue.
	 */
	rt2x00dev->workqueue = create_singlethread_workqueue(DRV_NAME);
	if (!rt2x00dev->workqueue)
		return -ENODEV;

	/*
	 * Initialize configuration work.
	 */
	INIT_DELAYED_WORK(&rt2x00dev->link.work, rt73usb_link_tuner);

	/*
	 * Reset current working type.
	 */
	rt2x00dev->interface.type = -EINVAL;

	/*
	 * Intialize scanning attributes.
	 */
	rt2x00dev->scan = NULL;

	/*
	 * Allocate ring array.
	 */
	if (rt73usb_alloc_rings(rt2x00dev))
		goto exit;

	/*
	 * Initialize hardware.
	 */
	if (rt73usb_init_eeprom(rt2x00dev)) {
		ERROR("Failed to initialize device.\n");
		goto exit;
	}

	/*
	 * Request firmware and wait with further
	 * initializing of the card until the firmware
	 * has been loaded.
	 */
	if (rt2x00lib_load_firmware(rt2x00dev, &rt2x00dev_usb(rt2x00dev)->dev,
		rt73usb_load_firmware))
		goto exit;

	return 0;

exit:
	rt73usb_free_dev(rt2x00dev);

	return -ENODEV;
}

/*
 * USB driver handlers.
 */
static int rt73usb_probe(struct usb_interface *usb_intf,
	const struct usb_device_id *id)
{
	struct usb_device *usb_dev = interface_to_usbdev(usb_intf);
	struct ieee80211_hw *hw;
	int status;

	usb_dev = usb_get_dev(usb_dev);

	hw = ieee80211_alloc_hw(sizeof(struct rt2x00_dev),
		&rt73usb_mac80211_ops);
	if (!hw) {
		ERROR("Failed to allocate hardware.\n");
		status = -ENOMEM;
		goto exit_put_device;
	}

	usb_set_intfdata(usb_intf, hw);

	status = rt73usb_alloc_dev(usb_intf, hw);
	if (status) {
		ERROR("Failed to allocate device.\n");
		goto exit_free_device;
	}

	ieee80211_netif_oper(hw, NETIF_ATTACH);

	return 0;

exit_free_device:
	ieee80211_free_hw(hw);

exit_put_device:
	usb_put_dev(usb_dev);

	return status;
}

static void rt73usb_disconnect(struct usb_interface *usb_intf)
{
	struct ieee80211_hw *hw = usb_get_intfdata(usb_intf);
	struct rt2x00_dev *rt2x00dev = hw->priv;

	/*
	 * Uninitialize the 80211 stack data.
	 */
	ieee80211_netif_oper(hw, NETIF_DETACH);
	ieee80211_unregister_hw(hw);

	/*
	 * Uninitialize and free the rt73usb driver data.
	 */
	rt73usb_disable_radio(rt2x00dev);
	rt73usb_uninitialize(rt2x00dev);
	rt73usb_free_dev(rt2x00dev);

	/*
	 * Free the 80211 stack data.
	 */
	ieee80211_free_hw(hw);

	/*
	 * Free the USB device data.
	 */
	usb_set_intfdata(usb_intf, NULL);
	usb_put_dev(interface_to_usbdev(usb_intf));
}

#ifdef CONFIG_PM
static int rt73usb_suspend(struct usb_interface *usb_intf, pm_message_t state)
{
	struct ieee80211_hw *hw = usb_get_intfdata(usb_intf);
	struct rt2x00_dev *rt2x00dev = hw->priv;
	int status;

	NOTICE("Going to sleep.\n");

	ieee80211_netif_oper(hw, NETIF_DETACH);

	/*
	 * Disable the radio.
	 */
	rt73usb_disable_radio(rt2x00dev);

	/*
	 * Set device mode to sleep for power management.
	 */
	status = rt73usb_set_state(rt2x00dev, STATE_SLEEP);
	if (status)
		return status;

	/*
	 * Uninitialize device and hardware.
	 */
	rt73usb_uninitialize(rt2x00dev);
	rt73usb_free_dev(rt2x00dev);

	/*
	 * Decrease usbdev refcount.
	 */
	usb_put_dev(interface_to_usbdev(usb_intf));

	return 0;
}

static int rt73usb_resume(struct usb_interface *usb_intf)
{
	struct ieee80211_hw *hw = usb_get_intfdata(usb_intf);
	struct rt2x00_dev *rt2x00dev = hw->priv;
	int status;

	NOTICE("Waking up.\n");

	/*
	 * Increase usbdev refcount.
	 */
	usb_get_dev(interface_to_usbdev(usb_intf));

	/*
	 * Initialize hardware.
	 */
	status = rt73usb_alloc_dev(usb_intf, hw);
	if (status) {
		ERROR("Failed to allocate device.\n");
		return status;
	}

	/*
	 * Set device mode to awake for power management.
	 */
	status = rt73usb_set_state(rt2x00dev, STATE_AWAKE);
	if (status)
		return status;

	ieee80211_netif_oper(hw, NETIF_ATTACH);
	return 0;
}
#endif /* CONFIG_PM */

/*
 * rt73usb module information.
 */
static char version[] =
	DRV_NAME " - " DRV_VERSION " (" DRV_RELDATE ") by " DRV_PROJECT;

static struct usb_device_id rt73usb_device_table[] = {
	/* AboCom */
	{ USB_DEVICE(0x07b8, 0xb21d) },
	/* Askey */
	{ USB_DEVICE(0x1690, 0x0722) },
	/* ASUS */
	{ USB_DEVICE(0x0b05, 0x1723) },
	/* Belkin */
	{ USB_DEVICE(0x050d, 0x7050) },
	{ USB_DEVICE(0x050d, 0x705a) },
	{ USB_DEVICE(0x050d, 0x905b) },
	/* Billionton */
	{ USB_DEVICE(0x1631, 0xc019) },
	/* CNet */
	{ USB_DEVICE(0x1371, 0x9022) },
	{ USB_DEVICE(0x1371, 0x9032) },
	/* Conceptronic */
	{ USB_DEVICE(0x14b2, 0x3c22) },
	/* D-Link */
	{ USB_DEVICE(0x07d1, 0x3c03) },
	{ USB_DEVICE(0x07d1, 0x3c04) },
	/* Gemtek */
	{ USB_DEVICE(0x15a9, 0x0004) },
	/* Gigabyte */
	{ USB_DEVICE(0x1044, 0x8008) },
	{ USB_DEVICE(0x1044, 0x800a) },
	/* Huawei-3Com */
	{ USB_DEVICE(0x1472, 0x0009) },
	/* Hercules */
	{ USB_DEVICE(0x06f8, 0xe010) },
	{ USB_DEVICE(0x06f8, 0xe020) },
	/* Linksys */
	{ USB_DEVICE(0x13b1, 0x0020) },
	{ USB_DEVICE(0x13b1, 0x0023) },
	/* MSI */
	{ USB_DEVICE(0x0db0, 0x6877) },
	{ USB_DEVICE(0x0db0, 0xa861) },
	{ USB_DEVICE(0x0db0, 0xa874) },
	/* Ralink */
	{ USB_DEVICE(0x148f, 0x2573) },
	{ USB_DEVICE(0x148f, 0x2671) },
	/* Qcom */
	{ USB_DEVICE(0x18e8, 0x6196) },
	{ USB_DEVICE(0x18e8, 0x6229) },
	/* Sitecom */
	{ USB_DEVICE(0x0df6, 0x9712) },
	{ USB_DEVICE(0x0df6, 0x90ac) },
	/* Surecom */
	{ USB_DEVICE(0x0769, 0x31f3) },
	/* Planex */
	{ USB_DEVICE(0x2019, 0xab01) },
	{ 0, }
};

MODULE_AUTHOR(DRV_PROJECT);
MODULE_VERSION(DRV_VERSION);
MODULE_DESCRIPTION("Ralink RT73 USB Wireless LAN driver.");
MODULE_SUPPORTED_DEVICE("Ralink RT2573 USB chipset based cards");
MODULE_DEVICE_TABLE(usb, rt73usb_device_table);
MODULE_LICENSE("GPL");

#ifdef CONFIG_RT2X00_DEBUG
module_param_named(debug, rt2x00_debug_level, bool, S_IWUSR | S_IRUGO);
MODULE_PARM_DESC(debug, "Set this parameter to 1 to enable debug output.");
#endif /* CONFIG_RT2X00_DEBUG */

static struct usb_driver rt73usb_driver = {
	.name		= DRV_NAME,
	.id_table	= rt73usb_device_table,
	.probe		= rt73usb_probe,
	.disconnect	= rt73usb_disconnect,
#ifdef CONFIG_PM
	.suspend	= rt73usb_suspend,
	.resume		= rt73usb_resume,
#endif /* CONFIG_PM */
};

static int __init rt73usb_init(void)
{
	printk(KERN_INFO "Loading module: %s.\n", version);
	return usb_register(&rt73usb_driver);
}

static void __exit rt73usb_exit(void)
{
	printk(KERN_INFO "Unloading module: %s.\n", version);
	usb_deregister(&rt73usb_driver);
}

module_init(rt73usb_init);
module_exit(rt73usb_exit);
