#!/bin/sh

set -e

iptables="iptables-1.2.9"
kernel="linux-2.4.26"
patch_o_matic="patch-o-matic-20031219"

dba=$(dpkg-architecture -qDEB_BUILD_ARCH)

pwd="$(pwd)"
build="${pwd}/debian/build"
doc="${build}/doc"
stamp="${build}/stamp"
patches="${pwd}/patches"
upstream="${pwd}/upstream"
arch_patches="${patches}/arch/${dba}"
kernel_profectio="${build}/${kernel}"
patch_o_matic_profectio="${build}/${patch_o_matic}"

unpack() {
  for tarball in "$@"; do
    tarball="${tarball}.tar.bz2"
    dinfo "$tarball"
    bunzip2 -dc "${upstream}/${tarball}" | ( cd "$build"; tar xf - )
  done
}

sgml() {
  for sgmldoc in "$@"; do
    sgmldoc="${sgmldoc}-HOWTO.sgml"
    dinfo "$sgmldoc"
    sgml2html "${upstream}/${sgmldoc}" >/dev/null 2>&1
  done
}

dinfo () {
  echo "${0} - ${*}"
}

test -f "${stamp}/prep" && {
  echo already prepared.
  exit 0
}

# package build paths
dinfo "installing source symlinks."
install -d "$doc" "$stamp"
(cd "$build"; ln -sf "$iptables" iptables_profectio; ln -sf "$kernel" kernel_profectio)

# unpack upstream tarballs
dinfo "unarchiving upstream source..."
unpack "$kernel"
unpack "$iptables"
unpack "$patch_o_matic"


# process sgml HOWTOS
dinfo "processing sgml documents..."
( cd ${doc}; sgml packet-filtering NAT netfilter-hacking netfilter-extensions )

# compile upstream changelog
dinfo "assembling changelog..."
rm -f "${doc}/changelog"
for i in $(cd ${upstream}; echo changes*txt | sort -r); do
  dinfo "$i"
  cat "${upstream}/${i}" >>  "${doc}/changelog"
done
for i in $(cd "$upstream"; echo changes*html | sort -r); do
  dinfo "$i"
  html2text -o - "${upstream}/$i" >> "${doc}/changelog"
done

# patch-o-matic patches
dinfo "applying patch-o-matic patches..."
for patch in $(cat patch-o-matic.accepted.list); do
  if test -f "$patch"; then
    dinfo "${patch##*/}"
    patch -p1 -s -f -d "$kernel_profectio" < "${pwd}/${patch}"
  fi
done

# local debian patches
dinfo "applying local debian patches..."
for patch in $(find "$patches" -maxdepth 1 -type f -name \*.patch | sort); do
  dinfo "${patch##*/patches/}"
  patch -p1 -s -d "$build" < "$patch"
done

if test -d "$arch_patches"; then
  for patch in $(find "$arch_patches" -maxdepth 1 -type f -name \*.patch | sort); do
    dinfo "${patch##*/patches/}"
    patch -p1 -s -d "$build" < "$patch"
  done
fi

touch "${stamp}/prep"

dinfo "done."
