/* This file is part of the KDE project
   Copyright (C) 2001 Wilco Greven <greven@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#ifndef CENTERINGSCROLLVIEW_H
#define CENTERINGSCROLLVIEW_H

#include <qpixmap.h>
#include <qscrollview.h>
#include <qptrvector.h>

/**
 * CenteringScrollview is a customized QScrollView, which can hold one
 * page. This page will be centered on the viewport.
 */
class CenteringScrollview : public QScrollView
{
    Q_OBJECT

public:
    CenteringScrollview( QWidget* parent = 0, const char* name = 0 );
    ~CenteringScrollview() { ; }

    void addChild( QPtrVector<QWidget> *wdgList );
    void addChild( QWidget *singleWdg );

    /** Sets the number of columns into which the widgets are
        aligned. If nothing is set, '2' is the default. */
    void setNrColumns( Q_UINT8 cols );
    void setNrRows( Q_UINT8 rows );

    void setContinuousViewMode(bool continuous);
    bool fullScreenMode() { return fullScreen; };
    bool singlePageFullScreenMode();
    bool overviewMode() { return !continuousViewmode && (nrRows > 1 || nrCols > 1); };

    /** Returns the number of columns into which the widgets are aligned. */
    Q_UINT8 getNrColumns() const { return nrCols; };
    Q_UINT8 getNrRows() const { return nrRows; };
    bool isContinuous() const { return continuousViewmode; };

    /** Return true if the top resp. bottom of the page is visible. */
    bool atTop()    const;
    bool atBottom() const;

    /** Distance between pages in pixels (this is independent of
        the zoom level). */
    int distanceBetweenPages() { return distanceBetweenWidgets; };

    /** Moves the viewport so that the widget is at the top left corner. */
    void moveViewportToWidget(QWidget* widget);

public slots:
    bool readUp();
    bool readDown();
    void scrollUp();
    void scrollDown();
    void scrollRight();
    void scrollLeft();
    void scrollBottom();
    void scrollTop();

    void setFullScreenMode(bool fullScreen);
    /** Turn the scrollbars on/off. */
    void slotShowScrollbars(bool);

    /** If the viewport is larger than the page, center the page on
        the viewport. */
    void centerContents();

signals:
    void viewSizeChanged( QSize size );
    void pageSizeChanged( QSize size );

    /** This signal is emitted when the scrollView receives a mouse
        wheel event. */
    void wheelEventReceived( QWheelEvent * );

protected:
    /** This method draws shadows about the child widgets that the
        scrollview holds. It is automatically called by QT whenever
        necessary. */
    void drawContents ( QPainter * p, int clipx, int clipy, int clipw, int cliph );

    void keyPressEvent( QKeyEvent* );

    /** Reimplemented from QScrollView to make sure that the page is
        centered when it fits in the viewport. */
    void viewportResizeEvent( QResizeEvent* );

    /** Reimplemented from QScrollView, because the kviepart needs to
        handle the wheel events itself. The wheel event is passed on by 
        emitting the singal "wheelEventReceived". Nothing else is done. */
    void contentsWheelEvent ( QWheelEvent * );

    void contentsMousePressEvent(QMouseEvent*);
    void contentsMouseReleaseEvent(QMouseEvent*);
    void contentsMouseMoveEvent(QMouseEvent*);

private:
    /** Stores the mouse position between two mouse events. This is used
        to implement the "grab and drag the viewport contents" feature. */
    QPoint   dragGrabPos;

    QPtrVector<QWidget>* widgetList;

    /** Used internally by the centerContents()-method. Set with the
        setNrColumns() method */
    Q_UINT8  nrCols;
    Q_UINT8  nrRows;

    bool continuousViewmode;
    bool fullScreen;

    /** Color used internally in the shadow drawing routines in the
        drawContents method. */
    QColor   viewportBackgroundColorForCorners;

    /** Internal storages used in the shadow drawing routines in the
        drawContents method.*/
    QPixmap  URShadow, BRShadow, BLShadow;

    QPtrVector<QWidget> widgetListIfOldAddChildIsUsed;

    /** This is set in slotShowScrollbars() to remember the status of the
        scrollbars when we switch from fullscreen mode back to normal mode. */
    bool showScrollBars;

    /** This int remembers the style of the frame of the centering
        scrollview when fullscreen mode is switched on. It is then
        restored when it is switched off. */
    int oldFrameStyle;

    /** color of the viewport's background. This is also
        stored on entering the fullscreen mode. */
    QColor backgroundColor;

    /** Distance between pages in pixels 
        (this is independent of the zoom level). */
    static const int distanceBetweenWidgets=10;
};

#endif
