/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA 02110-1301, USA.                                          *
 ***************************************************************************/

#ifndef INSTALL_H
#define INSTALL_H

//Qt
#include <qthread.h>

//KDE
#include <kassistantdialog.h>
class KMenu;
class KProgressDialog;

//Ui
#include "ui_install_assistant_intro.h"
#include "ui_install_assistant_choice.h"
#include "ui_install_assistant_hdd.h"
#include "ui_install_assistant_floppy.h"

//KGRUBEditor
#include "core/datatypes.h"

const QString installCommand( "grub-install" );
const QString mountpoint( QDir::tempPath() + "/rescue" );

/**
 * @short GRUB Installation Thread to Hard Disk.
 *
 * Installs GRUB on a hard disk's MBR (=Master Boot Record)
 * or on a partition boot sector.
 */
class HardDiskThread : public QThread
{
Q_OBJECT
public:
	/**
	* Constructor.
	*
	* @param location the location where GRUB will be installed.
	*/
	HardDiskThread( const QString &location, QObject *parent = 0 ) : QThread( parent ), m_location( location )
	{};
protected:
	/**
	* Initialises the installation process and using the exitCode(int)
	* signal, emits the exit code of the process.
	*
	* @see exitCode(int)
	*/
	void run();
signals:
	/**
	* Used in run().
	*
	* The possible exitCodes are as follows:
	* @li -2: the process could not be started.
	* @li -1: the process crashed.
	* @li 0: the process was successfully completed.
	* @li anything else: the exit code of the process.
	*
	* @see run()
	*/
	void exitCode( int exitCode );
private:
	const QString m_location;
};

/**
 * @short GRUB Installation Thread to Floppy.
 *
 * Installs GRUB on a floppy. Can be used either to create
 * a rescue floppy (with no menu file) or with a menu file
 * as an external means of booting GRUB, just as if GRUB was
 * installed on a Hard Disk.
 */
class RescueFloppyThread : public QThread
{
Q_OBJECT
public:
	/**
	* Constructor.
	*/
	RescueFloppyThread( QObject *parent = 0 ) : QThread( parent )
	{};
	/**
	* Sets the menu file that should be included in the floppy.
	* It is optional. If no menu file is specified, the floppy
	* serves as a rescue floppy containing the GRUB shell, where
	* the user may enter GRUB commands.
	*
	* @param menuFile the menu file that should be included in the floppy.
	*/
	inline void setMenuFile( const QString &menuFile ) { m_menuFile = menuFile; };
protected:
	/**
	* Initialises the installation process and using the exitCode(int)
	* signal, emits the exit code of the process.
	*
	* @see exitCode(int)
	*/
	void run();
signals:
	/**
	* Emits a description of the current process.
	* To be used with progress dialogs.
	*
	* @param process the currently running process.
	*/
	void currentProcess( const QString &process );
	/**
	* Emits the percent of the total installation process
	* that is completed.
	* To be used with progress dialogs.
	*
	* @param step the percent of the total installation process
	*             that is completed.
	*/
	void currentStep( int step );
	/**
	* Used in run().
	*
	* The possible exitCodes are as follows:
	* @li -1: an error occured while trying to format the floppy.
	* @li -2: an error occured while trying to create a mountpoint.
	* @li -3: an error occured while trying to mount the floppy.
	* @li -4: an error occured while installing GRUB.
	* @li -5: an error occured while trying to copy the (optional) menu file.
	*
	* @see run(), setMenuFile()
	*/
	void exitCode( int exitCode );
private:
	QString m_menuFile;
};

/**
 * @short Install Assistant.
 *
 * Installs GRUB step-to-step.
 */
class InstallAssistant : public KAssistantDialog
{
Q_OBJECT
public:
	/**
	* Sets up the GUI.
	*
	* @param parent QWidget parent. Self explanatory.
	*/
	InstallAssistant( QWidget *parent = 0 );
private:
	/**
	* Connects all signals to their slots.
	*/
	void setupConnections();
	/**
	* Checks if a device already exists in the menu of devices.
	* Makes sure that no duplicates are entered in this menu.
	*
	* @param device The device to check.
	*
	* @return whether the device already exists in the combobox or not.
	*/
	bool alreadyExists( const QString &device );
	/**
	* If QDir::tempPath() + "/rescue" exists, it tries to umount it 
	* (in case it is mounted) and then delete the directory.
	*
	* This is needed to assure that the floppy rescue installation process
	* does not fail due to an already existing directory. This could happen
	* if the process had already created the directory and crashed.
	*/
	void clearMountPoint();
private slots:
	/**
	* A suggestion was chosen from the drop-down list
	* of the suggestions button.
	*
	* Fills-in the corresponding KLineEdit with the appropriate
	* root device/partition (using standard GRUB device naming).
	*/
	void suggestionTriggered( QAction *action );

	/**
	* According to what the hard disk installation thread
	* returned, show a suitable message box and quit the assistant.
	*/
	void hddResult( int exitCode );

	/**
	* According to what the rescue floppy installation thread
	* returned, show a suitable message box and quit the assistant.
	*/
	void floppyResult( int exitCode );
	/**
	* Sets the label text of the floppy installation progress dialog.
	*/
	void setProgressLabelText( const QString &text );
	/**
	* Sets the value of the floppy installation progress dialog.
	*/
	void setProgressValue( int value );
protected slots:
	/**
	* Next: Used to show the appropriate page after the user
	* selected an installation location.
	*
	* Finish: If the user hasn't entered valid information,
	* reject them and prompt the user to enter valid.
	* Of course it is also used to install GRUB.
	*/
	void slotButtonClicked( int button );
private:
	Ui::install_assistant_intro ui_intro;
	Ui::install_assistant_choice ui_choice;
	Ui::install_assistant_hdd ui_hdd;
	Ui::install_assistant_floppy ui_floppy;

	KMenu *menu_suggestions;
	KMenu *menu_deviceSuggestions;
	KMenu *menu_partitionSuggestions;

	KPageWidgetItem *choicePage;
	KPageWidgetItem *hardPage;
	KPageWidgetItem *floppyPage;

	KProgressDialog *progressDialog;
};

#endif
