/***************************************************************************
 *   Copyright (C) 2003 by Sylvain Joyeux                                  *
 *   sylvain.joyeux@m4x.org                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#ifndef PACKAGEMANAGER_H
#define PACKAGEMANAGER_H

#include <qobject.h>

/** Base class for accessing package-manager specific
 * functionalities. ATM, it supports listing files in a
 * package and searching the package which owns a file.
 *
 * Support for online-search support is also included.
 *
 * @author Sylvain Joyeux
 */

class PackageManager : public QObject
{
  Q_OBJECT
public:
  PackageManager(QObject *parent = 0, const char *name = 0);
  virtual ~PackageManager();

  virtual bool search(const QString& file);
  virtual bool list(const QString& package);

  /** Checks what the package manager is capable of.
   * The \c cap parameter should be a or of exactly
   * one function (SEARCH, LIST, ...) and one access
   * (ONLINE, OFFLINE).
   * The function returns NOT_SUPPORTED if the specified query is not
   * supported. Otherwise, it returns cap with some
   * restrictions (EXHAUSTIVE, INSTALLED_ONLY, ...) if
   * they apply
   *
   * INSTALLED_ONLY meaning changes with the function considered.
   * With SEARCH_FILE, it means that SEARCH_FILE only finds
   * installed files. With LIST_FILES, it means that listing the
   * files of a package is possible only if the package is already
   * installed */
  virtual int capabilities(int query) const;

  virtual QString getOnlineForm();
  virtual QString getOnlineURL(const QString& query, const QMap<QString, QString>& options);

  enum Capabilities
  {
  	NOT_SUPPORTED = 0,

  	SEARCH_FILE = 0x01,
    LIST_FILES  = 0x02,
    SHOW = 0x04,
    OFFLINE = 0x10,
    ONLINE  = 0x20,

    INSTALLED_ONLY = 0x200
  };

signals:
  /** Tags:
   *    warning (warning text)
   *    error (error text)
   *    file (file_name)  [for dpkg]
   * end
   */
  void token(const QString& tag, const QString& value);
};

#endif
