/***************************************************************************
 *   Copyright (C) 2007 by Markus Leuthold   *
 *   <kusi at forum.titlis.org>   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Cambridge, MA 02110-1301, USA.        *
 ***************************************************************************/

// Qt includes.

#include <qwmatrix.h>
#include <qfileinfo.h>

#include <jpeglib.h>
#include <iostream>

// LibKDcraw includes.

#include <libkdcraw/rawfiles.h>
#include <libkdcraw/kdcraw.h>

// Local includes.

#include "timer.h"
#include "texture.h"


using namespace std;
using namespace KIPIviewer;

Texture::Texture()
{
	rotate_list[0]=90;
	rotate_list[1]=180;
	rotate_list[2]=270;
	rotate_list[3]=180;
	rotate_idx=0;
	reset();
}


Texture::~Texture()
{
	
}




/*!
    \fn Texture::height()
 */
int Texture::height()
{
    return glimage.height();
}


/*!
    \fn Texture::width()
 */
int Texture::width()
{
    return glimage.width();
}


/*!
    \fn Texture::load(QString fn, QSize size, GLuint tn)
    \param fn filename to load
	\param size size of image which is downloaded to texture mem
	\param tn texture id generated by glGenTexture
	if "size" is set to image size, scaling is only performed by the GPU but not by the CPU, however the AGP usage to texture memory is increased (20MB for a 5mp image)
	\todo support for raw format
 */
bool Texture::load(QString fn, QSize size, GLuint tn)
{
// 	cout << "enter Texture::load(string)" << endl;
	filename=fn;
	initial_size=size;
	_texnr=tn;

	// check if its a RAW file.
	QString rawFilesExt(raw_file_extentions);
	QFileInfo fileInfo(fn);
	if (rawFilesExt.upper().contains( fileInfo.extension(false).upper() )) {
		//it's a RAW file, use the libkdcraw loader
		KDcrawIface::KDcraw::loadDcrawPreview(qimage, fn);
	}	else {
		//use the standard loader
		qimage=QImage(fn);
	}
	
	if (qimage.isNull()) {
		return false;
	}
	
	_load();
	reset();
	rotate_idx=0;
	return true;
}

/*!
    \fn Texture::load(QImage im, int i, ltype t)
    \param im Qimage to be copied from
	\param size size of image which is downloaded to texture mem
	\param tn texture id generated by glGenTexture
	if "size" is set to image size, scaling is only performed by the GPU but not by the CPU, however the AGP usage to texture memory is increased (20MB for a 5mp image)
 */
bool Texture::load(QImage im, QSize size, GLuint tn)
{
// 	cout << "enter Texture::load(qimage)" << endl;
	qimage=im;
	initial_size=size;
	_texnr=tn;
	_load();
	reset();
	rotate_idx=0;
	return true;
}

/*!
    \fn Texture::load()
    internal load function
 */
bool Texture::_load()
{
	int w=initial_size.width();
	int h=initial_size.height();
	if (w==0 || w>qimage.width() || h>qimage.height()) {
		glimage=QGLWidget::convertToGLFormat(qimage);
	} else {
		glimage=QGLWidget::convertToGLFormat(qimage.scale(w,h,QImage::ScaleMin));
	}
	w=glimage.width();
	h=glimage.height();
	if (h < w) {
		rtx = 1;
		rty = float(h)/float(w);
	}
	else {
		rtx = float(w)/float(h);
		rty = 1;
	}	
	return true;
}

/*!
    \fn Texture::data()
 */
GLvoid * Texture::data()
{
	return glimage.bits();
}


/*!
    \fn Texture::texnr()
 */
GLuint Texture::texnr()
{
    return _texnr;
}


/*!
    \fn Texture::quality()
    deprecated
 */
Texture::Quality Texture::quality()
{
    return _quality;
}


/*!
    \fn Texture::zoom(float delta, QPoint mousepos)
    \param delta delta between previous zoom and current zoom
	\param mousepos mouse position returned by QT
	\TODO rename mousepos to something more generic
 */
void Texture::zoom(float delta, QPoint mousepos)
//u: start in texture, u=[0..1]
//z=[0..1], z=1 -> no zoom
//l: length of tex in glFrustum coordinate system
//rt: ratio of tex, rt<=1
//rd: ratio of display, rd>=1
//m: mouse pos normalized, cd=[0..rd]
//c:  mouse pos normalized to zoom*l, c=[0..1]		
{
// 	cout << "enter zoom" << endl;
	z*=delta;
	delta=z*(1.0/delta-1.0); //convert to real delta=z_old-z_new
	
	float mx=mousepos.x()/(float)display_x*rdx;
	float cx=(mx-rdx/2.0+rtx/2.0)/rtx;
	float vx=ux+cx*z;
	ux=ux+(vx-ux)*delta/z;	
	
	float my=mousepos.y()/(float)display_y*rdy;
	float cy=(my-rdy/2.0+rty/2.0)/rty;
	cy=1-cy;
	float vy=uy+cy*z;
	uy=uy+(vy-uy)*delta/z;
// 	cout << "z=" << z << " mousex=" << mousepos.x() << " mousey=" << mousepos.y() << endl ;
	calcVertex();
}


/*!
    \fn Texture::vertex_bottom()
 */
GLfloat Texture::vertex_bottom()
{
    return (GLfloat) vbottom;
}

/*!
    \fn Texture::vertex_top()
 */
GLfloat Texture::vertex_top()
{
    return (GLfloat) vtop;
}


/*!
    \fn Texture::vertex_left()
 */
GLfloat Texture::vertex_left()
{
    return (GLfloat) vleft;
}


/*!
    \fn Texture::vertex_right()
 */
GLfloat Texture::vertex_right()
{
    return (GLfloat) vright;
}


/*!
    \fn Texture::setViewport(int w, int h)
    \param w width of window
	\param h height of window
 */
void Texture::setViewport(int w, int h)
// ensures that rdx & rdy are always > 0
{
	if (h>w) {
		rdx=1.0;
		rdy=h/float(w);
	}
	else {
		rdx=w/float(h);
		rdy=1.0;	
	}
	display_x=w;
	display_y=h;
}


/*!
    \fn Texture::calcVertex()
    calculate vertices according internal state variables
 */
void Texture::calcVertex()
//calculate vertex positions according z,ux and uy
{
   	float lx=2*rtx/z;
	float wx=lx*(1-ux-z);
	vleft = -rtx-ux*lx; //left
	vright = rtx+wx; //right
	
	float ly=2*rty/z; 
	float wy=ly*(1-uy-z);
	vbottom = -rty-uy*ly; //bottom
	vtop = rty+wy; //top
	
// 	cout << "lx=" << lx <<" z="<<z<<" rtx="<<rtx<<"rty="<<rty<<endl;
// 	cout << "vleft=" << vleft << " vright=" << vright << endl;	
}


/*!
    \fn Texture::move(QPoint diff)
 */
void Texture::move(QPoint diff)
{
// 	cout << "enter tex::move " << "ux="<<ux<<" uy="<<uy<<endl;
	ux=ux-diff.x()/float(display_x)*z*rdx/rtx;
	uy=uy+diff.y()/float(display_y)*z*rdy/rty;
// 	cout << "ux=" << ux << "diff.x=" << diff.x() << "display_x=" << display_x << " rdx=" << rdx << " z=" << z  << " rdy=" << rdy << " rty="<<rty<<endl;
	calcVertex();
	
}


/*!
    \fn Texture::reset()
 */
void Texture::reset()
{
// 	cout << "enter texture::reset" << endl;
    ux=0;
	uy=0;
	z=1.0;
	float zoomdelta=0;
// 	cout << "rdx=" << rdx << " rdy="<< rdy << " rtx=" <<rtx<<" rty="<<rty<<endl;
	if ((rtx<rty) && (rdx<rdy) && (rtx/rty < rdx/rdy)) {
		zoomdelta=z-rdx/rdy;
	}
	if ((rtx<rty) && (rtx/rty > rdx/rdy)) {
		zoomdelta=z-rtx;
	}
	
	if ((rtx>=rty) && (rdy<rdx) && (rty/rtx < rdy/rdx)) {
		zoomdelta=z-rdy/rdx;
	}
	if ((rtx>=rty) && (rty/rtx > rdy/rdx)) {
		zoomdelta=z-rty;
	}
	QPoint p =  QPoint(display_x/2,display_y/2);
// 	cout << "z=" <<z<< " px=" << p.x() << endl;
	zoom(1.0-zoomdelta,p);
	
	calcVertex();
}

/*!
    \fn Texture::setSize(QSize size)
    \return true if size has changed, false otherwise
    set new texture size in order to reduce AGP bandwidth
 */
bool Texture::setSize(QSize size)
{
	if (glimage.width()==size.width()) {
// 		cout << "tex size  has not changed" << endl;
		return false;
	}
    int w=size.width();
	int h=size.height();
	if (w==0) {
		glimage=QGLWidget::convertToGLFormat(qimage);
	} else {
		glimage=QGLWidget::convertToGLFormat(qimage.scale(w,h,QImage::ScaleMin));
	}
	return true;
}


/*!
    \fn Texture::rotate()
    \brief smart image rotation
    
    since the two most frequent usecases are a CW or CCW rotation of 90, 
	perform these rotation with one (+90) or two (-90) calls of rotation()
 */
void Texture::rotate()
{
	QWMatrix r;
	r.rotate(rotate_list[rotate_idx%4]);		
	qimage=qimage.xForm(r);
	_load();
	reset();
	rotate_idx++;
}


/*!
    \fn Texture::setToOriginalSize()
    zoom image such that each pixel of the screen corresponds to a pixel in the jpg
    remember that OpenGL is not a pixel exact specification, and the image will still be filtered by OpenGL
 */
void Texture::zoomToOriginal()
{
	float zoomfactorToOriginal;
	reset();
// 	cout << "display x=" << display_x << " img=" << qimage.width() << endl;
// 	cout << "rdx=" << rdx << " rdy="<< rdy << " rtx=" <<rtx<<" rty="<<rty<<endl;
	
	if (qimage.width()/qimage.height() > float(display_x)/float(display_y)) {
		//image touches right and left edge of window
		zoomfactorToOriginal=float(display_x)/qimage.width();
	} else {
		//image touches upper and lower edge of window
		zoomfactorToOriginal=float(display_y)/qimage.height();
	}
		
	zoom(zoomfactorToOriginal,QPoint(display_x/2,display_y/2));
}
