/***************************************************************************
 *
 * knetworkmanager-dialogfab.cpp - A NetworkManager frontend for KDE 
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Timo Hoenig        <thoenig@suse.de>, <thoenig@nouse.net>
 *         Will Stephenson    <wstephenson@suse.de>, <wstephenson@kde.org>
 *         Valentine Sinitsyn <e_val@inbox.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include "knetworkmanager-dialogfab.h"
#include "knetworkmanager-devicestore.h"
#include "knetworkmanager-nminfo.h"

#include <qpushbutton.h>
#include <qbuttongroup.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qcombobox.h>
#include <qradiobutton.h>
#include <qwidgetstack.h>

#include <klocale.h>
#include <kurlrequester.h>

#include "cryptowidget.h"
#include "networkchoicewidget.h"
#include "networklabelwidget.h"
#include "settings.h"

#include "knetworkmanager-encryption.h"
#include "knetworkmanager-storage.h"

/* AcquirePasswordDialog */

void
AcquirePasswordDialog::slotOk ()
{
	ActivationStageNotifyNetwork* notify;

	emit sendPassphrase (_net);

	notify = new ActivationStageNotifyNetwork (_net->getEssid(), _ctx->getTray (), "ActivationStageNotify", false, 0, _ctx, _dev );
	KDialogBase::slotOk();
}

void
AcquirePasswordDialog::configureCryptoWidget()
{
	/* if we're asked for a password, user should not swtich be able to connect without supplying one */ 	
	cryptoWidget->groupEnc->setCheckable (false);
	comboEncryptionFill();

	/* WEP, WPA Personal */
	cryptoWidget->leditSecret->setEchoMode( cryptoWidget->cbShow->isChecked() ? QLineEdit::Normal : QLineEdit::Password );
	/* WPA Enterprise */
	cryptoWidget->EAPleditPassword->setEchoMode( cryptoWidget->EAPcbShow->isChecked() ? QLineEdit::Normal : QLineEdit::Password );
	cryptoWidget->EAPleditPrivatePassword->setEchoMode( cryptoWidget->EAPcbShow->isChecked() ? QLineEdit::Normal : QLineEdit::Password );

	cryptoWidget->groupEncAdvanced->setEnabled (true);
	cryptoWidget->stackEncAdvanced->setEnabled (true);

	/* set focus to leditSecret */
	cryptoWidget->leditSecret->setFocus ();
	
	/* fake select on encryption combo to update mode */
	comboEncryption_activated (0);

	if ( _net->getEncryption ()->hasStoredKey () )	
		cryptoWidget->leditSecret->setText( KNetworkManagerStorage::getInstance()->credentials( _essid )["password"] );
}

void
AcquirePasswordDialog::slotDeviceStoreChanged (DeviceStore *)
{
	if (!_net && ( _net = _dev->getNetwork (_net_path) )) {
		cryptoWidget->setEnabled(true);
		configureCryptoWidget();
	}
}

void
AcquirePasswordDialog::invalidateSecretMap ()
{
	SecretMap map;
	_net->getEncryption ()->setSecrets (map);
}

void
AcquirePasswordDialog::comboEncryptionFill ()
{
	int id = 0;
	int capabilities = _net->getCapabilities ();
	QComboBox* comboEnc = cryptoWidget->comboEncryption;

	/* let's find out the encyption of the network and update the model */
	if (capabilities & NM_802_11_CAP_PROTO_WEP) {
		comboEnc->insertItem (i18n ("WEP Passphrase"), id);
		_id_wep_passphrase = id++;
		comboEnc->insertItem (i18n ("WEP 40/104-bit hex"), id);
		_id_wep_hex = id++;
		comboEnc->insertItem (i18n ("WEP 40/104-bit ASCII"), id);
		_id_wep_ascii = id; 

		_encryptionMap [_id_wep_passphrase] = new EncryptionWEP  (WEP_PASSPHRASE);
		_encryptionMap [_id_wep_hex]        = new EncryptionWEP  (WEP_HEX);
		_encryptionMap [_id_wep_ascii]      = new EncryptionWEP  (WEP_ASCII);

		//cryptoWidget->stackEncAdvanced->raiseWidget (cryptoWidget->advancedWEPWidget);
		comboEnc->setCurrentItem (_id_wep_passphrase);
	} else if ((capabilities & NM_802_11_CAP_PROTO_WPA) || (capabilities & NM_802_11_CAP_PROTO_WPA2)) {
		QComboBox* comboProto = cryptoWidget->comboProtocol;
		int protoId = 0;
		bool wpa1_available = capabilities & NM_802_11_CAP_PROTO_WPA;
		bool wpa2_available = capabilities & NM_802_11_CAP_PROTO_WPA2;

		comboEnc->insertItem (i18n ("WPA Personal"), id);
		_id_wpa_wpa2personal = id++;
		comboEnc->insertItem (i18n ("WPA Enterprise"), id);
		_id_wpa_wpa2enterprise = id;

		_encryptionMap [_id_wpa_wpa2personal] = new EncryptionWPAPersonal ();
		_encryptionMap [_id_wpa_wpa2enterprise] = new EncryptionWPAEnterprise ();

		//EncryptionWPAPersonal* enc = (EncryptionWPAPersonal*) _encryptionMap [cryptoWidget->comboEncryption->currentItem ()];
		EncryptionWPAPersonal* encPersonal = (EncryptionWPAPersonal*) _encryptionMap [_id_wpa_wpa2personal];
		EncryptionWPAEnterprise* encEnterprise = (EncryptionWPAEnterprise*) _encryptionMap [_id_wpa_wpa2enterprise];

		comboProto->insertItem (i18n ("Automatic"), protoId);
		_id_wpa_personal_auto = protoId++;

		if (capabilities & NM_802_11_CAP_CIPHER_TKIP) {
			comboProto->insertItem (i18n ("Temporal Key Integrity Protocol (TKIP)"), protoId);
			_id_wpa_personal_tkip = protoId++;
		}
		if (capabilities & NM_802_11_CAP_CIPHER_CCMP) {
			comboProto->insertItem (i18n ("AES-Counter Mode CBC-MAC Protocol (AES-CCMP)"), protoId);
			_id_wpa_personal_ccmp = protoId++;
		}

		/* Let the user choose if WPA1 and WPA2 are both available. Else: Disable radio buttons */
		cryptoWidget->radioButtonWPA2->setEnabled (wpa1_available && wpa2_available);
		cryptoWidget->radioButtonWPA1->setEnabled (wpa1_available && wpa2_available);
		cryptoWidget->EAPrbWPA2->setEnabled (wpa1_available && wpa2_available);
		cryptoWidget->EAPrbWPA1->setEnabled (wpa1_available && wpa2_available);

		if (wpa1_available && !wpa2_available) {
			encPersonal->setVersion (WPA1);
			encEnterprise->setVersion (WPA1);
			cryptoWidget->radioButtonWPA1->setChecked (true);
			cryptoWidget->radioButtonWPA2->setChecked (false);
			cryptoWidget->EAPrbWPA1->setChecked (true);
			cryptoWidget->EAPrbWPA2->setChecked (false);
		} else if (!wpa1_available && wpa2_available) {
			encPersonal->setVersion (WPA2);
			encEnterprise->setVersion (WPA2);
			cryptoWidget->radioButtonWPA2->setChecked (true);
			cryptoWidget->radioButtonWPA1->setChecked (false);
			cryptoWidget->EAPrbWPA2->setChecked (true);
			cryptoWidget->EAPrbWPA1->setChecked (false);
		}

		comboEnc->setCurrentItem (_id_wpa_wpa2personal);
		/* Prefer NM_AUTH_TYPE_WPA_PSK_AUTO */
		comboProto->setCurrentItem (_id_wpa_personal_auto);
	}
}

AcquirePasswordDialog::AcquirePasswordDialog (QWidget* parent, const char *name, bool modal, WFlags fl, KNetworkManager* ctx, QString obj_path, QString net_path, QString essid)
                            : WirelessDialog (parent, name, modal, fl, ctx), _obj_path(obj_path), _net_path(net_path), _essid(essid)
{
	delete _net; // Free the object created in WirelessDialog constructor
	DeviceStore* store = _ctx->getDeviceStore ();

	_dev = store->getDevice (obj_path);
	_net = _dev->getNetwork (net_path);

	QWidget * mainWid = mainWidget();
	NetworkLabelWidget * labelWidget = new NetworkLabelWidget( mainWid );
	labelWidget->lblEssidVal->setText (essid);
	labelWidget->lblDeviceVal->setText ( QString("%1 %2 (%3)")
			.arg( _dev->getVendor () )
			.arg( _dev->getProduct () )
			.arg( _dev->getInterface () ) );

	cryptoWidget = new CryptoWidget( mainWid, "wirelessnetworkwidget" );
	connectCryptoWidget();
	if (_net) 
		configureCryptoWidget();
	else 
		cryptoWidget->setEnabled(false);
		
	enableButton( KDialogBase::Ok, false);	
	
	connect (store, SIGNAL (deviceStoreChanged (DeviceStore *)),
		 this, SLOT (slotDeviceStoreChanged (DeviceStore *)));
		 
	connect (this, SIGNAL (sendPassphrase (Network*)),
		 _ctx->getNetworkManagerInfo (), SLOT (sendPassphrase (Network*)));

	connect (this, SIGNAL (cancelClicked ()),
		 _ctx->getNetworkManagerInfo (), SLOT (sendPassphraseError ()));

	connect (this, SIGNAL (cancelClicked ()),
		 this, SLOT   (invalidateSecretMap ()));
}

AcquirePasswordDialog::~AcquirePasswordDialog ()
{

}

/* NewWirelessNetworkDialog - connecting to new or hidden wireless network */

void
NewWirelessNetworkDialog::slotOk()
{
	updateModel();
	DeviceStore* store = _ctx->getDeviceStore ();

	_dev->setPendingNetwork (_net);

	emit activateNetwork (_net, _dev);

	ActivationStageNotifyNetwork* notify = NULL;
	notify = new ActivationStageNotifyNetwork ( _net->getEssid(), this, "ActivationStageNotify", true, 0, _ctx, _dev );
	KDialogBase::slotOk();
}

void
NewWirelessNetworkDialog::leditEssid_textChanged (const QString& essid)
{
	_net->setEssid (essid);
	updateModel ();
}

void
NewWirelessNetworkDialog::comboEncryptionFill ()
{
	int id = 0;
	QComboBox* enc = cryptoWidget->comboEncryption;

	_id_disabled = -1;

	enc->insertItem (i18n ("WEP Passphrase"), id);
	_id_wep_passphrase = id++;
	enc->insertItem (i18n ("WEP 40/104-bit hex"), id);
	_id_wep_hex = id++;
	enc->insertItem (i18n ("WEP 40/104-bit ASCII"), id);
	_id_wep_ascii = id++; 
	enc->insertItem (i18n ("WPA Personal"), id);
	_id_wpa_wpa2personal = id++;
	enc->insertItem (i18n ("WPA Enterprise"), id);
	_id_wpa_wpa2enterprise = id++;

	_encryptionMap [_id_disabled]           = new EncryptionNone ();
	_encryptionMap [_id_wep_passphrase]     = new EncryptionWEP  (WEP_PASSPHRASE);
	_encryptionMap [_id_wep_hex]            = new EncryptionWEP  (WEP_HEX);
	_encryptionMap [_id_wep_ascii]          = new EncryptionWEP  (WEP_ASCII);
	_encryptionMap [_id_wpa_wpa2personal]   = new EncryptionWPAPersonal ();
	_encryptionMap [_id_wpa_wpa2enterprise] = new EncryptionWPAEnterprise ();

	id = 0;
	QComboBox* proto = cryptoWidget->comboProtocol;

	proto->insertItem (i18n ("Automatic"), id);
	_id_wpa_personal_auto = id++;
	proto->insertItem (i18n ("Temporal Key Integrity Protocol (TKIP)"), id);
	_id_wpa_personal_tkip = id++;
	proto->insertItem (i18n ("AES-Counter Mode CBC-MAC Protocol (AES-CCMP)"), id);
	_id_wpa_personal_ccmp = id++;
}

void
NewWirelessNetworkDialog::comboDeviceFill ()
{
	DeviceStore* store = _ctx->getDeviceStore ();
	DeviceList deviceList = store->getDeviceList ();
	Device* dev;
	DeviceList::iterator i;
	int id = 0;

	netWidget->comboDevice->clear ();
	
	for (i = deviceList.begin (); i != deviceList.end (); ++i) {
		dev = *i;
		if (dev->isWireless ()) {
			QString item;
			item = QString("%1 %2 (%3)").arg( dev->getVendor () )
						.arg( dev->getProduct () )
						.arg( dev->getInterface () );
			_deviceMap [id] = dev;
			netWidget->comboDevice->insertItem (item, id++);
		}
	}
}

void 
NewWirelessNetworkDialog::updateModel()
{
	/* set device */
	_dev = _deviceMap [netWidget->comboDevice->currentItem()];
	WirelessDialog::updateModel ();
}

NewWirelessNetworkDialog::NewWirelessNetworkDialog (QWidget* parent, const char *name, bool modal, WFlags fl, KNetworkManager* ctx)
                                  : WirelessDialog (parent, name, modal, fl, ctx)
{
	QWidget * mainWid = mainWidget();
	netWidget = new NetworkChoiceWidget( mainWid, "networkchoicewidget" );
	comboDeviceFill ();

	cryptoWidget = new CryptoWidget( mainWid, "wirelessnetworkwidget" );
	cryptoWidget->leditSecret->setEchoMode( cryptoWidget->cbShow->isChecked() ? QLineEdit::Normal : QLineEdit::Password );

	cryptoWidget->EAPleditPassword->setEchoMode (cryptoWidget->EAPcbShow->isChecked() ? QLineEdit::Normal : QLineEdit::Password );
	cryptoWidget->EAPleditPrivatePassword->setEchoMode (cryptoWidget->EAPcbShow->isChecked() ? QLineEdit::Normal : QLineEdit::Password );

	connectCryptoWidget ();
	comboEncryptionFill ();

	/* set up defaults */
	cryptoWidget->groupEncAdvanced->setEnabled (false);
	cryptoWidget->stackEncAdvanced->setEnabled (false);
	cryptoWidget->stackEncAdvanced->raiseWidget (cryptoWidget->advancedWEPWidget);

	/* deactivate "Connect" button as long as user input is invalid */
	enableButton (KDialogBase::Ok, false);
	_dev = 0;

	/* set focus to leditEssid */
	netWidget->leditEssid->setFocus ();

	_net->setEncryption (_encryptionMap [_id_disabled]);

	connect (netWidget->leditEssid, SIGNAL (textChanged (const QString &)),
		 this,                  SLOT   (leditEssid_textChanged (const QString &)));

	/* show and update activation progress */
	connect (this,                    SIGNAL (activateNetwork (Network*, Device*)),
		 _ctx->getDeviceStore (), SLOT   (activateNetwork (Network*, Device*)));
}

NewWirelessNetworkDialog::~NewWirelessNetworkDialog ()
{

}

/* WirelessDialog */

void
WirelessDialog::comboAuthentication_activated (int id)
{
	EncryptionWEP* enc = (EncryptionWEP*) _encryptionMap [cryptoWidget->comboEncryption->currentItem()];

	enc->setMethod (id ? WEP_SHARED_KEY : WEP_OPEN_SYSTEM);
}

void
WirelessDialog::groupWEPAdvanced_toggled (bool on)
{
	EncryptionWEP* enc = (EncryptionWEP*) _encryptionMap [cryptoWidget->comboEncryption->currentItem()];

	/* reset to default value and update UI */
	if (!on) {
		enc->setDefaults ();
		cryptoWidget->comboAuthentication->setCurrentItem (0);
	}
}

/* WPA Personal */
void
WirelessDialog::groupWPAAdvanced_toggled (bool on)
{
	EncryptionWPAPersonal* enc = (EncryptionWPAPersonal*) _encryptionMap [_id_wpa_wpa2personal];

	/* reset to default value and update UI */
	if (!on) {
		enc->setDefaults ();
		cryptoWidget->comboProtocol->setCurrentItem (0);
		cryptoWidget->radioButtonWPA1->setChecked (true);
	}
}

void
WirelessDialog::comboWPAProtocol_activated (int id)
{
	WPAProtocol proto = WPA_AUTO;
	EncryptionWPAPersonal* enc = (EncryptionWPAPersonal*) _encryptionMap [_id_wpa_wpa2personal];

	if (id == _id_wpa_personal_auto)
		proto = WPA_AUTO;
	else if (id == _id_wpa_personal_tkip)
		proto = WPA_TKIP;
	else if (id == _id_wpa_personal_ccmp)
		proto = WPA_CCMP_AES;

	enc->setProtocol (proto);
}

void
WirelessDialog::radioButtonWPA1_toggled (bool on)
{
	EncryptionWPAPersonal* enc = (EncryptionWPAPersonal*) _encryptionMap [_id_wpa_wpa2personal];
	enc->setVersion (on ? WPA1 : WPA2);
	if (cryptoWidget->radioButtonWPA2->isEnabled ())
		cryptoWidget->radioButtonWPA2->setChecked (!on);
}

void
WirelessDialog::radioButtonWPA2_toggled (bool on)
{
	EncryptionWPAPersonal* enc = (EncryptionWPAPersonal*) _encryptionMap [_id_wpa_wpa2personal];
	enc->setVersion (on ? WPA2 : WPA1);
	if (cryptoWidget->radioButtonWPA1->isEnabled ())
		cryptoWidget->radioButtonWPA1->setChecked (!on);
}

/* WPA Enterprise */

void
WirelessDialog::groupWPAEAPAdvanced_toggled (bool on)
{
	EncryptionWPAEnterprise* enc = (EncryptionWPAEnterprise*) _encryptionMap [_id_wpa_wpa2enterprise];

	/* reset to default value and update UI */
	if (!on) {
		enc->setDefaults ();
		cryptoWidget->EAPcomboMethod->setCurrentItem (0);
		cryptoWidget->EAPrbWPA1->setChecked (true);
	}
}
void
WirelessDialog::EAPrbWPA1_toggled (bool on)
{
	EncryptionWPAEnterprise* enc = (EncryptionWPAEnterprise*) _encryptionMap [_id_wpa_wpa2enterprise];
	enc->setVersion (on ? WPA1 : WPA2);
	if (cryptoWidget->EAPrbWPA2->isEnabled ())
		cryptoWidget->EAPrbWPA2->setChecked (!on);
}

void
WirelessDialog::EAPrbWPA2_toggled (bool on)
{
	EncryptionWPAEnterprise* enc = (EncryptionWPAEnterprise*) _encryptionMap [_id_wpa_wpa2enterprise];
	enc->setVersion (on ? WPA2 : WPA1);
	if (cryptoWidget->EAPrbWPA1->isEnabled ())
		cryptoWidget->EAPrbWPA1->setChecked (!on);
}


void
WirelessDialog::EAPcomboMethod_activated (int id)
{
	EAPMethod method = EAP_PEAP;
	EncryptionWPAEnterprise* enc = (EncryptionWPAEnterprise*) _encryptionMap [_id_wpa_wpa2enterprise]; 

	switch (id) {
		case 0:
			method = EAP_PEAP;
			break;
		case 1:
			method = EAP_TLS;
			break;
		case 2:
			method = EAP_TTLS;
			break;
		default:
			method = EAP_PEAP;
	}
	
	enc->setMethod (method);
}

void
WirelessDialog::EAPleditIdentity_textChanged (const QString & identity)
{
	EncryptionWPAEnterprise* enc = (EncryptionWPAEnterprise*) _encryptionMap [_id_wpa_wpa2enterprise]; 	
	enc->setIdentity (identity);
}

void
WirelessDialog::EAPleditPassword_textChanged (const QString & identityPassword)
{
	EncryptionWPAEnterprise* enc = (EncryptionWPAEnterprise*) _encryptionMap [_id_wpa_wpa2enterprise];
	SecretMap map = enc->getSecrets();
	map.insert( EncryptionWPAEnterprise::IdPasswordKey, identityPassword );
	enc->setSecrets (map);
}

void
WirelessDialog::EAPleditAnonIdentity_textChanged (const QString & anonIdentity)
{
	EncryptionWPAEnterprise* enc = (EncryptionWPAEnterprise*) _encryptionMap [_id_wpa_wpa2enterprise];
	enc->setAnonIdentity (anonIdentity);
}

void
WirelessDialog::EAPurlClient_textChanged (const QString & certClient)
{
	EncryptionWPAEnterprise* enc = (EncryptionWPAEnterprise*) _encryptionMap [_id_wpa_wpa2enterprise];
	enc->setCertClient (certClient);

}	

void
WirelessDialog::EAPurlCA_textChanged (const QString & certCA)
{
	EncryptionWPAEnterprise* enc = (EncryptionWPAEnterprise*) _encryptionMap [_id_wpa_wpa2enterprise];
	enc->setCertCA (certCA);
}

void
WirelessDialog::EAPurlPrivate_textChanged (const QString & certPrivate)
{
	EncryptionWPAEnterprise* enc = (EncryptionWPAEnterprise*) _encryptionMap [_id_wpa_wpa2enterprise];
	enc->setCertPrivate (certPrivate);
}

void
WirelessDialog::EAPleditPrivatePassword_textChanged (const QString & certPrivatePassword)
{
	EncryptionWPAEnterprise* enc = (EncryptionWPAEnterprise*) _encryptionMap [_id_wpa_wpa2enterprise];
	SecretMap map = enc->getSecrets();
	map.insert( EncryptionWPAEnterprise::CertPrivatePasswordKey, certPrivatePassword );
	enc->setSecrets (map);
}
	
void
WirelessDialog::EAPcbShow_toggled (bool on)
{
	cryptoWidget->EAPleditPassword->setEchoMode (on ? QLineEdit::Normal : QLineEdit::Password );
	cryptoWidget->EAPleditPrivatePassword->setEchoMode (on ? QLineEdit::Normal : QLineEdit::Password );
}

void
WirelessDialog::connectCryptoWidget()
{
	connect (cryptoWidget->cbShow,          SIGNAL (toggled (bool)),
			                        SLOT   (slotChangeSecretEchoMode (bool)));
	connect (cryptoWidget->groupEnc,        SIGNAL (toggled (bool)),
			                        SLOT   (slotEncryptionToggled (bool)));
	connect (cryptoWidget->leditSecret,     SIGNAL (textChanged (const QString &)),
		 this,                          SLOT   (leditSecret_textChanged (const QString &)));
	connect (cryptoWidget->comboEncryption, SIGNAL (activated (int)),
		 this,                          SLOT   (comboEncryption_activated (int)));

	/* WEP */
	connect (cryptoWidget->groupEncAdvanced,    SIGNAL (toggled (bool)),
		 this,                              SLOT   (groupWEPAdvanced_toggled (bool)));

	connect (cryptoWidget->comboAuthentication, SIGNAL (activated (int)),
		 this,                              SLOT   (comboAuthentication_activated (int)));

	/* WPA Personal */
	connect (cryptoWidget->groupEncAdvanced, SIGNAL (toggled (bool)),
		 this,                           SLOT   (groupWPAAdvanced_toggled (bool)));
	connect (cryptoWidget->comboProtocol,    SIGNAL (activated (int)),
		 this,                           SLOT   (comboWPAProtocol_activated (int)));
	connect (cryptoWidget->radioButtonWPA1,  SIGNAL (toggled (bool)),
		 this,                           SLOT   (radioButtonWPA1_toggled (bool)));
	connect (cryptoWidget->radioButtonWPA2,  SIGNAL (toggled (bool)),
		 this,                           SLOT   (radioButtonWPA2_toggled (bool)));

	/* WPA Enterprise */
	connect (cryptoWidget->groupEncAdvanced,        SIGNAL (toggled (bool)),
		 this,                                  SLOT   (groupWPAEAPAdvanced_toggled (bool)));
	connect (cryptoWidget->EAPcbShow,               SIGNAL (toggled (bool)),
		 this,                                  SLOT   (EAPcbShow_toggled (bool)));
	connect (cryptoWidget->EAPrbWPA1,               SIGNAL (toggled (bool)),
		 this,                                  SLOT   (EAPrbWPA1_toggled (bool)));
	connect (cryptoWidget->EAPrbWPA2,               SIGNAL (toggled (bool)),
		 this,                                  SLOT   (EAPrbWPA2_toggled (bool)));
	connect (cryptoWidget->EAPcomboMethod,          SIGNAL (activated (int)),
		 this,                                  SLOT   (EAPcomboMethod_activated (int)));
	connect (cryptoWidget->EAPleditIdentity,        SIGNAL (textChanged (const QString &)),
		 this,                                  SLOT   (EAPleditIdentity_textChanged (const QString &)));
	connect (cryptoWidget->EAPleditPassword,        SIGNAL (textChanged (const QString &)),
		 this,                                  SLOT   (EAPleditPassword_textChanged (const QString &)));
	connect (cryptoWidget->EAPleditAnonIdentity,    SIGNAL (textChanged (const QString &)),
		 this,                                  SLOT   (EAPleditAnonIdentity_textChanged (const QString &)));
	connect (cryptoWidget->EAPurlClient,            SIGNAL (textChanged (const QString &)),
		 this,                                  SLOT   (EAPurlClient_textChanged (const QString &)));
	connect (cryptoWidget->EAPurlCA,                SIGNAL (textChanged (const QString &)),
		 this,                                  SLOT   (EAPurlCA_textChanged (const QString &)));
	connect (cryptoWidget->EAPurlPrivate,           SIGNAL (textChanged (const QString &)),
		 this,                                  SLOT   (EAPurlPrivate_textChanged (const QString &)));
	connect (cryptoWidget->EAPleditPrivatePassword, SIGNAL (textChanged (const QString &)),
		 this,                                  SLOT   (EAPleditPrivatePassword_textChanged (const QString &)));
}

void
WirelessDialog::comboEncryption_activated (int id)
{
	Encryption* enc = _encryptionMap [id];

	SecretMap map = enc->getSecrets();
	map.insert( "password", cryptoWidget->leditSecret->text () );
	enc->setSecrets ( map );
	_net->setEncryption (enc);

	if (id == _id_wep_passphrase) {
		kdDebug() << k_funcinfo << "WEP passphrase" << endl;
		cryptoWidget->lblSecret->setText (i18n ("&Passphrase:"));
		cryptoWidget->stackEncAdvanced->raiseWidget(cryptoWidget->advancedWEPWidget);
	} else if ((id == _id_wep_hex) || (id == _id_wep_ascii)) {
		kdDebug() << k_funcinfo << "WEP hex or ascii" << endl;
		cryptoWidget->lblSecret->setText (i18n ("&Key:"));
		cryptoWidget->stackEncAdvanced->raiseWidget(cryptoWidget->advancedWEPWidget);
	} else if (id == _id_wpa_wpa2personal) {
		kdDebug() << k_funcinfo << "WPA Personal" << endl;
		cryptoWidget->lblSecret->setText (i18n ("P&assword:"));
		cryptoWidget->stackEncAdvanced->raiseWidget(cryptoWidget->advancedWPAPersonalWidget);
	} else if (id == _id_wpa_wpa2enterprise) {
		kdDebug() << k_funcinfo << "WPA Enterprise" << endl;
		cryptoWidget->stackEncAdvanced->raiseWidget(cryptoWidget->advancedWPAEnterpriseWidget);
	}

	/* WPA Enterprise does not use lblSecret, leditSecret and cbShow */
	bool on = (id == _id_wpa_wpa2enterprise) ? false : true;
	cryptoWidget->lblSecret->setEnabled (on);
	cryptoWidget->leditSecret->setEnabled (on);
	cryptoWidget->cbShow->setEnabled (on);

	updateModel ();
}

void
WirelessDialog::leditSecret_textChanged (const QString& secret)
{
	Encryption* enc = _net->getEncryption ();
	SecretMap map = enc->getSecrets();
	map.insert( "password", secret );
	enc->setSecrets (map);
	updateModel ();
}


void
WirelessDialog::slotChangeSecretEchoMode (bool on)
{
	if (on)
		cryptoWidget->leditSecret->setEchoMode (QLineEdit::Normal);
	else
		cryptoWidget->leditSecret->setEchoMode (QLineEdit::Password);
}

void
WirelessDialog::slotEncryptionToggled (bool on)
{
	if (on)
		_net->setEncryption (_encryptionMap [cryptoWidget->comboEncryption->currentItem()]);
	else 
		_net->setEncryption (_encryptionMap [_id_disabled]);

	cryptoWidget->groupEncAdvanced->setEnabled (on);
	cryptoWidget->stackEncAdvanced->setEnabled (on);
}

void
WirelessDialog::updateModel ()
{
	enableButton (KDialogBase::Ok, _net->getEncryption ()-> isValid (_net->getEssid ()));
}

WirelessDialog::WirelessDialog(QWidget * parent, const char *name, bool modal, WFlags /*fl*/, KNetworkManager* ctx)
                    : KDialogBase ( parent, name, modal, i18n("Connect to Wireless Network" ), KDialogBase::Help|KDialogBase::Ok|KDialogBase::Cancel, KDialogBase::Ok, false ),
		      _id_disabled (-1), _id_wep_passphrase (-1), _id_wep_hex (-1), _id_wep_ascii (-1), _id_wpa_wpa2personal (-1), _id_wpa_wpa2enterprise (-1)
{
	setButtonGuiItem(Ok,i18n("C&onnect"));
	_ctx = ctx;
	_net = new Network ();
	makeVBoxMainWidget();
	enableButton (KDialogBase::Ok, false);
}

WirelessDialog::~WirelessDialog()
{
	// TODO: make Encryption objects children of the applet, or else they could be deleted here while still in use by the rest of the program
/*	QMap<int, Encryption*>::Iterator it = _encryptionMap->begin(); 
	const QMap<int, Encryption*>::Iterator end = _encryptionMap->end;
	for ( ; it != end; +=it )
		delete it.data();*/
}

#include "knetworkmanager-dialogfab.moc"
