/***************************************************************************
 *
 * knetworkmanager-encryption.cpp - A NetworkManager frontend for KDE 
 *
 * Copyright (C) 2006 Novell, Inc.
 *
 * Author: Timo Hoenig        <thoenig@suse.de>, <thoenig@nouse.net>
 *         Will Stephenson    <wstephenson@suse.de>, <wstephenson@kde.org>
 *         Valentine Sinitsyn <e_val@inbox.ru> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include <kconfigbase.h>
#include <kdebug.h>

#include "knetworkmanager.h"
#include "knetworkmanager-encryption.h"
#include "knetworkmanager-network.h"
#include "knetworkmanager-storage.h"
#include <NetworkManager/dbus-helpers.h>

bool
Encryption::isValid (const QString & essid)
{
	QString input = this->_secret["password"];
	bool    status = false;

	if ((essid.length () == 0) || (input.length () == 0) || (!_cipherList) || (_cipherList->empty ())) {
		kdDebug() << "Encryption::isValid failed, bad inputs" << endl;
		kdDebug() << "isValid, essid: " << essid << ", input: " << input << ", cipherlist: " << _cipherList
			  << endl;
		return status;
	}

	CipherList::iterator iter;
	for (iter = _cipherList->begin (); iter != _cipherList->end (); ++iter) {
		if (ieee_802_11_cipher_validate (*iter, essid.ascii(), input.ascii() ) == 0) {
			this->_currentCipher = *iter;
			setWeCipher (ieee_802_11_cipher_get_we_cipher (_currentCipher));
			status = true;
		}
	}

	return status;
}
		
SecretMap
Encryption::getSecrets () const
{
	return _secret;
}

void
Encryption::setSecrets (const SecretMap & secret)
{
	_secret = secret;
	/* WPAEnterprise does not store secrets on disk so no need to sync */
	_dirty |= ( dynamic_cast<EncryptionWPAEnterprise *>(this) == NULL );
}

void
Encryption::setNetwork (Network* network)
{
	_network = network;
	_dirty = true;
}

bool
Encryption::hasStoredKey (void) const
{
	return KNetworkManagerStorage::getInstance()->hasCredentialsStored(_network->getEssid());
}

void
Encryption::slotCredentialsLoaded (QString /*id*/, QMap<QString, QString> map, bool canceled)
{
	bool result = false;

	if ( !map.isEmpty() ) {
		_secret = map;
		result = true;
	}
	emit keyRestored(result, canceled);

	// we got the desired information -> delete the request
	delete _credentialsRequest;
	_credentialsRequest = NULL;
}

void
Encryption::restoreKeyAsync (void)
{
	if ( hasStoredKey() && _network) {
		// if a credentialrequest for this encryption is already running, simply delete it
		if (_credentialsRequest) delete _credentialsRequest;

		// lets start a new credential request
		_credentialsRequest = KNetworkManagerStorage::getInstance( )->credentialsAsync(_network->getEssid());
		connect(_credentialsRequest, SIGNAL(credentialsLoaded(QString, QMap<QString, QString>, bool)), this, SLOT(slotCredentialsLoaded (QString, QMap<QString, QString>, bool)));
		_credentialsRequest->loadCredentials();
	} else {
		emit keyRestored(false, false);
	}
}

bool
Encryption::restoreKey (void)
{
	kdDebug() << k_funcinfo << endl;
	bool result = false;
	if ( hasStoredKey() && _network) {
		SecretMap secret = KNetworkManagerStorage::getInstance( )->credentials( _network->getEssid( ) );

		if ( !secret.isEmpty() ) {
			_secret = secret;
			result = true;
		}
	}
	return result;
}

bool
Encryption::persistKey (void) const
{
	if ( !_secret.isEmpty() && _network )
		return KNetworkManagerStorage::getInstance( )->storeCredentials( _network->getEssid( ), _secret);
	
	return hasStoredKey();
}

bool
Encryption::isModified (void) const
{
	return _dirty;
}

int
Encryption::getWeCipher (void) const
{
	return _we_cipher;
}

void
Encryption::setWeCipher (int we_cipher)
{
	_dirty |= (we_cipher != _we_cipher);
	_we_cipher = we_cipher;
}

Encryption::Encryption () : _currentCipher(0), _cipherList(0), _network(0), _credentialsRequest(0)
{
	_dirty = true;
	_we_cipher = -1;
}

void Encryption::clearCipherList()
{
    if (_cipherList)
        for (CipherList::Iterator it = _cipherList->begin(); it != _cipherList->end(); ++it)
        {
            ieee_802_11_cipher_unref(*it);
            (*it) = NULL;
        }
    delete _cipherList;
}

Encryption::~Encryption ()
{
    clearCipherList();
}


/* EncryptionNone */

bool
EncryptionNone::isValid (const QString & essid)
{
	if (essid.length ())
		return true;
	else
		return false;
}

bool
EncryptionNone::serialize (DBusMessage* msg, const QString & essid)
{
	kdDebug() << k_funcinfo << endl;
	bool status = false;

	if (!msg || !essid)
		return false;

	status = nmu_security_serialize_none_with_cipher (msg);

	return status;
}

bool 
EncryptionNone::deserialize( DBusMessageIter *, int we_cipher )
{
	return ( we_cipher == IW_AUTH_CIPHER_NONE );
}

void
EncryptionNone::setDefaults (void)
{
	return;
}

EncryptionNone::EncryptionNone ()
    : Encryption()
{

}

EncryptionNone::~EncryptionNone ()
{

}

void EncryptionNone::persist( KConfigBase * cfg, bool /*withKey*/ ) const
{
	cfg->writeEntry( "Encryption", "none" );
}

void EncryptionNone::restore( KConfigBase *, const char* /*version*/,  bool /*withKey*/)
{
	kdDebug() << k_funcinfo << endl;
}


/* EncryptionWEP */

bool
EncryptionWEP::serialize (DBusMessage* msg, const QString & essid)
{
	kdDebug() << "serialize: msg: " << msg << " essid: " << essid << " isValid(): " << isValid( essid )
		  << " secret: " << _secret["password"] << " method: " << _method << endl;
	bool status = false;

	/*
	 * There are two possibilities: we can serialize Encryption with or without a key.
	 * In the latter case, if we have a key in the secure storage, we can send the rest
	 * of the data to NM and supply a key on demand. This will show NM that we know
	 * the network we are connecting to and there is no need to ask user for a new key.
	 */
	if ( hasStoredKey() && _secret["password"].isEmpty() ) {
		if ( !msg || essid.isEmpty() )
			return false;

		/*
		 * We only need to pass we_cipher remembered since the last connection.
		 * Unfortunately, libnm-util functions accept only IEEE_802_11_Cipher object
		 * so we need to construct one with required we_cipher value.
		 */
		IEEE_802_11_Cipher *fake_cipher = 0;
		const char *fake_key = "";

		/*
		 * WEP ciphers differ only by key length regardless of type so we can use
		 * the corresponding _cipherList item as a fake cipher
		 */
		switch (_we_cipher)
		{
			case IW_AUTH_CIPHER_WEP40:
				fake_cipher = _cipherList->first();
				break;
			case IW_AUTH_CIPHER_WEP104:
				fake_cipher = _cipherList->last();
				break;
		}

		if (fake_cipher)
			status = nmu_security_serialize_wep_with_cipher (msg, fake_cipher, essid.utf8(), fake_key, _method);
	} else {
		if (!msg || essid.isEmpty() || !isValid( essid ) ) {
			return false;
		}

		status = nmu_security_serialize_wep_with_cipher (msg, _currentCipher, essid.utf8(), _secret["password"].utf8(), _method);
	}

	return status;
}

bool 
EncryptionWEP::deserialize( DBusMessageIter * iter, int we_cipher )
{
	char* key = 0;
	int   keyLen, authAlg;

	if ( iter == 0 )
		return false;
	if ( !( we_cipher == NM_AUTH_TYPE_WEP40 || we_cipher == NM_AUTH_TYPE_WEP104 ) )
		return false;

	if ( !nmu_security_deserialize_wep( iter, &key, &keyLen, &authAlg ) )
		return false;

	if ( !( authAlg == IW_AUTH_ALG_OPEN_SYSTEM || authAlg == IW_AUTH_ALG_SHARED_KEY ) )
		return false;

	setMethod( (WEPMethod) authAlg );
	setWeCipher( we_cipher );

	// we don't store this key, as it is the hashed version
	return true;
}

void
EncryptionWEP::setType( WEPType type )
{
	_type = type;
	_dirty = true;
	clearCipherList();
	_cipherList = new CipherList ();
	
	if (_type == WEP_ASCII) {
		_cipherList->append (cipher_wep64_ascii_new  ());
		_cipherList->append (cipher_wep128_ascii_new  ());
	} else if (_type == WEP_HEX) {
		_cipherList->append (cipher_wep64_hex_new  ());
		_cipherList->append (cipher_wep128_hex_new  ());
	} else if (_type == WEP_PASSPHRASE) {
		_cipherList->append (cipher_wep64_passphrase_new  ());
		_cipherList->append (cipher_wep128_passphrase_new  ());
	}
}

void
EncryptionWEP::setDefaults ()
{
	setMethod (WEP_OPEN_SYSTEM);
}

void
EncryptionWEP::setMethod (WEPMethod method)
{
	_dirty |= (_method != method);
	_method = method;
}

WEPMethod
EncryptionWEP::getMethod (void)
{
	return _method;
}


EncryptionWEP::EncryptionWEP (WEPType type)
{
	setType (type);
	setDefaults ();
}

EncryptionWEP::~EncryptionWEP ()
{

}

void EncryptionWEP::persist( KConfigBase * cfg, bool withKey ) const
{
	cfg->writeEntry( "Encryption", "WEP" );
	if ( WEP_ASCII == _type ) {
		cfg->writeEntry( "WEPType", "ASCII" );
	} else if ( WEP_HEX == _type ) {
		cfg->writeEntry( "WEPType", "HEX" );
	} else {
		cfg->writeEntry( "WEPType", "PASSPHRASE" );
	}
	if ( WEP_OPEN_SYSTEM == _method ) {
		cfg->writeEntry( "WEPMethod", "OpenSystem" );
	} else {
		cfg->writeEntry( "WEPMethod", "SharedKey" );
	}
	if ( withKey )
		this->persistKey( );
	cfg->writeEntry ( "Cipher", _we_cipher );
	_dirty = false;
}

void EncryptionWEP::restore( KConfigBase * cfg, const char* version, bool withKey)
{
	kdDebug() << k_funcinfo << endl;
	QString wepType = cfg->readEntry( "WEPType", "ASCII" );
	if ( "ASCII" == wepType ) {
		setType( WEP_ASCII );
	} else if ( "HEX" == wepType ) {
		setType( WEP_HEX );
	} else {
		setType( WEP_PASSPHRASE );
	}

	QString wepMethod = cfg->readEntry( "WEPMethod", "OpenSystem" );

	if ( "OpenSystem" == wepMethod )
		_method = WEP_OPEN_SYSTEM;
	else
		_method = WEP_SHARED_KEY;

	_dirty = false;

	if (strcmp(version, "0") == 0 ) {
		bool keyStored = this->restoreKey ();
		if (keyStored && _network)
			this->isValid (_network->getEssid ()); // set _we_cipher
		this->persist (cfg, true);
	} else {
		if (withKey && hasStoredKey())
			this->restoreKey ();
		_we_cipher = cfg->readNumEntry ("Cipher", -1);
	}
}


/* EncryptionWPAPersonal */

bool
EncryptionWPAPersonal::serialize (DBusMessage* msg, const QString & essid)
{
	kdDebug() << k_funcinfo << essid << endl;
	bool status = false;

	/*
	 * There are two possibilities: we can serialize Encryption with or without a key.
	 * In the latter case, if we have a key in the secure storage, we can send the rest
	 * of the data to NM and supply a key on demand. This will show NM that we know
	 * the network we are connecting to and there is no need to ask user for a new key.
	 */
	if ( hasStoredKey() && _secret["password"].isEmpty() ) {
		if ( !msg || essid.isEmpty() )
			return false;
		/*
		 * We only need to pass we_cipher remembered since the last connection.
		 * Unfortunately, libnm-util functions accept only IEEE_802_11_Cipher object
		 * so we need to construct one with required we_cipher value.
		 */
		IEEE_802_11_Cipher *fake_cipher = 0;
		const char *fake_key = "";

		/* 
		 * In case of WPA Personal, we can create any cipher_wpa_psk object we like
		 * and set the required we_cipher via set_we_cipher method.
		 */
		if ( _we_cipher != -1 ) {
			fake_cipher = cipher_wpa_psk_hex_new();
			cipher_wpa_psk_hex_set_we_cipher(fake_cipher, _we_cipher);
			status = nmu_security_serialize_wpa_psk_with_cipher (msg, fake_cipher, essid.utf8(), fake_key,
									     _version, IW_AUTH_KEY_MGMT_PSK);
			ieee_802_11_cipher_unref(fake_cipher);
		}
	} else {
		if ( !msg || !essid || !isValid( essid ) )
			return false;

		status = nmu_security_serialize_wpa_psk_with_cipher (msg, _currentCipher, essid.utf8(),
								     _secret["password"].utf8(), _version, IW_AUTH_KEY_MGMT_PSK);
	}

	return status;
}

bool 
EncryptionWPAPersonal::deserialize( DBusMessageIter * iter, int we_cipher )
{
	char* key = 0;
	int   keyLen, wpaVersion, keyManagement;

	if ( !iter )
		return false;
	if ( !(we_cipher == NM_AUTH_TYPE_WPA_PSK_TKIP || we_cipher == NM_AUTH_TYPE_WPA_PSK_CCMP ||
	       we_cipher == NM_AUTH_TYPE_WPA_PSK_AUTO) )
		return false;

	if (!nmu_security_deserialize_wpa_psk (iter, &key, &keyLen, &wpaVersion, &keyManagement) )
		return false;

	if ( !(wpaVersion == IW_AUTH_WPA_VERSION_WPA || wpaVersion == IW_AUTH_WPA_VERSION_WPA2 ) )
		return false;

	if ( keyManagement != IW_AUTH_KEY_MGMT_PSK )
		return false;

	setVersion( (WPAVersion)wpaVersion );
	setWeCipher( we_cipher );

	// we don't store this key, as it is the hashed version
	return true;
}

void
EncryptionWPAPersonal::setDefaults (void)
{
	/* Once NM can default to "no default protocol" change this */
	setProtocol (WPA_AUTO);
	setVersion  (WPA1);
}

void
EncryptionWPAPersonal::setProtocol (WPAProtocol protocol)
{
	_protocol = protocol;
	_dirty |= (_protocol != protocol);

	/* switch ciphers to match with protocol */
	cipher_wpa_psk_hex_set_we_cipher        ((IEEE_802_11_Cipher*) *_cipherList->at(0), _protocol);
	cipher_wpa_psk_passphrase_set_we_cipher ((IEEE_802_11_Cipher*) *_cipherList->at(1), _protocol);
}

WPAProtocol
EncryptionWPAPersonal::getProtocol (void)
{
	return _protocol;
}

void
EncryptionWPAPersonal::setVersion (WPAVersion version)
{
	_dirty |= (_version != version);
	_version = version;
}

WPAVersion
EncryptionWPAPersonal::getVersion (void)
{
	return _version;
}

EncryptionWPAPersonal::EncryptionWPAPersonal ()
{
	_cipherList = new CipherList (); 
        _cipherList->append (cipher_wpa_psk_hex_new ());
        _cipherList->append (cipher_wpa_psk_passphrase_new ());
	setDefaults ();
}

EncryptionWPAPersonal::~EncryptionWPAPersonal ()
{

}

void EncryptionWPAPersonal::persist( KConfigBase * cfg, bool withKey ) const
{
	cfg->writeEntry( "Encryption", "WPA" );
	if ( WPA_AUTO == _protocol ) {
		cfg->writeEntry( "WPAProtocol", "AUTO" );
	} if ( WPA_TKIP == _protocol ) {
		cfg->writeEntry( "WPAProtocol", "TKIP" );
	} else if (WPA_CCMP_AES == _protocol) {
		cfg->writeEntry( "WPAProtocol", "CCMPAES" );
	}

	if ( WPA1 == _version )
		cfg->writeEntry( "WPAVersion", "WPA1" );
	else
		cfg->writeEntry( "WPAVersion", "WPA2" );

	if ( withKey )
		this->persistKey( );

	cfg->writeEntry ("Cipher", _we_cipher);
	_dirty = false;
}

void EncryptionWPAPersonal::restore( KConfigBase * cfg, const char* version, bool withKey)
{
	kdDebug() << k_funcinfo << endl;

	QString wpaProtocol = cfg->readEntry( "WPAProtocol", "TKIP" );
	if ( "AUTO" == wpaProtocol ) {
		setProtocol (WPA_AUTO);
	} else if ( "TKIP" == wpaProtocol ) {
		setProtocol (WPA_TKIP);
	} else {
		setProtocol (WPA_CCMP_AES);
	}

	QString wpaVersion = cfg->readEntry( "WPAVersion", "WPA1" );
	if ( "WPA1" == wpaVersion )
		setVersion (WPA1);
	else
		setVersion (WPA2);

	_dirty = false;

	if (strcmp(version, "0") == 0) {
		bool keyStored = this->restoreKey ();
		if (keyStored && _network)
			this->isValid (_network->getEssid ()); // set _we_cipher
		this->persist (cfg, true);
	} else {
		if (withKey && hasStoredKey())
			this->restoreKey ();
		_we_cipher = cfg->readNumEntry ("Cipher", -1);
	}
}


/* EncryptionWPAEnterprise */

QString EncryptionWPAEnterprise::IdPasswordKey = "identityPassword";
QString EncryptionWPAEnterprise::CertPrivatePasswordKey = "certPrivatePassword";

bool
EncryptionWPAEnterprise::serialize (DBusMessage* msg, const QString & essid)
{
	kdDebug() << k_funcinfo << endl;
	DBusMessageIter iter;
	bool status = false;
	int eap_method = _method;
#ifdef HAVE_PHASE2_AUTH
	eap_method |= _phaseTwo;
#endif
	int keyType = NM_AUTH_TYPE_WPA_PSK_AUTO;

	switch(_protocol)
	{
		case WPA_AUTO:
			keyType = NM_AUTH_TYPE_WPA_PSK_AUTO;
			break;
		case WPA_TKIP:
			keyType = IW_AUTH_CIPHER_TKIP;
			break;
		case WPA_CCMP_AES:
			keyType = IW_AUTH_CIPHER_CCMP;
			break;
	}	
	if (!msg || !essid )
		return false;

	if (! (hasStoredKey() && _secret[IdPasswordKey].length() == 0 && _secret[CertPrivatePasswordKey].length() == 0)) {
		if (!isValid( essid ))
			return false;
	}

	/* FIXME: If the user denies access to kwallet we'd send NULL over the bus.  Avoid that. */
	if (!_secret[IdPasswordKey])
		_secret[IdPasswordKey] = "";

	if (!_secret[CertPrivatePasswordKey])
		_secret[CertPrivatePasswordKey] = "";

	kdDebug () << "method: " << _method << " identity: " << _identity << " password: " << _secret[IdPasswordKey] <<
		      " anon ident: " << _anonIdentity << " cert priv passwd: "  << _secret[CertPrivatePasswordKey]  <<
		      " cert priv: " << _certPrivate << " cert client: " << _certClient << " cert CA: " << _certCA   <<
		      " version: " << _version << endl;

	dbus_message_iter_init_append (msg, &iter);
	if (_method == EAP_LEAP )
	{
#ifdef HAVE_LEAP
		status = nmu_security_serialize_leap_with_cipher(&iter, _identity.utf8(), _secret[IdPasswordKey] ,
									_leapmethod );
#else
		return false;
#endif
	}
	else
	{
		status = nmu_security_serialize_wpa_eap_with_cipher (&iter, eap_method, keyType, _identity.utf8(),
								     _secret[IdPasswordKey].utf8(), _anonIdentity.utf8(),
								     _secret[CertPrivatePasswordKey].utf8(),
								     _certPrivate.utf8(), _certClient.utf8(), _certCA.utf8(),
								     _version);
	}

	return status;
}

bool 
EncryptionWPAEnterprise::deserialize( DBusMessageIter * iter, int we_cipher )
{
	int method;
	EAPMethod eapMethod;
	EAPPhaseTwo eapPhaseTwo;
	int keyType;
	char* leapmethod;
	char* identity = NULL;
	char* identityPassword = NULL;
	char* anonIdentity = NULL;
	char* certPrivatePassword = NULL;
	char* certPrivate = NULL;
	char* certClient = NULL;
	char* certCA = NULL;
	int version;

	if ( !iter )
		return false;

	if (method == EAP_LEAP)
	{
#ifdef HAVE_LEAP
		if (we_cipher != NM_AUTH_TYPE_LEAP)
			return false;
		if (!nmu_security_deserialize_leap (iter, &identity, &identityPassword, &leapmethod))
			return false;
#else
		return false;
#endif
	}
	else
	{
		if (we_cipher != NM_AUTH_TYPE_WPA_EAP)
			return false;

		if (!nmu_security_deserialize_wpa_eap (iter, &method, &keyType, &identity, &identityPassword, &anonIdentity,
						       &certPrivatePassword, &certPrivate, &certClient, &certCA, &version))
			return false;
	}

	if ( !(version == IW_AUTH_WPA_VERSION_WPA || version == IW_AUTH_WPA_VERSION_WPA2 ) )
		return false;
	
#ifdef HAVE_PHASE2_AUTH
	eapMethod = (EAPMethod) NM_EAP_TO_EAP_METHOD(method);
	eapPhaseTwo = (EAPPhaseTwo) NM_EAP_TO_PHASE2_METHOD(method);
#else
	eapMethod = (EAPMethod) method;
	eapPhaseTwo = PHASE2_NONE;
#endif

	if ( !(eapMethod == EAP_PEAP || eapMethod == EAP_TLS || eapMethod == EAP_TTLS || eapMethod == EAP_LEAP) )
		return false;
	if ( !(eapPhaseTwo == PHASE2_NONE || eapPhaseTwo == PHASE2_PAP || eapPhaseTwo == PHASE2_MSCHAP || eapPhaseTwo == PHASE2_MSCHAPV2 || eapPhaseTwo == PHASE2_GTC) )
		return false;

	setMethod ( eapMethod );
	setPhaseTwoAuth ( eapPhaseTwo );
	setIdentity (identity);
	setAnonIdentity (anonIdentity);
	setCertPrivate (certPrivate);
	setCertClient (certClient);
	setCertCA (certCA);
	setVersion ((WPAVersion) version);
	setWeCipher (we_cipher);
	SecretMap map;
	map.insert( IdPasswordKey, identityPassword );
	map.insert( CertPrivatePasswordKey, certPrivatePassword );
	setSecrets( map );
	
	return true;
}

bool
EncryptionWPAEnterprise::isValid (const QString & essid)
{
	bool hasSecret = (_secret[IdPasswordKey].length() > 0) || (_secret[CertPrivatePasswordKey].length() > 0);
	if (essid.length () && hasSecret)
		return true;
	else
		return false;
}

void
EncryptionWPAEnterprise::setDefaults (void)
{
	/* Once NM can default to "no default protocol" change this */
	setProtocol (WPA_EAP);
	setVersion  (WPA1);
	setMethod   (EAP_PEAP);
	setPhaseTwoAuth (PHASE2_NONE);
	setIdentity ("");
	setAnonIdentity ("");
	setCertPrivate ("");
	setCertClient ("");
	setCertCA ("");
	SecretMap map;
	map.insert( IdPasswordKey, "" );
	map.insert( CertPrivatePasswordKey, "" );
	setSecrets( map );
}

void
EncryptionWPAEnterprise::setIdentity (const QString & identity)
{
	kdDebug() << k_funcinfo << endl;
	_dirty |= (_identity != identity);
	_identity = identity;
}

QString
EncryptionWPAEnterprise::getIdentify (void) const
{
	kdDebug() << k_funcinfo << endl;
	return _identity;
}

void
EncryptionWPAEnterprise::setAnonIdentity (const QString & anonIdentity)
{
	kdDebug() << k_funcinfo << endl;
	_dirty |= (_anonIdentity != anonIdentity);
	_anonIdentity = anonIdentity;
}

QString
EncryptionWPAEnterprise::getAnonIdentity (void) const
{
	kdDebug() << k_funcinfo << endl;
	return _anonIdentity;
}

void
EncryptionWPAEnterprise::setCertClient (const QString & certClient)
{
	kdDebug() << k_funcinfo << endl;
	_dirty |= (_certClient != certClient);
	_certClient = certClient;
}

QString
EncryptionWPAEnterprise::getCertClient (void) const
{
	kdDebug() << k_funcinfo << endl;
	return _certClient;
}

void
EncryptionWPAEnterprise::setCertCA (const QString & certCA)
{
	kdDebug() << k_funcinfo << endl;
	_dirty |= (_certCA != certCA);
	_certCA = certCA;
}

QString
EncryptionWPAEnterprise::getCertCA (void) const
{
	kdDebug() << k_funcinfo << endl;
	return _certCA;
}

void
EncryptionWPAEnterprise::setCertPrivate (const QString & certPrivate)
{
	kdDebug() << k_funcinfo << endl;
	_dirty |= (_certPrivate != certPrivate);
	_certPrivate = certPrivate;
}

QString
EncryptionWPAEnterprise::getCertPrivate (void) const
{
	kdDebug() << k_funcinfo << endl;
	return _certPrivate;
}

void
EncryptionWPAEnterprise::setProtocol (WPAProtocol protocol)
{
	kdDebug() << k_funcinfo << endl;
	_dirty |= (_protocol != protocol);
	_protocol = protocol;
}

WPAProtocol
EncryptionWPAEnterprise::getProtocol (void)
{
	kdDebug() << k_funcinfo << endl;
	return _protocol;
}

void
EncryptionWPAEnterprise::setVersion (WPAVersion version)
{
	kdDebug() << k_funcinfo <<  " " << version << endl;
	_version = version;
}

WPAVersion
EncryptionWPAEnterprise::getVersion (void)
{
	kdDebug() << k_funcinfo << endl;
	return _version;
}

void
EncryptionWPAEnterprise::setMethod (EAPMethod method)
{
	kdDebug() << k_funcinfo << " " << method <<  endl;
	_method = method;
}

EAPMethod
EncryptionWPAEnterprise::getMethod (void)
{
	kdDebug() << k_funcinfo << endl;
	return _method;
}

void
EncryptionWPAEnterprise::setPhaseTwoAuth (EAPPhaseTwo auth)
{
	kdDebug() << k_funcinfo << " " << auth << endl;
	_phaseTwo = auth;
}

EAPPhaseTwo
EncryptionWPAEnterprise::getPhaseTwoAuth (void)
{
	kdDebug() << k_funcinfo << endl;
	return _phaseTwo;
}

QString
EncryptionWPAEnterprise::getLeapMethod (void)
{
	kdDebug() << k_funcinfo << endl;
	return _leapmethod;
}

void
EncryptionWPAEnterprise::setLeapMethod (const QString & leapmethod)
{
	kdDebug() << k_funcinfo << " " << leapmethod <<  endl;
	_leapmethod = leapmethod;
}

EncryptionWPAEnterprise::EncryptionWPAEnterprise ()
{
	kdDebug() << k_funcinfo << endl;
	/* no ciphers req. */
	setDefaults ();
}

EncryptionWPAEnterprise::~EncryptionWPAEnterprise ()
{
	kdDebug() << k_funcinfo << endl;

}

void EncryptionWPAEnterprise::persist( KConfigBase * cfg, bool withKey ) const
{
	kdDebug() << k_funcinfo << endl;

	cfg->writeEntry( "Encryption", "WPA-EAP" );
	if ( EAP_PEAP == _method ) {
		cfg->writeEntry( "Method", "PEAP" );
	} else if ( EAP_TLS == _method ) {
		cfg->writeEntry( "Method", "TLS" );
	} else if ( EAP_LEAP == _method ) {
		cfg->writeEntry( "Method", "LEAP" );
	} else {
		cfg->writeEntry( "Method", "TTLS" );
	}

	if ( PHASE2_PAP == _phaseTwo ) {
		cfg->writeEntry( "PhaseTwo", "PAP" );
	} else if ( PHASE2_MSCHAP == _phaseTwo ) {
		cfg->writeEntry( "PhaseTwo", "MSCHAP" );
	} else if ( PHASE2_MSCHAPV2 == _phaseTwo ) {
		cfg->writeEntry( "PhaseTwo", "MSCHAPV2" );
	} else if ( PHASE2_GTC == _phaseTwo ) {
		cfg->writeEntry( "PhaseTwo", "GTC" );
	} else {
		cfg->writeEntry( "PhaseTwo", "NONE");
	}

	cfg->writeEntry( "Identity", _identity );
	cfg->writeEntry( "AnonIdentity", _anonIdentity );
	cfg->writeEntry( "CertPrivate", _certPrivate );
	cfg->writeEntry( "CertClient", _certClient );
	cfg->writeEntry( "CertCA", _certCA );
	cfg->writeEntry( "LeapMethod", _leapmethod);

	if ( WPA_TKIP == _protocol )
		cfg->writeEntry( "WPAProtocol", "TKIP" );
	else if (WPA_CCMP_AES == _protocol)
		cfg->writeEntry( "WPAProtocol", "CCMPAES" );

	if ( WPA1 == _version )
		cfg->writeEntry( "WPAVersion", "WPA1" );
	else
		cfg->writeEntry( "WPAVersion", "WPA2" );

	if ( withKey )
		this->persistKey( );

	cfg->writeEntry ( "Cipher", _we_cipher );
	_dirty = false;
}

void EncryptionWPAEnterprise::restore( KConfigBase * cfg, const char* /*version*/, bool withKey)
{
	kdDebug() << k_funcinfo << endl;

	QString wpaProtocol = cfg->readEntry( "WPAProtocol", "TKIP" );
	if ( "AUTO" == wpaProtocol ) {
		setProtocol (WPA_AUTO);
	} else if ( "TKIP" == wpaProtocol ) {
		setProtocol (WPA_TKIP);
	} else {
		setProtocol (WPA_CCMP_AES);
	}
		
	QString wpaVersion = cfg->readEntry( "WPAVersion", "WPA1" );
	if ( "WPA1" == wpaVersion )
		setVersion (WPA1);
	else
		setVersion (WPA2);

	QString method = cfg->readEntry( "Method" );
	if ( "PEAP" == method ) {
		_method = EAP_PEAP;
	} else if ( "TLS" == method ) {
		_method = EAP_TLS;
	} else if ( "LEAP" == method ) {
		_method = EAP_LEAP;
	} else {
		_method = EAP_TTLS;
	}

	QString phaseTwo = cfg->readEntry( "PhaseTwo" );
	if ( "PAP" == phaseTwo ) {
		_phaseTwo = PHASE2_PAP;
	} else if ( "MSCHAP" == phaseTwo ) {
		_phaseTwo = PHASE2_MSCHAP;
	} else if ( "MSCHAPV2" == phaseTwo ) {
		_phaseTwo = PHASE2_MSCHAPV2;
	} else if ( "GTC" == phaseTwo ) {
		_phaseTwo = PHASE2_GTC;
	} else {
		_phaseTwo = PHASE2_NONE;
	}

	_identity = cfg->readEntry( "Identity" );
	_anonIdentity = cfg->readEntry( "AnonIdentity" );
	_certPrivate = cfg->readEntry( "CertPrivate" );
	_certClient = cfg->readEntry( "CertClient" );
	_certCA = cfg->readEntry( "CertCA" );

	_leapmethod = cfg->readEntry( "LeapMethod" );

	// _we_cipher is not used in WPAEnterprise, so no verison check required
	_we_cipher = cfg->readNumEntry( "Cipher", -1 );

	if ( withKey && hasStoredKey() )
		this->restoreKey( );

	_dirty = false;
}
