/* This file is part of the KDE project
 * Copyright (C) 2008 Thorsten Zachmann <zachmann@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (  at your option ) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "KPrPlaceholder.h"

#include <KoXmlReader.h>
#include <KoXmlWriter.h>
#include <KoXmlNS.h>
#include <KoUnit.h>
#include <KoPASavingContext.h>
#include <kdebug.h>

KPrPlaceholder::KPrPlaceholder()
{
}

KPrPlaceholder::~KPrPlaceholder()
{
}

bool KPrPlaceholder::loadOdf( const KoXmlElement & element, const QRectF & pageSize )
{
    if ( element.hasAttributeNS( KoXmlNS::presentation, "object" ) ) {
        m_presentationObject = element.attributeNS( KoXmlNS::presentation, "object" );
    }
    else {
        // TODO error message
        return false;
    }
    if ( element.hasAttributeNS( KoXmlNS::svg, "x" ) ) {
        m_relativeSize.setX( percent( element, "x", pageSize.width() ) );
    }
    if ( element.hasAttributeNS( KoXmlNS::svg, "y" ) ) {
        m_relativeSize.setY( percent( element, "y", pageSize.height() ) );
    }
    if ( element.hasAttributeNS( KoXmlNS::svg, "width" ) ) {
        m_relativeSize.setWidth( percent( element, "width", pageSize.width() ) );
    }
    if ( element.hasAttributeNS( KoXmlNS::svg, "height" ) ) {
        m_relativeSize.setHeight( percent( element, "height", pageSize.height() ) );
    }

    kDebug(33001) << "convert" << pageSize << m_relativeSize;

    return true;
}

void KPrPlaceholder::saveOdf( KoXmlWriter & xmlWriter )
{
    xmlWriter.startElement( "presentation:placeholder" );
    xmlWriter.addAttribute( "presentation:object", m_presentationObject );
    xmlWriter.addAttribute( "svg:x", QString( "%1%" ).arg( m_relativeSize.x() * 100.0 ) );
    xmlWriter.addAttribute( "svg:y", QString( "%1%" ).arg( m_relativeSize.y() * 100.0 ));
    xmlWriter.addAttribute( "svg:width", QString( "%1%" ).arg( m_relativeSize.width() * 100.0 ));
    xmlWriter.addAttribute( "svg:height", QString( "%1%" ).arg( m_relativeSize.height() * 100.0 ));
    xmlWriter.endElement();
}

QString KPrPlaceholder::presentationObject()
{
    return m_presentationObject;
}

QRectF KPrPlaceholder::rect( const QSizeF & pageSize )
{
    QRectF r;
    r.setX( pageSize.width() * m_relativeSize.x() );
    r.setY( pageSize.height() * m_relativeSize.y() );
    r.setWidth( pageSize.width() * m_relativeSize.width() );
    r.setHeight( pageSize.height() * m_relativeSize.height() );
    return r;
}

void KPrPlaceholder::fix( const QRectF & rect )
{
    if ( m_relativeSize.width() < 0 ) {
        m_relativeSize.setWidth( rect.width() );
    }
    if ( m_relativeSize.height() < 0 ) {
        m_relativeSize.setHeight( rect.height() );
    }
}

qreal KPrPlaceholder::percent( const KoXmlElement & element, const char * type, qreal absolute )
{
    qreal tmp = 0.0;
    QString value = element.attributeNS( KoXmlNS::svg, type, QString( "0%" ) );
    if ( value.indexOf( '%' ) > -1 ) { // percent value
        tmp = value.remove( '%' ).toDouble() / 100.0;
    }
    else { // fixed value
        tmp = KoUnit::parseValue( value ) / absolute;
    }

    return tmp;
}

bool KPrPlaceholder::operator==( const KPrPlaceholder & other ) const
{
    return m_presentationObject == other.m_presentationObject && m_relativeSize == other.m_relativeSize;
}

bool KPrPlaceholder::operator<( const KPrPlaceholder & other ) const
{
    if ( m_presentationObject == other.m_presentationObject ) {
        if ( m_relativeSize.x() == other.m_relativeSize.x() ) {
            if ( m_relativeSize.y() == other.m_relativeSize.y() ) {
                if ( m_relativeSize.width() == other.m_relativeSize.width() ) {
                    return m_relativeSize.height() < other.m_relativeSize.height();
                }
                return m_relativeSize.width() < other.m_relativeSize.width();
            }
            return m_relativeSize.y() < other.m_relativeSize.y();
        }
        return m_relativeSize.x() < other.m_relativeSize.x();
    }
    return m_presentationObject < other.m_presentationObject;
}
