/*
 *  This file is part of the KDE project
 *
 *  Copyright (c) 2007 Cyrille Berger <cberger@cberger.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "kis_basic_math_toolbox2.h"

#include "kis_generic_colorspace.h"
#include "kis_iterators_pixel.h"
#include "kis_transform_worker.h"
#include "kis_filter_strategy.h"

#include <kis_convolution_painter.h>
#include <kis_transaction.h>

// TODO stolen from KisMathToolbox, and updated
template<typename T>
float toDouble(const Q_UINT8* data, int channelpos )
{
    return (float)( *( ((T*)data) + channelpos) );
}

typedef float (*PtrToDouble)(const Q_UINT8*, int);

template<typename T, int min, int max>
void fromDoubleMinMax(Q_UINT8* data, int channelpos, float v )
{
    if( v < min ) v = min;
    else if( v > max) v = max;
    *((T*)(data) + channelpos) = (T)v;
}

template<typename T>
void fromDouble(Q_UINT8* data, int channelpos, float v )
{
    *((T*)(data) + channelpos) = (T)v;
}

typedef void (*PtrFromDouble)(Q_UINT8*, int, float);

// TODO: move to KisMathToolbox

KisColorSpace* createDoubleColorSpace(Q_INT32 depth)
{
    switch(depth)
    {
        case 1:
        return new KisGenericColorspace<float, 1>;
        break;
        case 3:
        return new KisGenericColorspace<float, 3>;
        break;
        case 4:
        return new KisGenericColorspace<float, 4>;
        break;
        default:
        return 0;
    }
}

KisPaintDeviceSP KisBasicMathToolbox2::toFloatDevice(KisPaintDeviceSP src, const QRect& rect)
{
    Q_INT32 depth = src->colorSpace()->nColorChannels();
    KisColorSpace* cs = createDoubleColorSpace(depth);
    
    KisPaintDeviceSP dst = new KisPaintDevice(cs);
    QValueVector<KisChannelInfo *> cis = src->colorSpace()->channels();
    
    PtrToDouble f[depth];
    for(Q_INT32 k = 0; k < depth; k++)
    {
        switch( cis[k]->channelValueType() )
        {
            case KisChannelInfo::UINT8:
                f[k] = toDouble<Q_UINT8>;
                break;
            case KisChannelInfo::UINT16:
                f[k] = toDouble<Q_UINT16>;
                break;
#ifdef HAVE_OPENEXR
            case KisChannelInfo::FLOAT16:
                f[k] = toDouble<half>;
                break;
#endif
            case KisChannelInfo::FLOAT32:
                f[k] = toDouble<float>;
                break;
            case KisChannelInfo::INT8:
                f[k] = toDouble<Q_INT8>;
                break;
            case KisChannelInfo::INT16:
                f[k] = toDouble<Q_INT16>;
                break;
            default:
                kdWarning() << "Unsupported value type in KisMathToolbox" << endl;
                return 0;
        }
    }
    
    KisHLineIterator itSrc = src->createHLineIterator(rect.x(), rect.y(), rect.width(), false);
    KisHLineIterator itDst = dst->createHLineIterator(0, 0, rect.width(), true);
    
    for(int i = rect.y(); i < rect.height(); i++)
    {
        while( ! itSrc.isDone() )
        {
            const Q_UINT8* v1 = itSrc.oldRawData();
            float* v2 = reinterpret_cast<float*>(itDst.rawData());
            for( int k = 0; k < depth; k++)
            {
                v2[k] = f[k](v1, k);
            }
            ++itSrc;
            ++itDst;
        }
        itSrc.nextRow();
        itDst.nextRow();
    }
    return dst;
}

void KisBasicMathToolbox2::fromFloatDevice(KisPaintDeviceSP src, KisPaintDeviceSP dst, const QRect& rect)
{
    QValueVector<KisChannelInfo *> cis = dst->colorSpace()->channels();
    Q_INT32 depth = dst->colorSpace()->nColorChannels();
    PtrFromDouble f[depth];
    for(Q_INT32 k = 0; k < depth; k++)
    {
        switch( cis[k]->channelValueType() )
        {
            case KisChannelInfo::UINT8:
                f[k] = fromDoubleMinMax<Q_UINT8, 0, 0xFF>;
                break;
            case KisChannelInfo::UINT16:
                f[k] = fromDoubleMinMax<Q_UINT16, 0, 0xFFFF>;
                break;
#ifdef HAVE_OPENEXR
            case KisChannelInfo::FLOAT16:
                f[k] = fromDouble<half>;
                break;
#endif          
            case KisChannelInfo::FLOAT32:
                f[k] = fromDouble<float>;
                break;
            case KisChannelInfo::INT8:
                f[k] = fromDoubleMinMax<Q_INT8,-127,128>;
                break;
            case KisChannelInfo::INT16:
                f[k] = fromDoubleMinMax<Q_INT16,  -32767, 32768>;
                break;
            default:
                kdWarning() << "Unsupported value type in KisMathToolbox" << endl;
                return;
        }
    }
    KisHLineIterator srcIt = src->createHLineIterator(0, 0, rect.width(), false);
    KisHLineIteratorPixel dstIt = dst->createHLineIterator(rect.x(), rect.y(), rect.width(), true );
    for(int i = rect.y(); i < rect.height(); i++)
    {
        while( ! dstIt.isDone() )
        {
            Q_UINT8* v1 = dstIt.rawData();
            const float* v2 = reinterpret_cast<const float*>(srcIt.oldRawData());
            for( int k = 0; k < depth; k++)
            {
                f[k](v1, k, v2[k]);
            }
            ++dstIt;
            ++srcIt;
        }
        dstIt.nextRow();
        srcIt.nextRow();
    }
}

// TODO: move to KisBasicMathToolbox
KisBasicMathToolbox2::Pyramid* KisBasicMathToolbox2::toSimplePyramid(KisPaintDeviceSP src, int nbLevels, const QRect& rect)
{
    Pyramid* pyramid = new Pyramid;
    Pyramid::Level previousLevel;
    previousLevel.device = toFloatDevice(src, rect );
    previousLevel.size = rect.size();
    pyramid->levels.push_back(previousLevel);
    for(int level = 1; level <= nbLevels; level++)
    {
        KisPaintDeviceSP newleveldevice = new KisPaintDevice(*previousLevel.device);
        KisTransformWorker worker(newleveldevice, 0.5, 0.5, 0.0, 0.0, 0.0, 0, 0, 0, new KisBoxFilterStrategy);
        worker.run();
        previousLevel.device = newleveldevice;
        previousLevel.size = 0.5 * previousLevel.size;
        pyramid->levels.push_back(previousLevel);
    }
    return pyramid;
}
void blur(KisPaintDeviceSP dev, KisKernelSP kernel, const QRect& rect)
{
    KisConvolutionPainter painter( dev );
    kernel->width = 5;
    kernel->height = 1;
    painter.applyMatrix(kernel, rect.x(), rect.y(), rect.width(), rect.height(), BORDER_REPEAT, KisChannelInfo::FLAG_COLOR_AND_ALPHA);
    KisTransaction("", dev);
    
    kernel->width = 1;
    kernel->height = 5;
    painter.applyMatrix(kernel, rect.x(), rect.y(), rect.width(), rect.height(), BORDER_REPEAT, KisChannelInfo::FLAG_COLOR_AND_ALPHA);
}

KisBasicMathToolbox2::Pyramid* KisBasicMathToolbox2::toGaussianPyramid(KisPaintDeviceSP src, int nbLevels, const QRect& rect, double a)
{
    // Compute the kernel
    KisKernelSP kernel = new KisKernel;
    kernel->width = 5;
    kernel->height = 1;
    kernel->offset = 0;
    kernel->factor = 0;
    kernel->data = new Q_INT32[kernel->width];
    // Kernel == [ 1/4-a/2; 1/4; a; 1/4; 1/4-a/2 ] a \in [0.3; 0.6]
    kernel->data[0] = (int)(100 * ( 0.25 - 0.5 * a));
    kernel->data[1] = 25;
    kernel->data[2] = (int)(100 * a);
    kernel->data[3] = 25;
    kernel->data[4] = kernel->data[0];
    kernel->factor = 2*(kernel->data[0] + kernel->data[1]) + kernel->data[2]; // due to rounding it might not be equal to 100
    // Compute the pyramid
    Pyramid* pyramid = new Pyramid;
    Pyramid::Level previousLevel;
    previousLevel.device = toFloatDevice(src, rect );
    previousLevel.size = rect.size();
    pyramid->levels.push_back(previousLevel);
    for(int level = 1; level <= nbLevels; level++)
    {
        KisPaintDeviceSP newleveldevice = new KisPaintDevice(*previousLevel.device);
        QSize newSize = 0.5*previousLevel.size;
        // Gaussian blur
        KisTransformWorker worker(newleveldevice, 0.5, 0.5, 0.0, 0.0, 0.0, 0.0, 0.0, 0, new KisBoxFilterStrategy);
        worker.run();
        blur(newleveldevice, kernel, QRect(QPoint(0,0), newSize));
        previousLevel.device = newleveldevice;
        previousLevel.size = newSize;
        pyramid->levels.push_back(previousLevel);
    }
    return pyramid;
}

KisBasicMathToolbox2::Pyramid* KisBasicMathToolbox2::toLaplacianPyramid(/*const*/ KisBasicMathToolbox2::Pyramid* gaussianPyramid)
{
    Pyramid* pyramid = new Pyramid;
    for(uint i = 1; i < gaussianPyramid->levels.size(); i++)
    {
        KisPaintDeviceSP dst = new KisPaintDevice(*gaussianPyramid->levels[i].device);
        int depth = dst->colorSpace()->nColorChannels();
        QSize s2 = gaussianPyramid->levels[i].size;
        QSize s1 = gaussianPyramid->levels[i-1].size;
        if(s1.width() != s2.width() and s1.height() != s2.height())
        {
            KisTransformWorker worker(dst, s1.width() / (double)s2.width(), s1.height() / (double)s2.height(), 0.0, 0.0, 0.0, 2, 2, 0, new KisBoxFilterStrategy); // TODO: in 2.0 check if we can get ride of the (2,2) offsetting
            worker.run();
        }
        KisHLineIterator itSrc = gaussianPyramid->levels[i-1].device->createHLineIterator(0, 0, s1.width(), false);
        KisHLineIterator itDst = dst->createHLineIterator(0, 0, s1.width(), true);
        for(int y = 0; y < s1.height(); y++)
        {
            while(not itSrc.isDone())
            {
                const float* srcA = reinterpret_cast<const float*>( itSrc.oldRawData() );
                float* dstA = reinterpret_cast<float*>( itDst.rawData() );
                for(int i = 0; i < depth; i++)
                {
                    dstA[i] = srcA[i] - dstA[i];
                }
                ++itSrc;
                ++itDst;
            }
            itSrc.nextRow();
            itDst.nextRow();
        }
        Pyramid::Level level;
        level.device = dst;
        level.size = s1;
        pyramid->levels.push_back(level);
    }
    pyramid->levels.push_back(gaussianPyramid->levels.last());
    return pyramid;
}
