/*
 * Copyright (c) 2001-2003 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 * $HEADER$
 *
 * $Id: ssi_coll_open.c,v 1.15 2003/05/31 22:28:51 jsquyres Exp $
 *
 */

#include <lam_config.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <all_list.h>
#include <lam-ssi.h>
#include <lam-ssi-coll.h>
#include <etc_misc.h>
#include <lamdebug.h>
#include <typical.h>


/*
 * The following file was created by configure.  It contains extern
 * statements and the definition of an array of pointers to each
 * module's public lam_ssi_rpi_t struct.
 */

#include "lam-ssi-coll-modules.h"


/*
 * Global variables
 */
int lam_ssi_coll_verbose = -1;
int lam_ssi_coll_did = -1;
int lam_ssi_coll_base_crossover = 4;
int lam_ssi_coll_base_associative = 0;
int lam_ssi_coll_base_reduce_crossover = 512;
LIST *lam_ssi_coll_base_available = NULL;


/*
 * Local variables
 */
static lam_debug_stream_info_t lds = {
  0, -1, -1, NULL, "coll:", -1, -1, -1, -1, "ssi-coll.txt"
};


/*
 * Local functions
 */
static int parse_names(char *names, LIST **names_list);


int
lam_ssi_coll_base_open(OPT *aod)
{
  char *e;
  const lam_ssi_t *ls;
  lam_ssi_module_t entry;
  int i, found, thread_min, thread_max;
  LIST *names_list = NULL;

  /* Call to base SSI open */

  lam_ssi_base_open(aod);

  /* Setup verbosity for this kind */

  lam_ssi_base_set_verbose("LAM_MPI_SSI_coll_verbose", &lds, aod,
			   &lam_ssi_coll_verbose, &lam_ssi_coll_did);
  if (lam_ssi_coll_verbose >= 10)
    lam_debug(lam_ssi_coll_did, "open: Opening");

  /* Look for environment variable overrides */

  if ((e = getenv("LAM_MPI_SSI_coll")) != NULL) {
    names_list = al_init(LAM_MPI_SSI_BASE_MAX_MODULE_NAME_LEN, strcmp);
    if (names_list == NULL) {
      show_help(NULL, "system-call-failed", "malloc", NULL);
      return LAMERROR;
    }
    if (parse_names(e, &names_list) != 0)
      return LAMERROR;
  }
  if ((e = getenv("LAM_MPI_SSI_coll_base_crossover")) != NULL)
    lam_ssi_coll_base_crossover = atoi(e);
  if ((e = getenv("LAM_MPI_SSI_coll_base_associative")) != NULL)
    lam_ssi_coll_base_associative = atoi(e);
  if ((e = getenv("LAM_MPI_SSI_coll_base_reduce_crossover")) != NULL)
    lam_ssi_coll_base_reduce_crossover = atoi(e);
  if (lam_ssi_coll_verbose >= 0) {
    lam_debug(lam_ssi_coll_did, "open:crossover: %d processes",
	      lam_ssi_coll_base_crossover);
    lam_debug(lam_ssi_coll_did, "open:associative: %d",
	      lam_ssi_coll_base_associative);
  }

  /* Initialize the list */

  lam_ssi_coll_base_available = 
    al_init(sizeof(lam_ssi_module_t), lam_ssi_base_module_compare);
  if (lam_ssi_coll_base_available == NULL) {
    show_help(NULL, "system-call-failed", "malloc", NULL);
    return LAMERROR;
  }

  /* In 64 bit mode, this struct can have empty padding */

  LAM_ZERO_ME(entry);

  /* Call the open function in every collective module and see if they
     want to run.  We don't call the module query functions now --
     those happen on a per-communicator basis. */

  for (found = i = 0; lam_ssi_coll_modules[i] != NULL; ++i) {
    ls = &(lam_ssi_coll_modules[i]->lsc_meta_info);

    /* Only look at this module if its name is on the user-specified
       list, or if the user-specified list is empty */

    if (names_list != NULL &&
        al_find(names_list, (char*) ls->ssi_module_name) == NULL) {
      if (lam_ssi_coll_verbose >= 0)
        lam_debug(lam_ssi_coll_did, "open: skipping non-selected module %s",
                  ls->ssi_module_name);
      continue;
    }

    if (ls->ssi_open_module == NULL ||
        (ls->ssi_open_module != NULL && ls->ssi_open_module(aod) == 1)) {

      /* Call the initial "what thread levels do you support" function
         to find out if this module has any restrictions on thread
         levels */

      if (lam_ssi_coll_modules[i]->lsc_thread_query(&thread_min,
                                                    &thread_max) != 0)
        continue;

      if (lam_ssi_coll_verbose >= 0)
	lam_debug(lam_ssi_coll_did, "open: collective %s available", 
		  ls->ssi_module_name);

      /* Save the results in the list.  The priority isn't relevant,
         because selection is decided at communicator-constructor
         time.  But we save the thread_min and thread_max arguments so
         that the initial selection algorithm can negotiate the
         overall thread level for this process. */

      entry.lsm_priority = 0;
      entry.lsm_thread_min = thread_min;
      entry.lsm_thread_max = thread_max;
      entry.lsm_module = (lam_ssi_t*) lam_ssi_coll_modules[i];
      al_insert(lam_ssi_coll_base_available, &entry);
      found = 1;
    }
  }

  /* If we have no collective modules available, it's an error.
     Thanks for playing! */

  if (found == 0) {
    al_free(lam_ssi_coll_base_available);
    lam_ssi_coll_base_available = NULL;

    if (lam_ssi_coll_verbose >= 0)
      lam_debug(lam_ssi_coll_did, "open: No collectives available!");
    show_help("ssi-coll", "none-available", NULL);
    return LAMERROR;
  }

  /* Free the list */

  if (names_list != NULL)
    al_free(names_list);

  /* All done */

  return 0;
}


static int
parse_names(char *names, LIST **names_list)
{
  char *comma, *start;
  LIST *nl = *names_list;
  char name[LAM_MPI_SSI_BASE_MAX_MODULE_NAME_LEN];

  /* No env names */

  if (nl == NULL)
    return 0;

  /* Loop over all names */
  /* (yes, this could be more clever, but it's nice and obvious this
     way!) */

  start = names;
  comma = strchr(start, ',');
  while (comma != NULL) {
    *comma = '\0';
    lam_strncpy(name, start, sizeof(name) - 1);
    name[sizeof(name) - 1] = '\0';
    al_insert(nl, name);

    start = comma + 1;
    comma = strchr(start, ',');
  }

  /* The last name */

  lam_strncpy(name, start, sizeof(name) - 1);
  name[sizeof(name) - 1] = '\0';
  al_insert(nl, name);

  /* All done */

  return 0;
}
