import sys, os, re, string, traceback, gettext, linda
from linda import root, clparser, outputobj
from linda.debug import dprint
from linda.parser.datavalidator import DataValidator
from linda.parser.debfilename import DebFilenameParser
_ = gettext.gettext

class LindaChecker:
    def __init__(self, lab, file, information):
        self.lab = lab
        self.err_dict = ErrDict()
        self.file = file
        debfn = DebFilenameParser(file)
        self.pkg_name = debfn.package
        self.version = debfn.version
        self.information = information
        if hasattr(self, 'init'):
            getattr(self, 'init')()
    
    def signal_error(self, tag=None, message=[]):
        self.err_dict.add_error(tag, message)

    def is_native(self):
        if self.version.find('-') != -1:
            return 0
        else:
            return 1
    
class ErrDict:
    def __init__(self):
        self.errs = {}

    def add_error(self, key, data=[]):
        if self.errs.has_key(key):
            self.errs[key].append(data)
        else:
            self.errs[key] = [data]

    def __getitem__(self, key): 
        return self.errs[key]

    def __delitem__(self, key): 
        del self.errs[key]

    def keys(self):
        return self.errs.keys()

    def clear(self):
        self.errs.clear()

class Checks:
    def __init__(self):
        self.registry = {}
        self.cur_file = ''

    def startup_register(self):
        self.register_all('%s/checks' % root)
        if not clparser['disable-local']:
            for dir in (os.path.expanduser('~/.linda/checks'), \
                '/etc/linda/checks'):
                if os.path.isdir(dir):
                    self.register_all(dir)
        dprint(_("Registry: %s") % self.registry, 3)
        for type in ('source', 'binary', 'changes', 'udeb'):
            for level in (1, 2):
                if self.registry.has_key(type):
                    if self.registry[type].has_key(level):
                        for klass in self.registry[type][level]:
                            dprint(_("%s: %s") % (klass.__name__, \
                                klass.__file__), 4)
    
    def register(self, klass):
        klass.__file__ = self.cur_file
        for type in ('source', 'binary', 'changes', 'udeb'):
            for level in (1, 2):
                if hasattr(klass, 'check_%s_%s' % (type, level)):
                    self.registry.setdefault(type, {}).setdefault(level, []).\
                        append(klass)
        
    def register_all(self, dir):
        for file in os.listdir(dir):
            file = os.path.join(dir, file)
            if file.endswith('.py'):
                try:
                    self.cur_file = file
                    local_checks = {'_': gettext.gettext, 'dprint': dprint, \
                        'DataValidator': DataValidator, 'os': os}
                    execfile(file, local_checks)
                except StandardError:
                    print _("Failed to import %s.") % file
                    traceback.print_exc(file=sys.stdout)
        
    def apply(self, lab, file, information, level):
        type = self.transliterate(os.path.splitext(file)[1][1:])
        if not self.registry.has_key(type):
            raise CheckRunningException('No checks registered with a type of %s.' % type)
        if not self.registry[type].has_key(level):
            raise CheckRunningException('No checks registered with a type of %s, and a level of %d.' % (type, level))
        errdicts = {}
        for klass in self.registry[type][level]:
            dprint(_("Instantiating check: %s") % klass.__name__)
            instance = klass(lab, file, information)
            method = getattr(instance, 'check_' + type + '_' + repr(level))
            if hasattr(outputobj, 'group_pre_print'):
                outputobj.add_type(type, level)
                outputobj.group_pre_print()
            dprint(_("Calling %s.%s().") % (klass.__name__, method.__name__))
            try:
                method()
            except KeyboardInterrupt:
                print _("Check %s interrupted.") % klass.__name__
                if not clparser['quiet'] and clparser['traceback']:
                    traceback.print_exc(file=sys.stdout)
            except Exception:
                if not clparser['quiet']:
                    excptype, value = sys.exc_info()[:2]
                    print _("Check %s failed. Exception %s thrown (%s).") % \
                        (klass.__name__, excptype.__name__, value)
                    if clparser['traceback']:
                        traceback.print_exc(file=sys.stdout)
            check_dir = os.path.split(os.path.split(instance.__file__)[0])[0]
            tmp_file = instance.__class__.__name__.lower()
            if tmp_file.endswith('check'):
                tmp_file = tmp_file[:-5]
            data_file = os.path.join(check_dir, 'data', '%s.data' % tmp_file)
            errdicts[data_file] = instance.err_dict
        return errdicts
    
    def purge(self):
        checks_to_run = clparser['checks'].replace(' ', '').split(',')
        list_to_cull = []
        for pkg_type in ('binary', 'source'):
            for level in (1, 2):
                for x in self.registry[pkg_type][level]:
                    class_str = x.__name__.lower()
                    not_found = 0
                    for y in checks_to_run:
                        if not re.match(y, class_str, re.IGNORECASE + \
                            re.DOTALL):
                            not_found += 1
                    if not_found == len(checks_to_run):
                        list_to_cull.append(x)
                for x in list_to_cull:
                    self.registry[pkg_type][level].remove(x)
                list_to_cull = []

    def print_checks(self):
        for pkg_type in clparser['list-checks'][0]:
            for level in clparser['list-checks'][1]:
                if self.registry[pkg_type][level]:
                    print "%s checks (level %d): " % (pkg_type.title(), level)
                for x in self.registry[pkg_type][level]:
                    if x.__doc__ is not None:
                        docstr = x.__doc__
                    else:
                        docstr = "No doc string."
                    print " - %s: %s" % (x.__name__, docstr)

    def transliterate(self, type):
        if type in ('deb', 'udeb'):
            return 'binary'
        elif type == 'dsc':
            return 'source'
        elif type == 'changes':
            return 'changes'

class CheckRunningException(Exception):
    pass

