import unittest, gettext, os, linda, tempfile, shutil, gzip, re
from po_count import POFileParser
from linda.collector import Collector
from linda.debug import dprint
from linda.parser.datavalidator import DataValidator
from linda.libchecks import Checks
from sets import Set

linda.checks = Checks()

class ChangelogCheckBinaryTest(unittest.TestCase):
    def setUp(self):
        execfile('checks/changelog.py', {'_': gettext.gettext, 'dprint': \
            dprint, 'DataValidator': DataValidator, 'os': os})
        for klass in linda.checks.registry['binary'][2]:
            self.instance = klass('/tmp', '/tmp/test_0.0.1-1_all.deb', {})
        self.instance.information['dir'] = tempfile.mkdtemp()
        self.po_keys = POFileParser('po/en.po')
        
    def tearDown(self):
        if self.instance.information.has_key('dir'):
            shutil.rmtree(self.instance.information['dir'])

    def test_usr_share_doc_sym(self):
        "Check usr_share_doc_sym() method of ChangelogCheck"
        method = getattr(self.instance, 'usr_share_doc_sym')
        os.makedirs('%s/unpacked/usr/share/doc' % \
            self.instance.information['dir'])
        self.instance.pkg_name = 'lala'
        os.symlink('%s/unpacked/usr/share/doc/bash' % \
            self.instance.information['dir'], \
            '%s/unpacked/usr/share/doc/lala' % \
            self.instance.information['dir'])
        self.instance.information['control'] = {'self': [{'depends': {}}]}
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        os.unlink('%s/unpacked/usr/share/doc/lala' % \
            self.instance.information['dir'])
        os.symlink('%s/unpacked/usr/share/doc/foobar' % \
            self.instance.information['dir'], \
            '%s/unpacked/usr/share/doc/lala' % \
            self.instance.information['dir'])
        method()
        self.assert_('usd-symlink-no-depends' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'usd-symlink-no-depends'), True)
    
    def test_search_changelogs(self):
        "Check search_changelogs() method of ChangelogCheck"
        method = getattr(self.instance, 'search_changelogs')
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/lib/changelog': '', '/usr/share/doc/lala/changelog': ''})
        os.makedirs('%s/unpacked/usr/share/doc/lala' % \
            self.instance.information['dir'])
        os.symlink('/no', os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'share', 'doc', 'lala', 'changelog'))
        method()
        self.assertEqual(self.instance.changelogs.keys(), \
            ['/usr/share/doc/lala/changelog'])
        self.instance.changelogs.clear()
        os.unlink(os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'share', 'doc', 'lala', 'changelog'))
        chg_file = open('%s/unpacked/usr/share/doc/lala/changelog' % \
            self.instance.information['dir'], 'w')
        chg_file.close()
        method()
        self.assertEqual(self.instance.changelogs.keys(), \
            ['/usr/share/doc/lala/changelog'])
        shutil.rmtree('%s/unpacked' % self.instance.information['dir'])

    def test_changelog_checks(self):
        "Check changelog_checks() method of ChangelogCheck"
        method = getattr(self.instance, 'changelog_checks')
        self.instance.usd_path = re.compile(r'/usr/(share/)?doc/kernel-patch-uml')
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        self.instance.changelogs = \
            {'/usr/share/doc/kernel-patch-uml/changelog.gz': {'path': \
            '/usr/share/doc/kernel-patch-uml', 'link': True, 'file': \
            'changelog.gz', 'size': 19919}, \
            '/usr/share/doc/kernel-patch-uml/changelog.Debian.gz': {'path': \
            '/usr/share/doc/kernel-patch-uml', 'link': False, 'file': \
            'changelog.Debian.gz', 'size':  3371}}
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/share/doc/kernel-patch-uml/changelog.gz': \
            "symbolic link to `../../kernel-patch-uml/changelog.gz'"})
        method('/usr/share/doc/kernel-patch-uml/changelog.gz')
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.changelogs = {'/usr/share/doc/lala/changelog': 
            {'size': 5647, 'link': 0, 'path': '/usr/share/doc/lala', 'file': \
            'changelog'}}
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/share/doc/lala/changelog': 'ASCII English text'})
        self.instance.usd_path = re.compile(r'/usr/(share/)?doc/lala')
        method('/usr/share/doc/lala/changelog')
        self.assert_('non-compressed-changelog' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'non-compressed-changelog'), True)
        self.instance.changelogs['/usr/share/doc/lala/changelog.gz'] = \
            {'size': 7657, 'link': 0, 'path': '/usr/share/doc/lala', 'file': \
            'changelog.gz'}
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/share/doc/lala/changelog.gz': 'gzip compressed data'})
        method('/usr/share/doc/lala/changelog.gz')
        self.assert_('not-max-compression-changelog' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'not-max-compression-changelog'), True)
        self.instance.changelogs['/usr/doc/lala/changelog.gz'] = {'size': \
            5876, 'link': 0, 'path': '/usr/doc/lala', 'file': 'changelog.gz'}
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/doc/lala/changelog.gz': 'gzip compressed data, max compression'})
        method('/usr/doc/lala/changelog.gz')
        self.assert_('changelog-usr-doc' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'changelog-usr-doc'), True)
        self.instance.pkg_name = 'lala'
        self.instance.changelogs = \
            {'/usr/share/doc/lala/files/doc/changelog.html': {'size': 85765, \
            'link': 0, 'path': '/usr/share/doc/lala/files/doc', 'file': \
            'changelog.html'}, '/usr/share/doc/lala/changelog.gz': {'size': \
            7558, 'link': 0, 'path': '/usr/share/doc/lala', 'file': \
            'changelog.gz'}}
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/share/doc/lala/files/doc/changelog.html': \
            'HTML document text', '/usr/share/doc/lala/changelog.gz': \
            'gzip compressed data, max compression'})
        method('/usr/share/doc/lala/files/doc/changelog.html')
        self.assert_('html-only-changelog' not in \
            self.instance.err_dict.keys())
        del self.instance.changelogs['/usr/share/doc/lala/changelog.gz']
        method('/usr/share/doc/lala/files/doc/changelog.html')
        self.assert_('html-only-changelog' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'html-only-changelog'), True)
        self.instance.changelogs = {'/usr/share/doc/lala/changelog.gz': \
            {'size': 7558, 'link': 0, 'path': '/usr/share/doc/lala', 'file': \
            'changelog.gz'}}
        self.instance.pkg_name = 'lala'
        method('/usr/share/doc/lala/changelog.gz')
        self.assertEqual(self.instance.info['found_upstream'], 1)
        self.instance.version = '0.0.1'
        self.instance.changelogs = {'/usr/share/doc/lala/changelog.Debian.gz': \
            {'size': 7558, 'link': 0, 'path': '/usr/share/doc/lala', 'file': \
            'changelog.Debian.gz'}}
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/share/doc/lala/changelog.Debian.gz': 'gzip compressed data, max compression'})
        os.makedirs('%s/unpacked/usr/share/doc/lala' % \
            self.instance.information['dir'])
        chg_file = gzip.open('%s/unpacked/usr/share/doc/lala/changelog.Debian.gz' % \
            self.instance.information['dir'], 'w')
        chg_file.close()
        self.instance.changelog = {'file': None, 'name': ''}
        method('/usr/share/doc/lala/changelog.Debian.gz')
        self.assert_('wrong-changelog-name' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'wrong-changelog-name'), True)
        self.instance.err_dict.clear()
        self.instance.changelogs['/usr/share/doc/lala/changelog.gz'] = \
            {'size': 7558, 'link': 0, 'path': '/usr/share/doc/lala', 'file': \
            'changelog.gz'}
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/share/doc/lala/changelog.gz': 'gzip compressed data, max compression', '/usr/share/doc/lala/changelog.Debian.gz': 'gzip compressed data, max compression'})
        chg_file = gzip.open('%s/unpacked/usr/share/doc/lala/changelog.gz' % \
            self.instance.information['dir'], 'w')
        chg_file.close()
        self.instance.changelog = {'file': None, 'name': ''}
        method('/usr/share/doc/lala/changelog.gz')
        method('/usr/share/doc/lala/changelog.Debian.gz')
        self.assertEqual(self.instance.err_dict.keys(), [])
        os.makedirs('%s/unpacked/usr/share/doc/pkg++' % \
            self.instance.information['dir'])
        chg_file = gzip.open('%s/unpacked/usr/share/doc/pkg++/changelog.gz' % \
            self.instance.information['dir'], 'w')
        chg_file.close()
        self.instance.changelogs['/usr/share/doc/pkg++/changelog.gz'] = \
            {'size': 7558, 'link': 0, 'path': '/usr/share/doc/pkg++', 'file': \
            'changelog.gz'}
        self.instance.pkg_name = "pkg++"
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/share/doc/pkg++/changelog.gz': 'gzip compressed data, max compression'})
        self.instance.err_dict.clear()
        method('/usr/share/doc/pkg++/changelog.gz')
        self.assertEqual(self.instance.err_dict.keys(), [])

    def test_specific_changelog_checks(self):
        "Check specific_changelog_checks() method of ChangelogCheck"
        method = getattr(self.instance, 'specific_changelog_checks')
        self.instance.info = {'found_upstream': 0, 'usr_share_doc_sym': 0}
        method()
        self.assert_('no-upstream-changelog' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'no-upstream-changelog'), True)
        self.instance.changelog['file'] = ''
        method()
        self.assert_('no-changelog' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'no-changelog'), True)
        os.makedirs('%s/unpacked/usr/share/doc/lala' % \
            self.instance.information['dir'])
        self.instance.changelog['file'] = gzip.open('%s/unpacked/usr/share/doc/lala/changelog.Debian.gz' % self.instance.information['dir'], 'w')
        self.instance.changelog['file'].write('add-log-mailing-address: stevenk@debian.org')
        self.instance.changelog['file'].close()
        self.instance.changelog['file'] = gzip.open('%s/unpacked/usr/share/doc/lala/changelog.Debian.gz' % self.instance.information['dir'])
        method()
        self.assert_('emacs-settings-in-changelog' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'emacs-settings-in-changelog'), True)

