import unittest, gettext, os, linda, tempfile, shutil
from po_count import POFileParser
from linda.collector import Collector
from linda.debug import dprint
from linda.funcs import run_external_cmd
from linda.parser.datavalidator import DataValidator
from linda.parser.menu import MenuParser
from linda.libchecks import Checks

linda.checks = Checks()

class MenuCheckBinaryTest(unittest.TestCase):
    def setUp(self):
        execfile('checks/menu.py', {'_': gettext.gettext, 'dprint': \
            dprint, 'DataValidator': DataValidator, 'os': os})
        for klass in linda.checks.registry['binary'][2]:
            self.instance = klass('/tmp', '/tmp/test_0.0.1-1_all.deb', {})
        self.instance.information['dir'] = tempfile.mkdtemp()
        run_external_cmd('tar zxCf %s tests/binary_menu.tar.gz' % \
            self.instance.information['dir'])
        self.po_keys = POFileParser('po/en.po')

    def tearDown(self):
        shutil.rmtree(self.instance.information['dir'])

    def test_load_menu_files(self):
        "Check load_menu_files() method of MenuCheck"
        method = getattr(self.instance, 'load_menu_files')
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/lib/menu/xine-ui': ''})
        os.makedirs(os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'lib', 'menu'))
        os.rename(os.path.join(self.instance.information['dir'], 'xine-ui'), \
            os.path.join(self.instance.information['dir'], 'unpacked', 'usr', \
            'lib', 'menu', 'xine-ui'))
        method()
        self.assertEqual(self.instance.menu_files.keys(), \
            ['/usr/lib/menu/xine-ui'])
        os.rename(os.path.join(self.instance.information['dir'], 'unpacked', \
            'usr', 'lib', 'menu', 'xine-ui'), \
            os.path.join(self.instance.information['dir'], 'xine-ui'))

    def test_check_paths(self):
        "Check check_paths() method of MenuCheck"
        method = getattr(self.instance, 'check_paths')
        self.instance.menu_files = {'/usr/share/menu/foobar': ''}
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.menu_files = {'/usr/lib/menu/foobar': ''}
        method()
        self.assert_('menu-files-in-usr-lib-menu' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'menu-files-in-usr-lib-menu'), True)
        
    def test_check_commands(self):
        "Check check_commands() method of MenuCheck"
        method = getattr(self.instance, 'check_commands')
        os.makedirs(os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'bin'))
        f = open(os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'bin', 'xine'), 'w')
        f.close()
        self.instance.menu_files = {'/usr/lib/menu/xine-ui': \
            MenuParser(os.path.join(self.instance.information['dir'], \
            'xine-ui'))}
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        shutil.rmtree(os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'bin'))
        method()
        self.assert_('command-not-exist' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'command-not-exist'), True)
        self.instance.err_dict.clear()
        self.instance.menu_files = {'/usr/lib/menu/paw': \
            MenuParser(os.path.join(self.instance.information['dir'], \
            'paw'))}
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])

    def test_check_section(self):
        "Check check_section() method of MenuCheck"
        method = getattr(self.instance, 'check_section')
        self.instance.menu_files = {'/usr/lib/menu/xine-ui': \
            MenuParser(os.path.join(self.instance.information['dir'], \
            'xine-ui'))}
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.menu_files['/usr/lib/menu/xine-ui'] = \
            MenuParser(os.path.join(self.instance.information['dir'], \
            'xine-ui.ctl'))
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.menu_files['/usr/lib/menu/xine-ui'] = \
            MenuParser(os.path.join(self.instance.information['dir'], \
            'xine-ui.broken'))
        method()
        self.assert_('unknown-menu-subsection' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'unknown-menu-subsection'), True)
        self.instance.menu_files['/usr/lib/menu/xine-ui'] = \
            MenuParser(os.path.join(self.instance.information['dir'], \
            'xine-ui.uk'))
        method()
        self.assert_('unknown-menu-section' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'unknown-menu-section'), True)
        self.instance.menu_files['/usr/lib/menu/xine-ui'] = \
            MenuParser(os.path.join(self.instance.information['dir'], \
            'xine-ui.tl'))
        method()
        self.assert_('top-level-section-bad' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'top-level-section-bad'), True)

    def test_check_quoting(self):
        "Check check_quoting() method of MenuCheck"
        method = getattr(self.instance, 'check_quoting')
        self.instance.menu_files['/usr/lib/menu/xine-ui'] = \
            MenuParser(os.path.join(self.instance.information['dir'], \
            'xine-ui.broken'))
        method()
        self.assertEqual(self.instance.err_dict.keys(), ['command-not-quoted'])
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'command-not-quoted'), True)

