import sys, os, unittest, shutil, tempfile, gettext, linda
from linda import clparser
from linda.funcs import run_external_cmd
from linda.debug import dprint
from linda.libchecks import Checks, CheckRunningException
from linda.parser.datavalidator import DataValidator
from sets import Set

linda.checks = Checks()

class ChecksBaseTest(unittest.TestCase):
    def setUp(self):
        self.dir = tempfile.mkdtemp()
        self.filename = tempfile.mkstemp()[1]
        self.local_checks = {'_': gettext.gettext, 'dprint': dprint, \
            'DataValidator': DataValidator, 'os': os}
        run_external_cmd('tar zxCf %s tests/base_checks.tar.gz' % self.dir)

    def tearDown(self):
        shutil.rmtree(self.dir)
        os.unlink(self.filename)
        
    def test_apply_empty(self):
        "Check that an empty registry raises CheckRunningException"
        linda.checks.registry = {}
        try:
            linda.checks.apply('/tmp', 'lala_0.0.1-1_all.deb', {}, 1)
        except CheckRunningException:
            pass
        else:
            self.fail()
        linda.checks.registry = {'binary': {}}
        try:
            linda.checks.apply('/tmp', 'lala_0.0.1-1_all.deb', {}, 1)
        except CheckRunningException:
            pass
        else:
            self.fail()

    def test_register(self):
        "Check registration of a check"
        linda.checks.registry = {}
        execfile(os.path.join(self.dir, 'test.py'), self.local_checks)
        self.assertEqual(linda.checks.registry.keys(), ['binary'])
        self.assertEqual(linda.checks.registry['binary'].keys(), [1])
        self.assertEqual(len(linda.checks.registry['binary'][1]), 1)

    def test_apply(self):
        "Check apply() method of Checks"
        linda.checks.registry = {}
        cur_dir = os.getcwd()
        execfile(os.path.join(self.dir, 'test.py'), self.local_checks)
        errdict = linda.checks.apply('/tmp', 'lala_0.0.1-1_all.deb', {}, 1)
        os.chdir(cur_dir)
        self.assertEqual(errdict.keys(), ['data/test.data'])
        
    def test_purge(self):
        "Check purge() method of Checks"
        linda.checks.registry = {'binary': {1: [FooCheck], 2: [BarCheck]}, \
            'source': {1: [BazCheck], 2: []}}
        clparser['checks'] = 'Foo, Baz'
        linda.checks.purge()
        types = Set(('binary', 'source'))
        self.assertEqual(types.intersection(linda.checks.registry.keys()), \
            types)
        keys = Set((1, 2))
        for type in types:
            self.assertEqual(keys.intersection(linda.checks.registry[type].keys()), keys)
        self.assertEqual(len(linda.checks.registry['binary'][1]), 1)
        self.assertEqual(len(linda.checks.registry['binary'][2]), 0)
        self.assertEqual(len(linda.checks.registry['source'][1]), 1)
        
    def test_print_checks_empty(self):
        "Check an empty registry causes print_checks() to say nothing"
        linda.checks.registry = {'binary': {1: [], 2: []}, 'source': {1: [], \
            2: []}}
        clparser['list-checks'] = (('binary',), (1,))
        f = open(self.filename, 'w')
        sys.stdout = f
        linda.checks.print_checks()
        f.close()
        sys.stdout = sys.__stdout__
        f = open(self.filename)
        data = []
        for line in f:
            data.append(line)
        self.assertEqual(len(data), 0)
        f.close()

    def test_print_checks(self):
        "Check print_checks() method of Checks"
        linda.checks.registry = {}
        cur_dir = os.getcwd()
        execfile(os.path.join(self.dir, 'test.py'), self.local_checks)
        clparser['list-checks'] = (('binary',), (1,))
        f = open(self.filename, 'w')
        sys.stdout = f
        linda.checks.print_checks()
        f.close()
        sys.stdout = sys.__stdout__
        f = open(self.filename)
        data = []
        for line in f:
            data.append(line[:-1])
        self.assertEqual(data, ['Binary checks (level 1): ', \
            ' - TestCheck: No doc string.'])
        f.close()

class FooCheck:
    def check_binary_1(self):
        pass

class BarCheck:
    def check_binary_2(self):
        pass

class BazCheck:
    def check_source_1(self):
        pass

