import unittest, gettext, os, linda, tempfile, shutil
from linda.collector import Collector
from linda.debug import dprint
from linda.funcs import run_external_cmd
from linda.parser.datavalidator import DataValidator
from linda.libchecks import Checks
from sets import Set

linda.checks = Checks()

class CopyrightCheckBinaryTest(unittest.TestCase):
    def setUp(self):
        execfile('checks/copyright.py', {'_': gettext.gettext, 'dprint': \
            dprint, 'DataValidator': DataValidator, 'os': os})
        for klass in linda.checks.registry['binary'][2]:
            self.instance = klass('/tmp', '/tmp/test_0.0.1-1_all.deb', {})
        self.instance.information['dir'] = tempfile.mkdtemp()
        run_external_cmd('tar zxCf %s tests/binary_copyright.tar.gz' % \
            self.instance.information['dir'])

    def tearDown(self):
        shutil.rmtree(self.instance.information['dir'])

    def test_copyright_exists(self):
        "Check copyright_exists() method of CopyrightCheck"
        method = getattr(self.instance, 'copyright_exists')
        method()
        self.assert_('no-copyright' in self.instance.err_dict.keys())
        os.makedirs(os.path.join(self.instance.information['dir'], \
            'unpacked/usr/share/doc/test'))
        os.symlink('/dev/null', os.path.join(self.instance.information['dir'],\
            'unpacked/usr/share/doc/test/copyright'))
        method()
        self.assert_('copyright-is-symlink' in self.instance.err_dict.keys())

    def test_copyright_location(self):
        "Check copyright_location() method of CopyrightCheck"
        method = getattr(self.instance, 'copyright_location')
        os.makedirs(os.path.join(self.instance.information['dir'], \
            'unpacked/usr/share/doc/copyright'))
        f = open(os.path.join(self.instance.information['dir'], \
            'unpacked/usr/share/doc/copyright/test'), 'w')
        f.write('')
        f.close()
        method()
        self.assert_('old-copyright-location' in self.instance.err_dict.keys())

    def test_compressed(self):
        "Check compressed() method of CopyrightCheck"
        method = getattr(self.instance, 'compressed')
        os.makedirs(os.path.join(self.instance.information['dir'], \
            'unpacked/usr/share/doc/test'))
        f = open(os.path.join(self.instance.information['dir'], \
            'unpacked/usr/share/doc/test', 'copyright'), 'w')
        f.write('')
        f.close()
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/share/doc/test/copyright': ''})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/share/doc/test/copyright': 'gzip compressed data'})
        method()
        self.assert_('compressed-copyright' in self.instance.err_dict.keys())

    def test_parse_copyright(self):
        "Check parse_copyright() method of CopyrightCheck"
        method = getattr(self.instance, 'parse_copyright')
        os.makedirs(os.path.join(self.instance.information['dir'], \
            'unpacked/usr/share/doc/test'))
        shutil.move(os.path.join(self.instance.information['dir'], \
            'copyright.blurb'), \
            os.path.join(self.instance.information['dir'], 'unpacked', \
            'usr/share/doc/test/copyright'))
        method()
        self.assert_('copyright-contains-full-gpl' in \
            self.instance.err_dict.keys())
        os.unlink(os.path.join(self.instance.information['dir'], 'unpacked', \
            'usr/share/doc/test/copyright'))
        shutil.move(os.path.join(self.instance.information['dir'], \
            'copyright.4tests'), \
            os.path.join(self.instance.information['dir'], 'unpacked', \
            'usr/share/doc/test/copyright'))
        method()
        expected = Set(('copyright-blurb-no-name', \
            'probable-dh-make-template', 'copyright-old-directory', \
            'common-licenses', 'upstream-authors'))
        self.assertEqual(expected.intersection(self.instance.err_dict.keys()), expected)
        shutil.rmtree(os.path.join(self.instance.information['dir'], \
            'unpacked'))

