import unittest, gettext, os, linda, tempfile, shutil
from linda.debug import dprint
from linda.funcs import run_external_cmd
from linda.parser.datavalidator import DataValidator
from linda.libchecks import Checks
from linda.parser.rfc822parser import RFC822Parser
from sets import Set

linda.checks = Checks()

class DebconfCheckBinaryTest(unittest.TestCase):
    def setUp(self):
        execfile('checks/debconf.py', {'_': gettext.gettext, 'dprint': \
            dprint, 'DataValidator': DataValidator, 'os': os})
        for klass in linda.checks.registry['binary'][1]:
            self.instance = klass('/tmp', '/tmp/test_0.0.1-1_all.deb', {})
        self.instance.information['dir'] = tempfile.mkdtemp()
        os.mkdir('%s/control' % self.instance.information['dir'])
        run_external_cmd('tar zxCf %s tests/binary_debconf.tar.gz' % \
            self.instance.information['dir'])

    def tearDown(self):
        shutil.rmtree(self.instance.information['dir'])

    def test_templates(self):
        "Check templates() method of DebconfCheck"
        method = getattr(self.instance, 'templates')
        f = open('%s/control/templates' % self.instance.information['dir'], 'w')
        f.close()
        f = open('%s/control/config' % self.instance.information['dir'], 'w')
        f.close()
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        os.unlink('%s/control/templates' % self.instance.information['dir'])
        method()
        self.assert_('config-no-templates' in self.instance.err_dict.keys())

    def test_check_template(self):
        "Check check_template() method of DebconfCheck"
        method = getattr(self.instance, 'check_template')
        template = RFC822Parser('/dev/null')
        template.data = [{'a': ''}]
        template.empty_stanzas = 0
        self.instance.parsed_templ = template
        method()
        expect_errs = Set(('debconf-no-type', 'debconf-no-template', \
            'unknown-field-templates'))
        self.assertEqual(expect_errs.intersection(self.instance.err_dict.keys()), expect_errs)
        self.instance.parsed_templ.data[0] = {'type': 'lala'}
        method()
        self.assert_('unknown-debconf-type' in self.instance.err_dict.keys())
        self.instance.parsed_templ.data[0]['type'] = 'select'
        method()
        self.assert_('debconf-select-without-choices' in \
            self.instance.err_dict.keys())
        for key in self.instance.err_dict.keys():
            del self.instance.err_dict[key]
        self.instance.parsed_templ.data[0]['type'] = 'multiselect'
        self.instance.parsed_templ.data[0]['template'] = 'a'
        self.instance.parsed_templ.data[0]['choices'] = 'foo'
        self.instance.information['control'] = {'self': [{'depends': \
            {'debconf': {'relation': [['>>', '0.5']], 'arch': [None]}}}]}
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['control']['self'][0]['depends']['debconf']['relation'][0][1] = '0.1.7'
        method()
        self.assert_('debconf-insufficient-version' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.instance.err_dict['debconf-insufficient-version'][0], ['multiselect', '0.2.26'])
        for key in self.instance.err_dict.keys():
            del self.instance.err_dict[key]
        self.instance.information['control'] = {'self': [{'depends': \
            {'debconf': {'relation': [['>>', '1.3.22']], 'arch': [None]}}}]}
        self.instance.parsed_templ.data[0]['title'] = 'stuff'
        self.instance.parsed_templ.data[0]['type'] = 'string'
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['control']['self'][0]['depends']['debconf']['relation'][0][1] = '0.1.7'
        method()
        self.assert_('debconf-insufficient-version' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.instance.err_dict['debconf-insufficient-version'][0], ['title', '1.3.22'])
        os.rename('%s/templates' % self.instance.information['dir'], \
            '%s/control/templates' % self.instance.information['dir'])
        getattr(self.instance, 'templates')()
        method()
        self.assert_('possible-empty-stanzas-templates' in \
            self.instance.err_dict.keys())
        os.unlink('%s/control/templates' % self.instance.information['dir'])
        
    def test_maint_scripts_checks(self):
        "Check maint_scripts_checks() method of DebconfCheck"
        method = getattr(self.instance, 'maint_scripts_checks')
        self.instance.information['control'] = {'self': [{'depends': \
            {'debconf': {'relation': ['>>', '0.1.5'], 'arch': ''}}}]}
        os.rename('%s/postinst' % self.instance.information['dir'], \
            '%s/control/postinst' % self.instance.information['dir'])
        method()
        expect_errs = Set(('postinst-db-input', \
            'debconf-isdefault-deprecated', 'seen-not-recent', \
            'obsolete-confmodule'))
        self.assertEqual(expect_errs.intersection(self.instance.err_dict.keys()), expect_errs)
        os.unlink('%s/control/postinst' % self.instance.information['dir'])
        os.rename('%s/postrm' % self.instance.information['dir'], \
            '%s/control/postrm' % self.instance.information['dir'])
        method()
        self.assertEqual(self.instance.maint_scripts['postrm']['purge'], 1)
        self.assertEqual(self.instance.maint_scripts['postrm']['confmodule'], 1)
        os.unlink('%s/control/postrm' % self.instance.information['dir'])
        for key in self.instance.err_dict.keys():
            del self.instance.err_dict[key]
        f = open('%s/control/postrm' % self.instance.information['dir'], 'w')
        f.close()
        self.instance.maint_scripts['postrm']['purge'] = 0
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        f = open('%s/control/templates' % self.instance.information['dir'], 'w')
        f.close()
        method()
        self.assert_('debconf-postrm-fails-purge' in \
            self.instance.err_dict.keys())
        os.unlink('%s/control/templates' % self.instance.information['dir'])
        f = open('%s/control/config' % self.instance.information['dir'], 'w')
        f.close()
        method()
        self.assert_('config-not-load-confmodule' in \
            self.instance.err_dict.keys())
        os.unlink('%s/control/config' % self.instance.information['dir'])
        self.instance.information['control']['self'][0]['pre-depends'] = {}
        self.instance.maint_scripts['preinst']['confmodule'] = 1
        f = open('%s/control/preinst' % self.instance.information['dir'], 'w')
        f.close()
        method()
        self.assert_('missing-pre-depends-debconf' in \
            self.instance.err_dict.keys())
        os.unlink('%s/control/preinst' % self.instance.information['dir'])
        for key in self.instance.err_dict.keys():
            del self.instance.err_dict[key]
        os.rename('%s/postinst.non-versioned' % self.instance.information['dir'], \
            '%s/control/postinst' % self.instance.information['dir'])
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        os.unlink('%s/control/postinst' % self.instance.information['dir'])
        os.rename('%s/postinst.versioned' % self.instance.information['dir'], \
            '%s/control/postinst' % self.instance.information['dir'])
        method()
        self.assert_('init-script-suggests-versioned-depends' in \
            self.instance.err_dict.keys())

