
/*
 * Linux device driver for ADMtek ADM8211 (IEEE 802.11b MAC/BBP)
 *
 * Copyright (c) 2003, Jouni Malinen <jkmaline@cc.hut.fi>
 * Copyright (c) 2004-2005, Michael Wu <flamingice@sourmilk.net>
 * Some parts copyright (c) 2003 by David Young <dyoung@pobox.com>
 * and used with permission.
 *
 * Much thanks to Infineon-ADMtek for their support of this driver.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. See README and COPYING for
 * more details.
 */

/*
 * Note!
 * - this is a pre-release of the driver; it is not yet finished and needs
 *   more testing.
 *
 * FIX:
 * - properly idle chip in parts that need it (goes w/ above)
 * - improve signal quality + RSSI stuff
 * - figure out why ping -s 5913 and up lose packets..
 *   (can be reduced with a mtu of 920..)
 * - check FCS for rev 0x20 and up?
 */


#include <linux/config.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/if.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/ioport.h>
#include <linux/pci.h>
#include <linux/if_arp.h>
#include <linux/delay.h>
#include <linux/crc32.h>
#include <linux/wireless.h>
#include <linux/dma-mapping.h>
#include <net/iw_handler.h>
#include <asm/delay.h>
#include <asm/unaligned.h>
#include <asm/types.h>

#include "adm8211.h"
#include "adm8211_ioctl.h"
#include "ieee80211.h"
#include "avs_caphdr.h"

#include "rdate.h"

MODULE_AUTHOR("Jouni Malinen <jkmaline@cc.hut.fi>, Michael Wu <flamingice@sourmilk.net>");
MODULE_DESCRIPTION("Driver for IEEE 802.11b wireless LAN cards based on ADMtek"
		   " ADM8211");
MODULE_SUPPORTED_DEVICE("ADM8211");
MODULE_LICENSE("GPL");


static const char *version = KERN_INFO "adm8211: Copyright 2003, "
"Jouni Malinen <jkmaline@cc.hut.fi>; "
"Copyright 2004-2005, Michael Wu <flamingice@sourmilk.net>\n";


static struct pci_device_id adm8211_pci_id_table[] __devinitdata = {
	/* ADMtek ADM8211 */
	{ 0x10B7, 0x6000, PCI_ANY_ID, PCI_ANY_ID }, /* 3Com 3CRSHPW796 */
	{ 0x1200, 0x8201, PCI_ANY_ID, PCI_ANY_ID }, /* ? */
	{ 0x1317, 0x8201, PCI_ANY_ID, PCI_ANY_ID }, /* ADM8211A */
	{ 0x1317, 0x8211, PCI_ANY_ID, PCI_ANY_ID }, /* ADM8211B/C */
	{ 0 }
};


#define ADM8211_INTMASK \
(ADM8211_IER_NIE | ADM8211_IER_AIE | ADM8211_IER_RCIE | ADM8211_IER_TCIE | \
ADM8211_IER_TDUIE | ADM8211_IER_GPTIE)


struct adm8211_desc {
	__le32 status;
	__le32 length;
	__le32 buffer1;
	__le32 buffer2;
};

#define RDES0_STATUS_OWN	(1 << 31)
#define RDES0_STATUS_ES		(1 << 30)
#define RDES0_STATUS_SQL	(1 << 29)
#define RDES0_STATUS_DE		(1 << 28)
#define RDES0_STATUS_FS		(1 << 27)
#define RDES0_STATUS_LS		(1 << 26)
#define RDES0_STATUS_PCF	(1 << 25)
#define RDES0_STATUS_SFDE	(1 << 24)
#define RDES0_STATUS_SIGE	(1 << 23)
#define RDES0_STATUS_CRC16E	(1 << 22)
#define RDES0_STATUS_RXTOE	(1 << 21)
#define RDES0_STATUS_CRC32E	(1 << 20)
#define RDES0_STATUS_ICVE	(1 << 19)
#define RDES0_STATUS_DA1	(1 << 17)
#define RDES0_STATUS_DA0	(1 << 16)
#define RDES0_STATUS_RXDR	((1 << 15) | (1 << 14) | (1 << 13) | (1 << 12))
#define RDES0_STATUS_FL		(0x00000fff)

#define RDES1_CONTROL_RER	(1 << 25)
#define RDES1_CONTROL_RCH	(1 << 24)
#define RDES1_CONTROL_RBS2	(0x00fff000)
#define RDES1_CONTROL_RBS1	(0x00000fff)

#define RDES1_STATUS_RSSI	(0x0000007f)


#define TDES0_CONTROL_OWN	(1 << 31)
#define TDES0_CONTROL_DONE	(1 << 30)
#define TDES0_CONTROL_TXDR	(0x0ff00000)

#define TDES0_STATUS_OWN	(1 << 31)
#define TDES0_STATUS_DONE	(1 << 30)
#define TDES0_STATUS_ES		(1 << 29)
#define TDES0_STATUS_TLT	(1 << 28)
#define TDES0_STATUS_TRT	(1 << 27)
#define TDES0_STATUS_TUF	(1 << 26)
#define TDES0_STATUS_TRO	(1 << 25)
#define TDES0_STATUS_SOFBR	(1 << 24)
#define TDES0_STATUS_ACR	(0x00000fff)

#define TDES1_CONTROL_IC	(1 << 31)
#define TDES1_CONTROL_LS	(1 << 30)
#define TDES1_CONTROL_FS	(1 << 29)
#define TDES1_CONTROL_TER	(1 << 25)
#define TDES1_CONTROL_TCH	(1 << 24)
#define TDES1_CONTROL_RBS2	(0x00fff000)
#define TDES1_CONTROL_RBS1	(0x00000fff)


#define PLCP_SIGNAL_1M		0x0a
#define PLCP_SIGNAL_2M		0x14
#define PLCP_SIGNAL_5M5		0x37
#define PLCP_SIGNAL_11M		0x6e


/* RX status - stored in skb->cb so this structure must be 48 bytes or less */
struct adm8211_rx_status {
	u8 rssi;
	u8 rate;
};


struct adm8211_tx_hdr {
	u8 da[6];
	u8 signal; /* PLCP signal / TX rate in 100 Kbps */
	u8 service;
	__le16 frame_body_size;
	__le16 frame_control;
	__le16 plcp_frag_tail_len;
	__le16 plcp_frag_head_len;
	__le16 dur_frag_tail;
	__le16 dur_frag_head;
	u8 address4[6];

#define ADM8211_TXHDRCTL_SHORT_PREAMBLE		(1 <<  0)
#define ADM8211_TXHDRCTL_MORE_FRAG		(1 <<  1)
#define ADM8211_TXHDRCTL_MORE_DATA		(1 <<  2)
#define ADM8211_TXHDRCTL_FRAG_NO		(1 <<  3) /* ? */
#define ADM8211_TXHDRCTL_ENABLE_RTS		(1 <<  4)
#define ADM8211_TXHDRCTL_ENABLE_WEP_ENGINE	(1 <<  5)
#define ADM8211_TXHDRCTL_ENABLE_EXTEND_HEADER	(1 << 15) /* ? */
	__le16 header_control;
#ifdef BIG_ENDIAN
	u32 retry_limit:8;
	u32 reserved_0:8;
	u32 frag_number:4;
	u32 frag_threshold:12;
#else
	u32 frag_threshold:12;
	u32 frag_number:4;
	u32 reserved_0:8;
	u32 retry_limit:8;
#endif

	u32 wep2key0;
	u32 wep2key1;
	u32 wep2key2;
	u32 wep2key3;

	u8 keyid;
	u8 reserved_1[3];
	u32 reserved_2;
} __attribute__ ((packed));



#define RX_COPY_BREAK 128

#define RX_PKT_SIZE 2400

/* SRAM offsets */
#define ADM8211_SRAM(x) (priv->revid < ADM8211_REV_BA ? \
        ADM8211_SRAM_A_ ## x : ADM8211_SRAM_B_ ## x)

#define ADM8211_SRAM_INDIV_KEY   0x0000
#define ADM8211_SRAM_A_SHARE_KEY 0x0160
#define ADM8211_SRAM_B_SHARE_KEY 0x00c0

#define ADM8211_SRAM_A_SSID      0x0180
#define ADM8211_SRAM_B_SSID      0x00d4
#define ADM8211_SRAM_SSID ADM8211_SRAM(SSID)

#define ADM8211_SRAM_A_SUPP_RATE 0x0191
#define ADM8211_SRAM_B_SUPP_RATE 0x00dd
#define ADM8211_SRAM_SUPP_RATE ADM8211_SRAM(SUPP_RATE)

#define ADM8211_SRAM_A_SIZE      0x0200
#define ADM8211_SRAM_B_SIZE      0x01c0
#define ADM8211_SRAM_SIZE ADM8211_SRAM(SIZE)

/* Serial EEPROM reading for 93C66/93C46 */
#define EE_ENB		(0x4000 | ADM8211_SPR_SRS | ADM8211_SPR_SCS)
#define EE_READ_CMD	(6)
#define eeprom_delay()	ADM8211_CSR_READ(SPR);


static u16 adm8211_eeprom_read_word(struct net_device *dev, int addr,
				    int addr_len)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	int i;
	u16 retval = 0;
	int read_cmd = addr | (EE_READ_CMD << addr_len);

	ADM8211_CSR_WRITE(SPR, EE_ENB & ~ADM8211_SPR_SCS);
	eeprom_delay();
	ADM8211_CSR_WRITE(SPR, EE_ENB);
	eeprom_delay();

	/* Shift the read command bits out. */
	for (i = 4 + addr_len; i >= 0; i--) {
		short dataval = (read_cmd & (1 << i)) ? ADM8211_SPR_SDI : 0;
		ADM8211_CSR_WRITE(SPR, EE_ENB | dataval);
		eeprom_delay();
		ADM8211_CSR_WRITE(SPR, EE_ENB | dataval | ADM8211_SPR_SCLK);
		eeprom_delay();
	}

	ADM8211_CSR_WRITE(SPR, EE_ENB);
	eeprom_delay();

	for (i = 16; i > 0; i--) {
		ADM8211_CSR_WRITE(SPR, EE_ENB | ADM8211_SPR_SCLK);
		eeprom_delay();
		retval <<= 1;
		retval |= ((ADM8211_CSR_READ(SPR) & ADM8211_SPR_SDO) ? 1 : 0);
		ADM8211_CSR_WRITE(SPR, EE_ENB);
		eeprom_delay();
	}

	/* Terminate the EEPROM access. */
	ADM8211_CSR_WRITE(SPR, EE_ENB & ~ADM8211_SPR_SCS);

	return retval;
}


static int adm8211_read_eeprom(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	u32 reg;
	int addr_len, words, i;

	reg = ADM8211_CSR_READ(CSR_TEST0);

	if (reg & ADM8211_CSR_TEST0_EPTYP) {
		printk(KERN_DEBUG "%s (adm8211): EEPROM type: 93C66\n", pci_name(priv->pdev));
		/* 256 * 16-bit = 512 bytes */
		addr_len = 8;
		words = 256;
	} else {
		printk(KERN_DEBUG "%s (adm8211): EEPROM type 93C46\n", pci_name(priv->pdev));
		/* 64 * 16-bit = 128 bytes */
		addr_len = 6;
		words = 64;
	}

	priv->eeprom_len = words * 2;
	priv->eeprom = kmalloc(priv->eeprom_len, GFP_KERNEL);
	if (priv->eeprom == NULL)
		return -ENOMEM;

	for (i = 0; i < words; i++)
		*((u16 *) &((u8 *)priv->eeprom)[i * 2]) =
			adm8211_eeprom_read_word(dev, i, addr_len);

	priv->rf_type = (le16_to_cpu(priv->eeprom->cr49) & (0x7 << 3)) >> 3;
	priv->bbp_type = le16_to_cpu(priv->eeprom->cr49) &  0x7;

	/* read country code from eeprom and make sure it's valid */
	if (priv->eeprom->country_code >= ARRAY_SIZE(cranges)) {
		printk(KERN_WARNING "%s (adm8211): Invalid country code (%d) in EEPROM, assuming ETSI\n",
		       pci_name(priv->pdev), priv->eeprom->country_code);

		priv->ieee80211.chan_range = cranges[2];
	} else
		priv->ieee80211.chan_range = cranges[priv->eeprom->country_code];

	printk(KERN_DEBUG "%s (adm8211): Channel range: %d - %d\n",
	       pci_name(priv->pdev), (int)priv->ieee80211.chan_range.min, (int)priv->ieee80211.chan_range.max);

	switch (priv->eeprom->specific_rftype) {
	case ADM8211_RFMD2948:
	case ADM8211_RFMD2958:
	case ADM8211_RFMD2958_RF3000_CONTROL_POWER:
	case ADM8211_MAX2820:
	case ADM8211_AL2210L:
		priv->transceiver_type = priv->eeprom->specific_rftype;
		break;

	default:
		if (priv->revid == ADM8211_REV_BA)
			priv->transceiver_type = ADM8211_RFMD2958_RF3000_CONTROL_POWER;
		else if (priv->revid == ADM8211_REV_CA)
			priv->transceiver_type = ADM8211_AL2210L;
		else if (priv->revid == ADM8211_REV_AB)
			priv->transceiver_type = ADM8211_RFMD2948;

		printk(KERN_WARNING "%s (adm8211): Invalid or unsupported transceiver: %d, assuming %d\n",
		       pci_name(priv->pdev), priv->eeprom->specific_rftype, priv->transceiver_type);

		break;
	}

	printk(KERN_DEBUG "%s (adm8211): RFtype=%d BBPtype=%d Specific BBP=%d Transceiver=%d\n",
	       pci_name(priv->pdev), priv->rf_type, priv->bbp_type,
	       priv->eeprom->specific_bbptype, priv->eeprom->specific_rftype);

	return 0;
}

static inline void adm8211_write_sram(struct net_device *dev, u32 addr, __le32 data)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	ADM8211_CSR_WRITE(WEPCTL, cpu_to_le32(addr | ADM8211_WEPCTL_TABLE_WR |
			  (priv->revid < ADM8211_REV_BA ?
			   0 : ADM8211_WEPCTL_SEL_WEPTABLE )) );
	ADM8211_CSR_READ(WEPCTL);
	mdelay(1);

	ADM8211_CSR_WRITE(WESK, data);
	ADM8211_CSR_READ(WESK);
	mdelay(1);
}

static void adm8211_write_sram_bytes(struct net_device *dev, int addr, u8 *buf, int len)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	u32 reg = ADM8211_CSR_READ(WEPCTL);
	int i;

	if (priv->revid < ADM8211_REV_BA) {
		for (i = 0; i < len; i += 2) {
			u16 val = buf[i] | buf[i + 1] << 8;
			adm8211_write_sram(dev, addr + i / 2, cpu_to_le32(val));
		}
	} else {
		for (i = 0; i < len; i += 4) {
			u32 val = (buf[i + 0] << 0 ) | (buf[i + 1] << 8 ) |
				  (buf[i + 2] << 16) | (buf[i + 3] << 24);
			adm8211_write_sram(dev, addr + i / 4, cpu_to_le32(val));
		}
	}

	ADM8211_CSR_WRITE(WEPCTL, reg);
}

static void adm8211_clear_sram(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	u32 reg = ADM8211_CSR_READ(WEPCTL);
	int addr;

	for (addr = 0; addr < ADM8211_SRAM_SIZE; addr++)
			adm8211_write_sram(dev, addr, 0);

	ADM8211_CSR_WRITE(WEPCTL, reg);
}

static void adm8211_link_change(struct net_device *dev, int link)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	u32 reg;

	/* Set aid for beacon TIM element decoding */
	reg = ADM8211_CSR_READ(FRCTL);
	reg &= ~ADM8211_FRCTL_AID;
	if (link) {
		reg |= ADM8211_FRCTL_AID_ON;
		reg |= priv->ieee80211.aid;
	}
	ADM8211_CSR_WRITE(FRCTL, reg);
}

static struct net_device_stats *adm8211_get_stats(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	return &priv->stats;
}

static struct iw_statistics *adm8211_get_wireless_stats(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	return &priv->wstats;
}

static void adm8211_set_rx_mode(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	int i, bit_nr;
	__le32 mc_filter[2];
	struct dev_mc_list *mclist;
	unsigned long flags;

	if (dev->flags & IFF_PROMISC) {
		priv->nar |= ADM8211_NAR_PR;
		priv->nar &= ~ADM8211_NAR_MM;
		mc_filter[1] = mc_filter[0] = __constant_cpu_to_le32(~0);
	} else if ((dev->flags & IFF_ALLMULTI) || (dev->mc_count > 32)) {
		priv->nar &= ~ADM8211_NAR_PR;
		priv->nar |= ADM8211_NAR_MM;
		mc_filter[1] = mc_filter[0] = __constant_cpu_to_le32(~0);
	} else {
		priv->nar &= ~(ADM8211_NAR_MM | ADM8211_NAR_PR);
		mc_filter[1] = mc_filter[0] = 0;
		mclist = dev->mc_list;
		for (i = 0; i < dev->mc_count; i+=1) {
			if (!mclist)
				break;
			bit_nr = ether_crc(ETH_ALEN, mclist->dmi_addr) >> 26;

			bit_nr &= 0x3F;
                        mc_filter[bit_nr >> 5] |= cpu_to_le32(1 << (bit_nr & 31));
			mclist = mclist->next;
		}
	}

	/* TODO: is this locking necessary? I don't think so.. */
	spin_lock_irqsave(&priv->lock, flags);

	/* make sure receive is stopped */
	if (priv->nar & ADM8211_NAR_SR) {
	ADM8211_CSR_WRITE(NAR, priv->nar & ~ADM8211_NAR_SR);
	ADM8211_CSR_READ(NAR);
	mdelay(20);
	}

	ADM8211_CSR_WRITE(MAR0, mc_filter[0]);
	ADM8211_CSR_WRITE(MAR1, mc_filter[1]);
	ADM8211_CSR_READ(NAR);
	ADM8211_CSR_WRITE(NAR, priv->nar);

	spin_unlock_irqrestore(&priv->lock, flags);
}

static int adm8211_set_mac_address(struct net_device *dev, void *p)
{
	struct adm8211_priv *priv = netdev_priv(dev);
        struct sockaddr *addr = p;
	u32 reg;

        memcpy(dev->dev_addr, addr->sa_data, dev->addr_len);

	if (!(dev->flags & IFF_UP))
		return 0;

	/* sure, we can change the MAC addr while running */
	reg = ADM8211_CSR_READ(NAR);

	/* make sure receive is stopped */
	if (priv->nar & ADM8211_NAR_SR) {
	ADM8211_CSR_WRITE(NAR, priv->nar & ~ADM8211_NAR_SR);
	mdelay(20);
	}

	ADM8211_CSR_WRITE(PAR0, *(u32 *)dev->dev_addr);
	ADM8211_CSR_WRITE(PAR1, *(u16 *)(dev->dev_addr + 4));
	ADM8211_CSR_WRITE(NAR, priv->nar);

	return 0;
}

static void adm8211_rx_skb(struct net_device *dev, struct sk_buff *skb)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;
	struct ieee80211_hdr *hdr;
	struct adm8211_rx_status *stat;
	struct iw_statistics *wstats = &priv->wstats;
	struct ieee80211_rx_status rx_status;
	static const u8 rate[] = {10, 20, 55, 110, 220};
	u16 fc;

	if (skb->len < 14)
		goto drop;

	/* TODO: Add crc checking here for cards/modes that need it */

	stat = (struct adm8211_rx_status *) skb->cb;
	memset(&rx_status, 0, sizeof(rx_status));
	if (priv->revid < ADM8211_REV_CA)
		rx_status.rssi = stat->rssi;
	else
		rx_status.rssi = 100 - stat->rssi;

	if (stat->rate <= 4)
		rx_status.rate = rate[stat->rate];

	hdr = (struct ieee80211_hdr *) skb->data;
	fc = le16_to_cpu(hdr->frame_control);

	if (priv->iw_mode == IW_MODE_MONITOR) {
		struct avs_caphdr *chdr;
		if (skb_headroom(skb) < sizeof(struct avs_caphdr)) {
			if (pskb_expand_head(skb, sizeof(struct avs_caphdr), 0,
					     GFP_ATOMIC)) {
				printk(KERN_DEBUG "%s: failed to allocate room for prism2 "
				       "header\n", dev->name);
				goto drop;
			}
		}
		memset(skb->cb, 0, sizeof(skb->cb));

		chdr = (struct avs_caphdr *) skb_push(skb, sizeof(struct avs_caphdr));
		chdr->version = cpu_to_be32(0x80211001);
		chdr->length = cpu_to_be32(sizeof(struct avs_caphdr));
		chdr->mactime = 0;
		chdr->hosttime = cpu_to_be64(jiffies);
		chdr->phytype = cpu_to_be32(4); /* phytype_dsss_dot11_b */
		chdr->channel = cpu_to_be32(data->channel);
		chdr->datarate = cpu_to_be32(rx_status.rate);
		chdr->antenna = 0; /* TODO: once antenna setting is possible.. */
		chdr->priority = 0; /* hmm, dunno if this is possible.. */
		chdr->ssi_type = cpu_to_be32(3); /* Raw RSSI */
		chdr->ssi_signal = cpu_to_be32(rx_status.rssi);
		chdr->ssi_noise = cpu_to_be32(0xFFFFFFFF);
		if (skb->len >= 14 + sizeof(struct avs_caphdr))
			chdr->preamble = cpu_to_be32(
					 le16_to_cpu(hdr->frame_control) & WLAN_CAPABILITY_SHORT_PREAMBLE
					 ? 2 : 1);
		else
			chdr->preamble = 0;
		chdr->encoding = cpu_to_be32(1); /* CCK */

		priv->stats.rx_bytes += skb->len;
		priv->stats.rx_packets += 1;

		skb->pkt_type = PACKET_OTHERHOST;
		skb->mac.raw = skb->data;

		netif_rx(skb);
		dev->last_rx = jiffies;
		return;
	}

	if (ieee80211_filter_duplicates(&priv->ieee80211, skb, fc))
		goto drop;

	/* remove FCS */
	if (dev->flags & IFF_PROMISC)
		skb_trim(skb, skb->len - FCS_LEN);

	/* Promiscuous mode disables hardware WEP RX decryption */
	if (priv->revid < ADM8211_REV_BA || dev->flags & IFF_PROMISC) {
		skb = ieee80211_wep_decode(&priv->ieee80211, skb);
		if (!skb) {
			priv->wstats.discard.code += 1;
			priv->stats.rx_errors += 1;
			return;
		}
	}

	skb = ieee80211_reassemble_frag(&priv->ieee80211, skb);
	if (!skb)
		return;

	/* FIX: this is a hack */
	wstats->qual.qual = stat->rssi;
	wstats->qual.updated = 1;

	switch (WLAN_FC_GET_TYPE(fc)) {
	case WLAN_FC_TYPE_MGMT:
		ieee80211_rx_mgmt(&priv->ieee80211, skb, &rx_status);
		break;
	case WLAN_FC_TYPE_DATA:
		skb = ieee80211_data_decaps(&priv->ieee80211, skb);
		if (skb) {
			skb->protocol = eth_type_trans(skb, dev);
			memset(skb->cb, 0, sizeof(skb->cb));
			priv->stats.rx_bytes+=skb->len;
			priv->stats.rx_packets+=1;
			netif_rx(skb);
			dev->last_rx = jiffies;
		}
		break;
	case WLAN_FC_TYPE_CTRL:
		goto drop;
	}

	return;

 drop:
	dev_kfree_skb(skb);
}


static void adm8211_rx_tasklet(unsigned long data)
{
	struct net_device *dev = (struct net_device *) data;
	struct adm8211_priv *priv = netdev_priv(dev);
	struct sk_buff *skb;

	while ((skb = skb_dequeue(&priv->rx_queue)) != NULL)
		adm8211_rx_skb(dev, skb);
}


static void adm8211_interrupt_tci(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	unsigned dirty_tx;

#if 0
	printk(KERN_DEBUG "TCI (dirty_tx=%d cur_tx=%d)\n",
	       priv->dirty_tx, priv->cur_tx);
#endif

	spin_lock(&priv->lock);

	for (dirty_tx = priv->dirty_tx;
	     priv->cur_tx - dirty_tx > 0; dirty_tx++) {
		unsigned entry = dirty_tx % TX_RING_SIZE;
		u32 status = le32_to_cpu(priv->tx_ring[entry].status);
		if (status & TDES0_CONTROL_OWN ||
		    !(status & TDES0_CONTROL_DONE))
			break;

		if (status & TDES0_STATUS_ES) {
			priv->stats.tx_errors += 1;

			if (status & (TDES0_STATUS_TUF | TDES0_STATUS_TRO))
				priv->stats.tx_fifo_errors += 1;
			if (status & (TDES0_STATUS_TLT | TDES0_STATUS_SOFBR))
				priv->wstats.discard.misc += 1;
			if (status & TDES0_STATUS_TRT)
				priv->wstats.discard.retries += 1;
		} else {
			priv->stats.tx_bytes += priv->tx_buffers[entry].skb->len;
			priv->stats.tx_packets += 1;
		}

		pci_unmap_single(priv->pdev, priv->tx_buffers[entry].mapping,
				 priv->tx_buffers[entry].skb->len,
				 PCI_DMA_TODEVICE);
		dev_kfree_skb_irq(priv->tx_buffers[entry].skb);
		priv->tx_buffers[entry].skb = NULL;
	}

	if (priv->cur_tx - dirty_tx < TX_RING_SIZE - 2)
		netif_wake_queue(dev);

	priv->dirty_tx = dirty_tx;
	spin_unlock(&priv->lock);
}


static void adm8211_interrupt_rci(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	int entry = priv->cur_rx % RX_RING_SIZE;
	u32 status;
	unsigned pktlen;
	struct sk_buff *skb, *newskb;
	int limit = RX_RING_SIZE;
	u8 rssi, rate;

#if 0
	printk(KERN_DEBUG "RCI\n");
#endif

	while (!(priv->rx_ring[entry].status &
		 __constant_cpu_to_le32(RDES0_STATUS_OWN))) {
		if (--limit < 0)
			break;

		status = le32_to_cpu(priv->rx_ring[entry].status);
		rate = (status & RDES0_STATUS_RXDR) >> 12;
		rssi = le32_to_cpu(priv->rx_ring[entry].length) &
			RDES1_STATUS_RSSI;

#if 0
		printk(KERN_DEBUG "%s: RX %02x RXDR=%d FL=%d RSSI=%d "
		       "%s%s%s%s\n",
		       dev->name, status, rate,
		       status & RDES0_STATUS_FL,
		       rssi,
		       status & RDES0_STATUS_ES ? "[ES]" : "",
		       status & RDES0_STATUS_FS ? "[FS]" : "",
		       status & RDES0_STATUS_LS ? "[LS]" : "",
		       status & RDES0_STATUS_CRC16E ? "[CRC16E]" : "");
#endif

		pktlen = status & RDES0_STATUS_FL;
		if (pktlen > RX_PKT_SIZE) {
			printk(KERN_DEBUG "%s: too long frame (pktlen=%d)\n",
			       dev->name, pktlen);
			pktlen = RX_PKT_SIZE;
		}

		if (!priv->soft_rx_crc && status & RDES0_STATUS_ES) {
#if 0
			printk(KERN_DEBUG "%s: dropped RX frame with error "
			       "(status=0x%x)\n", dev->name, status);
#endif
			skb = NULL; /* old buffer will be reused */
			priv->stats.rx_errors += 1;
			if (status & (RDES0_STATUS_SFDE |
				      RDES0_STATUS_SIGE | RDES0_STATUS_RXTOE))
				priv->wstats.discard.misc += 1;
			if (status & RDES0_STATUS_ICVE)
				priv->wstats.discard.code += 1;
			if (status & (RDES0_STATUS_CRC16E | RDES0_STATUS_CRC32E))
				priv->stats.rx_crc_errors += 1;

		} else if (pktlen < RX_COPY_BREAK) {
			skb = dev_alloc_skb(pktlen);
			if (skb) {
				skb->dev = dev;
				pci_dma_sync_single_for_cpu(
					priv->pdev,
					priv->rx_buffers[entry].mapping,
					pktlen, PCI_DMA_FROMDEVICE);
				memcpy(skb_put(skb, pktlen),
				       priv->rx_buffers[entry].skb->tail,
				       pktlen);
				pci_dma_sync_single_for_device(
					priv->pdev,
					priv->rx_buffers[entry].mapping,
					RX_PKT_SIZE, PCI_DMA_FROMDEVICE);
			}
		} else {
			newskb = dev_alloc_skb(RX_PKT_SIZE);
			if (newskb) {
				newskb->dev = dev;
				skb = priv->rx_buffers[entry].skb;
				skb_put(skb, pktlen);
				pci_unmap_single(
					priv->pdev,
					priv->rx_buffers[entry].mapping,
					RX_PKT_SIZE, PCI_DMA_FROMDEVICE);
				priv->rx_buffers[entry].skb = newskb;
				priv->rx_buffers[entry].mapping =
					pci_map_single(priv->pdev,
						       newskb->tail,
						       RX_PKT_SIZE,
						       PCI_DMA_FROMDEVICE);
			} else {
				skb = NULL;
				priv->stats.rx_dropped += 1;
			}

			priv->rx_ring[entry].buffer1 =
				cpu_to_le32(priv->rx_buffers[entry].mapping);
		}

		priv->rx_ring[entry].status = cpu_to_le32( RDES0_STATUS_OWN | RDES0_STATUS_SQL );
		priv->rx_ring[entry].length =
			cpu_to_le32(RX_PKT_SIZE |
				    (entry == RX_RING_SIZE - 1 ?
				     RDES1_CONTROL_RER : 0));

		if (skb) {
			struct adm8211_rx_status *stat =
				(struct adm8211_rx_status *) skb->cb;
#if 0
			{
				int i;
				printk(KERN_DEBUG "RX[%d/%d]",
				       pktlen, skb->len);
				for (i = 0; i < skb->len; i++)
					printk(" %02x", skb->data[i]);
				printk("\n");
			}
#endif
			stat->rssi = rssi;
			stat->rate = rate;
			skb->mac.raw = skb->data;
			skb->protocol = (__force unsigned short) __constant_htons(ETH_P_802_2);
			skb_queue_tail(&priv->rx_queue, skb);
			tasklet_schedule(&priv->rx_tasklet);
		}

		entry = (++priv->cur_rx) % RX_RING_SIZE;
	}

	priv->stats.rx_missed_errors += ADM8211_CSR_READ(LPC) & 0xFFFF;
}


static irqreturn_t adm8211_interrupt(int irq, void *dev_id,
				     struct pt_regs *regs)
{
#define ADM8211_INT(x) if (stsr & ADM8211_STSR_ ## x) printk(KERN_DEBUG "%s: " #x "\n", dev->name)

	struct net_device *dev = dev_id;
	struct adm8211_priv *priv = netdev_priv(dev);
	u32 stsr;
	int count = 0;

	do {
		stsr = ADM8211_CSR_READ(STSR);
		ADM8211_CSR_WRITE(STSR, stsr);
		if (stsr == 0xffffffff)
			return IRQ_HANDLED;

		if (!(stsr & (ADM8211_STSR_NISS | ADM8211_STSR_AISS)))
			break;

		/*if (stsr & ADM8211_STSR_TBTT) {
			priv->ieee80211.beacon_sync(dev);
		}*/

		if (stsr & ADM8211_STSR_RCI)
			adm8211_interrupt_rci(dev);
		if (stsr & ADM8211_STSR_TCI)
			adm8211_interrupt_tci(dev);
		if (stsr & (ADM8211_STSR_RCI | ADM8211_STSR_TCI))
			mod_timer(&priv->timer, jiffies + HZ);

		if ((stsr & (ADM8211_STSR_LinkOn | ADM8211_STSR_LinkOff))
			 != (ADM8211_STSR_LinkOn | ADM8211_STSR_LinkOff)) {
			if (stsr & ADM8211_STSR_LinkOn) {
				printk(KERN_DEBUG "%s: LinkOn\n", dev->name);
				priv->ieee80211.flags |= LINK_ON;
			}
		
			if (stsr & ADM8211_STSR_LinkOff) {
				printk(KERN_DEBUG "%s: LinkOff\n", dev->name);
				priv->ieee80211.flags &= ~LINK_ON;

				if (dev->flags & IFF_UP)
					ieee80211_start(&priv->ieee80211);
			}
		}
		
		ADM8211_INT(PCF);
		/*ADM8211_INT(BCNTC);*/
		ADM8211_INT(GPINT);
		ADM8211_INT(ATIMTC);
		/*ADM8211_INT(TSFTF);*/
		ADM8211_INT(TSCZ);
		ADM8211_INT(SQL);
		ADM8211_INT(WEPTD);
		ADM8211_INT(ATIME);
		/*ADM8211_INT(TBTT);*/
		ADM8211_INT(TEIS);
		ADM8211_INT(FBE);
		ADM8211_INT(REIS);
		ADM8211_INT(GPTT);
		ADM8211_INT(RPS);
		ADM8211_INT(RDU);
		ADM8211_INT(TUF);
		/*ADM8211_INT(TRT);*/
		/*ADM8211_INT(TLT);*/
		/*ADM8211_INT(TDU);*/
		ADM8211_INT(TPS);

	} while (count++ < 20);

	return IRQ_RETVAL(count);

#undef ADM8211_INT
}

static int adm8211_rf_write_syn_max2820 (struct net_device *dev, u8 addr, u16 value)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	int i;
	u32 reg;

	value &= 0xFFF;
	addr  &= 0xF;
	value = value | (addr << 12);

	ADM8211_CSR_WRITE(SYNRF, ADM8211_SYNRF_IF_SELECT_1);
	ADM8211_CSR_READ(SYNRF);
	ADM8211_CSR_WRITE(SYNRF, ADM8211_SYNRF_IF_SELECT_0);
	ADM8211_CSR_READ(SYNRF);

	ADM8211_CSR_WRITE(SYNRF, ADM8211_SYNRF_WRITE_SYNDATA_0);
	ADM8211_CSR_READ(SYNRF);

	for (i = 0; i <= 15; i+=1) {
		if ( value & (1 << (15 - i)) )
			reg = ADM8211_SYNRF_WRITE_SYNDATA_1;
		else
			reg = ADM8211_SYNRF_WRITE_SYNDATA_0;

		ADM8211_CSR_WRITE(SYNRF, reg);
		ADM8211_CSR_READ(SYNRF);

		ADM8211_CSR_WRITE(SYNRF, reg | ADM8211_SYNRF_WRITE_CLOCK_1);
		ADM8211_CSR_READ(SYNRF);
		ADM8211_CSR_WRITE(SYNRF, reg | ADM8211_SYNRF_WRITE_CLOCK_0);
		ADM8211_CSR_READ(SYNRF);
	}

	ADM8211_CSR_WRITE(SYNRF, reg | ADM8211_SYNRF_IF_SELECT_0);
	ADM8211_CSR_READ(SYNRF);

	ADM8211_CSR_WRITE(SYNRF, 0);
	ADM8211_CSR_READ(SYNRF);

	return 0;
}

static int adm8211_rf_write_syn_al2210l (struct net_device *dev, u8 addr, u32 value)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	int i;
	u32 reg;

	value &= 0xFFFFF;
	addr  &= 0xF;
	value = (value << 4) | addr;

	ADM8211_CSR_WRITE(SYNRF, ADM8211_SYNRF_IF_SELECT_1);
	ADM8211_CSR_READ(SYNRF);
	ADM8211_CSR_WRITE(SYNRF, ADM8211_SYNRF_IF_SELECT_0);
	ADM8211_CSR_READ(SYNRF);

	ADM8211_CSR_WRITE(SYNRF, ADM8211_SYNRF_WRITE_SYNDATA_0);
	ADM8211_CSR_READ(SYNRF);

	for (i = 0; i <= 23; i+=1) {
		if ( value & (1 << (23 - i)) )
			reg = ADM8211_SYNRF_WRITE_SYNDATA_1;
		else
			reg = ADM8211_SYNRF_WRITE_SYNDATA_0;

		ADM8211_CSR_WRITE(SYNRF, reg);
		ADM8211_CSR_READ(SYNRF);

		ADM8211_CSR_WRITE(SYNRF, reg | ADM8211_SYNRF_WRITE_CLOCK_1);
		ADM8211_CSR_READ(SYNRF);
		ADM8211_CSR_WRITE(SYNRF, reg | ADM8211_SYNRF_WRITE_CLOCK_0);
		ADM8211_CSR_READ(SYNRF);
	}

	ADM8211_CSR_WRITE(SYNRF, reg | ADM8211_SYNRF_IF_SELECT_0);
	ADM8211_CSR_READ(SYNRF);

	ADM8211_CSR_WRITE(SYNRF, 0);
	ADM8211_CSR_READ(SYNRF);

	return 0;
}

static int adm8211_rf_write_syn_rfmd2958 (struct net_device *dev, u16 addr, u32 value)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	int i;
	u32 reg;

	value &= 0x3FFFF;
	addr  &= 0x1F;
	value = value | (addr << 18);

	ADM8211_CSR_WRITE(SYNRF, ADM8211_SYNRF_IF_SELECT_1);
	ADM8211_CSR_READ(SYNRF);
	ADM8211_CSR_WRITE(SYNRF, ADM8211_SYNRF_IF_SELECT_0);
	ADM8211_CSR_READ(SYNRF);

	for (i = 0; i <= 23; i+=1) {
		if ( value & (1 << (23 - i)) )
			reg = ADM8211_SYNRF_WRITE_SYNDATA_1;
		else
			reg = ADM8211_SYNRF_WRITE_SYNDATA_0;

		ADM8211_CSR_WRITE(SYNRF, reg);
		ADM8211_CSR_READ(SYNRF);

		ADM8211_CSR_WRITE(SYNRF, reg | ADM8211_SYNRF_WRITE_CLOCK_1);
		ADM8211_CSR_READ(SYNRF);
		ADM8211_CSR_WRITE(SYNRF, reg | ADM8211_SYNRF_WRITE_CLOCK_0);
		ADM8211_CSR_READ(SYNRF);
	}

	ADM8211_CSR_WRITE(SYNRF, reg | ADM8211_SYNRF_IF_SELECT_0);
	ADM8211_CSR_READ(SYNRF);

	ADM8211_CSR_WRITE(SYNRF, 0);
	ADM8211_CSR_READ(SYNRF);

	return 0;
}

static int adm8211_rf_write_syn_rfmd2948 (struct net_device *dev, u8 addr, u16 value)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	int i;
	u32 reg, bits;

	value &= 0xFFFF;
	addr  &= 0xF;
	bits = (value << 4) | addr;

	ADM8211_CSR_WRITE(SYNRF, ADM8211_SYNRF_IF_SELECT_1);
	ADM8211_CSR_READ(SYNRF);
	ADM8211_CSR_WRITE(SYNRF, ADM8211_SYNRF_IF_SELECT_0);
	ADM8211_CSR_READ(SYNRF);

	for (i = 0; i <= 21; i+=1) {
		if ( bits & (1 << (21 - i)) )
			reg = ADM8211_SYNRF_WRITE_SYNDATA_1;
		else
			reg = ADM8211_SYNRF_WRITE_SYNDATA_0;

		ADM8211_CSR_WRITE(SYNRF, reg);
		ADM8211_CSR_READ(SYNRF);

		ADM8211_CSR_WRITE(SYNRF, reg | ADM8211_SYNRF_WRITE_CLOCK_1);
		ADM8211_CSR_READ(SYNRF);
		ADM8211_CSR_WRITE(SYNRF, reg | ADM8211_SYNRF_WRITE_CLOCK_0);
		ADM8211_CSR_READ(SYNRF);
	}

	ADM8211_CSR_WRITE(SYNRF, reg | ADM8211_SYNRF_IF_SELECT_1);
	ADM8211_CSR_READ(SYNRF);

	ADM8211_CSR_WRITE(SYNRF, 0);
	ADM8211_CSR_READ(SYNRF);

	return 0;
}

static int adm8211_write_bbp(struct net_device *dev, u8 addr, u8 data)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	int timeout;
	u32 reg;

	timeout = 10;
	while (timeout > 0) {
		reg = ADM8211_CSR_READ(BBPCTL);
		if (!(reg & (ADM8211_BBPCTL_WR | ADM8211_BBPCTL_RD)))
			break;
		timeout--;
		mdelay(2);
	}

	if (timeout == 0) {
		printk(KERN_DEBUG "%s: adm8211_write_bbp(%d,%d) failed prewrite "
		       "(reg=0x%08x)\n",
		       dev->name, addr, data, reg);
		return -ETIMEDOUT;
	}

	switch (priv->bbp_type) {
	case ADM8211_TYPE_INTERSIL:
		reg = ADM8211_BBPCTL_MMISEL;	/* three wire interface */
		break;
	case ADM8211_TYPE_RFMD:
		reg = (0x20<<24) | ADM8211_BBPCTL_TXCE | ADM8211_BBPCTL_CCAP |
		      (0x01<<18);
		break;
	case ADM8211_TYPE_ADMTEK:
		reg = (0x20<<24) | ADM8211_BBPCTL_TXCE | ADM8211_BBPCTL_CCAP |
		      (0x05<<18);
		break;
	}
	reg |= ADM8211_BBPCTL_WR | (addr << 8) | data;

	ADM8211_CSR_WRITE(BBPCTL, reg);

	timeout=10;
	while (timeout > 0) {
		reg = ADM8211_CSR_READ(BBPCTL);
		if (!(reg & ADM8211_BBPCTL_WR))
			break;
		timeout--;
		mdelay(2);
	}

	if (timeout == 0) {
		ADM8211_CSR_WRITE(BBPCTL, ADM8211_CSR_READ(BBPCTL) &
				  (~ADM8211_BBPCTL_WR));
		printk(KERN_DEBUG "%s: adm8211_write_bbp(%d,%d) failed postwrite "
		       "(reg=0x%08x)\n",
		       dev->name, addr, data, reg);
		return -ETIMEDOUT;
	}

	return 0;
}

int adm8211_rf_set_channel(struct net_device *dev, int channel)
{
	static const u32 adm8211_rfmd2958_reg5[] =
		{0x22BD, 0x22D2, 0x22E8, 0x22FE, 0x2314, 0x232A, 0x2340,
		 0x2355, 0x236B, 0x2381, 0x2397, 0x23AD, 0x23C2, 0x23F7};
	static const u32 adm8211_rfmd2958_reg6[] =
		{0x05D17, 0x3A2E8, 0x2E8BA, 0x22E8B, 0x1745D, 0x0BA2E, 0x00000,
		 0x345D1, 0x28BA2, 0x1D174, 0x11745, 0x05D17, 0x3A2E8, 0x11745};

	struct adm8211_priv *priv = netdev_priv(dev);
	u8 ant_power = priv->ant_power > 0x3F ?
		priv->eeprom->antenna_power[channel-1] : priv->ant_power;
	u8 tx_power = priv->tx_power > 0x3F ?
		priv->eeprom->tx_power[channel-1] : priv->tx_power;
	u8 lpf_cutoff = priv->lpf_cutoff == 0xFF ?
		priv->eeprom->lpf_cutoff[channel-1] : priv->lpf_cutoff;
	u8 lnags_thresh = priv->lnags_threshold == 0xFF ?
		priv->eeprom->lnags_threshold[channel-1] : priv->lnags_threshold;
	u32 reg;

	if (channel < 1 || channel > 14)
		return -EINVAL;

	/* Stop transmit */
	ADM8211_CSR_WRITE(NAR, priv->nar & ~(ADM8211_NAR_ST | ADM8211_NAR_SR));
	ADM8211_CSR_READ(NAR);
	mdelay(20);

	/* Program synthesizer to new channel */
	switch (priv->transceiver_type) {
	case ADM8211_RFMD2958:
	case ADM8211_RFMD2958_RF3000_CONTROL_POWER:
		adm8211_rf_write_syn_rfmd2958(dev, 0x00, 0x04007);
		adm8211_rf_write_syn_rfmd2958(dev, 0x02, 0x00033);

		adm8211_rf_write_syn_rfmd2958(dev, 0x05,
					adm8211_rfmd2958_reg5[channel-1]);
		adm8211_rf_write_syn_rfmd2958(dev, 0x06,
					adm8211_rfmd2958_reg6[channel-1]);
		break;

	case ADM8211_RFMD2948:
		adm8211_rf_write_syn_rfmd2948(dev, SI4126_MAIN_CONF, SI4126_MAIN_XINDIV2);
		adm8211_rf_write_syn_rfmd2948(dev, SI4126_POWERDOWN,
				     SI4126_POWERDOWN_PDIB | SI4126_POWERDOWN_PDRB);
		adm8211_rf_write_syn_rfmd2948(dev, SI4126_PHASE_DET_GAIN, 0);
		adm8211_rf_write_syn_rfmd2948(dev, SI4126_RF2_N_DIV,
				     (channel == 14 ? 2110 : 2033 + (channel * 5)));
		adm8211_rf_write_syn_rfmd2948(dev, SI4126_IF_N_DIV, 1496);
		adm8211_rf_write_syn_rfmd2948(dev, SI4126_RF2_R_DIV, 44);
		adm8211_rf_write_syn_rfmd2948(dev, SI4126_IF_R_DIV, 44);
		break;

	case ADM8211_MAX2820:
		adm8211_rf_write_syn_max2820(dev, 0x3,
			(channel == 14 ? 0x054 : 0x7 + (channel * 5)));
		break;

	case ADM8211_AL2210L:
		adm8211_rf_write_syn_al2210l(dev, 0x0,
			(channel == 14 ? 0x229B4 : 0x22967 + (channel * 5)));
		break;

	default:
		printk(KERN_DEBUG "%s: unsupported transceiver type %d\n",
		       dev->name, priv->transceiver_type);
		break;
	}

	/* write BBP regs */
	if (priv->bbp_type == ADM8211_TYPE_RFMD) {

	/* SMC 2635W specific? adm8211b doesn't use the 2948 though.. */
	/* TODO: remove if SMC 2635W doesn't need this */
	if (priv->transceiver_type == ADM8211_RFMD2948) {
		reg = ADM8211_CSR_READ(GPIO);
		reg &= 0xfffc0000;
		reg |= ADM8211_CSR_GPIO_EN0;
		if (channel != 14)
			reg |= ADM8211_CSR_GPIO_O0;
		ADM8211_CSR_WRITE(GPIO, reg);
	}

	if (priv->transceiver_type == ADM8211_RFMD2958) {
		/* set PCNT2 */
		adm8211_rf_write_syn_rfmd2958(dev, 0x0B, 0x07100);
		/* set PCNT1 P_DESIRED/MID_BIAS */
		reg = le16_to_cpu(priv->eeprom->cr49);
		reg >>= 13;
		reg <<= 15;
		reg |= ant_power<<9;
		adm8211_rf_write_syn_rfmd2958(dev, 0x0A, reg);
		/* set TXRX TX_GAIN */
		adm8211_rf_write_syn_rfmd2958(dev, 0x09, 0x00050 |
			(priv->revid < ADM8211_REV_CA ? tx_power : 0));
	} else {
		reg = ADM8211_CSR_READ(PLCPHD);
		reg &= 0xff00ffff;
		reg |= tx_power<<18;
		ADM8211_CSR_WRITE(PLCPHD, reg);
	}

	ADM8211_CSR_WRITE(SYNRF, ADM8211_SYNRF_SELRF | ADM8211_SYNRF_PE1 |
			  ADM8211_SYNRF_PHYRST);
	ADM8211_CSR_READ(SYNRF);
	mdelay(30);

	/* RF3000 BBP */
	if (priv->transceiver_type != ADM8211_RFMD2958)
		adm8211_write_bbp(dev, RF3000_TX_VAR_GAIN__TX_LEN_EXT,
				  tx_power<<2);
	adm8211_write_bbp(dev, RF3000_LOW_GAIN_CALIB, lpf_cutoff);
	adm8211_write_bbp(dev, RF3000_HIGH_GAIN_CALIB, lnags_thresh);
	adm8211_write_bbp(dev, 0x1c, priv->revid == ADM8211_REV_BA
			  ? priv->eeprom->cr28 : 0);
	adm8211_write_bbp(dev, 0x1d, priv->eeprom->cr29);

	ADM8211_CSR_WRITE(SYNRF, 0);

	} else if (priv->bbp_type != ADM8211_TYPE_ADMTEK) {	/* Nothing to do for ADMtek BBP */
		printk(KERN_DEBUG "%s: unsupported BBP type %d\n",
		       dev->name, priv->bbp_type);
	}

	ADM8211_CSR_WRITE(NAR, priv->nar);

	/* update current channel for adhoc (and maybe AP mode) */
	reg = ADM8211_CSR_READ(CAP0);
	reg &= ~0xF;
	reg |= channel;
	ADM8211_CSR_WRITE(CAP0, reg);

	return 0;
}

void adm8211_write_wepkey(struct net_device *dev, int index)
{
#define ADM8211_WEP_ENABLE	(1 << 7)
#define ADM8211_WEP_A_104	(1 << 6)
#define ADM8211_WEP_B_104	(1 << 4)

	struct adm8211_priv *priv = netdev_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;
	int addr;
	u8 buf[32];
	memset(buf, 0, sizeof(buf));

	if (priv->revid < ADM8211_REV_BA) {
		addr = (index*7) + ADM8211_SRAM_A_SHARE_KEY;

		/* control entry */
		if (data->wep_data[index].len > 5)
			buf[1] = ADM8211_WEP_ENABLE | ADM8211_WEP_A_104;
		else if (data->wep_data[index].len > 0)
			buf[1] = ADM8211_WEP_ENABLE;

		buf[0]=data->wep_data[index].key[0];

		if (data->wep_data[index].len > 0)
			memcpy(buf+2, &(data->wep_data[index].key[1]), data->wep_data[index].len-1);

		adm8211_write_sram_bytes(dev, addr, buf, 14);
	} else {
		addr = (index*5) + ADM8211_SRAM_B_SHARE_KEY;

		/* control entry */
		if (data->wep_data[index].len > 5)
			*(__le32 *)buf = cpu_to_le32(ADM8211_WEP_ENABLE | ADM8211_WEP_B_104);
		else if (data->wep_data[index].len > 0)
			*(__le32 *)buf = cpu_to_le32(ADM8211_WEP_ENABLE);

		if (data->wep_data[index].len > 0)
			memcpy(buf+4, data->wep_data[index].key, data->wep_data[index].len);

		adm8211_write_sram_bytes(dev, addr, buf, 17);
	}
}

void adm8211_update_wep(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;
	u32 reg;

	reg = ADM8211_CSR_READ(MACTEST);
	if (data->flags & WEP_ENABLED) {
		reg &= ~(3<<20);
		reg |= (1<<22);
		reg |= (data->default_key << 20);
	} else
		reg &= ~(7<<20);
	ADM8211_CSR_WRITE(MACTEST, reg);

	reg = ADM8211_CSR_READ(WEPCTL);
	if (data->flags & WEP_ENABLED)
		reg |= ADM8211_WEPCTL_WEPENABLE;
	else
		reg &= ~ADM8211_WEPCTL_WEPENABLE;

	/* no hardware WEP RX decryption on the ADM8211A */
	if (priv->revid < ADM8211_REV_BA)
		reg |= ADM8211_WEPCTL_WEPRXBYP;

	ADM8211_CSR_WRITE(WEPCTL, reg);
}

void adm8211_update_mode(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;

	/* make sure we're stopped */
	if (priv->nar & (ADM8211_NAR_SR | ADM8211_NAR_ST)) {
		ADM8211_CSR_WRITE(NAR, priv->nar & ~(ADM8211_NAR_SR | ADM8211_NAR_ST));
		ADM8211_CSR_READ(NAR);
		mdelay(20);
	}

	priv->soft_rx_crc = 0;
	switch (priv->iw_mode) {
	case IW_MODE_INFRA:
		priv->nar &= ~(ADM8211_NAR_PR | ADM8211_NAR_EA);
		priv->nar |= ADM8211_NAR_ST | ADM8211_NAR_SR;
		ADM8211_CSR_WRITE(CAP1, data->capab<<16);
		break;
	case IW_MODE_ADHOC:
		priv->nar &= ~ADM8211_NAR_PR;
		priv->nar |= ADM8211_NAR_EA | ADM8211_NAR_ST | ADM8211_NAR_SR;

		ADM8211_CSR_WRITE(CAP1, data->capab<<16);

		/* don't trust the error bits on rev 0x20 and up in adhoc */
		if (priv->revid >= ADM8211_REV_BA)
			priv->soft_rx_crc = 1;
		break;
	case IW_MODE_MONITOR:
		priv->nar &= ~(ADM8211_NAR_EA | ADM8211_NAR_ST);
		priv->nar |= ADM8211_NAR_PR | ADM8211_NAR_SR;
		ADM8211_CSR_WRITE(CAP1, 0);
		break;
	}

	ADM8211_CSR_WRITE(NAR, priv->nar);
}

static void adm8211_hw_init_syn(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	switch (priv->transceiver_type) {
	case ADM8211_RFMD2958:
	case ADM8211_RFMD2958_RF3000_CONTROL_POWER:
		adm8211_rf_write_syn_rfmd2958(dev, 0x1F, 0x00000);
		adm8211_rf_write_syn_rfmd2958(dev, 0x0C, 0x3001F);
		adm8211_rf_write_syn_rfmd2958(dev, 0x01, 0x29C03);
		adm8211_rf_write_syn_rfmd2958(dev, 0x03, 0x1FF6F);
		adm8211_rf_write_syn_rfmd2958(dev, 0x04, 0x29403);
		adm8211_rf_write_syn_rfmd2958(dev, 0x07, 0x1456F);
		adm8211_rf_write_syn_rfmd2958(dev, 0x09,
			(priv->transceiver_type == ADM8211_RFMD2958
			? 0x10050 : 0x00050) );
		adm8211_rf_write_syn_rfmd2958(dev, 0x08, 0x3FFF8);
		break;

	case ADM8211_MAX2820:
		adm8211_rf_write_syn_max2820(dev, 0x1, 0x01E);
		adm8211_rf_write_syn_max2820(dev, 0x2, 0x001);
		adm8211_rf_write_syn_max2820(dev, 0x3, 0x054);
		adm8211_rf_write_syn_max2820(dev, 0x4, 0x310);
		adm8211_rf_write_syn_max2820(dev, 0x5, 0x000);
		break;

	case ADM8211_AL2210L:
		adm8211_rf_write_syn_al2210l(dev, 0x0, 0x0196C);
		adm8211_rf_write_syn_al2210l(dev, 0x1, 0x007CB);
		adm8211_rf_write_syn_al2210l(dev, 0x2, 0x3582F);
		adm8211_rf_write_syn_al2210l(dev, 0x3, 0x010A9);
		adm8211_rf_write_syn_al2210l(dev, 0x4, 0x77280);
		adm8211_rf_write_syn_al2210l(dev, 0x5, 0x45641);
		adm8211_rf_write_syn_al2210l(dev, 0x6, 0xEA130);
		adm8211_rf_write_syn_al2210l(dev, 0x7, 0x80000);
		adm8211_rf_write_syn_al2210l(dev, 0x8, 0x7850F);
		adm8211_rf_write_syn_al2210l(dev, 0x9, 0xF900C);
		adm8211_rf_write_syn_al2210l(dev, 0xA, 0x00000);
		adm8211_rf_write_syn_al2210l(dev, 0xB, 0x00000);
		break;

	case ADM8211_RFMD2948:
	default:
		break;
	}
}

static int adm8211_hw_init_bbp(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	u32 reg;

	/* write addresses */
	if (priv->bbp_type == ADM8211_TYPE_INTERSIL) {
		ADM8211_CSR_WRITE(MMIWA,  0x100E0C0A);
		ADM8211_CSR_WRITE(MMIRD0, 0x00007c7e);
		ADM8211_CSR_WRITE(MMIRD1, 0x00100000);
	} else if (priv->bbp_type == ADM8211_TYPE_RFMD ||
		   priv->bbp_type == ADM8211_TYPE_ADMTEK) {

	/* check specific BBP type */
	switch (priv->eeprom->specific_bbptype) {
	case ADM8211_BBP_RFMD3000:
	case ADM8211_BBP_RFMD3002:
		ADM8211_CSR_WRITE(MMIWA,  0x00009101);
		ADM8211_CSR_WRITE(MMIRD0, 0x00000301);
		break;

	case ADM8211_BBP_ADM8011:
		ADM8211_CSR_WRITE(MMIWA,  0x00008903);
		ADM8211_CSR_WRITE(MMIRD0, 0x00001716);

		reg = ADM8211_CSR_READ(BBPCTL);
		reg &= ~ADM8211_BBPCTL_TYPE;
		reg |= 0x5 << 18;
		ADM8211_CSR_WRITE(BBPCTL, reg);
		break;
	}

	switch (priv->revid) {
	case ADM8211_REV_CA:
		if (priv->transceiver_type == ADM8211_RFMD2958 ||
		    priv->transceiver_type == ADM8211_RFMD2958_RF3000_CONTROL_POWER ||
		    priv->transceiver_type == ADM8211_RFMD2948)
			ADM8211_CSR_WRITE(SYNCTL, (0x1 << 22));
		else if (priv->transceiver_type == ADM8211_MAX2820 ||
			 priv->transceiver_type == ADM8211_AL2210L)
			ADM8211_CSR_WRITE(SYNCTL, (0x3 << 22));
		break;

	case ADM8211_REV_BA:
		reg  = ADM8211_CSR_READ(MMIRD1);
		reg &= 0x0000FFFF;
		reg |= 0x7e100000;
		ADM8211_CSR_WRITE(MMIRD1, reg);
		break;

	case ADM8211_REV_AB:
	case ADM8211_REV_AF:
	default:
		ADM8211_CSR_WRITE(MMIRD1, 0x7e100000);
		break;
	}

	/* For RFMD */
	ADM8211_CSR_WRITE(MACTEST, 0x800);
	}

	adm8211_hw_init_syn(dev);

	/* Set RF Power control IF pin to PE1+PHYRST# */
	ADM8211_CSR_WRITE(SYNRF, ADM8211_SYNRF_SELRF | ADM8211_SYNRF_PE1 |
			  ADM8211_SYNRF_PHYRST);
	ADM8211_CSR_READ(SYNRF);
	mdelay(20);

	/* write BBP regs */
	if (priv->bbp_type == ADM8211_TYPE_RFMD) {
		/* RF3000 BBP */
		/* another set:
		 * 11: c8
		 * 14: 14
		 * 15: 50 (chan 1..13; chan 14: d0)
		 * 1c: 00
		 * 1d: 84
		 */
		adm8211_write_bbp(dev, RF3000_CCA_CTRL, 0x80);
		adm8211_write_bbp(dev, RF3000_DIVERSITY__RSSI, 0x80); /* antenna selection: diversity */
		adm8211_write_bbp(dev, RF3000_TX_VAR_GAIN__TX_LEN_EXT, 0x74);
		adm8211_write_bbp(dev, RF3000_LOW_GAIN_CALIB, 0x38);
		adm8211_write_bbp(dev, RF3000_HIGH_GAIN_CALIB, 0x40);

		if (priv->eeprom->major_version < 2) {
			adm8211_write_bbp(dev, 0x1c, 0x00);
			adm8211_write_bbp(dev, 0x1d, 0x80);
		} else {
			if (priv->revid == ADM8211_REV_BA)
				adm8211_write_bbp(dev, 0x1c, priv->eeprom->cr28);
			else
				adm8211_write_bbp(dev, 0x1c, 0x00);

			adm8211_write_bbp(dev, 0x1d, priv->eeprom->cr29);
		}
	} else if (priv->bbp_type == ADM8211_TYPE_ADMTEK) {
	adm8211_write_bbp(dev, 0x00, 0xFF);	/* reset baseband */
	adm8211_write_bbp(dev, 0x07, 0x0A);	/* antenna selection: diversity */

	/* TODO: find documentation for this */
	switch (priv->transceiver_type) {
	case ADM8211_RFMD2958:
	case ADM8211_RFMD2958_RF3000_CONTROL_POWER:
		adm8211_write_bbp(dev, 0x00, 0x00);
		adm8211_write_bbp(dev, 0x01, 0x00);
		adm8211_write_bbp(dev, 0x02, 0x00);
		adm8211_write_bbp(dev, 0x03, 0x00);
		adm8211_write_bbp(dev, 0x06, 0x0f);
		adm8211_write_bbp(dev, 0x09, 0x00);
		adm8211_write_bbp(dev, 0x0a, 0x00);
		adm8211_write_bbp(dev, 0x0b, 0x00);
		adm8211_write_bbp(dev, 0x0c, 0x00);
		adm8211_write_bbp(dev, 0x0f, 0xAA);
		adm8211_write_bbp(dev, 0x10, 0x8c);
		adm8211_write_bbp(dev, 0x11, 0x43);
		adm8211_write_bbp(dev, 0x18, 0x40);
		adm8211_write_bbp(dev, 0x20, 0x23);
		adm8211_write_bbp(dev, 0x21, 0x02);
		adm8211_write_bbp(dev, 0x22, 0x28);
		adm8211_write_bbp(dev, 0x23, 0x30);
		adm8211_write_bbp(dev, 0x24, 0x2d);
		adm8211_write_bbp(dev, 0x28, 0x35);
		adm8211_write_bbp(dev, 0x2a, 0x8c);
		adm8211_write_bbp(dev, 0x2b, 0x81);
		adm8211_write_bbp(dev, 0x2c, 0x44);
		adm8211_write_bbp(dev, 0x2d, 0x0A);
		adm8211_write_bbp(dev, 0x29, 0x40);
		adm8211_write_bbp(dev, 0x60, 0x08);
		adm8211_write_bbp(dev, 0x64, 0x01);
		break;

	case ADM8211_MAX2820:
		adm8211_write_bbp(dev, 0x00, 0x00);
		adm8211_write_bbp(dev, 0x01, 0x00);
		adm8211_write_bbp(dev, 0x02, 0x00);
		adm8211_write_bbp(dev, 0x03, 0x00);
		adm8211_write_bbp(dev, 0x06, 0x0f);
		adm8211_write_bbp(dev, 0x09, 0x05);
		adm8211_write_bbp(dev, 0x0a, 0x02);
		adm8211_write_bbp(dev, 0x0b, 0x00);
		adm8211_write_bbp(dev, 0x0c, 0x0f);
		adm8211_write_bbp(dev, 0x0f, 0x55);
		adm8211_write_bbp(dev, 0x10, 0x8d);
		adm8211_write_bbp(dev, 0x11, 0x43);
		adm8211_write_bbp(dev, 0x18, 0x4a);
		adm8211_write_bbp(dev, 0x20, 0x20);
		adm8211_write_bbp(dev, 0x21, 0x02);
		adm8211_write_bbp(dev, 0x22, 0x23);
		adm8211_write_bbp(dev, 0x23, 0x30);
		adm8211_write_bbp(dev, 0x24, 0x2d);
		adm8211_write_bbp(dev, 0x2a, 0x8c);
		adm8211_write_bbp(dev, 0x2b, 0x81);
		adm8211_write_bbp(dev, 0x2c, 0x44);
		adm8211_write_bbp(dev, 0x29, 0x4a);
		adm8211_write_bbp(dev, 0x60, 0x2b);
		adm8211_write_bbp(dev, 0x64, 0x01);
		break;

	case ADM8211_AL2210L:
		adm8211_write_bbp(dev, 0x00, 0x00);
		adm8211_write_bbp(dev, 0x01, 0x00);
		adm8211_write_bbp(dev, 0x02, 0x00);
		adm8211_write_bbp(dev, 0x03, 0x00);
		adm8211_write_bbp(dev, 0x06, 0x0f);
		adm8211_write_bbp(dev, 0x07, 0x05);
		adm8211_write_bbp(dev, 0x08, 0x03);
		adm8211_write_bbp(dev, 0x09, 0x00);
		adm8211_write_bbp(dev, 0x0a, 0x00);
		adm8211_write_bbp(dev, 0x0b, 0x00);
		adm8211_write_bbp(dev, 0x0c, 0x10);
		adm8211_write_bbp(dev, 0x0f, 0x55);
		adm8211_write_bbp(dev, 0x10, 0x8d);
		adm8211_write_bbp(dev, 0x11, 0x43);
		adm8211_write_bbp(dev, 0x18, 0x4a);
		adm8211_write_bbp(dev, 0x20, 0x20);
		adm8211_write_bbp(dev, 0x21, 0x02);
		adm8211_write_bbp(dev, 0x22, 0x23);
		adm8211_write_bbp(dev, 0x23, 0x30);
		adm8211_write_bbp(dev, 0x24, 0x2d);
		adm8211_write_bbp(dev, 0x2a, 0xaa);
		adm8211_write_bbp(dev, 0x2b, 0x81);
		adm8211_write_bbp(dev, 0x2c, 0x44);
		adm8211_write_bbp(dev, 0x29, 0xfa);
		adm8211_write_bbp(dev, 0x60, 0x2d);
		adm8211_write_bbp(dev, 0x64, 0x01);
		break;

	case ADM8211_RFMD2948:
		break;

	default:
		printk(KERN_DEBUG "%s: unsupported transceiver type %d\n",
		       dev->name, priv->transceiver_type);
		break;
	}
	} else {
		printk(KERN_DEBUG "%s: unsupported BBP type %d\n",
		       dev->name, priv->bbp_type);
	}

	ADM8211_CSR_WRITE(SYNRF, 0);

	/* Set RF CAL control source to MAC control */
	reg = ADM8211_CSR_READ(SYNCTL);
	reg |= ADM8211_SYNCTL_SELCAL;
	ADM8211_CSR_WRITE(SYNCTL, reg);

	return 0;
}

static int adm8211_set_rate(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;
	u32 reg;
	int i = 0;
	u8 rate_buf[12] = {0};

	/* write supported rates */
	if (priv->revid != ADM8211_REV_BA) {
		rate_buf[0] = data->num_supp_rates;
		for (i = 0; i < data->num_supp_rates; i+=1)
			rate_buf[i+1] = data->supp_rates[i] |
					(data->supp_rates[i] <= data->rate ? 0x80 : 0);
	} else {
		/* workaround for rev BA specific bug */
		rate_buf[0]=4;
		rate_buf[1]=0x82;
		rate_buf[2]=0x04;
		rate_buf[3]=0x0b;
		rate_buf[4]=0x16;
	}
	
	adm8211_write_sram_bytes(dev, ADM8211_SRAM_SUPP_RATE, rate_buf, data->num_supp_rates+1);

	priv->plcp_signal = data->rate * 5;

	/* keep bits 0-23 */
	reg = ADM8211_CSR_READ(PLCPHD) & 0x00FFFFFF;

	/* short preamble */
	reg |= (1 << 15);

	reg |= priv->plcp_signal << 24;
	ADM8211_CSR_WRITE(PLCPHD, reg);

	/* MTMLT   = 512 TU (max TX MSDU lifetime)
	 * BCNTSIG = plcp_signal (beacon, probe resp, and atim TX rate)
	 * SRTYLIM = 224 (short retry limit, value in TX header used by default) */
	ADM8211_CSR_WRITE(TXLMT, (512<<16) | (priv->plcp_signal<<8) | (224<<0));

	if (priv->revid >= ADM8211_REV_BA)
		adm8211_set_beacon(dev);

	return 0;
}

static void adm8211_hw_init(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	u32 reg;
	u8 cacheline;

	reg = ADM8211_CSR_READ(PAR);
	reg |= ADM8211_PAR_MRLE | ADM8211_PAR_MRME;
	reg &= ~(ADM8211_PAR_BAR | ADM8211_PAR_CAL);

	if (!pci_set_mwi(priv->pdev)) {
		reg |= (0x1<<24);
		pci_read_config_byte(priv->pdev, PCI_CACHE_LINE_SIZE, &cacheline);

		switch (cacheline) {
		case  0x8: reg |= (0x1<<14);
			   break;
		case 0x16: reg |= (0x2<<14);
			   break;
		case 0x32: reg |= (0x3<<14);
			   break;
		  default: reg |= (0x0<<14);
			   break;
		}
	}

	ADM8211_CSR_WRITE(PAR, reg);

	reg = ADM8211_CSR_READ(CSR_TEST1);
	reg &= ~(0xF<<28);
	reg |= ((1 << 28) | (1 << 31));
	ADM8211_CSR_WRITE(CSR_TEST1, reg);

	reg = (0x07 << 21) | (1 << 20) | (1 << 8);
	ADM8211_CSR_WRITE(WCSR, reg);

	/* Disable APM, enable receive FIFO threshold, and set drain receive
	 * threshold to store-and-forward */
	reg = ADM8211_CSR_READ(CMDR);
	reg &= ~(ADM8211_CMDR_APM | ADM8211_CMDR_DRT);
	reg |= ADM8211_CMDR_RTE | ADM8211_CMDR_DRT_SF;
	ADM8211_CSR_WRITE(CMDR, reg);

	adm8211_set_rate(dev);

	/* 4-bit values:
	 * PWR1UP   = 8 * 2 ms
	 * PWR0PAPE = 8 us or 5 us
	 * PWR1PAPE = 1 us or 3 us
	 * PWR0TRSW = 5 us
	 * PWR1TRSW = 12 us
	 * PWR0PE2  = 13 us
	 * PWR1PE2  = 1 us
	 * PWR0TXPE = 8 or 6 */
	if (priv->revid < ADM8211_REV_CA)
		ADM8211_CSR_WRITE(TOFS2, 0x8815cd18);
	else
		ADM8211_CSR_WRITE(TOFS2, 0x8535cd16);

	/* Enable store and forward for transmit */
	priv->nar = ADM8211_NAR_SF | ADM8211_NAR_PB;
	ADM8211_CSR_WRITE(NAR, priv->nar);

	/* Reset RF */
	ADM8211_CSR_WRITE(SYNRF, ADM8211_SYNRF_RADIO);
	ADM8211_CSR_READ(SYNRF);
	mdelay(10);
	ADM8211_CSR_WRITE(SYNRF, 0);
	ADM8211_CSR_READ(SYNRF);
	mdelay(5);

	/* Set CFP Max Duration to 0x10 TU */
	reg = ADM8211_CSR_READ(CFPP);
	reg &= ~(0xffff<<8);
	reg |= 0x0010<<8;
	ADM8211_CSR_WRITE(CFPP, reg);

	/* USCNT = 0x16 (number of system clocks, 22 MHz, in 1us
	 * TUCNT = 0x3ff - Tu counter 1024 us  */
	ADM8211_CSR_WRITE(TOFS0, (0x16 << 24) | 0x3ff);

	/* SLOT=20 us, SIFS=110 cycles of 22 MHz (5 us),
	 * DIFS=50 us, EIFS=100 us */
	if (priv->revid < ADM8211_REV_CA)
		ADM8211_CSR_WRITE(IFST, (20 << 23) | (110 << 15) |
					(50 << 9)  | 100);
	else
		ADM8211_CSR_WRITE(IFST, (20 << 23) | (24 << 15) |
					(50 << 9)  | 100);

	/* PCNT = 1 (MAC idle time awake/sleep, unit S)
	 * RMRD = 2346 * 8 + 1 us (max RX duration)  */
	ADM8211_CSR_WRITE(RMD, (1 << 16) | 18769);

	/* MART=65535 us, MIRT=256 us, TSFTOFST=0 us */
	ADM8211_CSR_WRITE(RSPT, 0xffffff00);

	/* Initialize BBP (and SYN) */
	adm8211_hw_init_bbp(dev);

	/* make sure interrupts are off */
	ADM8211_CSR_WRITE(IER, 0);

	/* ACK interrupts */
	ADM8211_CSR_WRITE(STSR, ADM8211_CSR_READ(STSR));

	/* Setup WEP */
	adm8211_write_wepkey(dev, 0);
	adm8211_write_wepkey(dev, 1);
	adm8211_write_wepkey(dev, 2);
	adm8211_write_wepkey(dev, 3);
	adm8211_update_wep(dev);
}

static int adm8211_hw_reset(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	u32 reg;
	int timeout = 50;

	/* Power-on issue */
	ADM8211_CSR_WRITE(FRCTL, 0);

	/* Reset the chip */
	reg = ADM8211_CSR_READ(PAR);
	ADM8211_CSR_WRITE(PAR, ADM8211_PAR_SWR);

	while ((ADM8211_CSR_READ(PAR) & ADM8211_PAR_SWR) && timeout--)
		mdelay(100);

	if (timeout <= 0)
		return -ETIMEDOUT;

	ADM8211_CSR_WRITE(PAR, reg);

	if (priv->revid == ADM8211_REV_BA &&
	    ( priv->transceiver_type == ADM8211_RFMD2958_RF3000_CONTROL_POWER
	   || priv->transceiver_type == ADM8211_RFMD2958)) {
		reg = ADM8211_CSR_READ(CSR_TEST1);
		reg |= (1 << 4) | (1 << 5);
		ADM8211_CSR_WRITE(CSR_TEST1, reg);
	} else if (priv->revid == ADM8211_REV_CA) {
		reg = ADM8211_CSR_READ(CSR_TEST1);
		reg &= ~((1 << 4) | (1 << 5));
		ADM8211_CSR_WRITE(CSR_TEST1, reg);
	}

	ADM8211_CSR_WRITE(FRCTL, 0);

	reg = ADM8211_CSR_READ(CSR_TEST0);
	reg |= ADM8211_CSR_TEST0_EPRLD;
	ADM8211_CSR_WRITE(CSR_TEST0, reg);

	adm8211_clear_sram(dev);

	return 0;
}

static void adm8211_set_tbtt (struct net_device *dev, u16 tbtt)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	/* TSFTOFSR (RX TSFT Offset) = 1 us
	 * TBTTPRE (prediction time) = tbtt TU
	 * TBTTOFS (Wake up time offset before TBTT) = 20 TU */
	ADM8211_CSR_WRITE(TOFS1, (1 << 24) | (tbtt << 8) | 20);
	ADM8211_CSR_READ(TOFS1);
}

static u64 adm8211_get_tsft (struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	u32 tsftl;
	u64 tsft;

	tsftl = ADM8211_CSR_READ(TSFTL);
	tsft = ADM8211_CSR_READ(TSFTH);
	tsft <<= 32;
	tsft |= tsftl;

	return tsft;
}

static void adm8211_set_interval(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;
	u32 reg;

	/* BP (beacon interval) = data->beacon_interval
	 * LI (listen interval) = data->listen_interval (in beacon intervals) */
	reg = (data->beacon_interval << 16) | data->listen_interval;
	ADM8211_CSR_WRITE(BPLI, reg);

	/* lose link after 7 lost beacons */
	reg = ADM8211_CSR_READ(WCSR) & ~(0xFF<<21);
	ADM8211_CSR_WRITE(WCSR, reg | (0x03<<21));
	ADM8211_CSR_READ(WCSR);
}

static int adm8211_set_bssid(struct net_device *dev, u8 *bssid)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	u32 reg;

	reg = bssid[0] | (bssid[1] << 8) | (bssid[2] << 16) | (bssid[3] << 24);
	ADM8211_CSR_WRITE(BSSID0, reg);
	reg = ADM8211_CSR_READ(ABDA1);
	reg &= 0x0000ffff;
	reg |= (bssid[4] << 16) | (bssid[5] << 24);
	ADM8211_CSR_WRITE(ABDA1, reg);

	return 0;
}

static int adm8211_set_ssid(struct net_device *dev, u8 *ssid, size_t ssid_len)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	u8 buf[36];

	if (ssid_len > 32)
		return -EINVAL;

	memset(buf, 0, sizeof(buf));
	buf[0] = ssid_len;
	memcpy(buf + 1, ssid, ssid_len);
	adm8211_write_sram_bytes(dev, ADM8211_SRAM_SSID, buf, 33);
	adm8211_set_beacon(dev);

	return 0;
}

void adm8211_set_beacon (struct net_device *dev) {
	struct adm8211_priv *priv = netdev_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;
	unsigned int blen, len, rem;
	u32 reg;

	if (priv->iw_mode == IW_MODE_ADHOC)
		blen = 24 +
			8 + 2 + 2 + 2 + data->ssid_len + 2 + data->num_supp_rates +
			3 + 4 + WLAN_FCS_LEN;
	else
		blen = 0;

	if (priv->revid < ADM8211_REV_BA) {
		/* 11M PLCP length */
		blen *= 8;
		rem   = blen % 11;
		len   = blen / 11;
		if (rem) {
			len += 1;
			if (rem <= 3)
				len |= 1<<7;
		}
		reg = len << 16;

		/* 5.5M PLCP length */
		rem   = (blen*2) % 11;
		len   = (blen*2) / 11;
		if (rem)
			len += 1;
		reg |= len << 8;

		reg |= blen;

		ADM8211_CSR_WRITE(BCNT, reg);
	} else {
		len = blen;
		rem = (blen*80) % priv->plcp_signal;
		len = (blen*80) / priv->plcp_signal;
		if (rem) {
			len += 1;
			if (data->rate == 0x16 && ((blen*8) % 11) <= 3)
				len |= 1<<15;
		}

		len &= 0xFFFF;

		ADM8211_CSR_WRITE(BCNT, len);

		reg = ADM8211_CSR_READ(MMIRD1);
		reg &= 0xFFFF0000;
		reg |= len;
		ADM8211_CSR_WRITE(MMIRD1, reg);
	}
}

static void adm8211_init_rings(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	int i;
	struct adm8211_desc *desc;
	struct adm8211_ring_info *info;

	for (i = 0; i < RX_RING_SIZE; i++) {
		desc = &priv->rx_ring[i];
		desc->status = 0;
		desc->length = cpu_to_le32(RX_PKT_SIZE);
		priv->rx_buffers[i].skb = NULL;
	}
	/* Mark the end of RX ring; hw returns to base address after this
	 * descriptor */
	desc->length |= cpu_to_le32(RDES1_CONTROL_RER);


	for (i = 0; i < RX_RING_SIZE; i++) {
		desc = &priv->rx_ring[i];
		info = &priv->rx_buffers[i];

		info->skb = dev_alloc_skb(RX_PKT_SIZE);
		if (info->skb == NULL)
			break;
		info->mapping = pci_map_single(priv->pdev, info->skb->tail,
					       RX_PKT_SIZE,
					       PCI_DMA_FROMDEVICE);
		info->skb->dev = dev;
		desc->buffer1 = cpu_to_le32(info->mapping);
		desc->status = cpu_to_le32( RDES0_STATUS_OWN | RDES0_STATUS_SQL );
	}

	/* Setup TX ring. TX buffers descriptors will be filled in as needed */
	for (i = 0; i < TX_RING_SIZE; i++) {
		desc = &priv->tx_ring[i];
		info = &priv->tx_buffers[i];

		info->skb = NULL;
		info->mapping = 0;
		desc->status = 0;
	}
	desc->length = cpu_to_le32(TDES1_CONTROL_TER);

	priv->cur_rx = priv->cur_tx = priv->dirty_tx = 0;
	ADM8211_CSR_WRITE(RDB, priv->rx_ring_dma);
	ADM8211_CSR_WRITE(TDBD, priv->tx_ring_dma);
}

static void adm8211_free_rings(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	int i;

	for (i = 0; i < RX_RING_SIZE; i++) {
		if (!priv->rx_buffers[i].skb)
			continue;

		pci_unmap_single(
			priv->pdev,
			priv->rx_buffers[i].mapping,
			RX_PKT_SIZE, PCI_DMA_FROMDEVICE);

		dev_kfree_skb(priv->rx_buffers[i].skb);
	}

	for (i = 0; i < TX_RING_SIZE; i++) {
		if (!priv->tx_buffers[i].skb)
			continue;

		pci_unmap_single(
			priv->pdev,
			priv->tx_buffers[i].mapping,
			priv->tx_buffers[i].skb->len, PCI_DMA_TODEVICE);

		dev_kfree_skb(priv->tx_buffers[i].skb);
	}
}

static int adm8211_open(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	int retval;

	/* Power up MAC and RF chips */
	retval = adm8211_hw_reset(dev);
	if (retval) {
		printk(KERN_ERR "%s: hardware reset failed\n", dev->name);
		goto fail;
	}

	adm8211_init_rings(dev);

	/* Init hardware */
	adm8211_hw_init(dev);
	adm8211_rf_set_channel(dev, max(priv->ieee80211.pref_channel,
					priv->ieee80211.chan_range.min));

	/* set mac address */
	ADM8211_CSR_WRITE(PAR0, *(u32 *)dev->dev_addr);
	ADM8211_CSR_WRITE(PAR1, *(u16 *)(dev->dev_addr + 4));

	adm8211_set_rx_mode(dev);

	retval = request_irq(dev->irq, &adm8211_interrupt,
			     SA_SHIRQ, dev->name, dev);
	if (retval) {
		printk(KERN_ERR "%s: failed to register IRQ handler\n",
		       dev->name);
		goto fail;
	}

	ADM8211_CSR_WRITE(IER, ADM8211_INTMASK);
	ADM8211_CSR_READ(IER);

	adm8211_update_mode(dev);

	/* should be more than enough time for the card to settle down */
	mdelay(200);

	ieee80211_start(&priv->ieee80211);

	netif_start_queue(dev);
	return 0;

fail:
	return retval;
}


static int adm8211_stop(struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	del_singleshot_timer_sync(&priv->timer);

	ieee80211_stop(&priv->ieee80211);

	netif_stop_queue(dev);

	ADM8211_CSR_WRITE(IER, 0);
	priv->nar &= ~(ADM8211_NAR_ST | ADM8211_NAR_PR | ADM8211_NAR_PB |
		       ADM8211_NAR_SR | ADM8211_NAR_MM );
	ADM8211_CSR_WRITE(NAR, priv->nar);
	ADM8211_CSR_READ(NAR);

	free_irq(dev->irq, dev);

	adm8211_free_rings(dev);

	skb_queue_purge(&priv->rx_queue);

	adm8211_hw_reset(dev);
	return 0;
}

static void adm8211_timer(unsigned long ptr)
{
	struct net_device *dev = (struct net_device *) ptr;
	struct adm8211_priv *priv = netdev_priv(dev);
	u32 reg;
	u16 rra, rwa;

	if (priv->revid != ADM8211_REV_BA ||
	   !(dev->flags & IFF_UP))
		return;

	/* checks for stalls on adm8211b */
	reg = ADM8211_CSR_READ(CSR_TEST1);
	rra = (reg >> 12) & 0x1FF;
	rwa = (reg >> 2 ) & 0x1FF;

	if ( (rra != rwa) && !(reg & (1<<1)) ) {
		printk(KERN_DEBUG "%s: stalled. resetting card\n", dev->name);
		adm8211_stop(dev);
		adm8211_open(dev);
	}
}

static void adm8211_calc_durations(int *dur, int *plcp, size_t payload_len, int len,
				   int plcp_signal, int short_preamble)
{
	/* Alternative calculation from NetBSD: */

/* IEEE 802.11b durations for DSSS PHY in microseconds */
#define IEEE80211_DUR_DS_LONG_PREAMBLE	144
#define IEEE80211_DUR_DS_SHORT_PREAMBLE	72
#define IEEE80211_DUR_DS_FAST_PLCPHDR	24
#define IEEE80211_DUR_DS_SLOW_PLCPHDR	48
#define IEEE80211_DUR_DS_SLOW_ACK	112
#define IEEE80211_DUR_DS_FAST_ACK	56
#define IEEE80211_DUR_DS_SLOW_CTS	112
#define IEEE80211_DUR_DS_FAST_CTS	56
#define IEEE80211_DUR_DS_SLOT		20
#define IEEE80211_DUR_DS_SIFS		10

	int remainder;

	*dur = (80 * (WLAN_ADDR3_HDR_LEN + payload_len) + plcp_signal - 1)
		/ plcp_signal;

	if (plcp_signal <= PLCP_SIGNAL_2M) {
		/* 1-2Mbps WLAN: send ACK/CTS at 1Mbps */
		*dur += 3 * (IEEE80211_DUR_DS_SIFS +
			     IEEE80211_DUR_DS_SHORT_PREAMBLE +
			     IEEE80211_DUR_DS_FAST_PLCPHDR) +
			     IEEE80211_DUR_DS_SLOW_CTS + IEEE80211_DUR_DS_SLOW_ACK;
	} else {
		/* 5-11Mbps WLAN: send ACK/CTS at 2Mbps */
		*dur += 3 * (IEEE80211_DUR_DS_SIFS +
			     IEEE80211_DUR_DS_SHORT_PREAMBLE +
			     IEEE80211_DUR_DS_FAST_PLCPHDR) +
			     IEEE80211_DUR_DS_FAST_CTS + IEEE80211_DUR_DS_FAST_ACK;
	}

	/* lengthen duration if long preamble */
	if (!short_preamble) {
		*dur +=
			3 * (IEEE80211_DUR_DS_LONG_PREAMBLE -
			     IEEE80211_DUR_DS_SHORT_PREAMBLE) +
			3 * (IEEE80211_DUR_DS_SLOW_PLCPHDR -
			     IEEE80211_DUR_DS_FAST_PLCPHDR);
	}


	*plcp = (80 * len) / plcp_signal;
	remainder = (80 * len) % plcp_signal;
	if (plcp_signal == PLCP_SIGNAL_11M &&
	    remainder <= 30 && remainder > 0) {
		*plcp = (*plcp | 0x8000) + 1;
	} else if (remainder)
		(*plcp)++;
}

/* TODO: add addr4 to this */
/* Transmit skb (802.11 header created by hardware) */
static int adm8211_tx(struct net_device *dev, struct sk_buff *skb,
		      __le16 fc, int short_preamble, u8 *dst)
{
	struct adm8211_priv *priv = netdev_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;
	struct adm8211_tx_hdr *txhdr;
	int entry;
	dma_addr_t mapping;
	u32 flag;
	unsigned long flags;
	u16 fc_;
	size_t payload_len;
	int plcp, dur, len;

	int plcp_signal;

	fc_ = le16_to_cpu(fc);
	if (WLAN_FC_GET_TYPE(fc_) != WLAN_FC_TYPE_DATA) {
		if (fc_ & WLAN_FC_ISWEP && wep_encrypt(skb, 0, data)) {
			printk(KERN_DEBUG "%s: failed to WEP encrypt frame\n", dev->name);
			dev_kfree_skb(skb);
			return 0;
		}
		fc_ &= ~WLAN_FC_ISWEP;
	}

	payload_len = skb->len;

	if (skb_headroom(skb) < sizeof(struct adm8211_tx_hdr)) {
		if (pskb_expand_head(skb, sizeof(struct adm8211_tx_hdr), 0,
				     GFP_ATOMIC)) {
			printk(KERN_DEBUG "%s: failed to allocate room for TX "
			       "header\n", dev->name);
			dev_kfree_skb(skb);
			return 0;
		}
	}

	plcp_signal = ieee80211_get_rate(&priv->ieee80211, dst) * 5;

	txhdr = (struct adm8211_tx_hdr *) skb_push(skb, sizeof(*txhdr));
	memset(txhdr, 0, sizeof(*txhdr));
	memcpy(txhdr->da, dst, ETH_ALEN);
	txhdr->signal = plcp_signal;
	txhdr->frame_body_size = cpu_to_le16(payload_len);
	txhdr->frame_control = fc;

	len = WLAN_ADDR3_HDR_LEN + payload_len + WLAN_FCS_LEN;
	if (fc_ & WLAN_FC_ISWEP)
		len += 8;

	if (len > priv->frag_thresh && is_valid_ether_addr(dst)) {
	txhdr->frag_threshold = cpu_to_le16(priv->frag_thresh);

	len = WLAN_ADDR3_HDR_LEN + priv->frag_thresh + WLAN_FCS_LEN;
	if (fc_ & WLAN_FC_ISWEP)
		len += 8;
	adm8211_calc_durations(&dur, &plcp, priv->frag_thresh,
				len, plcp_signal, short_preamble);
	txhdr->plcp_frag_head_len = cpu_to_le16(plcp);
	txhdr->dur_frag_head = cpu_to_le16(dur);

	if (payload_len % priv->frag_thresh) {
		len = WLAN_ADDR3_HDR_LEN +
		      (payload_len % priv->frag_thresh) + WLAN_FCS_LEN;
		if (fc_ & WLAN_FC_ISWEP)
			len += 8;

		adm8211_calc_durations(&dur, &plcp, payload_len % priv->frag_thresh,
				       len, plcp_signal, short_preamble);

		txhdr->plcp_frag_tail_len = cpu_to_le16(plcp);
		txhdr->dur_frag_tail = cpu_to_le16(dur);
	} else {
		txhdr->plcp_frag_tail_len = txhdr->plcp_frag_head_len;
		txhdr->dur_frag_tail = txhdr->dur_frag_head;
	}

	} else {
		txhdr->frag_threshold = cpu_to_le16(0x0FFF);
		adm8211_calc_durations(&dur, &plcp, payload_len,
				       len, plcp_signal, short_preamble);
		txhdr->plcp_frag_head_len = cpu_to_le16(plcp);
		txhdr->plcp_frag_tail_len = cpu_to_le16(plcp);
		txhdr->dur_frag_head = cpu_to_le16(dur);
		txhdr->dur_frag_tail = cpu_to_le16(dur);
	}

	txhdr->header_control = cpu_to_le16(ADM8211_TXHDRCTL_ENABLE_EXTEND_HEADER);

	if (short_preamble)
		txhdr->header_control |= cpu_to_le16(ADM8211_TXHDRCTL_SHORT_PREAMBLE);

	if ((payload_len + (fc_ & WLAN_FC_ISWEP ? 8 : 0)) > priv->rts_thresh
	    && is_valid_ether_addr(dst))
		txhdr->header_control |= cpu_to_le16(ADM8211_TXHDRCTL_ENABLE_RTS);

	if (fc_ & WLAN_FC_ISWEP)
		txhdr->header_control |= cpu_to_le16(ADM8211_TXHDRCTL_ENABLE_WEP_ENGINE);

	txhdr->retry_limit = priv->retry_limit;
	mapping = pci_map_single(priv->pdev, skb->data, skb->len,
				 PCI_DMA_TODEVICE);

	spin_lock_irqsave(&priv->lock, flags);

	entry = priv->cur_tx % TX_RING_SIZE;

	priv->tx_buffers[entry].skb = skb;
	priv->tx_buffers[entry].mapping = mapping;
	priv->tx_ring[entry].buffer1 = cpu_to_le32(mapping);

	if (priv->cur_tx - priv->dirty_tx < TX_RING_SIZE / 2)
		flag = TDES1_CONTROL_LS | TDES1_CONTROL_FS;
	else if (priv->cur_tx - priv->dirty_tx == TX_RING_SIZE / 2)
		flag = TDES1_CONTROL_IC | TDES1_CONTROL_LS | TDES1_CONTROL_FS;
	else if (priv->cur_tx - priv->dirty_tx < TX_RING_SIZE - 2)
		flag = TDES1_CONTROL_LS | TDES1_CONTROL_FS;
	else {
		flag = TDES1_CONTROL_IC | TDES1_CONTROL_LS | TDES1_CONTROL_FS;
		netif_stop_queue(dev);
	}
	if (entry == TX_RING_SIZE - 1)
		flag |= TDES1_CONTROL_TER;
	priv->tx_ring[entry].length = cpu_to_le32(flag | skb->len);

	/* Set TX rate (SIGNAL field in PLCP PPDU format) */
	flag = TDES0_CONTROL_OWN | (plcp_signal << 20) | 8 /* ? */;
	priv->tx_ring[entry].status = cpu_to_le32(flag);

	priv->cur_tx++;

	spin_unlock_irqrestore(&priv->lock, flags);

	/* Trigger transmit poll */
	ADM8211_CSR_WRITE(TDR, 0);

	dev->trans_start = jiffies;

	return 0;
}


static int adm8211_hard_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct adm8211_priv *priv = netdev_priv(dev);

	return ieee80211_data_tx(&priv->ieee80211, skb);
}


int adm8211_80211_header_parse(struct sk_buff *skb, unsigned char *haddr)
{
	memcpy(haddr, skb->mac.raw + 10 + sizeof(struct avs_caphdr), ETH_ALEN); /* addr2 */
	return ETH_ALEN;
}

static int __devinit adm8211_probe(struct pci_dev *pdev,
				   const struct pci_device_id *id)
{
	struct net_device *dev;
	struct adm8211_priv *priv;
	unsigned long mem_addr, mem_len;
	unsigned int io_addr, io_len;
	unsigned int ring_size;
	int err;
	u32 reg;

#ifndef MODULE
	static unsigned int cardidx;
	if (!cardidx++) {
		printk(version);
		printk(KERN_INFO "adm8211: release " RELEASE_DATE "\n");
	}
#endif

	err = pci_enable_device(pdev);
	if (err) {
		printk(KERN_ERR "%s (adm8211): Cannot enable new PCI device\n", pci_name(pdev));
		return err;
	}

	io_addr = pci_resource_start(pdev, 0);
	io_len = pci_resource_len(pdev, 0);
	mem_addr = pci_resource_start(pdev, 1);
	mem_len = pci_resource_len(pdev, 1);
	if (io_len < 256 || mem_len < 1024) {
		printk(KERN_ERR "%s (adm8211): Too short PCI resources\n", pci_name(pdev));
		goto err_disable_pdev;
	}


	/* check signature */
	pci_read_config_dword(pdev, 0x80 /* CR32 */, &reg);
	if (reg != ADM8211_SIG1 && reg != ADM8211_SIG2) {
		printk(KERN_ERR "%s (adm8211): Invalid signature (0x%x)\n", pci_name(pdev), reg);
		goto err_disable_pdev;
	}

	err = pci_request_regions(pdev, "adm8211");
	if (err) {
		printk(KERN_ERR "%s (adm8211): Cannot obtain PCI resources\n", pci_name(pdev));
		goto err_disable_pdev;
	}

	if (pci_set_dma_mask(pdev, DMA_32BIT_MASK)) {
		printk(KERN_ERR "%s (adm8211): No suitable DMA available\n", pci_name(pdev));
		goto err_free_reg;
	}
	pci_set_consistent_dma_mask(pdev, DMA_32BIT_MASK);

	pci_set_master(pdev);

	dev = alloc_etherdev(sizeof(struct adm8211_priv));
	if (!dev) {
		printk(KERN_ERR "%s (adm8211): Etherdev alloc failed\n", pci_name(pdev));
		err = -ENOMEM;
		goto err_free_reg;
	}
	priv = netdev_priv(dev);
	priv->pdev = pdev;

	spin_lock_init(&priv->lock);

	SET_MODULE_OWNER(dev);
	SET_NETDEV_DEV(dev, &pdev->dev);

	pci_set_drvdata(pdev, dev);

	priv->map = pci_iomap(pdev, 1, mem_len);
	if (!priv->map)
		priv->map = pci_iomap(pdev, 0, io_len);

	if (!priv->map) {
		printk(KERN_ERR "%s (adm8211): Cannot map device memory\n", pci_name(pdev));
		goto err_free_dev;
	}

	dev->irq = pdev->irq;

	/* Allocate TX/RX descriptors */
	ring_size = sizeof(struct adm8211_desc) * RX_RING_SIZE +
		sizeof(struct adm8211_desc) * TX_RING_SIZE;
	priv->rx_ring = pci_alloc_consistent(pdev, ring_size,
					     &priv->rx_ring_dma);
	if (!priv->rx_ring) {
		printk(KERN_ERR "%s (adm8211): Cannot allocate TX/RX ring\n", pci_name(pdev));
		goto err_iounmap;
	}
	priv->tx_ring = (struct adm8211_desc *) (priv->rx_ring + RX_RING_SIZE);
	priv->tx_ring_dma = priv->rx_ring_dma +
		sizeof(struct adm8211_desc) * RX_RING_SIZE;

	skb_queue_head_init(&priv->rx_queue);
	priv->rx_tasklet.func = adm8211_rx_tasklet;
	priv->rx_tasklet.data = (unsigned long) dev;

	pci_read_config_byte(pdev, PCI_CLASS_REVISION, &priv->revid);

	/* Power-on issue */
	ADM8211_CSR_WRITE(FRCTL, 0);
	ADM8211_CSR_READ(FRCTL);
	ADM8211_CSR_WRITE(FRCTL, 1);
	ADM8211_CSR_READ(FRCTL);
	mdelay(100);

	/* Clear the missed-packet counter. */
	ADM8211_CSR_READ(LPC);

	put_unaligned(ADM8211_CSR_READ(PAR0), (u32 *) dev->dev_addr);
	put_unaligned(ADM8211_CSR_READ(PAR1) & (__force u32) __constant_cpu_to_le32(0xffff),
		      (u16 *) &dev->dev_addr[4]);

	if (!is_valid_ether_addr(dev->dev_addr)) {
		printk(KERN_WARNING "%s (adm8211): Invalid hwaddr! Using randomly generated hwaddr\n", pci_name(pdev));
		random_ether_addr(dev->dev_addr);
	}

	//dev->features |= NETIF_F_LLTX;
	dev->open = adm8211_open;
	dev->stop = adm8211_stop;
	dev->get_stats = adm8211_get_stats;
	dev->set_multicast_list = adm8211_set_rx_mode;
	dev->set_mac_address = adm8211_set_mac_address;
	dev->hard_start_xmit = adm8211_hard_start_xmit;

	dev->get_wireless_stats = adm8211_get_wireless_stats;
	dev->wireless_handlers =
		(struct iw_handler_def *) &adm8211_iw_handler_def;

	init_timer(&priv->timer);
	priv->timer.data = (unsigned long) dev;
	priv->timer.function = adm8211_timer;

	priv->iw_mode = IW_MODE_INFRA;

	priv->ieee80211.dev = dev;
	priv->ieee80211.set_channel = adm8211_rf_set_channel;
	priv->ieee80211.set_bssid = adm8211_set_bssid;
	priv->ieee80211.set_ssid = adm8211_set_ssid;
	priv->ieee80211.tx = adm8211_tx;
	priv->ieee80211.set_interval = adm8211_set_interval;
	priv->ieee80211.set_tbtt = adm8211_set_tbtt;
	priv->ieee80211.get_tsft = adm8211_get_tsft;
	priv->ieee80211.link_change = adm8211_link_change;
	priv->ieee80211.set_rate = adm8211_set_rate;
	priv->ieee80211.supp_rates = (u8 *)IEEE80211_B_RATES;
	priv->ieee80211.capab = WLAN_CAPABILITY_SHORT_PREAMBLE;
	if (ieee80211_init(&priv->ieee80211)) {
		printk(KERN_ERR "%s (adm8211): Cannot load ARC4 cipher\n", pci_name(pdev));
		goto err_free_desc;
	}

	priv->rts_thresh = 2347;
	priv->frag_thresh = 2346;
	priv->plcp_signal = priv->ieee80211.rate * 5;
	priv->retry_limit = 3;
	priv->ant_power = 0x40;
	priv->tx_power = 0x40;
	priv->lpf_cutoff = 0xFF;
	priv->lnags_threshold = 0xFF;

	err = adm8211_read_eeprom(dev);
	if (err) {
		printk(KERN_ERR "%s (adm8211): Cannot allocate eeprom buffer\n", pci_name(pdev));
		goto err_free_desc;
	}

	err = register_netdev(dev);
	if (err) {
		printk(KERN_ERR "%s (adm8211): Cannot register netdevice\n", pci_name(pdev));
		goto err_free_desc;
	}

	printk("%s: hwaddr %02x:%02x:%02x:%02x:%02x:%02x, IRQ %d, Rev 0x%02x\n",
	       dev->name, dev->dev_addr[0], dev->dev_addr[1], dev->dev_addr[2],
	       dev->dev_addr[3], dev->dev_addr[4], dev->dev_addr[5],
	       dev->irq, priv->revid);

	return 0;

 err_free_desc:
	pci_free_consistent(pdev, ring_size, priv->rx_ring, priv->rx_ring_dma);

 err_iounmap:
	pci_iounmap(pdev, priv->map);

 err_free_dev:
	pci_set_drvdata(pdev, NULL);
	free_netdev(dev);

 err_free_reg:
	pci_release_regions(pdev);

 err_disable_pdev:
	pci_disable_device(pdev);
	return err;
}


static void __devexit adm8211_remove(struct pci_dev *pdev)
{
	struct net_device *dev = pci_get_drvdata(pdev);
	struct adm8211_priv *priv;

	if (!dev)
		return;

	unregister_netdev(dev);

	priv = netdev_priv(dev);
	ieee80211_deinit(&priv->ieee80211);

	pci_free_consistent(pdev,
			    sizeof(struct adm8211_desc) * RX_RING_SIZE +
			    sizeof(struct adm8211_desc) * TX_RING_SIZE,
			    priv->rx_ring, priv->rx_ring_dma);

	kfree(priv->eeprom);
	pci_iounmap(pdev, priv->map);
	pci_release_regions(pdev);
	pci_disable_device(pdev);
	free_netdev(dev);
}


#ifdef CONFIG_PM
static int adm8211_suspend(struct pci_dev *pdev, pm_message_t state)
{
	struct net_device *dev = pci_get_drvdata(pdev);
	netif_device_detach(dev);

	if (dev->flags & IFF_UP)
		dev->stop(dev);

	pci_save_state(pdev);
	pci_set_power_state(pdev, 3);
	return 0;
}

static int adm8211_resume(struct pci_dev *pdev)
{
	struct net_device *dev = pci_get_drvdata(pdev);
	pci_set_power_state(pdev, 0);
	pci_restore_state(pdev);

	if (dev->flags & IFF_UP)
		dev->open(dev);

	netif_device_attach(dev);
	return 0;
}
#endif /* CONFIG_PM */


MODULE_DEVICE_TABLE(pci, adm8211_pci_id_table);

/* TODO: enable_wake */
static struct pci_driver adm8211_driver = {
	.name		= "adm8211",
	.id_table	= adm8211_pci_id_table,
	.probe		= adm8211_probe,
	.remove		= __devexit_p(adm8211_remove),
#ifdef CONFIG_PM
	.suspend	= adm8211_suspend,
	.resume		= adm8211_resume,
#endif /* CONFIG_PM */
};



static int __init adm8211_init(void)
{
#ifdef MODULE
	printk(version);
	printk(KERN_INFO "adm8211: release " RELEASE_DATE "\n");
#endif

	return pci_register_driver(&adm8211_driver);
}


static void __exit adm8211_exit(void)
{
	pci_unregister_driver(&adm8211_driver);

	printk(KERN_INFO "adm8211: Driver unloaded\n");
}


module_init(adm8211_init);
module_exit(adm8211_exit);
