/*
 * Mac Mini "toonie" mixer control
 *
 * Copyright (c) 2005 by Benjamin Herrenschmidt <benh@kernel.crashing.org>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <sound/driver.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/i2c-dev.h>
#include <linux/kmod.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <sound/core.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/machdep.h>
#include <asm/pmac_feature.h>

#include "pmac.h"
#include "snd-pmac-gpio.h"

#undef DEBUG

#ifdef DEBUG
#define DBG(fmt...) printk(fmt)
#else
#define DBG(fmt...)
#endif

struct pmac_toonie
{
	snd_pmac_gpio_t		hp_detect_gpio;
	snd_pmac_gpio_t		hp_mute_gpio;
	snd_pmac_gpio_t		amp_mute_gpio;

	int			auto_mute_notify;
	struct work_struct	detect_work;
};


static void toonie_reset_audio(pmac_t *chip)
{
	snd_pmac_gpio_t gpio;

	if (snd_pmac_get_gpio("audio-hw-reset", "hw-reset", &gpio))
		return;

	DBG("(I) codec normal reset !\n");

	snd_pmac_write_gpio(&gpio, 0);
	msleep(200);
	snd_pmac_write_gpio(&gpio, 1);
	msleep(100);
	snd_pmac_write_gpio(&gpio, 0);
	msleep(100);

	snd_pmac_free_gpio(&gpio);
}


enum { TOONIE_MUTE_HP, TOONIE_MUTE_AMP };

static int toonie_get_mute_switch(snd_kcontrol_t *kcontrol,
				  snd_ctl_elem_value_t *ucontrol)
{
	pmac_t *chip = snd_kcontrol_chip(kcontrol);
	struct pmac_toonie *mix = chip->mixer_data;
	snd_pmac_gpio_t *gp;

	if (mix == NULL)
		return -ENODEV;

	switch(kcontrol->private_value) {
	case TOONIE_MUTE_HP:
		gp = &mix->hp_mute_gpio;
		break;
	case TOONIE_MUTE_AMP:
		gp = &mix->amp_mute_gpio;
		break;
	default:
		return -EINVAL;;
	}
	ucontrol->value.integer.value[0] = !snd_pmac_gpio_internal_state(gp);
	return 0;
}

static int toonie_put_mute_switch(snd_kcontrol_t *kcontrol,
				   snd_ctl_elem_value_t *ucontrol)
{
	pmac_t *chip = snd_kcontrol_chip(kcontrol);
	struct pmac_toonie *mix = chip->mixer_data;
	snd_pmac_gpio_t *gp;

	if (chip->update_automute && chip->auto_mute)
		return 0; /* don't touch in the auto-mute mode */

	if (mix == NULL)
		return -ENODEV;

	switch(kcontrol->private_value) {
	case TOONIE_MUTE_HP:
		gp = &mix->hp_mute_gpio;
		break;
	case TOONIE_MUTE_AMP:
		gp = &mix->amp_mute_gpio;
		break;
	default:
		return -EINVAL;;
	}
	if (!snd_pmac_gpio_internal_state(gp) != ucontrol->value.integer.value[0]) {
		snd_pmac_write_gpio(gp, ! ucontrol->value.integer.value[0]);
		return 1;
	}
	return 0;
}

static snd_kcontrol_new_t toonie_hp_sw __initdata = {
	.iface = SNDRV_CTL_ELEM_IFACE_MIXER,
	.name = "Headphone Playback Switch",
	.info = snd_pmac_boolean_mono_info,
	.get = toonie_get_mute_switch,
	.put = toonie_put_mute_switch,
	.private_value = TOONIE_MUTE_HP,
};
static snd_kcontrol_new_t toonie_speaker_sw __initdata = {
	.iface = SNDRV_CTL_ELEM_IFACE_MIXER,
	.name = "PC Speaker Playback Switch",
	.info = snd_pmac_boolean_mono_info,
	.get = toonie_get_mute_switch,
	.put = toonie_put_mute_switch,
	.private_value = TOONIE_MUTE_AMP,
};

/*
 * auto-mute stuffs
 */
static int toonie_detect_headphone(pmac_t *chip)
{
	struct pmac_toonie *mix = chip->mixer_data;
	int detect = 0;

	snd_pmac_read_gpio(&mix->hp_detect_gpio, &detect);

	return detect;
}

static int toonie_do_mute(pmac_t *chip, snd_pmac_gpio_t *gp, int val,
			   int do_notify, snd_kcontrol_t *sw)
{
	snd_pmac_write_gpio(gp, val);

	if (do_notify)
		snd_ctl_notify(chip->card, SNDRV_CTL_EVENT_MASK_VALUE,
			       &sw->id);

	return val;
}

static void toonie_detect_handler(void *self)
{
	pmac_t *chip = (pmac_t*) self;
	struct pmac_toonie *mix;
	int headphone;

	if (!chip)
		return;

	mix = chip->mixer_data;
	snd_assert(mix, return);

	headphone = toonie_detect_headphone(chip);

	DBG("headphone: %d\n", headphone);

	if (headphone) {
		/* unmute headphone/lineout & mute speaker */
		toonie_do_mute(chip, &mix->hp_mute_gpio, 0,
			       mix->auto_mute_notify, chip->master_sw_ctl);
		toonie_do_mute(chip, &mix->amp_mute_gpio, 1,
			       mix->auto_mute_notify, chip->speaker_sw_ctl);
	} else {
		/* unmute speaker, mute others */
		toonie_do_mute(chip, &mix->amp_mute_gpio, 0,
			       mix->auto_mute_notify, chip->speaker_sw_ctl);
		toonie_do_mute(chip, &mix->hp_mute_gpio, 1,
			       mix->auto_mute_notify, chip->master_sw_ctl);
	}
	if (mix->auto_mute_notify) {
		snd_ctl_notify(chip->card, SNDRV_CTL_EVENT_MASK_VALUE,
				       &chip->hp_detect_ctl->id);
	}
}

static void toonie_update_automute(pmac_t *chip, int do_notify)
{
	if (chip->auto_mute) {
		struct pmac_toonie *mix;
		mix = chip->mixer_data;
		snd_assert(mix, return);
		mix->auto_mute_notify = do_notify;
		schedule_work(&mix->detect_work);
	}
}

/* interrupt - headphone plug changed */
static void toonie_hp_intr(void *devid)
{
	pmac_t *chip = devid;

	if (chip->update_automute && chip->initialized)
		chip->update_automute(chip, 1);
}

static void toonie_cleanup(pmac_t *chip)
{
	struct pmac_toonie *mix = chip->mixer_data;
	if (! mix)
		return;

	snd_pmac_free_gpio(&mix->hp_mute_gpio);
	snd_pmac_free_gpio(&mix->amp_mute_gpio);
	snd_pmac_free_gpio(&mix->hp_detect_gpio);

	kfree(mix);
	chip->mixer_data = NULL;
}

int snd_pmac_toonie_init(pmac_t *chip)
{
	struct pmac_toonie *mix;

	mix = kzalloc(sizeof(*mix), GFP_KERNEL);
	if (! mix)
		return -ENOMEM;

	chip->mixer_data = mix;
	chip->mixer_free = toonie_cleanup;

	/* Atleast have to have these two */
	if (snd_pmac_get_gpio("headphone-mute", "lineout-mute",
			      &mix->hp_mute_gpio) ||
	    snd_pmac_get_gpio("amp-mute", NULL,
			      &mix->amp_mute_gpio)) {
		return -ENODEV;
	}

	strcpy(chip->card->mixername, "PowerMac Toonie");

	chip->master_sw_ctl = snd_ctl_new1(&toonie_hp_sw, chip);
	snd_ctl_add(chip->card, chip->master_sw_ctl);

	chip->speaker_sw_ctl = snd_ctl_new1(&toonie_speaker_sw, chip);
	snd_ctl_add(chip->card, chip->speaker_sw_ctl);

	INIT_WORK(&mix->detect_work, toonie_detect_handler, (void *)chip);

	if (!snd_pmac_get_gpio("headphone-detect", "lineout-detect",
			       &mix->hp_detect_gpio)) {
		snd_pmac_add_automute(chip);

		chip->detect_headphone = toonie_detect_headphone;
		chip->update_automute = toonie_update_automute;
		toonie_update_automute(chip, 0);

		snd_pmac_request_irq(&mix->hp_detect_gpio, toonie_hp_intr, chip);
	}

	toonie_reset_audio(chip);

	return 0;
}
