/*
 * Line6 PODxt Pro USB driver - 0.5
 *
 * Copyright (C) 2004, 2005 Markus Grabner (grabner@icg.tu-graz.ac.at)
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation, version 2.
 *
 */

#ifndef DRIVER_H
#define DRIVER_H


#include <linux/spinlock.h>
#include <linux/usb.h>
#include <linux/wait.h>

#include <sound/driver.h>
#include <sound/core.h>


/*
	Sizes of URBs
*/
#define PODXTPRO_BUFSIZE_LISTEN   32
#define PODXTPRO_BUFSIZE_DUMPREQ   7

/*
	PODxt Pro endpoints
*/
#define PODXTPRO_EP_WRITE_AUDIO 0x01
#define PODXTPRO_EP_WRITE_MIDI  0x03
#define PODXTPRO_EP_READ_AUDIO  0x82
#define PODXTPRO_EP_READ_MIDI   0x84

/*
	PODxt Pro control commands
*/
#define PODXTPRO_PARAM_CHANGE   0xb0
#define PODXTPRO_PROGRAM_CHANGE 0xc0
#define PODXTPRO_SYSEX_BEGIN    0xf0
#define PODXTPRO_SYSEX_END      0xf7

#define PODXTPRO_OFFSET_ECHO 0
#define PODXTPRO_OFFSET_ORIG 2
#define PODXTPRO_OFFSET_UNKNOWN 5  /* don't know yet what this is good for */

/*
	Locate name in binary program dump
*/
#define	PODXTPRO_NAME_OFFSET 0
#define	PODXTPRO_NAME_LENGTH 16

/*
	Some atomic flags
*/
#define PODXTPRO_BUSY_READ     0
#define PODXTPRO_BUSY_WRITE    1
#define PODXTPRO_CHANNEL_DIRTY 2
#define PODXTPRO_SAVE_PRESSED  3
#define PODXTPRO_BUSY_MIDISEND 4

/*
	Other constants
*/
#define PODXTPRO_CONTROL_SIZE 0x80
#define PODXTPRO_MESSAGE_MAXLEN 0xb0
#define PODXTPRO_TIMEOUT 1
#define PODXTPRO_MAX_DEVICES 8
#define PODXTPRO_RAW_CHUNK_SIZE 32


/**
	 Data structure for values that need to be requested explicitly.
	 This is the case for system and tuner settings.
*/
struct ValueWait
{
	unsigned short value;
	wait_queue_head_t wait;
};

/**
	 Binary PodXT Pro program dump
*/
struct podxtpro_program {
	/**
		 Header information (including program name).
	*/
	unsigned char header[0x20];

	/**
		 Program parameters.
	*/
	unsigned char control[PODXTPRO_CONTROL_SIZE];
};

struct usb_podxtpro {
	/**
		 USB device.
	*/
	struct usb_device *usbdev;

	/**
		 Name of this device.
	*/
	const char *devname;

	/**
		 Device to send messages to (via dev_*).
	*/
	struct device *msgdev;

	/**
		 PODxt Pro sound card data structure.
	*/
	snd_card_t *card;

	/**
		 PODxt Pro PCM device data structure.
	*/
	struct snd_podxtpro_pcm *chip;

	/**
		 PODxt Pro MIDI device data structure.
	*/
	struct snd_podxtpro_midi *midi;

	/**
		 Current program number.
	*/
	unsigned char channel_num;

	/**
		 Current program settings.
	*/
	struct podxtpro_program prog_data;

	/**
		 Buffer for data retrieved from or to be stored on PODxt Pro.
	*/
	struct podxtpro_program prog_data_buf;

	/**
		 Flag to indicate modification of current program settings.
	*/
	int dirty;

	/**
		 URB for listening to PODxt Pro control endpoint.
	*/
	struct urb *urb_listen;

#if USE_CHARACTER_DEVICE
	/**
		 Status of character device;
		 <0:  not registered
		 >=0: successfully registered
	*/
	int chrdev;
#endif

	/**
		 Buffer for listening to PODxt Pro control endpoint.
	*/
	unsigned char *buffer_listen;

	/**
		 Buffer for program dumps.
	*/
	unsigned char *buffer_dumpreq;

	/**
		 Buffer for composition of messages.
	*/
	unsigned char *buffer_message;

	/**
		 Position in message composition buffer.
	*/
	int message_pos;
	
	/**
		 Length of message to be composed.
		 0:  no current message, determine length with next received byte
		 >0: total length of current message (if known in advance)
		 -1: length unknown (wait for sysex end)
	*/
	int message_length;

	/**
		 Wait queue for access to program dump data.
	*/
	wait_queue_head_t dump_wait;

	/**
		 Indicates an unfinished program dump request.
		 0: no dump
		 1: dump current settings
		 2: dump arbitrary program memory
	*/
	int dump_in_progress;

	/**
		 Wait queue for access to batch data.
	*/
	wait_queue_head_t batch_wait;

	/**
		 Tuner mute mode.
	*/
	struct ValueWait tuner_mute;

	/**
		 Tuner base frequency (typically 440Hz).
	*/
	struct ValueWait tuner_freq;

	/**
		 Note received from tuner.
	*/
	struct ValueWait tuner_note;

	/**
		 Pitch value received from tuner.
	*/
	struct ValueWait tuner_pitch;

	/**
		 Instrument monitor level.
	*/
	struct ValueWait monitor_level;

	/**
		 Audio routing mode.
		 0: send processed guitar
		 1: send clean guitar
		 2: send clean guitar re-amp playback
		 3: send re-amp playback
	*/
	struct ValueWait routing;

	/**
		 Dirty flags for access to parameter data.
	*/
	unsigned long param_dirty[PODXTPRO_CONTROL_SIZE / sizeof(unsigned long)];

#if USE_CHARACTER_DEVICE
	/**
		 Buffer for reading from character device.
	*/
	unsigned char chrdev_readbuf[9];

	/**
		 Buffer for writing to character device.
	*/
	unsigned char chrdev_writebuf[3];

	/**
		 Length of current message;
	*/
	short chrdev_msglen;

	/**
		 Current position in character device read buffer.
	*/
	short chrdev_readpos;

	/**
		 Current position in character device write buffer.
	*/
	short chrdev_writepos;
#endif

	/**
		 Some atomic flags.
	*/
	unsigned long atomic_flags;

	/**
		 Timer for delayed dump request.
	*/
	struct timer_list dumpreq_timer;

	/**
		 Flag if initial dump request has been successful.
	*/
	int dumpreq_ok;

	/**
		 Serial number of device.
	*/
	int serial_number;
};

typedef struct usb_podxtpro usb_podxtpro_t;


extern void podxtpro_invalidate_current(struct usb_podxtpro *podxtpro);
extern void podxtpro_set_parameter(struct usb_podxtpro *podxtpro, int param, int value);
extern void podxtpro_get_parameter(struct usb_podxtpro *podxtpro, int param);
extern int podxtpro_wait_dump(struct usb_podxtpro *podxtpro, int nonblock);
extern void podxtpro_write_hexdump(struct usb_podxtpro *podxtpro, char dir, const unsigned char *buffer, int size);


#endif
