/*
 * Copyright (c) 2004 Chris Kemp <ck231@cam.ac.uk>
 * Driver for the wacom tablet on the Compaq TC1100
 *
 * Based heavily on 8250_acpi.c :
 *   Copyright (c) 2002-2003 Matthew Wilcox for Hewlett-Packard
 *   Copyright (C) 2004 Hewlett-Packard Co
 *	  Bjorn Helgaas <bjorn.helgaas@hp.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <linux/acpi.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/tty.h>
#include <linux/serial.h>
#include <linux/tty.h>
#include <linux/serial_core.h>

#include <acpi/acpi_bus.h>

#include <asm/io.h>
#include <asm/serial.h>

struct wacom_private {
  int	line;
};

static acpi_status acpi_wacom_port(struct serial_struct *req,
				    struct acpi_resource_io *io)
{
  if (io->range_length) {
    //printk(KERN_ERR "Length: %d Base: %d \n", io->range_length, io->min_base_address);
    //printk(KERN_ERR "MaxBase: %d io_dec: %d \n", io->max_base_address, io->io_decode);
    //printk(KERN_ERR "align: %d \n", io->alignment);
    req->io_type = SERIAL_IO_PORT;
    req->port=io->min_base_address;
  } else
    printk(KERN_ERR "%s: zero-length IO port range?\n", __FUNCTION__);
  return AE_OK;
}

static acpi_status acpi_wacom_irq(struct serial_struct *req,
				   struct acpi_resource_irq *irq)
{
	if (irq->number_of_interrupts > 0) {
#ifdef CONFIG_IA64
		req->irq = acpi_register_irq(irq->interrupts[0],
	                  irq->active_high_low, irq->edge_level);
#else
		req->irq = irq->interrupts[0];
#endif
	}
	return AE_OK;
}

static acpi_status acpi_serial_resource(struct acpi_resource *res, void *data)
{
  struct serial_struct *serial_req = (struct serial_struct *) data;
  struct acpi_resource_address64 addr;
  acpi_status status;
  
  status = acpi_resource_to_address64(res, &addr);
  if (ACPI_SUCCESS(status))
    {
      printk(KERN_ERR "Wacom acpi setup: shouldn't get a possible memio in _PRS\n");
      return -ENODEV;
    }
  else if (res->id == ACPI_RSTYPE_IO)
    {
      return acpi_wacom_port(serial_req, &res->data.io);
    }
  else if (res->id == ACPI_RSTYPE_EXT_IRQ)
    {
      printk(KERN_ERR "Wacom acpi setup: shouldn't get a possible ext_irq in _PRS\n");
      return -ENODEV;
    }
  else if (res->id == ACPI_RSTYPE_IRQ)
    {
      return acpi_wacom_irq(serial_req, &res->data.irq);
    }
  return AE_OK;
}

static int acpi_wacom_add(struct acpi_device *device)
{
	struct wacom_private *priv;
	acpi_status status;
	struct serial_struct serial_req;
	int result;
	struct acpi_buffer prs={ACPI_ALLOCATE_BUFFER,NULL};
	void* new;

	memset(&serial_req, 0, sizeof(serial_req));
	
	priv = kmalloc(sizeof(struct wacom_private), GFP_KERNEL);
	if (!priv) {
		result = -ENOMEM;
		goto fail;
	}
	memset(priv, 0, sizeof(*priv));
	
	status = acpi_walk_resources(device->handle, METHOD_NAME__PRS,
				     acpi_serial_resource, &serial_req);
	if (ACPI_FAILURE(status)) {
	  result = -ENODEV;
	  goto fail;
	}
	
        if (!serial_req.port) {
	  printk(KERN_ERR "%s: no port address in %s _PRS\n",
		 __FUNCTION__, device->pnp.bus_id);
	  result = -ENODEV;
	  goto fail;
	}

	printk(KERN_ERR "%s: got port %d and irq %d\n", __FUNCTION__, serial_req.port, serial_req.irq);


	// Get the prs (again)
	status=acpi_get_possible_resources(device->handle, &prs);
	if (ACPI_FAILURE(status)) {
	  printk(KERN_ERR "%s: failed to get resources\n",__FUNCTION__);
	  result = -ENODEV;
	  goto fail;
	}
	
	// And set them
	// the acpi code does not decode properly but assumes locations
	// so we have to fix things...very ugly
	new=ACPI_NEXT_RESOURCE(((struct acpi_resource *)prs.pointer));
	prs.length=prs.length-(new-prs.pointer);
	prs.pointer=new;
	status=acpi_set_current_resources(device->handle, &prs);
	if (ACPI_FAILURE(status)) {
	  printk(KERN_ERR "%s: failed to set resources\n",__FUNCTION__);
	  result = -ENODEV;
	  goto fail;
	}	

	serial_req.baud_base = BASE_BAUD;
	serial_req.flags = UPF_SKIP_TEST | UPF_BOOT_AUTOCONF;
	
	priv->line = register_serial(&serial_req);
	if (priv->line < 0) {
	  printk(KERN_WARNING "Couldn't register wacom serial port %s: %d\n",
		 device->pnp.bus_id, priv->line);
	  result = -ENODEV;
	  goto fail;
	}

	acpi_driver_data(device) = priv;
	return 0;

fail:
	kfree(priv);

	return result;
}

static int acpi_wacom_remove(struct acpi_device *device, int type)
{
	struct wacom_private *priv;

	if (!device || !acpi_driver_data(device))
		return -EINVAL;

	priv = acpi_driver_data(device);
	unregister_serial(priv->line);
	kfree(priv);

	return 0;
}

static struct acpi_driver acpi_wacom_driver = {
  .name =		"wacom",
  .class =	"",
  .ids =		"WACF005",
  .ops =	
  {
    .add =		acpi_wacom_add,
    .remove =	acpi_wacom_remove,
  },
};

static int __init acpi_wacom_init(void)
{
  return acpi_bus_register_driver(&acpi_wacom_driver);
}

static void __exit acpi_wacom_exit(void)
{
  acpi_bus_unregister_driver(&acpi_wacom_driver);
}

module_init(acpi_wacom_init);
module_exit(acpi_wacom_exit);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Compaq TC1100 Wacom driver");
