/*
 * Realtime Capabilities Linux Security Module
 *
 *  Copyright (C) 2003 Torben Hohn
 *  Copyright (C) 2003, 2004 Jack O'Quin
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/security.h>
#include <linux/file.h>
#include <linux/mm.h>
#include <linux/mman.h>
#include <linux/pagemap.h>
#include <linux/swap.h>
#include <linux/smp_lock.h>
#include <linux/skbuff.h>
#include <linux/netlink.h>
#include <linux/ptrace.h>

#ifdef CONFIG_SECURITY

#include <linux/vermagic.h>
MODULE_INFO(vermagic,VERMAGIC_STRING);

/* module parameters */
static int any = 0;			/* if TRUE, any process is realtime */
module_param(any, uint, 0400);
MODULE_PARM_DESC(any, " grant realtime privileges to any process.");

static int gid = -1;			/* realtime group id, or NO_GROUP */
module_param(gid, uint, 0400);
MODULE_PARM_DESC(gid, " the group ID with access to realtime privileges.");

static int mlock = 1;			/* enable mlock() privileges */
module_param(mlock, uint, 0400);
MODULE_PARM_DESC(mlock, " enable memory locking privileges.");

static int allcaps = 0;			/* enable all capabilities */
module_param(allcaps, uint, 0400);
MODULE_PARM_DESC(allcaps, " enable all capabilities, including CAP_SETPCAP.");

static kernel_cap_t cap_bset_save;	/* place to save cap-bound */

int realtime_bprm_set_security(struct linux_binprm *bprm)
{
	/* Copied from security/commoncap.c: cap_bprm_set_security()... */
	/* Copied from fs/exec.c:prepare_binprm. */
	/* We don't have VFS support for capabilities yet */
	cap_clear(bprm->cap_inheritable);
	cap_clear(bprm->cap_permitted);
	cap_clear(bprm->cap_effective);

	/*  If a non-zero `any' parameter was specified, we grant
	 *  realtime privileges to every process.  If the `gid'
	 *  parameter was specified and it matches the group id of the
	 *  executable, of the current process or any supplementary
	 *  groups, we grant realtime capabilites.
	 */

	if (any || (gid != -1)) {

		int rt_ok = 1;

		/* check group permissions */
		if ((gid != -1) &&
		    (gid != bprm->e_gid) &&
		    (gid != current->gid)) {
			int i;
			rt_ok = 0;
#ifdef NGROUPS_SMALL			/* using new groups struct? */
			get_group_info(current->group_info);
			for (i = 0; i < current->group_info->ngroups; ++i) {
				if (gid == GROUP_AT(current->group_info, i)) {
 					rt_ok = 1;
 					break;
 				}
 			}
			put_group_info(current->group_info);
#else  /* old task struct */
			for (i = 0; i < NGROUPS; ++i) {
				if (gid == current->groups[i]) {
					rt_ok = 1;
					break;
				}
			}
#endif /* NGROUPS_SMALL */
		}

		if (rt_ok)  {
			cap_raise(bprm->cap_effective, CAP_SYS_NICE);
			cap_raise(bprm->cap_permitted, CAP_SYS_NICE);
			if (mlock) {
				cap_raise(bprm->cap_effective, CAP_IPC_LOCK);
				cap_raise(bprm->cap_permitted, CAP_IPC_LOCK);
				cap_raise(bprm->cap_effective,
					  CAP_SYS_RESOURCE);
				cap_raise(bprm->cap_permitted,
					  CAP_SYS_RESOURCE);
			}
		}
	}

	/*  To support inheritance of root-permissions and suid-root
	 *  executables under compatibility mode, we raise all three
	 *  capability sets for the file.
	 *
	 *  If only the real uid is 0, we only raise the inheritable
	 *  and permitted sets of the executable file.
	 */

	if (bprm->e_uid == 0 || current->uid == 0) {
		cap_set_full (bprm->cap_inheritable);
		cap_set_full (bprm->cap_permitted);
	}
	if (bprm->e_uid == 0)
		cap_set_full (bprm->cap_effective);

	return 0;
}

static struct security_operations capability_ops = {
	.ptrace =			cap_ptrace,
	.capget =			cap_capget,
	.capset_check =			cap_capset_check,
	.capset_set =			cap_capset_set,
	.capable =			cap_capable,
	.netlink_send =			cap_netlink_send,
	.netlink_recv =			cap_netlink_recv,

#ifdef LSM_UNSAFE_SHARE			/* version >= 2.6.6 */
	.bprm_apply_creds =		cap_bprm_apply_creds,
#else
	.bprm_compute_creds =		cap_bprm_compute_creds,
#endif
	.bprm_set_security =		realtime_bprm_set_security,
	.bprm_secureexec =		cap_bprm_secureexec,

	.task_post_setuid =		cap_task_post_setuid,
	.task_reparent_to_init =	cap_task_reparent_to_init,

	.syslog =                       cap_syslog,

	.vm_enough_memory =             cap_vm_enough_memory,
};

#define MY_NAME THIS_MODULE->name

/* flag to keep track of how we were registered */
static int secondary;


static int __init capability_init(void)
{
	/* register ourselves with the security framework */
	if (register_security(&capability_ops)) {

		/* try registering with primary module */
		if (mod_reg_security(MY_NAME, &capability_ops)) {
			printk(KERN_INFO "Failure registering capabilities "
			       "with primary security module.\n");
			printk(KERN_INFO "Realtime: is kernel configured "
			       "with CONFIG_SECURITY_CAPABILITIES=m?\n");
			return -EINVAL;
		}
		secondary = 1;
	}

	cap_bset_save = cap_bset;	/* save cap-bound */
	if (allcaps) {
		cap_bset = to_cap_t(~0);
		printk(KERN_INFO "Realtime LSM enabling all capabilities\n");
	}

	if (any)
		printk(KERN_INFO
		       "Realtime LSM initialized (all groups, mlock=%d)\n",
			mlock);
	else if (gid == -1)
		printk(KERN_INFO
		       "Realtime LSM initialized (no groups, mlock=%d)\n",
		       mlock);
	else
		printk(KERN_INFO
		       "Realtime LSM initialized (group %d, mlock=%d)\n",
		       gid, mlock);
		
	return 0;
}

static void __exit capability_exit(void)
{
	cap_bset = cap_bset_save;	/* restore cap-bound */

	/* remove ourselves from the security framework */
	if (secondary) {
		if (mod_unreg_security(MY_NAME, &capability_ops))
			printk(KERN_INFO "Failure unregistering capabilities "
				"with primary module.\n");

	} else if (unregister_security(&capability_ops)) {
		printk(KERN_INFO
		       "Failure unregistering capabilities with the kernel\n");
	}
	printk(KERN_INFO "Realtime Capability LSM exiting\n");
}

security_initcall(capability_init);
module_exit(capability_exit);

MODULE_DESCRIPTION("Realtime Capabilities Security Module");
MODULE_LICENSE("GPL");

#endif	/* CONFIG_SECURITY */
