/** -*- C++ -*-
 * @file cache/component/history.h
 * @author Peter Rockai <me@mornfall.net>
 */

#include <apt-front/cache/cache.h>
#include <apt-front/cache/component/base.h>
#include <apt-front/error.h>
#include <list>
#include <iostream>

#ifndef CACHE_COMPONENT_HISTORY_H
#define CACHE_COMPONENT_HISTORY_H

namespace aptFront {
namespace cache {
namespace component {

/**
   This class template implements an undo history for an arbitrary
   cache component. However, this component absolutely must have
   a deep-copying copy constructor!
 */
template<typename T>
class History : public Implementation<History<T> >, public Observer {
protected:
    typedef std::list<T *> TQueue;
    typename TQueue::iterator m_current;
    TQueue m_queue;
    int m_limit;
    bool m_winding:1;
    bool m_autoCheckpoint;

public:

    History( T *initial = 0, int limit = 16 )
        : m_limit( limit ), m_winding( false ), m_autoCheckpoint( false )
    {
        if (initial)
            m_queue.push_front( initial );
        m_current = m_queue.begin();
        // std::cerr << "history constructed..." << std::endl;
    }
    virtual ~History() {}

    virtual bool allowClose() {
        return false;
    }

    void setAutoCheckpoint( bool b ) {
        m_autoCheckpoint = b;
    }

    virtual void notifyPreRebuild( component::Base * ) {
        std::cerr << "pre rebuild, winding = " << m_winding << std::endl;
        if (!m_winding) {
            clear(); // nuke the old stuff
        }
    }

    virtual void notifyPreChange( component::Base * ) {
        if ((!m_winding) && m_autoCheckpoint)
            checkpoint();
    }

    virtual void notifyPostRebuild( component::Base * ) {
        std::cerr << "post rebuild, winding = " << m_winding << std::endl;
    }

    void setOwnerCache( Cache *c ) {
        if (m_winding)
            throw exception::InternalError( "history tried to wind while winding" );
        Implementation<History<T> >::setOwnerCache( c );
        // OBSERVER API NEEDS TO BE FUCKING FIXED
        /* and the set(Owner)Cache crap synced somewhat */
        setCache( c ); // for Observer
        observeComponent< T >(); // this really should NOT go to ctor
        clear();
    }

    T &current() {
        T *cur = &(this->ownerCache()->template component<T>());
        m_current = std::find( m_queue.begin(), m_queue.end(), cur );
        if (m_current == m_queue.end())
            clear();
        return **m_current;
    }

    T &previous() {
        typename TQueue::iterator i = m_current;
        return **--i;
    }

    void undo(int steps = 1) {
        if (m_winding)
            throw exception::InternalError( "history tried to wind while winding" );
        if (!canUndo())
            throw exception::InternalError( "history tried to unwind into void" );
        std::advance( m_current, steps );
        m_winding = true;
        this->ownerCache()->addComponent( *m_current , Cache::KeepOld);
        m_winding = false;
    }

    void redo(int steps = 1) {
        if (m_winding)
            throw exception::InternalError( "history tried to wind while winding" );
        if (!canRedo())
            throw exception::InternalError( "history tried to unwind into void" );
        std::advance( m_current, -steps );
        m_winding = true;
        this->ownerCache()->addComponent( *m_current, Cache::KeepOld );
        m_winding = false;
    }

    bool canUndo(int steps = 1) {
        typename TQueue::iterator i = m_current;
        std::advance( i, steps );
        if (i == m_queue.end())
            return false;
        return true;
    }

    bool canRedo( int steps = 1) {
        typename TQueue::iterator i = m_current;
        std::advance( i, steps - 1 );
        if (i == m_queue.begin())
            return false;
        return true;
    }

    bool winding() const {
        return m_winding;
    }

    struct kill {
        Cache *c;
        T *cur;
        void operator()( T *t ) {
            if (t == &(c->template component<T>())) return;
            // c->deleteComponent( t );
            delete t;
        }
        kill( Cache *_c, T *_t ) : c( _c ), cur( _t ) {}
    };

    void checkpoint() {
        typename TQueue::iterator i = m_queue.begin();
        if (i != m_current) {
            std::cerr << "history dropping entries..." << std::endl;
            std::for_each( i, m_current,
                           kill( this->ownerCache(), *m_current ) );
            m_queue.erase( i, m_current );
        }
        i = m_current;
        // std::cerr << "history duplicating current..." << std::endl;
        T *x;
        m_queue.insert( ++i, x = new T( current() ) );
        x->setAllowDelete( false );
        // std::cerr << "history cleaning up overlimit..." << std::endl;
        while (m_queue.size() > m_limit)
           m_queue.pop_back();
    }

    void setSizeLimit( int x ) {
        m_limit = x;
    }

    void clear() {
        std::for_each( m_queue.begin(), m_queue.end(),
                       kill( this->ownerCache(), *m_current ) );
        m_queue.clear();
        m_queue.push_front( &(this->ownerCache()->template component<T>()) );
        m_current = m_queue.begin();
        (*m_current)->setAllowDelete( false );
        // there is probably a leak in there... and i don't really
        // care atm
    }

};

}
}
} 

#endif
