#ifndef CACHE_COMPONENT_TAGS_H
#define CACHE_COMPONENT_TAGS_H

/** @file
 * @author Enrico Zini <enrico@enricozini.org>
 * Tag vocabulary access
 */

/*
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <apt-front/cache/component/base.h>
#include <apt-front/cache/entity/tag.h>

#include <string>
#include <vector>
#include <map>

namespace aptFront {
namespace cache {
namespace component {

class Tags : public Implementation<Tags>
{
protected:
	struct FacetInfo
	{
		unsigned int id;
		unsigned int firstTag;
		unsigned int tagCount;
		FacetInfo(unsigned int id, unsigned int firstTag) : id(id), firstTag(firstTag), tagCount(0) {}
	};
	
	std::vector<entity::Facet::Data> m_facets;
	std::vector<entity::Tag::Data> m_tags;

	// Once there is a guarantee that facets and tags are alphabetically sorted
	// in the vocabulary, one can get rid of these maps and do binary searches
	// on the vectors
	std::map<std::string, FacetInfo> m_facetinfo;
	std::map<std::string, unsigned int> m_taginfo;

	void init(const std::string& filename);

public:
	Tags();
	Tags(const std::string& filename);

	/**
	 * Check if the vocabulary contains the facet `name'
	 */
	bool hasFacet(const std::string& name) const
	{
		return m_facetinfo.find(name) != m_facetinfo.end();
	}

	/**
	 * Check if the vocabulary contains the tag `fullname'
	 */
	bool hasTag(const std::string& fullname) const
	{
		return m_taginfo.find(fullname) != m_taginfo.end();
	}

	/**
	 * Return the facet with the given name
	 */
	entity::Facet facetByName(const std::string& name) const;

	/**
	 * Return the tag with the given full name
	 */
	entity::Tag tagByName(const std::string& fullname) const;

	/**
	 * Return all the facets in the vocabulary
	 */
	Tagcoll::OpSet<entity::Facet> facets() const;

	/**
	 * Return all the tags in the vocabulary
	 */
	Tagcoll::OpSet<entity::Tag> tags() const;

	/**
	 * Return the tags in the given facet
	 */
	Tagcoll::OpSet<entity::Tag> tags(const std::string& facetName) const;

#if 0
	/// Get the DerivedTagList with the Equates: expressions read from the vocabulary
	const DerivedTagList& getEquations() const throw () { return equations; }
	
	/// Get a set with all the facets present in the vocabulary that are matched by `filter'
	FacetSet facets(const FacetMatcher& filter) const throw () { return getFiltered(filter); }
#endif

	// These functions are here just to be used by Facet and Tag.  I'm not
	// making them private because I don't want Facet and Tag to access other
	// Tags member, and other classes can't use these anyway as Facet::Data and
	// Tag::Data are protected
	const entity::Facet::Data& facetData(int idx) { return m_facets[idx]; }
	const entity::Tag::Data& tagData(int idx) { return m_tags[idx]; }
};

}
}
}

// vim:set ts=4 sw=4:
#endif
