#ifndef APTFRONT_CACHE_ENTITY_TAG_H
#define APTFRONT_CACHE_ENTITY_TAG_H

/** \file
 * Debtags facets and tags
 */

/*
 * Copyright (C) 2005,2006  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

//#include <apt-front/cache/entity/entity.h>

#include <tagcoll/OpSet.h>

#if 0
#include <string>
#include <debtags/fdecls.h>
#endif

namespace aptFront {
namespace cache {

namespace component { class Tags; }

namespace entity {

class Tag;

/**
 * Representation of a facet.
 *
 * Tagcoll::Facet represents a Facet with all its informations.  It is
 * implemented via a reference-counted smart pointer, so it can be passed
 * around freely and efficiently without worrying about memory management
 * issues.
 *
 * The class is normally instantiated using a Vocabulary:
 * \code
 * 	Facet facet = vocabulary.faceByNamet("made-of");
 * \endcode
 *
 * Facets can contain a "false" value, in which case using any of their
 * methonds produce a null-pointer dereference segfault.  The "null" facets are
 * useful as "none" return values:
 *
 * \code
 *    Facet facet = vocabulary.facetByName("made-of");
 *    if (!facet)
 *       throw ConsistencyCheckException("facet \"made-of\" has not been defined");
 * \endcode
 */
class Facet
{
protected:
	const component::Tags* m_tags;
	int m_id;

	Facet(const component::Tags *tags, int id) : m_tags(tags), m_id(id) {}

public:
	Facet() : m_tags(0), m_id(-1) {}
	~Facet() {}

	bool operator==(const Facet& f) const { return m_id == f.m_id; }
	bool operator!=(const Facet& f) const { return m_id != f.m_id; }
	bool operator<(const Facet& f) const { return m_id < f.m_id; }

	/**
	 * Return true if the facet is valid
	 */
	operator bool() const { return m_id != -1; }
	bool valid() const { return m_id != -1; }

	/**
	 * Return the ID of this facet
	 */
	int id() const { return m_id; }

	/**
	 * Return the name of the facet
	 */
	std::string name() const;
	std::string name(const std::string& d) const;

	/**
	 * Return the short description of the facet
	 */
	std::string shortDescription() const;
	std::string shortDescription(const std::string& d) const;

	/**
	 * Return the long description of the facet
	 */
	std::string longDescription() const;
	std::string longDescription(const std::string& d) const;

	/**
	 * Return true if the facet has a tag with the given name (name, not fullname)
	 */
	bool hasTag(const std::string& name) const;

	/**
	 * Return the list of tags in this facet
	 */
	Tagcoll::OpSet<Tag> tags() const;

	friend class component::Tags;
};

/**
 * Representation of a tag.
 *
 * Tagcoll::Tag represents a Tag with all its informations.  It is implemented
 * via a reference-counted smart pointer, so it can be passed around freely and
 * efficiently without worrying about memory management issues.
 *
 * The class is normally instantiated using a Vocabulary:
 * \code
 * 	Tag tag = vocabulary.tagByName("made-of::lang:c++");
 * \endcode
 *
 * Tags can contain a "false" value, in which case using any of their
 * methonds produce a null-pointer dereference segfault.  The "null" tags are
 * useful as "none" return values:
 *
 * \code
 *    Tag tag = vocabulary.tagByName("made-of");
 *    if (!tag)
 *       throw ConsistencyCheckException("tag \"mytag\" has not been defined");
 * \endcode
 */
class Tag
{
protected:
	const component::Tags* m_tags;
	int m_id;

	Tag(const component::Tags *tags, int id) : m_tags(tags), m_id(id) {}

public:
	typedef Tagcoll::OpSet<Tag> Set;

	Tag() : m_tags(0), m_id(-1) {}
	~Tag() {}

	bool operator==(const Tag& f) const { return m_id == f.m_id; }
	bool operator!=(const Tag& f) const { return m_id != f.m_id; }
	bool operator<(const Tag& f) const { return m_id < f.m_id; }

	operator bool() const { return m_id != -1; }
	bool valid() const { return m_id != -1; }

	/**
	 * Return the ID of this tag
	 */
	int id() const { return m_id; }

	Facet facet() const;

	/**
	 * Return the name of the tag, without the facet:: prefix
	 */
	std::string name() const;
	std::string name(const std::string& d) const;

	/**
	 * Return the name of the tag, with the facet:: prefix
	 */
	std::string fullname() const;
	std::string fullname(const std::string& d) const;

	/**
	 * Return the short description of the tag
	 */
	std::string shortDescription() const;
	std::string shortDescription(const std::string& d) const;

	/**
	 * Return the long description of the tag
	 */
	std::string longDescription() const;
	std::string longDescription(const std::string& d) const;

	friend class component::Tags;
};

}
}
}

// vim:set ts=3 sw=3:
#endif
