/*
 * Copyright (C) 2007  Enrico Zini <enrico@enricozini.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <tests/test-utils.h>
#include <ept/apt/apt.h>
#include <set>
#include <algorithm>

namespace tut {
using namespace std;
using namespace ept;
using namespace ept::apt;

struct ept_apt_apt_shar {
	Apt apt;
};
TESTGRP(ept_apt_apt);

// Check that iterations iterates among some packages
template<> template<>
void to::test<1>()
{
	Apt::iterator i = apt.begin();
	ensure(i != apt.end());
	
	size_t count = 0;
	for (; i != apt.end(); ++i)
		++count;

	ensure(count > 100);
}

// Check that iteration gives some well-known packages
template<> template<>
void to::test<2>()
{
	set<string> packages;

	std::copy(apt.begin(), apt.end(), inserter(packages, packages.begin()));

	ensure(packages.find("apt") != packages.end());
	ensure(packages.find("") == packages.end());
}

// Check that timestamp gives some meaningful timestamp
template<> template<>
void to::test<3>()
{
	time_t ts = apt.timestamp();
	ensure(ts > 1000000);
}

// Check the package validator
template<> template<>
void to::test<4>()
{
	ensure(apt.isValid("apt"));
	ensure(!apt.isValid("this-package-does-not-really-exists"));
}

// Check the version instantiators
template<> template<>
void to::test<5>()
{
	std::string pkg("apt");
	Version ver = apt.candidateVersion(pkg);
	ensure(ver.isValid());
	
	ver = apt.installedVersion(pkg);
	ensure(ver.isValid());

	ver = apt.anyVersion(pkg);
	ensure(ver.isValid());

	std::string pkg1("this-package-does-not-really-exists");
	ver = apt.candidateVersion(pkg1);
	ensure(!ver.isValid());
	
	ver = apt.installedVersion(pkg1);
	ensure(!ver.isValid());

	ver = apt.anyVersion(pkg1);
	ensure(!ver.isValid());
}

// Check the version validator
template<> template<>
void to::test<6>()
{
	Version ver = apt.candidateVersion("apt");
	ensure(apt.validate(ver) == ver);

	ver = Version("this-package-does-not-really-exists", "0.1");
	ensure(!apt.validate(ver).isValid());

	ver = Version("apt", "0.31415");
	ensure(!apt.validate(ver).isValid());
}

// Check the raw record accessor
template<> template<>
void to::test<7>()
{
	string pkg("apt");
	Version ver = apt.candidateVersion(pkg);
	ensure(apt.validate(ver) == ver);

	string record = apt.rawRecord(ver);
	ensure(record.find("Package: apt") != string::npos);
	ensure(record.find("Build-Essential: yes") != string::npos);

	record = apt.rawRecord(Version("apt", "0.31415"));
	ensure_equals(record, string());

	ensure_equals(apt.rawRecord(pkg), apt.rawRecord(apt.anyVersion(pkg)));
}

// Check the package state accessor
template<> template<>
void to::test<8>()
{
	PackageState s = apt.state("apt");
	ensure(s.isValid());
	ensure(s.isInstalled());

	s = apt.state("this-package-does-not-really-exists");
	ensure(!s.isValid());
}

// Check the record iterator (accessing with *)
template<> template<>
void to::test<9>()
{
	size_t count = 0;
	for (Apt::record_iterator i = apt.recordBegin();
			i != apt.recordEnd(); ++i)
	{
		ensure((*i).size() > 8);
		ensure_equals((*i).substr(0, 8), "Package:");
		++count;
	}
	ensure(count > 200);
}

// Check the record iterator (accessing with ->)
template<> template<>
void to::test<10>()
{
	size_t count = 0;
	for (Apt::record_iterator i = apt.recordBegin();
			i != apt.recordEnd(); ++i)
	{
		ensure(i->size() > 8);
		ensure_equals(i->substr(0, 8), "Package:");
		++count;
	}
	ensure(count > 200);
}

// Check that the iterators can be used with the algorithms
template<> template<>
void to::test<11>()
{
	vector<string> out;
	std::copy(apt.begin(), apt.end(), back_inserter(out));
}

// Check that the iterators can be used with the algorithms
template<> template<>
void to::test<12>()
{
	vector<string> out;
	std::copy(apt.recordBegin(), apt.recordEnd(), back_inserter(out));
}

}

// vim:set ts=4 sw=4:
