/* GNOME DB library
 * Copyright (C) 1999-2001 The Free Software Foundation
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <libgnomedb/gnome-db-combo.h>

#define PARENT_TYPE GTK_TYPE_COMBO

struct _GnomeDbComboPrivate {
	GdaDataModel *model;
	gint col;
};

static void gnome_db_combo_class_init   (GnomeDbComboClass *klass);
static void gnome_db_combo_init         (GnomeDbCombo *combo,
					   GnomeDbComboClass *klass);
static void gnome_db_combo_set_property (GObject *object,
					   guint paramid,
					   const GValue *value,
					   GParamSpec *pspec);
static void gnome_db_combo_get_property (GObject *object,
					   guint param_id,
					   GValue *value,
					   GParamSpec *pspec);
static void gnome_db_combo_finalize     (GObject *object);
static void model_changed_cb 		(GdaDataModel *model,
					 gpointer user_data);

enum {
	PROP_0,
	PROP_MODEL
};

static GObjectClass *parent_class = NULL;

/*
 * GnomeDbCombo class implementation
 */

static void
gnome_db_combo_class_init (GnomeDbComboClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->set_property = gnome_db_combo_set_property;
	object_class->get_property = gnome_db_combo_get_property;
	object_class->finalize = gnome_db_combo_finalize;

	/* add class properties */
	g_object_class_install_property (
		object_class, PROP_MODEL,
		g_param_spec_object ("model", NULL, NULL,
				     GDA_TYPE_DATA_MODEL,
				     (G_PARAM_READABLE | G_PARAM_WRITABLE)));
}

static void
gnome_db_combo_init (GnomeDbCombo *combo, GnomeDbComboClass *klass)
{
	g_return_if_fail (GNOME_DB_IS_COMBO (combo));

	/* allocate private structure */
	combo->priv = g_new0 (GnomeDbComboPrivate, 1);
	combo->priv->model = NULL;
}

static void
gnome_db_combo_set_property (GObject *object,
			     guint param_id,
			     const GValue *value,
			     GParamSpec *pspec)
{
	GnomeDbCombo *combo = (GnomeDbCombo *) object;

	g_return_if_fail (GNOME_DB_IS_COMBO (combo));

	switch (param_id) {
	case PROP_MODEL :
		gnome_db_combo_set_model (combo,
					  GDA_DATA_MODEL (g_value_get_object (value)),
					  0);
		break;
	default :
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_combo_get_property (GObject *object,
			     guint param_id,
			     GValue *value,
			     GParamSpec *pspec)
{
	GnomeDbCombo *combo = (GnomeDbCombo *) object;

	g_return_if_fail (GNOME_DB_IS_COMBO (combo));

	switch (param_id) {
	case PROP_MODEL :
		g_value_set_object (value, G_OBJECT (combo->priv->model));
		break;
	default :
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_combo_finalize (GObject *object)
{
	GnomeDbCombo *combo = (GnomeDbCombo *) object;

	g_return_if_fail (GNOME_DB_IS_COMBO (combo));

	/* free memory */
	if (GDA_IS_DATA_MODEL (combo->priv->model)) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (combo->priv->model), model_changed_cb, combo);
		g_object_unref (G_OBJECT (combo->priv->model));
		combo->priv->model = NULL;
	}

	g_free (combo->priv);
	combo->priv = NULL;

	/* chain to parent class */
	parent_class->finalize (object);
}

GType
gnome_db_combo_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbComboClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_combo_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbCombo),
			0,
			(GInstanceInitFunc) gnome_db_combo_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbCombo", &info, 0);
	}
	return type;
}

/**
 * gnome_db_combo_new
 *
 * Create a new GnomeDbCombo widget.
 *
 * Returns: the newly-created widget.
 */
GtkWidget *
gnome_db_combo_new (void)
{
	GnomeDbCombo *combo;

	combo = g_object_new (GNOME_DB_TYPE_COMBO, NULL);
	return GTK_WIDGET (combo);
}

static void
model_changed_cb (GdaDataModel *model, gpointer user_data)
{
	GnomeDbCombo *combo = GNOME_DB_COMBO (user_data);
	gint rows, i, cols;
	const GdaValue *value;
	GList *list = NULL;

	g_return_if_fail (GDA_IS_DATA_MODEL (model));
	g_return_if_fail (GNOME_DB_IS_COMBO (combo));

	cols = gda_data_model_get_n_columns (model);
	rows = gda_data_model_get_n_rows (model);
	if (rows == 0 || cols < combo->priv->col) {
		list = g_list_append (list, "");
		gtk_combo_set_popdown_strings (GTK_COMBO (combo), list);
		g_list_free (list);
		return;
	}

	for (i = 0; i < rows; i++) {
		value = gda_data_model_get_value_at (model, combo->priv->col, i);
		list = g_list_append (list, gda_value_stringify (value));
	}

	gtk_combo_set_popdown_strings (GTK_COMBO (combo), list);
	g_list_foreach (list, (GFunc) g_free, NULL);
	g_list_free (list);
}

/**
 * gnome_db_combo_set_model
 * @combo: a #GnomeDbCombo widget.
 * @model: a #GdaDataModel object.
 * @col: column in the model to be shown.
 *
 * Associate a #GdaDataModel with the given combo widget. Doing so makes the
 * combo widget refresh its list of values and display the values contained
 * in the model, in the given position. A NULL @model will make the combo empty
 * and disassociate the previous model, if any.
 */
void
gnome_db_combo_set_model (GnomeDbCombo *combo, GdaDataModel *model, gint col)
{
	GList *list = NULL;

	g_return_if_fail (GNOME_DB_IS_COMBO (combo));
	g_return_if_fail (model == NULL || GDA_IS_DATA_MODEL (model));

	if (GDA_IS_DATA_MODEL (combo->priv->model)) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (combo->priv->model), model_changed_cb, combo);
		g_object_unref (G_OBJECT (combo->priv->model));
	}

	combo->priv->model = model;
	if (model != NULL) {
		g_object_ref (G_OBJECT (model));
		g_signal_connect (G_OBJECT (combo->priv->model), "changed",
				  G_CALLBACK (model_changed_cb), combo);

		combo->priv->col = col;
		model_changed_cb (model, combo);
	} else {
		list = g_list_append (list, "");
		gtk_combo_set_popdown_strings (GTK_COMBO (combo), list);
		g_list_free (list);
	}
}

