/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2009-2010 Canonical Services Ltd (www.canonical.com)
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <libxml/HTMLparser.h>
#include <glib/gi18n.h>
#include <gst/gst.h>
#include <gst/pbutils/pbutils.h>
#include <webkit/webkit.h>
#include <libsoup/soup-gnome-features.h>
#include <libsoup/soup-uri.h>
#include <libsyncdaemon/libsyncdaemon.h>
#include "oauth.h"
#include "u1-music-store.h"
#include "u1-marshal.h"
#include "u1-codec-installer.h"
#include "u1-request-chrome.h"

#define U1_MUSIC_LIBRARY_LOCATION  ".ubuntuone/Purchased from Ubuntu One"

#define U1_STORE_URL               "/music/login"
#define U1_LIBRARY_URL             "/music/store/library"
#define U1_NOT_LOGGED_IN_STORE_URL "/music/store-no-token"
#define U1_TOKEN_INVALID_URL       "/music/store-invalid-token"
#define U1_NOT_REGISTERED_URL      "https://one.ubuntu.com/music/notregistered?returnUrl="

#define U1_DEFAULT_ERROR_PAGE "load_error.html"
#define U1_OUTSIDE_DOMAIN_ERROR_PAGE "outside_domain.html"
#define U1_TRYING_SSO_ERROR_PAGE "trying_sso.html"
#define U1_IN_DEVELOPMENT_PAGE "in_development.html"
#define U1_CONNECTING_PAGE "connecting.html" 
#define U1_INITIAL_PAGE "<html><body>Loading Ubuntu One music store</body></html>"

#define DEFAULT_ENCODING "UTF-8"
#define MAXIMUM_AUTH_RETRIES 5

struct _U1MusicStorePrivate {
	SyncdaemonDaemon *syncdaemon;
	gchar *base_url;

	GtkWidget *scroll;
	GtkWidget *initial_web_viewer;
	GtkWidget *web_viewer;
	WebKitWebFrame *frame_to_use;
	GtkWidget *progress;

	/* Alert bar widgets
	 * We use two button widgets here, to avoid overlapping conditions
	 */
	GtkWidget *alertbar;
	GtkWidget *alert_label;
	GtkWidget *subscribe_btn;
	GtkWidget *install_btn;
	GtkWidget *install_progress;
	gboolean folder_subscribed;
	gboolean codec_installed;

	/* The installer object */
	U1CodecInstaller *installer;

	/* GStreamer pipeline used to check for the MP3 codec */
	GstElement *pipeline;

	guint watch_id;
	guint idle_cb;
	GHashTable *watched_downloads;

	gint auth_retries;
};

G_DEFINE_TYPE(U1MusicStore, u1_music_store, GTK_TYPE_VBOX)

enum {
	PREVIEW_MP3_SIGNAL,
	PLAY_LIBRARY_SIGNAL,
	URL_LOADED_SIGNAL,
	DOWNLOAD_FINISHED_SIGNAL,
	LAST_SIGNAL
};

static guint u1_music_store_signals[LAST_SIGNAL] = { 0, };

static void
u1_music_store_finalize (GObject *object)
{
	U1MusicStore *music_store = U1_MUSIC_STORE (object);

	if (music_store->priv != NULL) {
		if (music_store->priv->idle_cb != 0)
			g_source_remove (music_store->priv->idle_cb);

		if (music_store->priv->watch_id != 0)
			g_source_remove (music_store->priv->watch_id);

		if (music_store->priv->watched_downloads != NULL)
			g_hash_table_destroy (music_store->priv->watched_downloads);

		if (music_store->priv->base_url != NULL)
			g_free (music_store->priv->base_url);

		if (music_store->priv->syncdaemon != NULL)
			g_object_unref (G_OBJECT (music_store->priv->syncdaemon));

		if (music_store->priv->installer != NULL)
			g_object_unref (music_store->priv->installer);

		g_free (music_store->priv);
	}

	G_OBJECT_CLASS (u1_music_store_parent_class)->finalize (object);
}

static void
u1_music_store_class_init (U1MusicStoreClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	/* Signals */
	u1_music_store_signals[PREVIEW_MP3_SIGNAL] = g_signal_new ("preview-mp3",
								   G_TYPE_FROM_CLASS (klass),
								   (GSignalFlags) G_SIGNAL_RUN_LAST,
								   G_STRUCT_OFFSET (U1MusicStoreClass, preview_mp3),
								   NULL,
								   NULL,
								   _u1_marshal_VOID__STRING_STRING,
								   G_TYPE_NONE, 2,
								   G_TYPE_STRING, G_TYPE_STRING);
	u1_music_store_signals[PLAY_LIBRARY_SIGNAL] = g_signal_new ("play-library",
								    G_TYPE_FROM_CLASS (klass),
								    (GSignalFlags) G_SIGNAL_RUN_LAST,
								    G_STRUCT_OFFSET (U1MusicStoreClass, play_library),
								    NULL,
								    NULL,
								    g_cclosure_marshal_VOID__STRING,
								    G_TYPE_NONE, 1,
								    G_TYPE_STRING);
	u1_music_store_signals[URL_LOADED_SIGNAL] = g_signal_new ("url-loaded",
								  G_TYPE_FROM_CLASS (klass),
								  (GSignalFlags) G_SIGNAL_RUN_LAST,
								  G_STRUCT_OFFSET (U1MusicStoreClass, url_loaded),
								  NULL,
								  NULL,
								  g_cclosure_marshal_VOID__STRING,
								  G_TYPE_NONE, 1,
								  G_TYPE_STRING);
	u1_music_store_signals[DOWNLOAD_FINISHED_SIGNAL] = g_signal_new ("download-finished",
									 G_TYPE_FROM_CLASS (klass),
									 (GSignalFlags) G_SIGNAL_RUN_LAST,
									 G_STRUCT_OFFSET (U1MusicStoreClass, download_finished),
									 NULL,
									 NULL,
									 g_cclosure_marshal_VOID__STRING,
									 G_TYPE_NONE, 1,
									 G_TYPE_STRING);

	object_class->finalize = u1_music_store_finalize;
}

static gchar *
get_internal_html_page_url (WebKitWebView *web_view, const gchar *file_name, const gchar *reload_url)
{
	gchar *calculated_file_path, *calculated_file_url, *calculated_file_url_noreload;

	calculated_file_path = g_build_path ("/", U1_JAVASCRIPT_DIR, file_name, NULL);
	if (reload_url == NULL) {
		calculated_file_url = g_filename_to_uri (calculated_file_path, NULL, NULL);
	} else {
		calculated_file_url_noreload = g_filename_to_uri (calculated_file_path, NULL, NULL);
		calculated_file_url = g_strdup_printf ("%s?%s", 
			calculated_file_url_noreload,
			reload_url);
		g_free (calculated_file_url_noreload);
	}

	g_free (calculated_file_path);

	return calculated_file_url;
}

static void
load_internal_html_page (WebKitWebView *web_view, const gchar *file_name, const gchar *reload_url)
{
	gchar *calculated_file_url;

	calculated_file_url = get_internal_html_page_url (web_view, file_name, reload_url);
	webkit_web_view_open (web_view, calculated_file_url);
	g_free (calculated_file_url);
}

static void
parse_oauth_string (const gchar *string,
		    gchar **oauth_consumer_token,
		    gchar **oauth_consumer_secret,
		    gchar **oauth_token,
		    gchar **oauth_token_secret)
{
	gchar **items;

	*oauth_consumer_token = NULL;
	*oauth_consumer_secret = NULL;
	*oauth_token = NULL;
	*oauth_token_secret = NULL;

	items = g_strsplit (string, "&", 2);
	if (items) {
		gint i;

		for (i = 0; items[i] != NULL; i++) {
			gchar **kv_pair;

			kv_pair = g_strsplit (items[i], "=", 2);
			if (kv_pair) {
				if (g_strcmp0 ((const gchar *) kv_pair[0], "oauth_token") == 0)
					*oauth_token = g_strdup (kv_pair[1]);
				else if (g_strcmp0 ((const gchar *) kv_pair[0], "oauth_token_secret") == 0)
					*oauth_token_secret = g_strdup (kv_pair[1]);
				else if (g_strcmp0 ((const gchar *) kv_pair[0], "oauth_consumer_token") == 0)
					*oauth_consumer_token = g_strdup (kv_pair[1]);
				else if (g_strcmp0 ((const gchar *) kv_pair[0], "oauth_consumer_secret") == 0)
					*oauth_consumer_secret = g_strdup (kv_pair[1]);

				g_strfreev (kv_pair);
			}
		}

		g_strfreev (items);
	}
}

static void
get_credentials (U1MusicStore *music_store,
		 gchar **oauth_consumer_token,
		 gchar **oauth_consumer_secret,
		 gchar **oauth_token,
		 gchar **oauth_token_secret)
{
	SyncdaemonCredentials *credentials;

	*oauth_consumer_token = *oauth_consumer_secret = *oauth_token = *oauth_token_secret = NULL;

	/* Get the OAuth token from the SSO client */
	if ((credentials = syncdaemon_authentication_find_credentials (
		     syncdaemon_daemon_get_authentication (music_store->priv->syncdaemon)))) {
		*oauth_consumer_token = g_strdup (syncdaemon_credentials_get_consumer_key (credentials));
		*oauth_consumer_secret = g_strdup (syncdaemon_credentials_get_consumer_secret (credentials));
		*oauth_token = g_strdup (syncdaemon_credentials_get_token (credentials));
		*oauth_token_secret = g_strdup (syncdaemon_credentials_get_token_secret (credentials));
	}
}

typedef struct {
	U1MusicStore *music_store;
	gchar *not_registered_url;
} NotRegisteredReplacementData;

static gchar *
get_url_to_use (U1MusicStore *music_store)
{
	gchar *oauth_consumer_token, *oauth_consumer_secret, *oauth_token = NULL, *oauth_token_secret = NULL, *real_url;

	/* OAUth sign the URL */
	if (g_str_has_prefix (music_store->priv->base_url, "http://localhost") || g_str_has_prefix (music_store->priv->base_url, "http://127.0.0.1")) {
		const gchar *oauthfile;

		oauthfile = g_getenv ("OAUTHKEYFILE");
		if (oauthfile != NULL) {
			gchar *oauthstring;
			gsize len;

			if (g_file_get_contents (oauthfile, &oauthstring, &len, NULL)) {
				parse_oauth_string (oauthstring,
						    &oauth_consumer_token,
						    &oauth_consumer_secret,
						    &oauth_token,
						    &oauth_token_secret);
				g_free (oauthstring);
			}
		} else {
			webkit_web_view_load_string (WEBKIT_WEB_VIEW (music_store->priv->web_viewer),
						     "Fail! If you specify U1MUSICSTOREURL you must "
						     "also specify OAUTHKEYFILE as a file with OAuth keys in it",
						     "text/html", "utf-8", "file:///");
			return NULL;
		}
	} else {
		get_credentials (music_store,
				 &oauth_consumer_token,
				 &oauth_consumer_secret,
				 &oauth_token,
				 &oauth_token_secret);
	}

	if (oauth_token == NULL || oauth_token_secret == NULL)
		real_url = g_strdup_printf ("%s%s", music_store->priv->base_url, U1_NOT_LOGGED_IN_STORE_URL);
	else {
		gchar *s = g_strdup_printf ("%s%s", music_store->priv->base_url, U1_STORE_URL);

		real_url = oauth_sign_url2 (s, NULL, OA_HMAC, "GET",
					    oauth_consumer_token, oauth_consumer_secret,
					    oauth_token, oauth_token_secret);
		g_free (s);

		/* Connect to syncdaemon now that we have tokens */
		if (syncdaemon_daemon_is_ready (music_store->priv->syncdaemon))
			syncdaemon_daemon_connect (music_store->priv->syncdaemon);
	}

	g_free (oauth_consumer_token);
	g_free (oauth_consumer_secret);
	g_free (oauth_token);
	g_free (oauth_token_secret);

	return real_url;
}



static void
got_new_credentials_cb (SyncdaemonAuthentication *auth, SyncdaemonCredentials *credentials, gpointer user_data)
{
	gchar *oauth_consumer_token, *oauth_consumer_secret, *oauth_token, *oauth_token_secret, *real_url;
	gchar *url_to_use = NULL;
	gchar **uri_parts;
	NotRegisteredReplacementData *nrrd = (NotRegisteredReplacementData *) user_data;

	g_debug ("Got new credentials from ubuntu-sso");

	oauth_consumer_token = g_strdup (syncdaemon_credentials_get_consumer_key (credentials));
	oauth_consumer_secret = g_strdup (syncdaemon_credentials_get_consumer_secret (credentials));
	oauth_token = g_strdup (syncdaemon_credentials_get_token (credentials));
	oauth_token_secret = g_strdup (syncdaemon_credentials_get_token_secret (credentials));

	/* Get the return_url parameter from the not-registered url */
	uri_parts = g_strsplit (nrrd->not_registered_url, "?", 0);
	if (uri_parts != NULL && uri_parts[1] != NULL) {
		gchar **uri_args;

		uri_args = g_strsplit (uri_parts[1], "?", 0);
		if (uri_args != NULL) {
			gint i;

			for (i = 0; uri_args[i] != NULL; i++) {
				gchar **this_arg;

				if (g_str_has_prefix (uri_args[i], "returnUrl") ||
				    g_str_has_prefix (uri_args[i], "next")) {
					this_arg = g_strsplit (uri_args[i], "=", 2);

					url_to_use = g_strdup_printf ("%s%s?forward_on_to_url=%s",
								      nrrd->music_store->priv->base_url,
								      U1_STORE_URL,
								      this_arg[1]);
					g_strfreev (this_arg);
					break;
				}
			}

			g_strfreev (uri_args);
		}

		g_strfreev (uri_parts);
	} else
		url_to_use = g_strdup_printf ("%s%s", nrrd->music_store->priv->base_url, U1_STORE_URL);

	/* Now sign the URL and load the new page in the view */
	real_url = oauth_sign_url2 (url_to_use, NULL, OA_HMAC, "GET",
				    oauth_consumer_token, oauth_consumer_secret,
				    oauth_token, oauth_token_secret);
	webkit_web_view_open (WEBKIT_WEB_VIEW (nrrd->music_store->priv->web_viewer), real_url);

	/* Free memory */
	g_signal_handlers_disconnect_by_func (auth, got_new_credentials_cb, nrrd);

	g_free (url_to_use);
	g_free (real_url);
	g_free (oauth_consumer_token);
	g_free (oauth_consumer_secret);
	g_free (oauth_token);
	g_free (oauth_token_secret);

	g_free (nrrd->not_registered_url);
	g_free (nrrd);
}

static void authorization_cancelled_cb (SyncdaemonAuthentication *auth, gpointer user_data)
{
	NotRegisteredReplacementData *nrrd = (NotRegisteredReplacementData *) user_data;

	g_debug ("SSO authentication cancelled");

	/* Free memory */
	g_signal_handlers_disconnect_by_func (auth, authorization_cancelled_cb, nrrd);
	g_free (nrrd->not_registered_url);
	g_free (nrrd);

}

static void
credentials_error_cb (SyncdaemonAuthentication *auth, const gchar *error, gpointer user_data)
{
	GtkWidget *dialog;
	NotRegisteredReplacementData *nrrd = (NotRegisteredReplacementData *) user_data;

	g_debug ("SSO authentication error: %s", error);

	/* Show an error message */
	dialog = gtk_message_dialog_new (GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (nrrd->music_store))),
					 0,
					 GTK_MESSAGE_ERROR,
					 GTK_BUTTONS_CLOSE,
					 _("Error while getting credentials:\n%s"),
					 error);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

typedef struct {
	WebKitWebView *web_view;
	char *uri;
} DelayedLoadData;

static gboolean
html_delayed_load_cb (gpointer user_data)
{
	DelayedLoadData *delayed_data = (DelayedLoadData *) user_data;

	webkit_web_view_open (delayed_data->web_view, delayed_data->uri);

	g_object_unref (G_OBJECT (delayed_data->web_view));
	g_free (delayed_data->uri);
	g_free (delayed_data);

	return FALSE;
}

static void
load_delayed_page (WebKitWebView *web_view, const char *uri, WebKitWebPolicyDecision *policy_decision)
{
	DelayedLoadData *delayed_data;

	delayed_data = g_new0 (DelayedLoadData, 1);
	delayed_data->web_view = g_object_ref (web_view);
	delayed_data->uri = g_strdup (uri);

	g_idle_add ((GSourceFunc) html_delayed_load_cb, delayed_data);

	webkit_web_policy_decision_ignore (policy_decision);
}

static void
fire_authentication (U1MusicStore *music_store, WebKitNetworkRequest *request, WebKitWebPolicyDecision *policy_decision)
{
	NotRegisteredReplacementData *nrrd;
	gchar *internal_uri;
	SyncdaemonAuthentication *auth;

	if (music_store->priv->auth_retries > MAXIMUM_AUTH_RETRIES) {
		GtkWidget *dialog;
		gchar *home_url;

		/* Tell the user authentication has failed */
		dialog = gtk_message_dialog_new (GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (music_store))),
						 0,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_CLOSE,
						 _("Authentication to the music store failed"));
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);

		/* And move back to the store's home page */
		home_url = g_strdup_printf ("%s%s", music_store->priv->base_url, U1_NOT_LOGGED_IN_STORE_URL);
		webkit_web_view_open (WEBKIT_WEB_VIEW (music_store->priv->web_viewer), home_url);
		g_free (home_url);

		return;
	}

	music_store->priv->auth_retries++;

	g_debug ("firing authentication");

	internal_uri = get_internal_html_page_url (WEBKIT_WEB_VIEW (music_store->priv->web_viewer), U1_CONNECTING_PAGE, NULL);
	load_delayed_page (WEBKIT_WEB_VIEW (music_store->priv->web_viewer), internal_uri, policy_decision);
	g_free (internal_uri);

	nrrd = g_new0 (NotRegisteredReplacementData, 1);
	nrrd->music_store = music_store;
	nrrd->not_registered_url = g_strdup (webkit_network_request_get_uri (request));

	auth = syncdaemon_daemon_get_authentication (music_store->priv->syncdaemon);

	g_signal_connect (auth, "credentials_found",
			  G_CALLBACK (got_new_credentials_cb), nrrd);
	g_signal_connect (auth, "authorization_cancelled",
			  G_CALLBACK (authorization_cancelled_cb), nrrd);
	g_signal_connect (auth, "error",
			  G_CALLBACK (credentials_error_cb), nrrd);
	syncdaemon_authentication_login_or_register (auth);
}

static gboolean
navigation_requested_cb (WebKitWebView *web_view,
			 WebKitWebFrame *frame,
			 WebKitNetworkRequest *request,
			 WebKitWebNavigationAction *navigation_action,
			 WebKitWebPolicyDecision *policy_decision,
			 gpointer user_data)
{
	U1MusicStore *music_store = U1_MUSIC_STORE (user_data);
	SoupURI *parsed_uri;
	gchar *internal_uri;
	gboolean return_val = FALSE;

	/* Hide the alertbar in case we should show something else */
	gtk_widget_hide (music_store->priv->alertbar);

	/* Remove watching callback for the page */
	if (music_store->priv->watch_id != 0) {
		g_source_remove (music_store->priv->watch_id);
		music_store->priv->watch_id = 0;
	}

	if (music_store->priv->watched_downloads != NULL) {
		g_hash_table_destroy (music_store->priv->watched_downloads);
		music_store->priv->watched_downloads = NULL;
	}

	/* Capture URLs we're interested in */
	parsed_uri = soup_uri_new (webkit_network_request_get_uri (request));
	if (parsed_uri == NULL)
		return FALSE;

	if (g_str_has_prefix (webkit_network_request_get_uri (request), U1_NOT_REGISTERED_URL)) {
		fire_authentication (music_store, request, policy_decision);
	} else if (g_str_has_prefix (parsed_uri->path, "/auth/login") &&
		   g_strrstr (parsed_uri->query, "next=") != NULL) {
		fire_authentication (music_store, request, policy_decision);
	} else if (g_str_has_prefix (parsed_uri->path, U1_TOKEN_INVALID_URL)) {
		syncdaemon_authentication_clear_token (syncdaemon_daemon_get_authentication (music_store->priv->syncdaemon));
		music_store->priv->auth_retries = 0;
		fire_authentication (music_store, request, policy_decision);
	} else if (g_strrstr (parsed_uri->path, "library.aspx") != NULL ||
		 g_str_has_prefix (parsed_uri->path, "/music/store/library")) {
		//g_str_has_prefix (webkit_network_request_get_uri (request), "https://one.ubuntu.com/music/store/library")) {
		const gchar *library_override;

		/* If the URL being loaded is the 7d library page, and we have a library
		   override set, load ours instead. */
		library_override = g_getenv ("U1MUSICLIBRARYURL");
		if (library_override != NULL &&
		    g_strcmp0 (library_override, webkit_network_request_get_uri (request)) != 0) {
			g_debug ("overriding the library page to ours");
			load_delayed_page (web_view, library_override, policy_decision);

			return_val = TRUE;
		}
	} else if (!g_strcmp0 ((const gchar *) parsed_uri->host, "www.7digital.com")) {
		/* host is 7digital.com. It must be in our store. */
		if (!g_str_has_prefix (parsed_uri->path, "/stores/")) {
			gchar *to_and_from_error_uris;

			/* a 7digital URL not in our store. Die */
			g_debug ("Hitting the unbranded store, so throwing an error");
			to_and_from_error_uris = g_strdup_printf ("%s:::%s", 
								  webkit_web_view_get_uri (web_view),
								  webkit_network_request_get_uri (request));

			internal_uri = get_internal_html_page_url (web_view, U1_OUTSIDE_DOMAIN_ERROR_PAGE, to_and_from_error_uris);
			load_delayed_page (web_view, internal_uri, policy_decision);

			g_free (internal_uri);
			g_free (to_and_from_error_uris);

			return_val = TRUE;
		}
	} else if (!g_strcmp0 ((const gchar *) parsed_uri->host, "one.ubuntu.com")) {
		/* host is one.ubuntu.com. It must not be the login page */
		if (g_str_has_prefix (parsed_uri->path, "/auth")) {
			gchar *real_url;
			gchar *to_and_from_error_uris;
			
			/* trying to log the user into o.u.c inside the webview. Die. */
			real_url = get_url_to_use (music_store);
			to_and_from_error_uris = g_strdup_printf ("%s:::%s:::%s", 
								  webkit_web_view_get_uri (web_view),
								  webkit_network_request_get_uri (request),
								  real_url);
			g_debug("Being led through Ubuntu One login inside Rhythmbox, so throwing an error");
			if (real_url != NULL) {
				internal_uri = get_internal_html_page_url (web_view, U1_TRYING_SSO_ERROR_PAGE, to_and_from_error_uris);
				g_free (real_url);
			} else {
				internal_uri = get_internal_html_page_url (web_view, U1_TRYING_SSO_ERROR_PAGE, NULL);
			}

		        load_delayed_page (web_view, internal_uri, policy_decision);

			g_free (internal_uri);
			g_free (to_and_from_error_uris);

			return_val = TRUE;
		}
	} else if (!g_strcmp0 ((const gchar *) parsed_uri->host, "login.ubuntu.com")) {
		gchar *real_url;
		gchar *to_and_from_error_uris;

		/* host is SSO. Die. */
		real_url = get_url_to_use (music_store);
		to_and_from_error_uris = g_strdup_printf ("%s:::%s:::%s", 
							  webkit_web_view_get_uri (web_view),
							  webkit_network_request_get_uri (request),
							  real_url);
		g_debug("Being led through Ubuntu One login inside Rhythmbox, so throwing an error");
		if (real_url != NULL) {
			internal_uri = get_internal_html_page_url (web_view, U1_TRYING_SSO_ERROR_PAGE, to_and_from_error_uris);
			g_free (real_url);
		} else {
			internal_uri = get_internal_html_page_url (web_view, U1_TRYING_SSO_ERROR_PAGE, NULL);
		}

		load_delayed_page (web_view, internal_uri, policy_decision);

		g_free (internal_uri);
		g_free (to_and_from_error_uris);

		return_val = TRUE;
	}

	/* Free memory */
	soup_uri_free (parsed_uri);

	return return_val;
}

static void
execute_script (WebKitWebView *web_view, const gchar *script_name, const gchar *replacement)
{
	gchar *file_contents, *script, *path;
	gsize length;
	GError *error = NULL;

	path = g_build_path ("/", U1_JAVASCRIPT_DIR, script_name, NULL);

	/* Load script from known location */
	if (g_file_get_contents (path, &file_contents, &length, &error)) {
		if (replacement != NULL)
			script = g_strdup_printf (file_contents, replacement);
		else
			script = g_strdup (file_contents);

		webkit_web_view_execute_script (web_view, script);

		g_free (script);
		g_free (file_contents);
	} else {
		g_warning ("Could not load %s: %s", path, error->message);
		g_error_free (error);
	}

	g_free (path);
}

static void
parse_html_node (U1MusicStore *music_store, WebKitWebView *web_view, htmlNodePtr node, GHashTable *current_downloads)
{
	htmlNodePtr subnode;
	GList *keys;
	static GHashTable *downloads_in_progress = NULL;

	if (node == NULL)
		return;

	/* We keep track of all the files being downloaded */
	if (downloads_in_progress == NULL)
		downloads_in_progress = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);

	keys = g_hash_table_get_keys (current_downloads);
	while (keys != NULL) {
		g_hash_table_insert (downloads_in_progress, g_strdup (keys->data), g_strdup (keys->data));
		keys = g_list_remove (keys, keys->data);
	}

	if (music_store->priv->watched_downloads == NULL)
		music_store->priv->watched_downloads = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);

	/* Search HTML for downloads */
	for (subnode = node->children; subnode != NULL; subnode = subnode->next) {
		if (!g_strcmp0 ((const gchar *) subnode->name, "li")) {
			xmlChar *prop;
			gchar *script = NULL, *full_path;
			GString *escaped_prop;
			gint i;
			SyncdaemonTransferInfo *download;

			prop = xmlGetProp (subnode, (xmlChar *) "data-path");
			if (prop == NULL)
				continue;

			/* Escape 's in the file name */
			escaped_prop = g_string_new (NULL);
			for (i = 0; i < strlen ((gchar *) prop); i++) {
				if (prop[i] == '\'')
					escaped_prop = g_string_append (escaped_prop, "\\\'");
				else
					escaped_prop = g_string_append_c (escaped_prop, prop[i]);
			}

			full_path = g_build_filename (g_get_home_dir (),
						      U1_MUSIC_LIBRARY_LOCATION,
						      prop, NULL);

			download = g_hash_table_lookup (current_downloads, full_path);
			if (download != NULL) {
				gdouble percent;
				glong total_size, bytes_read;

				total_size = syncdaemon_transfer_info_get_total_size (download);
				bytes_read = syncdaemon_transfer_info_get_bytes_transferred (download);

				if (total_size != 0)
					percent = (gdouble) bytes_read / (gdouble) total_size;
				else
					percent = 0.0;
				script = g_strdup_printf ("setProgressBar('%s', %.2f, '%.1fMB of %.1fMB downloaded');",
							  escaped_prop->str, percent,
							  (gdouble) bytes_read / (gdouble) 1024 / (gdouble) 1024,
							  (gdouble) total_size / (gdouble) 1024 / (gdouble) 1024);
			} else {
				/* If the file exists, the download is done */
				if (g_file_test (full_path, G_FILE_TEST_EXISTS)) {
					if (!g_hash_table_lookup (music_store->priv->watched_downloads, (gconstpointer) full_path)) {
						if (g_hash_table_lookup (downloads_in_progress, full_path)) {
							script = g_strdup_printf ("setProgressBar('%s', 1.0, 'Completed', false);",
										  escaped_prop->str);
							g_hash_table_remove (downloads_in_progress, full_path);
						} else {
							script = g_strdup_printf ("setProgressBar('%s', 1.0, 'Completed', true);",
										  escaped_prop->str);
						}

						g_signal_emit (music_store, u1_music_store_signals[DOWNLOAD_FINISHED_SIGNAL], 0,
							       (const gchar *) full_path);

						/* Add the file to watched downloads so that we don't signal again */
						g_hash_table_insert (music_store->priv->watched_downloads,
								     g_strdup (full_path),
								     g_strdup (full_path));
					}
				} else
					script = g_strdup_printf ("setProgressBar('%s', 0.0, 'Transferring to your Ubuntu One storage');",
								  escaped_prop->str);
			}

			if (script != NULL) {
				webkit_web_view_execute_script (web_view, script);
				g_free (script);
			}

			g_free (full_path);
			g_string_free (escaped_prop, TRUE);
		} else
			parse_html_node (music_store, web_view, subnode, current_downloads);
	}
}

static void
update_status_in_library_page (U1MusicStore *music_store, WebKitWebView *web_view, WebKitWebFrame *frame)
{
	GString *html;
	WebKitWebDataSource *data_source;
	htmlDocPtr html_doc;
	htmlNodePtr root_node;
	GSList *list;
	SyncdaemonInterface *interface;
	GHashTable *current_downloads;

	/* Only continue if we can make calls to SyncDaemon */
	if (!syncdaemon_daemon_is_ready (music_store->priv->syncdaemon))
		return;

	/* Retrieve contents of the page */
	data_source = webkit_web_frame_get_data_source (frame);
	html = webkit_web_data_source_get_data (data_source);

	/* Get list of current downloads */
	interface = syncdaemon_daemon_get_status_interface (music_store->priv->syncdaemon);
	if (interface == NULL) {
		g_warning ("Could not get /status interface for SyncDaemon");
		return;
	}

        list = syncdaemon_status_interface_get_current_downloads (SYNCDAEMON_STATUS_INTERFACE (interface));

	current_downloads = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, (GDestroyNotify) g_object_unref);

	while (list != NULL) {
		SyncdaemonTransferInfo *transfer_info = (SyncdaemonTransferInfo *) list->data;

		g_hash_table_insert (current_downloads,
				     g_strdup (syncdaemon_transfer_info_get_path (transfer_info)),
				     g_object_ref (G_OBJECT (transfer_info)));

		list = g_slist_remove (list, transfer_info);
	}

	/* Parse HTML */
	html_doc = htmlParseDoc ((xmlChar *) html->str, DEFAULT_ENCODING);
	root_node = xmlDocGetRootElement (html_doc);
	parse_html_node (music_store, web_view, root_node, current_downloads);

	/* Free memory */
	xmlFreeDoc (html_doc);
	g_hash_table_destroy (current_downloads);
}

static gboolean
poll_downloads_cb (gpointer user_data)
{
	U1MusicStore *music_store = U1_MUSIC_STORE (user_data);

	update_status_in_library_page (music_store,
				       WEBKIT_WEB_VIEW (music_store->priv->web_viewer),
				       music_store->priv->frame_to_use);

	return TRUE;
}

static void
_show_missing_codec_warning (U1MusicStore *music_store)
{
	gtk_label_set_text (GTK_LABEL (music_store->priv->alert_label),
			    _("MP3 playback support is not available. It must be installed to play Previews and Purchased Music on this computer."));

	if (music_store->priv->install_progress)
		gtk_widget_hide (music_store->priv->install_progress);

	gtk_widget_hide (music_store->priv->subscribe_btn);
	gtk_widget_show (music_store->priv->install_btn);
	gtk_widget_show (music_store->priv->alertbar);
}

static void
load_finished_cb (WebKitWebView *web_view, WebKitWebFrame *frame, gpointer user_data)
{
	const gchar *library_override;
	U1MusicStore *music_store = U1_MUSIC_STORE (user_data);
	SoupURI * uri;

	uri = soup_uri_new (webkit_web_view_get_uri (web_view));

	/* Remove the initial view if still around */
	if (music_store->priv->initial_web_viewer != NULL) {
		gtk_widget_destroy (music_store->priv->initial_web_viewer);
		music_store->priv->initial_web_viewer = NULL;

		/* And show the real store viewer */
		gtk_widget_show (music_store->priv->web_viewer);
		gtk_container_add (GTK_CONTAINER (music_store->priv->scroll), music_store->priv->web_viewer);
	}

	/* Inject some JavaScript to enable previews and hide the Flash player */
	execute_script (web_view, "u1-preview.js", NULL);

	/* Allow us to specifically override the library page links */
	library_override = g_getenv ("U1MUSICLIBRARYURL");
	if (library_override != NULL)
		execute_script (web_view, "u1-library-override.js", library_override);

	/* If the URL just loaded is the library page, update download status */
	if (g_str_has_prefix (uri->path, U1_LIBRARY_URL)) {
		/* Show the alert bar if folder is unsubscribed */
		if (!music_store->priv->folder_subscribed) {
			gtk_label_set_text (GTK_LABEL (music_store->priv->alert_label),
					    _("Your 'Purchased Music' folder is not subscribed. New purchases will not download to this computer."));
			gtk_widget_hide (music_store->priv->install_btn);
			gtk_widget_show (music_store->priv->subscribe_btn);
			gtk_widget_show (music_store->priv->alertbar);
		}

		/* Inject JavaScript to make songs clickable 
		   Needs to be done before update-status-in-library-page */
		execute_script (web_view, "u1-songs-clickable.js", NULL);

		music_store->priv->frame_to_use = frame;
		update_status_in_library_page (music_store, web_view, frame);

		/* Poll syncdaemon for downloads while the page is loaded */
		music_store->priv->watch_id = g_timeout_add (1000, (GSourceFunc) poll_downloads_cb, music_store);
	} else if (g_strcmp0 (uri->path, "/default.aspx") == 0 ||
		   g_strcmp0 (uri->path, "/stores/default.aspx") == 0) {
		/* Show the alert bar for codec installation */
		if (!music_store->priv->codec_installed) {
			_show_missing_codec_warning (music_store);
 		}
	}

	/* Signal users the URL finished loading */
	gtk_widget_hide (music_store->priv->progress);

	g_signal_emit (music_store, u1_music_store_signals[URL_LOADED_SIGNAL], 0,
		       (const gchar *) webkit_web_view_get_uri (WEBKIT_WEB_VIEW (music_store->priv->web_viewer)));

	soup_uri_free (uri);
}

static void
status_bar_text_changed_cb (WebKitWebView *web_view, const gchar *text, gpointer user_data)
{
	U1MusicStore *music_store = U1_MUSIC_STORE (user_data);

	if (g_str_has_prefix (text, "u1preview:::")) {
		gchar **args;

		/* Show the alert bar for codec installation */
		if (!music_store->priv->codec_installed) {
			_show_missing_codec_warning (music_store);
			return;
 		}

		args = g_strsplit (text, ":::", 0);
		if (args != NULL) {
			g_signal_emit (music_store, u1_music_store_signals[PREVIEW_MP3_SIGNAL], 0, args[1], args[2]);
			g_strfreev (args);
		}
	} else if (g_str_has_prefix (text, "u1playlibrary:::")) {
		gchar **args;

		/* Show the alert bar for codec installation */
		if (!music_store->priv->codec_installed) {
			_show_missing_codec_warning (music_store);
			return;
 		}

		args = g_strsplit (text, ":::", 2);
		if (args != NULL) {
			g_signal_emit (music_store, u1_music_store_signals[PLAY_LIBRARY_SIGNAL], 0, args[1]);
			g_strfreev (args);
		}
	} else if (g_str_has_prefix (text, "u1showcertificate:::")) {
		/* should show certificate here in its own window. */
	}

	
}

static gboolean
load_error_cb (WebKitWebView *web_view, WebKitWebFrame *frame, const gchar *uri, GError *error, gpointer user_data)
{
	if (error->domain == WEBKIT_NETWORK_ERROR) {
		load_internal_html_page (web_view, U1_DEFAULT_ERROR_PAGE, uri);
		return TRUE;
	}

	return FALSE;
}

static gboolean
load_real_store_cb (gpointer user_data)
{
	gchar *real_url;
	U1MusicStore *music_store = U1_MUSIC_STORE (user_data);

	g_debug ("Loading the real store page");
	real_url = get_url_to_use (music_store);
	if (real_url != NULL) {
		webkit_web_view_open (WEBKIT_WEB_VIEW (music_store->priv->web_viewer), real_url);

		/* Free memory */
		g_free (real_url);
	}

	music_store->priv->idle_cb = 0;

	return FALSE;
}

static void
viewer_property_changed_cb (GObject *object, GParamSpec *pspec, gpointer user_data)
{
	U1MusicStore *music_store = U1_MUSIC_STORE (user_data);

	if (g_strcmp0 (pspec->name, "progress") == 0) {
		gdouble progress = webkit_web_view_get_progress (WEBKIT_WEB_VIEW (music_store->priv->web_viewer));

		if (progress > 0.0 && progress < 1.0) {
			gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (music_store->priv->progress), progress);
			gtk_widget_show (music_store->priv->progress);
		} else
			gtk_widget_hide (music_store->priv->progress);
	}
}

static void
sd_folder_subscribed_cb (SyncdaemonDaemon *daemon, gboolean success,
			 SyncdaemonFolderInfo *folder_info, gpointer user_data)
{
	U1MusicStore *music_store = U1_MUSIC_STORE (user_data);
	const gchar *new_path = syncdaemon_folder_info_get_path (folder_info);
	const gchar *lib_path = u1_music_store_get_library_location (music_store);

	if (success && g_strcmp0 (new_path, lib_path) == 0) {
		music_store->priv->folder_subscribed = TRUE;
		gtk_widget_hide (music_store->priv->subscribe_btn);
		gtk_widget_hide (music_store->priv->alertbar);
	}
}

static void
got_gst_element_message (GstBus *bus, GstMessage *msg, gpointer user_data)
{
	U1MusicStore *music_store = U1_MUSIC_STORE (user_data);
	gboolean is_missing = FALSE;

	is_missing = gst_is_missing_plugin_message (msg);
	if (is_missing) {
		g_debug ("MP3 playback is missing.");
		gst_element_set_state (music_store->priv->pipeline,
				       GST_STATE_NULL);
		music_store->priv->codec_installed = FALSE;
	}
}

static void
got_gst_eos_message (GstBus *bus, GstMessage *msg, gpointer user_data)
{
	U1MusicStore *music_store = U1_MUSIC_STORE (user_data);

	gst_element_set_state (music_store->priv->pipeline, GST_STATE_NULL);
	if (g_getenv ("U1INSTALLMP3ANYWAY") != NULL)
		music_store->priv->codec_installed = FALSE;
	else
		music_store->priv->codec_installed = TRUE;
}

static gboolean
check_mp3_support (gpointer user_data)
{
	U1MusicStore *music_store = U1_MUSIC_STORE (user_data);
	GstBus *bus;
	GError *error = NULL;
	gchar *empty_file;
	gchar *empty_uri;
	gchar *launch;

	empty_file = g_build_filename (U1_JAVASCRIPT_DIR, "empty.mp3", NULL);
	empty_uri = g_filename_to_uri (empty_file, NULL, NULL);
	g_free (empty_file);

	launch = g_strdup_printf ("uridecodebin uri=%s ! fakesink", empty_uri);
	g_free (empty_uri);

	music_store->priv->pipeline = gst_parse_launch (launch, &error);
	if (error != NULL) {
		g_warning ("Error checking for MP3 support: %s",
			   error->message);
		music_store->priv->codec_installed = FALSE;
		return FALSE;
	}
	bus = gst_element_get_bus (music_store->priv->pipeline);
	gst_bus_add_signal_watch (bus);
	g_signal_connect (G_OBJECT (bus), "message::element",
			  G_CALLBACK (got_gst_element_message), music_store);
	g_signal_connect (G_OBJECT (bus), "message::eos",
			  G_CALLBACK (got_gst_eos_message), music_store);
	gst_element_set_state (music_store->priv->pipeline, GST_STATE_PLAYING);

	return FALSE;
}

static void
_install_codec_error (U1CodecInstaller *installer, GError *error,
		      gpointer user_data)
{
	U1MusicStore *music_store = U1_MUSIC_STORE (user_data);

	music_store->priv->codec_installed = FALSE;
	if (music_store->priv->install_progress) {
		gtk_widget_hide (music_store->priv->install_progress);
	}

	if (music_store->priv->install_btn)
		gtk_widget_hide (music_store->priv->install_btn);

	if (error->domain == U1_CODEC_INSTALLER_ERROR &&
	    error->code == U1_CODEC_INSTALLER_ERROR_UNSUPPORTED) {
		gtk_label_set_text (GTK_LABEL (music_store->priv->alert_label),
				    _("MP3 codec cannot be installed. Install is only supported on Ubuntu Linux."));
	} else {
		_show_missing_codec_warning (music_store);
	}

	g_error_free (error);
}

static void
_install_codec_started (U1CodecInstaller *installer, gpointer user_data)
{
	U1MusicStore *music_store = U1_MUSIC_STORE (user_data);

	if (music_store->priv->install_progress == NULL) {
		music_store->priv->install_progress = u1_codec_installer_get_progress_widget (installer);
		gtk_box_pack_end (GTK_BOX (music_store->priv->alertbar),
				  music_store->priv->install_progress,
				  TRUE, TRUE, 0);
	}

	gtk_widget_hide (music_store->priv->install_btn);
	gtk_widget_show (music_store->priv->install_progress);
}

static void
_install_codec_finished (U1CodecInstaller *installer, gpointer user_data)
{
	U1MusicStore *music_store = U1_MUSIC_STORE (user_data);

	gtk_widget_hide (music_store->priv->alertbar);
	gtk_widget_hide (music_store->priv->install_btn);
	gtk_widget_hide (music_store->priv->install_progress);

	music_store->priv->codec_installed = TRUE;
}

static void
_install_missing_codec (GtkButton *button, gpointer user_data)
{
	U1MusicStore *music_store = U1_MUSIC_STORE (user_data);

	u1_codec_installer_install_codec (music_store->priv->installer,
					  GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (music_store))));
}

static void
subscribe_purchased_folder (GtkButton *button, gpointer user_data)
{
	U1MusicStore *music_store = U1_MUSIC_STORE (user_data);
	SyncdaemonFolderInfo *folder_info;

	folder_info = syncdaemon_daemon_get_folder_info (music_store->priv->syncdaemon,
							 u1_music_store_get_library_location (music_store));
	syncdaemon_daemon_subscribe_folder (music_store->priv->syncdaemon,
					    syncdaemon_folder_info_get_volume_id (folder_info));
}

static void
_u1_purchased_enumerate (GObject *source, GAsyncResult *result,
                         gpointer user_data)
{
	U1MusicStore *music_store = U1_MUSIC_STORE (user_data);
	GFile *dir = G_FILE (source);
	GFileEnumerator *files;
	GFileInfo *info;
	GError *error = NULL;

	files = g_file_enumerate_children_finish (dir, result, &error);

	if (error != NULL) {
		g_warning ("Error rescanning Purchased Music: %s", error->message);
		g_error_free (error);
		return;
	}

	while ((info = g_file_enumerator_next_file (files, NULL, NULL)) != NULL) {
		gchar *path;

		path = g_build_filename (g_file_get_path (dir),
					 g_file_info_get_name (info),
					 NULL);
		if (g_file_test (path, G_FILE_TEST_IS_DIR)) {
			GFile *subdir = g_file_new_for_path (path);
			g_file_enumerate_children_async (subdir,
							 G_FILE_ATTRIBUTE_STANDARD_NAME,
                                        		 G_FILE_QUERY_INFO_NOFOLLOW_SYMLINKS,
                                        		 G_PRIORITY_DEFAULT, NULL,
                                        		 (GAsyncReadyCallback) _u1_purchased_enumerate,
                                        		 music_store);
		} else if (g_file_test (path, G_FILE_TEST_IS_REGULAR)) {
			g_signal_emit (music_store,
				       u1_music_store_signals[DOWNLOAD_FINISHED_SIGNAL], 0,
                                       (const gchar *) path);
		}

		g_free (path);
		g_object_unref (info);
	}
	g_file_enumerator_close (files, NULL, NULL);
	g_object_unref (dir);
}

static void
_u1_music_store_rescan_purchased_folder (U1MusicStore *music_store)
{
	GFile *dir;
	const gchar *path;

	path = u1_music_store_get_library_location (music_store);
	dir = g_file_new_for_path (path);
	g_file_enumerate_children_async (dir,
					 G_FILE_ATTRIBUTE_STANDARD_NAME,
					 G_FILE_QUERY_INFO_NOFOLLOW_SYMLINKS,
					 G_PRIORITY_DEFAULT, NULL,
					 (GAsyncReadyCallback) _u1_purchased_enumerate,
					 music_store);
}

static void
u1_music_store_init (U1MusicStore *music_store)
{
	gchar *new_user_agent;
	const gchar *url_to_use;
	SoupSession *session;
	GtkWidget *alert_icon;
	SyncdaemonFolderInfo *folder_info;

	if (!g_thread_get_initialized ())
		g_thread_init (NULL);

	gst_init (NULL, NULL);

	music_store->priv = g_new0 (U1MusicStorePrivate, 1);
	music_store->priv->watched_downloads = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
	music_store->priv->syncdaemon = syncdaemon_daemon_new ();

	g_signal_connect (G_OBJECT (music_store->priv->syncdaemon),
			  "folder_subscribed",
			  G_CALLBACK (sd_folder_subscribed_cb), music_store);

	/* Set up the installer in case we need it */
	music_store->priv->installer = u1_codec_installer_new ();
	g_signal_connect (G_OBJECT (music_store->priv->installer), "error",
			  G_CALLBACK (_install_codec_error), music_store);
	g_signal_connect (G_OBJECT (music_store->priv->installer), "started",
			  G_CALLBACK (_install_codec_started), music_store);
	g_signal_connect (G_OBJECT (music_store->priv->installer), "finished",
			  G_CALLBACK (_install_codec_finished), music_store);

	/* If U1MUSICSTOREURL is defined, use that instead of the real URL */
	if (! (url_to_use = g_getenv ("U1MUSICSTOREURL")))
		url_to_use = "https://one.ubuntu.com";

	music_store->priv->base_url = g_strdup (url_to_use);

	/* Create web viewer object */
	music_store->priv->scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (music_store->priv->scroll),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_widget_show (music_store->priv->scroll);

	music_store->priv->web_viewer = webkit_web_view_new ();

	new_user_agent = g_strdup_printf ("%s U1MusicStore/" VERSION ,
					  webkit_web_settings_get_user_agent (
						  webkit_web_view_get_settings (WEBKIT_WEB_VIEW (music_store->priv->web_viewer))));
	g_object_set (G_OBJECT (webkit_web_view_get_settings (WEBKIT_WEB_VIEW (music_store->priv->web_viewer))),
		      "user-agent", new_user_agent,
		      "default-encoding", DEFAULT_ENCODING,
		      "enable-default-context-menu", g_getenv ("U1SHOWCONTEXTMENU") != NULL,
		      "enable-plugins", FALSE,
		      NULL);
	g_free(new_user_agent);

	session = webkit_get_default_session ();

	/* Use GNOME proxy settings */
	soup_session_add_feature_by_type (session, SOUP_TYPE_PROXY_RESOLVER_GNOME);

	/* Add our handler for u1chrome: URIs */
	soup_session_add_feature_by_type (session, U1_TYPE_REQUEST_CHROME);

	g_signal_connect (G_OBJECT (music_store->priv->web_viewer), "navigation-policy-decision-requested",
				  G_CALLBACK (navigation_requested_cb), music_store);
	g_signal_connect (G_OBJECT (music_store->priv->web_viewer), "load-finished",
			  G_CALLBACK (load_finished_cb), music_store);
	g_signal_connect (G_OBJECT (music_store->priv->web_viewer), "status-bar-text-changed",
			  G_CALLBACK (status_bar_text_changed_cb), music_store);
	g_signal_connect (G_OBJECT (music_store->priv->web_viewer), "load-error",
			  G_CALLBACK (load_error_cb), music_store);

	/* Show a temp view when loading the initial store page */
	music_store->priv->initial_web_viewer = webkit_web_view_new ();
	g_object_set (G_OBJECT (webkit_web_view_get_settings (WEBKIT_WEB_VIEW (music_store->priv->initial_web_viewer))),
		      "user-agent", new_user_agent,
		      "default-encoding", DEFAULT_ENCODING,
		      "enable-default-context-menu", g_getenv ("U1SHOWCONTEXTMENU") != NULL,
		      "enable-plugins", FALSE,
		      NULL);
	load_internal_html_page (WEBKIT_WEB_VIEW (music_store->priv->initial_web_viewer), 
	                         U1_CONNECTING_PAGE, NULL);
	gtk_widget_show (music_store->priv->initial_web_viewer);
	gtk_container_add (GTK_CONTAINER (music_store->priv->scroll), music_store->priv->initial_web_viewer);

	/* Create the alert bar */
	music_store->priv->alertbar = gtk_hbox_new (FALSE, 12);
	alert_icon = gtk_image_new_from_stock (GTK_STOCK_DIALOG_WARNING,
					       GTK_ICON_SIZE_MENU);
	gtk_box_pack_start (GTK_BOX (music_store->priv->alertbar), alert_icon,
			    FALSE, FALSE, 0);
	gtk_widget_show (alert_icon);

	music_store->priv->alert_label = gtk_label_new ("");
	gtk_misc_set_alignment (GTK_MISC (music_store->priv->alert_label),
				0.0, 0.5);
	gtk_box_pack_start (GTK_BOX (music_store->priv->alertbar),
			    music_store->priv->alert_label,
			    TRUE, TRUE, 0);
	gtk_widget_show (music_store->priv->alert_label);

	/* Button to install MP3 codec if missing */
	music_store->priv->install_btn = gtk_button_new_with_label (_("Install"));
	gtk_box_pack_end (GTK_BOX (music_store->priv->alertbar),
			  music_store->priv->install_btn,
			  FALSE, FALSE, 6);
	g_signal_connect (G_OBJECT (music_store->priv->install_btn), "clicked",
			  G_CALLBACK (_install_missing_codec), music_store);

	/* Check that MP3 codec is available */
	g_timeout_add_seconds (3, (GSourceFunc) check_mp3_support, music_store);

	/* Button to subscribe the Purchased Music folder */
	music_store->priv->subscribe_btn = gtk_button_new_with_label (_("Subscribe"));
	gtk_box_pack_end (GTK_BOX (music_store->priv->alertbar),
			  music_store->priv->subscribe_btn,
			  FALSE, FALSE, 6);
	g_signal_connect (G_OBJECT (music_store->priv->subscribe_btn), "clicked",
			  G_CALLBACK (subscribe_purchased_folder), music_store);

	/* Figure out if the Purchased Music folder is subscribed */
	folder_info = syncdaemon_daemon_get_folder_info (music_store->priv->syncdaemon, u1_music_store_get_library_location (music_store));
	music_store->priv->folder_subscribed = syncdaemon_folder_info_get_subscribed (folder_info);

	/* Create a hidden progress bar */
	music_store->priv->progress = gtk_progress_bar_new ();
	
	g_signal_connect (G_OBJECT (music_store->priv->web_viewer), "notify",
			  G_CALLBACK (viewer_property_changed_cb), music_store);

	/* And load the real store in the background */
	music_store->priv->idle_cb = g_idle_add ((GSourceFunc) load_real_store_cb, music_store);

	gtk_box_pack_start (GTK_BOX (music_store), music_store->priv->alertbar, FALSE, FALSE, 6);
	gtk_box_pack_start (GTK_BOX (music_store), music_store->priv->scroll, TRUE, TRUE, 0);
	gtk_box_pack_end (GTK_BOX (music_store), music_store->priv->progress, FALSE, FALSE, 0);

	_u1_music_store_rescan_purchased_folder (music_store);
}

/**
 * u1_music_store_new:
 *
 * Create a new #U1MusicStore widget.
 *
 * Return value: the newly created widget.
 */
GtkWidget *
u1_music_store_new (void)
{
	U1MusicStore *music_store;

	music_store = g_object_new (U1_TYPE_MUSIC_STORE, NULL);

	return (GtkWidget *) music_store;
}

/**
 * u1_music_store_get_library_location:
 * @music_store: A #U1MusicStore object.
 *
 * Return the location where purchased music is stored on the user's
 * computer.
 *
 * Return value: Location of the music library.
 */
const gchar *
u1_music_store_get_library_location (U1MusicStore *music_store)
{
	static gchar *full_path = NULL;

	if (full_path == NULL) {
		full_path = g_build_filename (g_get_home_dir (),
					      U1_MUSIC_LIBRARY_LOCATION,
					      NULL);
	}

	return (const gchar *) full_path;
}

/**
 * u1_music_store_load_store_link:
 * @music_store: A #U1MusicStore object
 * @url: The URL to load
 *
 * Load a music store link on the given #U1MusicStore object, passing through
 * authentication of the user.
 */
void
u1_music_store_load_store_link (U1MusicStore *music_store, const gchar *url)
{
	gchar *real_url, *oauth_consumer_token, *oauth_consumer_secret, *oauth_token, *oauth_token_secret;

	g_return_if_fail (U1_IS_MUSIC_STORE (music_store));
	g_return_if_fail (url != NULL);

	/* If the load_real_store callback is set up, disable it first */
	if (music_store->priv->idle_cb > 0) {
		g_source_remove (music_store->priv->idle_cb);
		music_store->priv->idle_cb = 0;
	}

	real_url = g_strdup_printf ("%s%s?forward_on_to_url=%s",
				    music_store->priv->base_url,
				    U1_STORE_URL,
				    url);

	/* Sign the URL if we have OAuth tokens */
	get_credentials (music_store,
			 &oauth_consumer_token,
			 &oauth_consumer_secret,
			 &oauth_token,
			 &oauth_token_secret);
	if (oauth_consumer_token != NULL && oauth_consumer_secret != NULL &&
	    oauth_token != NULL && oauth_token_secret != NULL) {
		gchar *signed_url;

		signed_url = oauth_sign_url2 (real_url, NULL, OA_HMAC, "GET",
					      oauth_consumer_token, oauth_consumer_secret,
					      oauth_token, oauth_token_secret);
		webkit_web_view_open (WEBKIT_WEB_VIEW (music_store->priv->web_viewer), signed_url);

		g_free (signed_url);
	} else
		webkit_web_view_open (WEBKIT_WEB_VIEW (music_store->priv->web_viewer), real_url);

	g_free (real_url);
}
