/*
 * Copyright (C) 2011 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

using GLib;

namespace Unity {

/*
 * Previews
 */

public abstract class Preview : Object, Dee.Serializable
{
  public signal void closed ();

  public string title
  {
    get { return _raw.title; }
    set { _raw.title = value; }
  }
  public string subtitle
  {
    get { return _raw.subtitle; }
    set { _raw.subtitle = value; }
  }
  public string description_markup
  {
    get { return _raw.description; }
    set { _raw.description = value; }
  }
  public string image_source_uri
  {
    get { return _raw.image_source_uri; }
    set { _raw.image_source_uri = value; }
  }
  public Icon? image
  {
    get { return _raw.image; }
    set { _raw.image = value; }
  }

  private Protocol.Preview? _raw;
  internal unowned Protocol.Preview? get_raw ()
  {
    return _raw;
  }

  construct
  {
    _raw = create_raw () as Protocol.Preview;
    warn_if_fail (_raw != null);
  }

  // as a virtual method this will get into our main .h file and we don't want
  // to add dep on the unity-protocol library, therefore it returns Object
  // instead of Protocol.Preview
  internal abstract Object create_raw ();

  private GenericArray<PreviewAction> _actions =
    new GenericArray<PreviewAction> ();

  public void add_action (PreviewAction action)
  {
    _actions.add (action);

    _raw.add_action_with_hints (action.id, action.display_name,
                                action.icon_hint, action.layout_hint,
                                action.get_hints_internal ());
  }

  public void add_info (InfoHint info_hint)
  {
    info_hint.ref_sink ();
    _raw.add_info_hint (info_hint.id, info_hint.display_name,
                        info_hint.icon_hint, info_hint.data);
  }

  internal virtual async HashTable<string, Variant> update_property (HashTable<string, Variant> values)
  {
    _raw.update_property (values);

    Variant? action_v = values["base-preview-action"];
    if (action_v != null && action_v.get_string () == "closed")
    {
      closed (); //signal closing of the preview
    }
    return new HashTable<string, Variant> (str_hash, str_equal);
  }

  internal unowned GenericArray<PreviewAction> get_actions ()
  {
    return _actions;
  }

  private Variant serialize ()
  {
    return _raw.serialize ();
  }

  // preserving ABI compability in Vala? why not!
  internal virtual void dummy1 () {}
  internal virtual void dummy2 () {}
  internal virtual void dummy3 () {}
  internal virtual void dummy4 () {}
}

/* This is 1:1 copy of Protocol.LayoutHint, but we need to expose this
 * to our gir, we don't want to depend on UnityProtocol's gir */
public enum LayoutHint
{
  NONE,
  LEFT,
  RIGHT,
  TOP,
  BOTTOM
}

public class PreviewAction : Object, Dee.Serializable // TODO: Implement GLib.Action
{
  public string id { get; construct; }
  public string display_name { get; construct; }
  public string extra_text { get; set; }
  public Icon? icon_hint { get; construct; }
  public LayoutHint layout_hint { get; construct; }
  public HashTable<string, Variant>? hints { get { return hints_; } }

  private HashTable<string, Variant> hints_ =
    new HashTable<string, Variant> (str_hash, str_equal);

  public PreviewAction (string id, string display_name, Icon? icon_hint)
  {
    Object (id: id, display_name: display_name, icon_hint: icon_hint);
  }

  public PreviewAction.with_layout_hint (string id, string display_name,
                                         Icon? icon_hint, LayoutHint layout)
  {
    Object (id: id, display_name: display_name, icon_hint: icon_hint,
            layout_hint: layout);
  }

  public signal ActivationResponse activated (string uri);

  private Variant serialize ()
  {
    // FIXME: we should use PreviewActionRaw, but this is faster
    Variant tuple[5];

    tuple[0] = id;
    tuple[1] = display_name;
    tuple[2] = new Variant.string (icon_hint != null ? icon_hint.to_string () : "");
    tuple[3] = (uint) layout_hint;
    tuple[4] = get_hints_internal ();

    return new Variant.tuple (tuple);
  }

  internal unowned HashTable<string, Variant> get_hints_internal ()
  {
    if (extra_text != null && extra_text[0] != '\0')
      hints["extra-text"] = extra_text;

    return hints;
  }

  static construct
  {
    Dee.Serializable.register_parser (typeof (PreviewAction),
                                      new VariantType ("(sssua{sv})"),
                                      (data) =>
    {
      unowned string icon_hint = data.get_child_value (2).get_string ();
      Icon? icon = null;
      if (icon_hint != null && icon_hint != "")
      {
        try
        {
          icon = Icon.new_for_string (icon_hint);
        }
        catch (Error err)
        {
          warning ("Failed to deserialize GIcon: %s", err.message);
        }
      }

      return new PreviewAction.with_layout_hint (
        data.get_child_value (0).get_string (),
        data.get_child_value (1).get_string (),
        icon,
        (LayoutHint) data.get_child_value (3).get_uint32 ());
    });
  }

  // preserving ABI compability in Vala? why not!
  internal virtual void dummy1 () {}
  internal virtual void dummy2 () {}
  internal virtual void dummy3 () {}
  internal virtual void dummy4 () {}
}


public class InfoHint : InitiallyUnowned
{
  public string id { get; construct; }
  public string display_name { get; construct; }
  public Icon? icon_hint { get; construct; }
  public Variant data { get; construct; }

  public InfoHint (string id, string display_name, Icon? icon_hint,
                   string data)
  {
    Object (id: id, display_name: display_name, icon_hint: icon_hint,
            data: new Variant.string (data));
  }

  public InfoHint.with_variant (string id, string display_name,
                                Icon? icon_hint, Variant data)
  {
    Object (id: id, display_name: display_name, icon_hint: icon_hint,
            data: data);
  }
}


public class GenericPreview : Preview
{
  public GenericPreview (string title,
                         string description,
                         Icon? image)
  {
    Object (title: title, image: image,
            description_markup: description);
  }

  internal static GenericPreview empty ()
  {
    var preview = new GenericPreview ("", "", null);
    preview.get_raw ().set_no_details (true);

    return preview;
  }

  internal override Object create_raw ()
  {
    return new Protocol.GenericPreview ();
  }
}


public class ApplicationPreview : Preview
{
  public Icon app_icon
  {
    get { return _raw.app_icon; }
    set { _raw.app_icon = value; }
  }
  public string license
  {
    get { return _raw.license; }
    set { _raw.license = value; }
  }
  public string copyright
  {
    get { return _raw.copyright; }
    set { _raw.copyright = value; }
  }
  public string last_update
  {
    get { return _raw.last_update; }
    set { _raw.last_update = value; }
  }

  public ApplicationPreview (string title,
                             string subtitle,
                             string description,
                             Icon? icon,
                             Icon? screenshot)
  {
    Object (title: title, subtitle: subtitle, image: screenshot,
            description_markup: description, app_icon: icon);
  }

  public void set_rating (float rating, uint num_ratings)
  {
    _raw.rating = rating;
    _raw.num_ratings = num_ratings;
  }

  private unowned Protocol.ApplicationPreview _raw;
  internal override Object create_raw ()
  {
    var raw = new Protocol.ApplicationPreview ();
    _raw = raw;
    return _raw;
  }
}


public class MusicPreview : Preview
{
  /* Keep in sync with Protocol.PlayState! */
  public enum TrackState
  {
    STOPPED,
    PLAYING,
    PAUSED
  }

  private Dee.SharedModel _track_data;
  private float _current_progress;
  private string _current_track_uri;
  private TrackState _current_track_state;

  public string current_track_uri
  {
    get { return _current_track_uri; }
    set
    {
      if (_current_track_uri != value)
      {
        _current_track_uri = value;
        update_track_state ();
      }
    }
  }
  public float current_progress
  {
    get { return _current_progress; }
    set { _current_progress = value; update_track_state (); }
  }
  public TrackState current_track_state
  {
    get { return _current_track_state; }
    set
    {
      _current_track_state = value;
      if (_current_track_state == TrackState.STOPPED)
      {
        _current_progress = 0.0f;
      }
      update_track_state ();
    }
  }

  public MusicPreview (string title,
                       string subtitle,
                       Icon? image)
  {
    Object (title: title, subtitle: subtitle, image: image);
  }

  private unowned Protocol.MusicPreview _raw;
  internal override Object create_raw ()
  {
    var raw = new Protocol.MusicPreview ();
    _raw = raw;
    return _raw;
  }

  public void add_track (TrackMetadata track)
  {
    init_model ();

    _track_data.append (track.uri, track.track_no, track.title,
                        track.length, TrackState.STOPPED, 0.0);
  }

  private void update_track_state () requires (_track_data != null)
  {
    var model = _track_data;
    var iter = model.get_first_iter ();
    var end_iter = model.get_last_iter ();

    bool found_last_playing = false;
    // linear search... kindof eek, but there should be only ~20 tracks
    while (iter != end_iter)
    {
      if (model.get_string (iter, TrackDataColumns.URI) == _current_track_uri)
      {
        var row = model.get_row (iter);
        row[TrackDataColumns.PLAY_STATE] = new Variant.uint32 (_current_track_state);
        row[TrackDataColumns.PROGRESS] = (double) _current_progress;

        model.set_row (iter, row);
        if (found_last_playing) break;
      }
      else if (model.get_uint32 (iter, TrackDataColumns.PLAY_STATE) != TrackState.STOPPED)
      {
        found_last_playing = true;
        var row = model.get_row (iter);
        row[TrackDataColumns.PLAY_STATE] = new Variant.uint32 (TrackState.STOPPED);
        row[TrackDataColumns.PROGRESS] = 0.0;

        model.set_row (iter, row);
      }
      iter = model.next (iter);
    }
  }

  internal override async HashTable<string, Variant> update_property (HashTable<string, Variant> values)
  {
    Variant? action_v = values["action"];
    Variant? uri_v = values["uri"];
    if (action_v != null && uri_v != null)
    {
      switch (action_v.get_string ())
      {
        case "play":
          play (uri_v.get_string ());
          break;
        case "pause":
          pause (uri_v.get_string ());
          break;
        default:
          warning ("Unknown MusicPreview action: %s", action_v.get_string ());
          break;
      }
    }
    return yield base.update_property (values);
  }

  public signal void play (string uri);
  public signal void pause (string uri);

  // use add_info to add total number of tracks and "tags"

  private enum TrackDataColumns
  {
    URI,
    TRACK_NO,
    TITLE,
    LENGTH,
    PLAY_STATE,
    PROGRESS
  }

  private void init_model ()
  {
    if (_track_data == null)
    {
      string name = ("com.canonical.Unity.Scope.TrackData.T%" + int64.FORMAT)
        .printf (get_monotonic_time ());
      // FIXME: we'll probably want LEADER_WRITABLE access mode
      _track_data = new Dee.SharedModel (name);
      _track_data.set_schema ("s", "i", "s", "u", "u", "d");

      _raw.track_data_swarm_name = name;
    }
  }
}


public class PaymentPreview : Preview
{
  public enum Type {
    APPLICATION,
    MUSIC,
    ERROR
  }

  public string header
  {
    get { return _raw.header; }
    set { _raw.header = value;}
  }

  public string email
  {
    get { return _raw.email; }
    set { _raw.email = value; }
  }

  public string payment_method
  {
    get { return _raw.payment_method; }
    set { _raw.payment_method = value; }
  }

  public string purchase_prize
  {
    get { return _raw.purchase_prize; }
    set { _raw.purchase_prize = value; }
  }

  public string purchase_type
  {
    get { return _raw.purchase_type; }
    set { _raw.purchase_type = value; }
  }

  public Type preview_type
  {
    get { return (Type) _raw.preview_type; }
    set { _raw.preview_type = (Unity.Protocol.PreviewPaymentType) value; }
  }

  public PaymentPreview (string title, string subtitle, Icon? image)
  {
    Object (title: title, subtitle: subtitle, image: image);
  }

  public PaymentPreview.for_type (string title, string subtitle, Icon? image,
    Type type)
  {
    this (title, subtitle, image);
    preview_type = type;
  }

  public PaymentPreview.for_application (string title, string subtitle,
    Icon? image)
  {
    this.for_type (title, subtitle, image, Type.APPLICATION);
  }

  public PaymentPreview.for_music (string title, string subtitle, Icon? image)
  {
    this.for_type (title, subtitle, image, Type.MUSIC);
  }

  public PaymentPreview.for_error (string title, string subtitle, Icon? image)
  {
    this.for_type (title, subtitle, image, Type.ERROR);
  }

  static construct
  {
    // perform assertion so that we do not have bad castings if
    // someone forgot to keep the enums synced
    static_assert ((int) Protocol.PreviewPaymentType.APPLICATION ==
                   (int) Type.APPLICATION);
    static_assert ((int) Protocol.PreviewPaymentType.MUSIC ==
                   (int) Type.MUSIC);
    static_assert ((int) Protocol.PreviewPaymentType.ERROR ==
                   (int) Type.ERROR);
  }

  private unowned Protocol.PaymentPreview _raw;
  internal override Object create_raw ()
  {
    var raw = new Protocol.PaymentPreview ();
    _raw = raw;
    return _raw;
  }
}


public class MoviePreview : Preview
{
  public string year
  {
    get { return _raw.year; }
    set { _raw.year = value; }
  }

  public MoviePreview (string title, string subtitle, string description,
                       Icon? image)
  {
    Object (title: title, subtitle: subtitle, description_markup: description,
            image: image);
  }

  public void set_rating (float rating, uint num_ratings)
  {
    _raw.rating = rating;
    _raw.num_ratings = num_ratings;
  }

  private unowned Protocol.MoviePreview _raw;
  internal override Object create_raw ()
  {
    var raw = new Protocol.MoviePreview ();
    _raw = raw;
    return _raw;
  }
}


public class SocialPreview : Preview
{
  public class Comment : InitiallyUnowned
  {
    public string id { get; construct; }
    public string name { get; construct; }
    public string text { get; construct; }
    public string time { get; construct; }

    public Comment (string id, string name, string text, string time)
    {
      Object (id: id, name: name, text: text, time: time);
    }
  }

  public Icon avatar
  {
    get { return _raw.avatar; }
    set { _raw.avatar = value; }
  }
  public string content
  {
    get { return _raw.description; }
    set { _raw.description = value; }
  }
  public string sender
  {
    get { return _raw.sender; }
    set { _raw.sender = value; }
  }

  public SocialPreview (string sender,
                             string subtitle,
                             string content,
                             Icon? avatar)
  {
    Object (title: sender, subtitle: subtitle, content: content,
            avatar: avatar);
  }

  private unowned Protocol.SocialPreview _raw;
  internal override Object create_raw ()
  {
    var raw = new Protocol.SocialPreview ();
    _raw = raw;
    return _raw;
  }

  public void add_comment (Comment comment)
  {
    _raw.add_comment (comment.id, comment.name, comment.text, comment.time);
  }

}

/* No support for SeriesPreview yet */
internal class SeriesItem : Object
{
  public string uri { get; construct; }
  public string title { get; construct; }
  public Icon? icon_hint { get; construct; }

  public SeriesItem (string uri, string title, Icon? icon_hint)
  {
    Object (uri: uri, title: title, icon_hint: icon_hint);
  }
}

internal class SeriesPreview : Preview, Dee.Serializable
{
  private string _current_item_uri;
  public string current_item_uri
  {
    get { return _current_item_uri; }
    construct { _current_item_uri = value; }
  }

  // FIXME: why do we even keep this?
  private int _selected_item = -1;

  public SeriesPreview (SeriesItem[] items, string selected_item_uri)
  {
    // careful current_item_uri will be set before items are
    Object (current_item_uri: selected_item_uri);
    foreach (unowned SeriesItem item in items)
    {
      add_item (item);
    }
  }

  private void add_item (SeriesItem item)
  {
    _raw.add_series_item (item.title, item.uri, item.icon_hint);
  }

  /* Emits request_item_preview and updates internal state */
  private void update_child_preview (string uri, int item_index = -1)
  {
    var preview = request_item_preview (uri);
    // FIXME: this needs to handle AsyncPreview
    if (preview != null)
    {
      if (item_index < 0)
      {
        // try to auto-update _selected_item
        int i = 0;
        foreach (unowned Protocol.SeriesItemRaw item in _raw.get_items ())
        {
          if (item.uri == uri)
          {
            _selected_item = i;
            break;
          }
          i++;
        }
        if (_selected_item < 0) warning ("Unable to find URI in series items");
      }
      else _selected_item = item_index;

      _child_preview = preview;
      _raw.child_preview = preview.get_raw ();
      if (_selected_item >= 0 && _raw.selected_item != _selected_item)
        _raw.selected_item = _selected_item;
      _current_item_uri = uri;
    }
  }

  private Preview _child_preview;

  public signal Preview? request_item_preview (string uri);

  internal override async HashTable<string, Variant> update_property (HashTable<string, Variant> values)
  {
    var response = yield base.update_property (values);
    int new_selected_item = _raw.selected_item;
    Preview? preview = null;
    if (new_selected_item != _selected_item &&
        new_selected_item < _raw.get_items ().length)
    {
      unowned Protocol.SeriesItemRaw item = _raw.get_items ()[new_selected_item];
      update_child_preview (item.uri, new_selected_item);
    }
    preview = _child_preview;
    response["preview"] = preview.serialize ();
    return response;
  }

  internal Preview? get_active_preview ()
  {
    return _child_preview;
  }

  private Variant serialize ()
  {
    // FIXME: this won't work with AsyncPreview, but we can't really emit
    // the request_item_preview signal sooner, because the user of this API
    // needs time to construct the instance and connect to the signal
    // We seem to need extra init() method.

    // ensures that we emit selected_item_changed shortly after construction
    if (_selected_item < 0 && _raw.get_items ().length > 0)
    {
      update_child_preview (current_item_uri);
    }
    return _raw.serialize ();
  }

  private unowned Protocol.SeriesPreview _raw;
  internal override Object create_raw ()
  {
    var raw = new Protocol.SeriesPreview ();
    _raw = raw;
    return _raw;
  }
}


public class AsyncPreview : Preview
{
  public AsyncPreview ()
  {
    Object ();
  }

  public AsyncPreview.with_cancellable (Cancellable cancellable)
  {
    Object (cancellable: cancellable);
  }

  public Cancellable cancellable { get; construct set; }

  public signal void preview_ready (Preview? preview);

  // simple emit method, so it's easy to use in C/Python
  [CCode (cname = "unity_async_preview_preview_ready")]
  public void emit_preview_ready (Preview? preview)
    requires (!(preview is AsyncPreview))
  {
    preview_ready (preview);
  }

  internal override Object create_raw ()
  {
    return new Protocol.GenericPreview ();
  }

  internal async Preview? wait_for_signal ()
  {
    Preview? preview = null;
    var sig_id = this.preview_ready.connect ((preview_) =>
    {
      if (preview_ != null && !(preview_ is AsyncPreview))
      {
        preview = preview_;
      }
      wait_for_signal.callback ();
    });

    ulong canc_id = 0;
    Cancellable? canc = null;
    if (cancellable != null)
    {
      canc = cancellable;
      canc_id = canc.connect (() =>
      {
        SignalHandler.block (this, sig_id);
        // the cancellable's lock is held in the callback, we'd deadlock if we
        // tried to continue the callback and disconnect right away
        Idle.add (wait_for_signal.callback);
      });
    }

    yield;

    // cleanup
    SignalHandler.disconnect (this, sig_id);
    if (canc != null) canc.disconnect (canc_id);

    return preview;
  }
}

} /* namespace */
