# Copyright 2012-2015 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

""":class:`NodeResult` model."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

str = None

__metaclass__ = type
__all__ = [
    'NodeResult',
    ]


from django.db.models import (
    CharField,
    ForeignKey,
    IntegerField,
)
from maasserver.models.cleansave import CleanSave
from maasserver.models.timestampedmodel import TimestampedModel
from metadataserver import DefaultMeta
from metadataserver.enum import (
    RESULT_TYPE,
    RESULT_TYPE_CHOICES,
)
from metadataserver.fields import BinaryField


class NodeResult(CleanSave, TimestampedModel):
    """Storage for data returned from node commissioning/installation.

    Commissioning/Installing a node results in various bits of data that
    need to be stored, such as lshw output.  This model allows storing of
    this data as unicode text, with an arbitrary name, for later retrieval.

    :ivar node: The context :class:`Node`.
    :ivar script_result: If this data results from the execution of a script,
        this is the status of this execution.  This can be "OK", "FAILED" or
        "WORKING" for progress reports.
    :ivar result_type: This can be either commissioning or installation.
    :ivar name: A unique name to use for the data being stored.
    :ivar data: The file's actual data, unicode only.
    """

    class Meta(DefaultMeta):
        unique_together = ('node', 'name')

    node = ForeignKey(
        'maasserver.Node', null=False, editable=False, unique=False)
    script_result = IntegerField(editable=False)
    result_type = IntegerField(
        choices=RESULT_TYPE_CHOICES, editable=False,
        default=RESULT_TYPE.COMMISSIONING)
    name = CharField(max_length=255, unique=False, editable=False)
    data = BinaryField(
        max_length=1024 * 1024, editable=True, blank=True, default=b'',
        null=False)
