/*
 * Copyright © 2015 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored By: Alan Griffiths <alan@octopull.co.uk>
 */

#include "default_shell.h"
#include "mir/shell/input_targeter.h"
#include "mir/scene/placement_strategy.h"
#include "mir/scene/prompt_session.h"
#include "mir/scene/prompt_session_manager.h"
#include "mir/scene/session_coordinator.h"
#include "mir/scene/session.h"
#include "mir/scene/surface.h"
#include "mir/scene/surface_configurator.h"
#include "mir/scene/surface_coordinator.h"
#include "mir/scene/surface_creation_parameters.h"

namespace mf = mir::frontend;
namespace ms = mir::scene;
namespace msh = mir::shell;

msh::DefaultShell::DefaultShell(
    std::shared_ptr<InputTargeter> const& input_targeter,
    std::shared_ptr<scene::SurfaceCoordinator> const& surface_coordinator,
    std::shared_ptr<scene::SessionCoordinator> const& session_coordinator,
    std::shared_ptr<scene::PromptSessionManager> const& prompt_session_manager,
    std::shared_ptr<ms::PlacementStrategy> const& placement_strategy,
    std::shared_ptr<ms::SurfaceConfigurator> const& surface_configurator) :
    AbstractShell(input_targeter, surface_coordinator, session_coordinator, prompt_session_manager),
    placement_strategy{placement_strategy},
    surface_configurator{surface_configurator}
{
}

std::shared_ptr<ms::Session> msh::DefaultShell::open_session(
    pid_t client_pid,
    std::string const& name,
    std::shared_ptr<mf::EventSink> const& sink)
{
    auto const new_session = AbstractShell::open_session(client_pid, name, sink);
    set_focus_to(new_session);
    return new_session;
}

void msh::DefaultShell::close_session(
    std::shared_ptr<ms::Session> const& session)
{
    AbstractShell::close_session(session);
    set_focus_to(session_coordinator->successor_of(std::shared_ptr<ms::Session>()));
}

void msh::DefaultShell::handle_surface_created(
    std::shared_ptr<ms::Session> const& session)
{
    AbstractShell::handle_surface_created(session);
    set_focus_to(session);
}

mf::SurfaceId msh::DefaultShell::create_surface(std::shared_ptr<ms::Session> const& session, ms::SurfaceCreationParameters const& params)
{
    return AbstractShell::create_surface(session, placement_strategy->place(*session, params));
}

int msh::DefaultShell::set_surface_attribute(
    std::shared_ptr<ms::Session> const& session,
    std::shared_ptr<ms::Surface> const& surface,
    MirSurfaceAttrib attrib,
    int value)
{
    auto const configured_value = surface_configurator->select_attribute_value(*surface, attrib, value);

    auto const result = AbstractShell::set_surface_attribute(session, surface, attrib, configured_value);

    surface_configurator->attribute_set(*surface, attrib, result);

    return result;
}

void msh::DefaultShell::setting_focus_to(std::shared_ptr<ms::Surface> const& surface)
{
    if (surface)
        surface_coordinator->raise(surface);
}
