//
// RSAOAEPKeyExchangeDeformatterTest.cs - NUnit Test Cases for RSAOAEPKeyExchangeDeformatter
//
// Author:
//	Sebastien Pouliot (sebastien@ximian.com)
//
// (C) 2002, 2003 Motus Technologies Inc. (http://www.motus.com)
// Copyright (C) 2004-2006 Novell, Inc (http://www.novell.com)
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

using NUnit.Framework;
using System;
using System.Security.Cryptography;

namespace MonoTests.System.Security.Cryptography {

	[TestFixture]
	public class RSAOAEPKeyExchangeDeformatterTest : Assertion {

		protected static RSA key;

		[SetUp]
		public void SetUp () 
		{
			// generating a keypair is REALLY long and the framework
			// makes sure that we generate one (even if create an object
			// to import an existing key)
			if (key == null) {
				key = RSA.Create ();
				key.ImportParameters (AllTests.GetRsaKey (true));
			}
		}

		public void AssertEquals (string msg, byte[] array1, byte[] array2) 
		{
			AllTests.AssertEquals (msg, array1, array2);
		}

		// LAMESPEC: RSAOAEPKeyExchangeDeformatter.RNG versus RSAOAEPKeyExchangeFormatter.Rng
		[Test]
		public void Properties () 
		{
			RSAOAEPKeyExchangeDeformatter keyex = new RSAOAEPKeyExchangeDeformatter ();
			keyex.SetKey (key);
			Assertion.AssertNull("RSAOAEPKeyExchangeDeformatter.Parameters", keyex.Parameters);
			Assertion.AssertEquals("RSAOAEPKeyExchangeDeformatter.ToString()", "System.Security.Cryptography.RSAOAEPKeyExchangeDeformatter", keyex.ToString ());
		}

		// TestExchangeMin (1)
		// Test vector (EM) generated by CryptoAPI on Windows XP
		[Test]
		public void CapiExchangeMin () 
		{
			byte[] M = { 0x01 };
			byte[] EM = { 0x4D, 0x5F, 0x26, 0x67, 0x88, 0x07, 0xF1, 0xB5, 0x85, 0x33, 0xDA, 0x8B, 0xD6, 0x9E, 0x9B, 0x56, 0xE6, 0x54, 0x49, 0xF3, 0x4A, 0x8B, 0x5B, 0xB2, 0xCB, 0x10, 0x50, 0x91, 0x85, 0xBC, 0xE1, 0x5E, 0xDC, 0x8A, 0xAE, 0x63, 0xAD, 0x57, 0xC3, 0x03, 0x78, 0xDD, 0xCE, 0xCD, 0x46, 0x71, 0x43, 0x2F, 0x7F, 0xA4, 0x6A, 0x93, 0xE4, 0xB7, 0x4C, 0x36, 0x92, 0x5C, 0xFF, 0xAC, 0x32, 0xE8, 0x71, 0x1E, 0x69, 0xD6, 0x8C, 0xB5, 0x65, 0xA5, 0x83, 0xC2, 0x3F, 0xED, 0x35, 0x6F, 0x03, 0x0D, 0x4B, 0xE1, 0x0F, 0x7B, 0x5D, 0x0A, 0x68, 0x90, 0x71, 0xA0, 0xD6, 0xC0, 0xEC, 0x6B, 0x24, 0x45, 0x3D, 0x9C, 0x11, 0xD9, 0xC8, 0xFB, 0xC8, 0x8F, 0xDB, 0x48, 0x70, 0x1F, 0xC4, 0x7D, 0x83, 0x58, 0x95, 0xDC, 0x36, 0xA0, 0x68, 0xE1, 0xCF, 0x2D, 0xA9, 0x8A, 0xF0, 0xB8, 0xA6, 0x0D, 0x6E, 0xA6, 0x9C, 0xBC };

			AsymmetricKeyExchangeDeformatter keyback = new RSAOAEPKeyExchangeDeformatter (key);
			try {
				byte[] Mback = keyback.DecryptKeyExchange (EM);
				AssertEquals ("RSAOAEPKeyExchangeDeformatter 1", M, Mback);
			}
			catch (CryptographicException ce) {
				// not supported by every version of Windows - Minimum: Windows XP
				Console.WriteLine (ce.Message + " (" + Environment.OSVersion.ToString () + ")");
			}
		}

		// test with a message 128 bits (16 bytes) long
		// Test vector (EM) generated by CryptoAPI on Windows XP
		[Test]
		public void CapiExchange128 () 
		{
			byte[] M = { 0xd4, 0x36, 0xe9, 0x95, 0x69, 0xfd, 0x32, 0xa7, 0xc8, 0xa0, 0x5b, 0xbc, 0x90, 0xd3, 0x2c, 0x49 };
			byte[] EM = { 0x6D, 0xAA, 0xBC, 0x7C, 0xE7, 0x7C, 0xE6, 0x16, 0x72, 0x6F, 0xBE, 0xAF, 0x26, 0xBF, 0x87, 0x77, 0xB5, 0x47, 0x25, 0x37, 0x01, 0xC7, 0xD8, 0x18, 0xA8, 0x09, 0x0F, 0x1F, 0x13, 0x24, 0xD0, 0x2A, 0xD4, 0xDC, 0x5A, 0x2D, 0x91, 0x6D, 0x11, 0x6C, 0x1A, 0x67, 0xB2, 0xFC, 0xA3, 0xE7, 0xF4, 0xEF, 0xDC, 0xEC, 0xB2, 0xAF, 0x42, 0xB5, 0xD9, 0x7A, 0x6D, 0xE5, 0x76, 0xBA, 0x7D, 0x04, 0xCA, 0x17, 0xB0, 0x06, 0x88, 0x4D, 0xE8, 0xCC, 0x82, 0xEA, 0x13, 0xC8, 0x12, 0x05, 0x05, 0xC6, 0xB0, 0x4B, 0x53, 0x1D, 0x07, 0xFA, 0x09, 0x64, 0x30, 0xCD, 0x5C, 0xE3, 0x41, 0xB1, 0x25, 0x94, 0x05, 0x5A, 0xC2, 0xD8, 0x2E, 0xC4, 0x39, 0x51, 0x73, 0x75, 0x0C, 0x4B, 0x2F, 0x4B, 0xAD, 0x04, 0x54, 0x32, 0x30, 0x94, 0xF4, 0xFE, 0x92, 0xF4, 0xB4, 0x54, 0x07, 0x22, 0xCD, 0xB5, 0xB2, 0x01, 0x95, 0xA0 };

			AsymmetricKeyExchangeDeformatter keyback = new RSAOAEPKeyExchangeDeformatter (key);
			try {
				byte[] Mback = keyback.DecryptKeyExchange (EM);
				AssertEquals ("RSAOAEPKeyExchangeFormatter 1", M, Mback);
			}
			catch (CryptographicException ce) {
				// not supported by every version of Windows - Minimum: Windows XP
				Console.WriteLine (ce.Message + " (" + Environment.OSVersion.ToString () + ")");
			}
		}

		// test with a message 160 bits (20 bytes) long
		// Test vector (EM) generated by CryptoAPI on Windows XP
		[Test]
		public void CapiExchange192 () 
		{
			byte[] M = { 0xd4, 0x36, 0xe9, 0x95, 0x69, 0xfd, 0x32, 0xa7, 0xc8, 0xa0, 0x5b, 0xbc, 0x90, 0xd3, 0x2c, 0x49, 0x00, 0x00, 0x00, 0x00 };
			byte[] EM = { 0x72, 0x0A, 0x0E, 0xA0, 0x8C, 0x40, 0x69, 0x9D, 0x78, 0xBC, 0x94, 0x4B, 0x2C, 0x14, 0xC0, 0xC8, 0x13, 0x8B, 0x6D, 0x2F, 0x01, 0xE3, 0x45, 0x33, 0xE7, 0x9B, 0x87, 0xDB, 0x37, 0xBD, 0xA9, 0x63, 0xCC, 0x94, 0x8F, 0xBB, 0x1D, 0x61, 0xA1, 0x5E, 0x6A, 0x33, 0xBD, 0xD5, 0xC8, 0x22, 0x40, 0xB1, 0x32, 0xFB, 0x6F, 0x2A, 0x3B, 0x8A, 0x15, 0x4E, 0xC1, 0x25, 0xF0, 0x34, 0x17, 0x9A, 0x68, 0x34, 0x27, 0x5B, 0x49, 0xC5, 0xEA, 0x4D, 0x7F, 0x07, 0x4C, 0xAC, 0xF8, 0xE3, 0xD7, 0x9E, 0x72, 0xB0, 0xD1, 0xAD, 0x9E, 0x9C, 0xBB, 0xC6, 0x79, 0x14, 0x63, 0x5E, 0x13, 0xD1, 0x02, 0xDE, 0x57, 0xB5, 0xBC, 0x57, 0x68, 0xAC, 0xE3, 0xEF, 0x79, 0xED, 0xAF, 0xC0, 0xBB, 0xFE, 0xB3, 0xA6, 0x4C, 0xE8, 0x79, 0xE2, 0x3A, 0xB3, 0x37, 0x97, 0x90, 0x05, 0xCF, 0xB9, 0x1A, 0x92, 0xEC, 0x73, 0xC5, 0x8D };

			AsymmetricKeyExchangeDeformatter keyback = new RSAOAEPKeyExchangeDeformatter (key);
			try {
				byte[] Mback = keyback.DecryptKeyExchange (EM);
				AssertEquals ("RSAOAEPKeyExchangeFormatter 1", M, Mback);
			}
			catch (CryptographicException ce) {
				// not supported by every version of Windows - Minimum: Windows XP
				Console.WriteLine (ce.Message + " (" + Environment.OSVersion.ToString () + ")");
			}
		}

		// Max = (key size in bytes) - 2 * (hash length) - 2
		// Test vector (EM) generated by CryptoAPI on Windows XP
		[Test]
		public void CapiExchangeMax () 
		{
			byte[] M = new byte [(key.KeySize >> 3) - 2 * 20 - 2];
			byte[] EM = { 0x45, 0x31, 0xA7, 0x4A, 0xE5, 0x8B, 0xC7, 0x0C, 0x5C, 0x8B, 0xE6, 0xAB, 0xC5, 0x99, 0xF3, 0x31, 0xB9, 0x64, 0xFA, 0x19, 0x4A, 0x41, 0xF3, 0x49, 0x05, 0x0B, 0x28, 0xD5, 0x96, 0x43, 0x0E, 0xEB, 0x62, 0x0B, 0x22, 0xFE, 0x5F, 0x92, 0x4F, 0x60, 0xB9, 0xAE, 0x7F, 0xAA, 0xC8, 0x82, 0x66, 0xD7, 0x19, 0xCF, 0xC0, 0x69, 0x1C, 0xAA, 0x7E, 0x95, 0x5D, 0x70, 0x3D, 0x4A, 0x1D, 0x3B, 0x55, 0xBC, 0x7D, 0x36, 0xCF, 0x98, 0x09, 0xF8, 0x20, 0x92, 0x34, 0x79, 0x0B, 0x1A, 0x91, 0xE5, 0xFB, 0x94, 0xAD, 0x2A, 0x15, 0x6E, 0x3D, 0xF4, 0xA5, 0x6F, 0x6B, 0xAE, 0x77, 0x80, 0x3C, 0xF5, 0xCC, 0x84, 0x3A, 0xF9, 0xCE, 0x9F, 0x06, 0xF6, 0xCC, 0xA8, 0x75, 0xE1, 0x55, 0x56, 0xA5, 0x76, 0x50, 0x28, 0xA7, 0x3E, 0x91, 0x11, 0x5C, 0x82, 0x7C, 0x1A, 0x92, 0x02, 0x74, 0xCC, 0xA9, 0x6C, 0xFC, 0xA4 };

			AsymmetricKeyExchangeDeformatter keyback = new RSAOAEPKeyExchangeDeformatter (key);
			try {
				byte[] Mback = keyback.DecryptKeyExchange (EM);
				AssertEquals ("RSAOAEPKeyExchangeFormatter 1", M, Mback);
			}
			catch (CryptographicException ce) {
				// not supported by every version of Windows - Minimum: Windows XP
				Console.WriteLine (ce.Message + " (" + Environment.OSVersion.ToString () + ")");
			}
		}

		// TestExchangeMin (1)
		// Test vector (EM) generated by Mono on Windows
		[Test]
		public void MonoExchangeMin() 
		{
			byte[] M = { 0x01 };
			byte[] EM = { 0x75, 0x06, 0x33, 0xB4, 0x47, 0xDB, 0x9C, 0x72, 0x1A, 0x09, 0x48, 0xEA, 0xF3, 0xED, 0xCF, 0xA7, 0x6D, 0x09, 0x8E, 0xF5, 0xC9, 0x70, 0x18, 0xF7, 0x22, 0xCC, 0xEC, 0xCB, 0xD3, 0x2B, 0xE7, 0xE3, 0x0A, 0x47, 0xF4, 0xE1, 0x66, 0x73, 0xEA, 0x7F, 0x5B, 0x27, 0x32, 0x62, 0x9D, 0x0C, 0x2F, 0x88, 0x2E, 0x3A, 0x37, 0x6C, 0x0C, 0x09, 0x1D, 0xF3, 0xE1, 0x80, 0x66, 0x26, 0x29, 0x97, 0x3D, 0xB7, 0xE6, 0x01, 0xD0, 0x88, 0x9C, 0x68, 0x3F, 0x62, 0x7C, 0x71, 0x36, 0xDA, 0xC3, 0xE2, 0x93, 0x20, 0x95, 0xF8, 0x15, 0x89, 0x73, 0x24, 0xDF, 0xFB, 0x72, 0xD9, 0x63, 0x19, 0x84, 0x41, 0x92, 0x3C, 0x71, 0x36, 0xB9, 0x09, 0x24, 0xD6, 0x8E, 0x2D, 0x5A, 0xF2, 0x4B, 0xF4, 0xCD, 0x58, 0xEB, 0x4E, 0xE4, 0x97, 0xB3, 0x09, 0xED, 0xCF, 0x6A, 0x6D, 0xBC, 0xE4, 0xD4, 0x8D, 0xEC, 0xA6, 0x35, 0x53 };

			AsymmetricKeyExchangeDeformatter keyback = new RSAOAEPKeyExchangeDeformatter (key);
			try {
				byte[] Mback = keyback.DecryptKeyExchange (EM);
				AssertEquals ("RSAOAEPKeyExchangeDeformatter Min", M, Mback);
			}
			catch (CryptographicException ce) {
				// not supported by every version of Windows - Minimum: Windows XP
				Console.WriteLine (ce.Message + " (" + Environment.OSVersion.ToString () + ")");
			}
		}

		// test with a message 128 bits (16 bytes) long
		// Test vector (EM) generated by Mono on Windows
		[Test]
		public void MonoExchange128() 
		{
			byte[] M = { 0xd4, 0x36, 0xe9, 0x95, 0x69, 0xfd, 0x32, 0xa7, 0xc8, 0xa0, 0x5b, 0xbc, 0x90, 0xd3, 0x2c, 0x49 };
			byte[] EM = { 0x3A, 0x18, 0xE7, 0xF7, 0x45, 0x32, 0x5B, 0x60, 0x46, 0x2A, 0x20, 0x1A, 0x69, 0x98, 0x84, 0x68, 0x76, 0xC0, 0x01, 0x7D, 0x37, 0xE7, 0xEB, 0x72, 0x18, 0xD7, 0xF3, 0xE5, 0x1B, 0x2C, 0xB2, 0x47, 0x86, 0x9D, 0x90, 0xE8, 0x38, 0x43, 0x0C, 0x58, 0x59, 0xDB, 0x2A, 0x46, 0xBA, 0x15, 0xD9, 0x79, 0x77, 0xB5, 0x8B, 0xA7, 0x06, 0x15, 0xE1, 0xBF, 0xA2, 0xA7, 0x69, 0xC6, 0x6A, 0x6C, 0x4F, 0x87, 0xA3, 0xA1, 0xBC, 0x21, 0x46, 0x68, 0x61, 0xEC, 0x1E, 0xE1, 0x2B, 0xEF, 0xD7, 0xE0, 0x61, 0xAF, 0xF5, 0xDA, 0x27, 0xCB, 0xFC, 0x98, 0x0C, 0x12, 0x59, 0x45, 0x64, 0x75, 0x58, 0xB5, 0x5B, 0x7A, 0x9B, 0x76, 0x68, 0x14, 0x9F, 0xAB, 0x64, 0xC7, 0x4E, 0xB7, 0xFF, 0x7D, 0x3D, 0xA3, 0x11, 0x9E, 0xE8, 0x06, 0xAF, 0x5D, 0xA3, 0xEB, 0x8E, 0x1E, 0x38, 0x5D, 0x0D, 0xC3, 0x8C, 0xC3, 0xF0, 0x57 };

			AsymmetricKeyExchangeDeformatter keyback = new RSAOAEPKeyExchangeDeformatter (key);
			try {
				byte[] Mback = keyback.DecryptKeyExchange (EM);
				AssertEquals ("RSAOAEPKeyExchangeFormatter 128", M, Mback);
			}
			catch (CryptographicException ce) {
				// not supported by every version of Windows - Minimum: Windows XP
				Console.WriteLine (ce.Message + " (" + Environment.OSVersion.ToString () + ")");
			}
		}

		// test with a message 160 bits (20 bytes) long
		// Test vector (EM) generated by Mono on Windows
		[Test]
		public void MonoExchange192() 
		{
			byte[] M = { 0xd4, 0x36, 0xe9, 0x95, 0x69, 0xfd, 0x32, 0xa7, 0xc8, 0xa0, 0x5b, 0xbc, 0x90, 0xd3, 0x2c, 0x49, 0x00, 0x00, 0x00, 0x00 };
			byte[] EM = { 0xAD, 0xFA, 0x95, 0x2E, 0xCE, 0x23, 0xBE, 0xA5, 0x77, 0x81, 0x34, 0x98, 0x64, 0x58, 0x02, 0xAB, 0x62, 0x28, 0x97, 0x84, 0x6D, 0xF3, 0xE1, 0x9D, 0xB2, 0x04, 0xA8, 0xB0, 0x1D, 0x48, 0xFC, 0x94, 0x1A, 0x3A, 0xB1, 0x72, 0x39, 0xE1, 0xFE, 0x25, 0xF6, 0x6E, 0x64, 0xA5, 0x84, 0xA3, 0x2F, 0x3D, 0x49, 0x4F, 0x7F, 0xD1, 0x45, 0x74, 0xFB, 0x42, 0x6F, 0x50, 0x5C, 0x19, 0x44, 0xB8, 0x0F, 0x3B, 0x31, 0x94, 0x8E, 0xC2, 0x44, 0x47, 0xA2, 0xE9, 0x6E, 0x7F, 0x58, 0x49, 0x38, 0xB9, 0x2C, 0xB8, 0x0E, 0xA9, 0x7A, 0x86, 0x2B, 0xDB, 0xED, 0x5A, 0x16, 0x48, 0x41, 0x84, 0x3B, 0xE3, 0xA8, 0x26, 0x01, 0xAE, 0x09, 0x41, 0xB3, 0x95, 0xC5, 0xA4, 0x5A, 0x82, 0xC3, 0x37, 0x57, 0xD9, 0x03, 0x53, 0x8D, 0x28, 0x24, 0xA4, 0x37, 0x2A, 0xA9, 0xC7, 0x66, 0xAD, 0xAC, 0x5F, 0x3A, 0xF0, 0xE5, 0x90 };

			AsymmetricKeyExchangeDeformatter keyback = new RSAOAEPKeyExchangeDeformatter (key);
			try {
				byte[] Mback = keyback.DecryptKeyExchange (EM);
				AssertEquals ("RSAOAEPKeyExchangeFormatter 192", M, Mback);
			}
			catch (CryptographicException ce) {
				// not supported by every version of Windows - Minimum: Windows XP
				Console.WriteLine (ce.Message + " (" + Environment.OSVersion.ToString () + ")");
			}
		}

		// Max = (key size in bytes) - 2 * (hash length) - 2
		// Test vector (EM) generated by Mono on Windows
		[Test]
		public void MonoExchangeMax() 
		{
			byte[] M = new byte [(key.KeySize >> 3) - 2 * 20 - 2];
			byte[] EM = { 0x6C, 0x3E, 0x71, 0x14, 0xAD, 0xDE, 0x46, 0x26, 0x42, 0xA8, 0x84, 0x40, 0xCC, 0x06, 0x73, 0xCC, 0x88, 0x76, 0x40, 0x08, 0x93, 0xE5, 0x5F, 0xFF, 0x7D, 0x02, 0x88, 0xE9, 0x2D, 0xF6, 0x90, 0xA1, 0x8F, 0x64, 0x9A, 0x79, 0x67, 0x63, 0xA5, 0xBE, 0xCE, 0x7F, 0x48, 0x65, 0x8F, 0x53, 0x24, 0x70, 0x4F, 0x2A, 0x61, 0x7C, 0x95, 0xB6, 0xD0, 0x1D, 0x6F, 0x92, 0xA5, 0x2B, 0xB9, 0x13, 0x6B, 0xD2, 0xEB, 0x1D, 0x4F, 0x1E, 0x51, 0x6D, 0x65, 0xA6, 0x97, 0xE8, 0x60, 0x4B, 0x19, 0xE4, 0x23, 0xDC, 0x22, 0xED, 0x23, 0x02, 0x5E, 0x0C, 0x0B, 0x99, 0x5D, 0xBA, 0xFC, 0xBD, 0x75, 0x2F, 0x3E, 0xCD, 0x33, 0xBF, 0x08, 0xD5, 0x31, 0x68, 0x7C, 0x51, 0x2E, 0xBF, 0x8A, 0xBF, 0xA9, 0x8F, 0x0A, 0xDF, 0xB0, 0x91, 0xB1, 0x95, 0x03, 0x37, 0x3C, 0x77, 0x61, 0x75, 0x06, 0x61, 0xD8, 0x94, 0x04, 0x42 };

			AsymmetricKeyExchangeDeformatter keyback = new RSAOAEPKeyExchangeDeformatter (key);
			try {
				byte[] Mback = keyback.DecryptKeyExchange (EM);
				AssertEquals ("RSAOAEPKeyExchangeFormatter Max", M, Mback);
			}
			catch (CryptographicException ce) {
				// not supported by every version of Windows - Minimum: Windows XP
				Console.WriteLine (ce.Message + " (" + Environment.OSVersion.ToString () + ")");
			}
		}

		// TestExchangeTooBig
		[Test]
		[ExpectedException (typeof (CryptographicException))]
		public void ExchangeTooBig() 
		{
			AsymmetricKeyExchangeDeformatter keyex = new RSAOAEPKeyExchangeDeformatter (key);
			byte[] EM = new byte [(key.KeySize >> 3) + 1];
			// invalid format
			byte[] M = keyex.DecryptKeyExchange (EM);
		}

		[Test]
		public void Parameters () 
		{
			RSAOAEPKeyExchangeDeformatter keyex = new RSAOAEPKeyExchangeDeformatter (key);
			keyex.Parameters = "Mono";
			AssertNull ("Parameters", keyex.Parameters);
		}

		[Test]
#if NET_2_0
		[ExpectedException (typeof (CryptographicUnexpectedOperationException))]
#else
		[ExpectedException (typeof (NullReferenceException))]
#endif
		public void ExchangeNoKey () 
		{
			AsymmetricKeyExchangeDeformatter keyex = new RSAOAEPKeyExchangeDeformatter ();
			byte[] M = keyex.DecryptKeyExchange (new byte [(key.KeySize >> 3) - 2 * 20 - 2]);
		}

		[Test]
		[ExpectedException (typeof (InvalidCastException))]
		public void ExchangeDSAKey () 
		{
			DSA dsa = DSA.Create ();
			AsymmetricKeyExchangeDeformatter keyex = new RSAOAEPKeyExchangeDeformatter (dsa);
		}

		[Test]
		[ExpectedException (typeof (CryptographicException))]
		public void Bug79320 ()
		{
			string s = "hdphq/mn8goBi43YGPkmOfPj5vXjHrKPJkT4mLT3l+XzLttHMLC4/yBYkuzlXtbrl2jtAJRb6oA8UcQFalUMnCa09LDZrgNU2yySn7YbiG8raSq7u2nfDCbPu+c8T9fyHxrCHrX0zeqqImX33csIn6rIrQZ8HKcMsoQso4qtS2A=";
			byte [] bytes = Convert.FromBase64String (s);
			RSACryptoServiceProvider r = new RSACryptoServiceProvider ();
			r.FromXmlString ("<RSAKeyValue><Modulus>iSObDmmhDgrl4NiLaviFcpv4NdysBWJcqiVz3AQbPdajtXaQQ8VJdfRkixah132yKOFGCWZhHS3EuPMh8dcNwGwta2nh+m2IV6ktzI4+mZ7CSNAsmlDY0JI+H8At1vKvNArlC5jkVGuliYroJeSU/NLPLNYgspi7TtXGy9Rfug8=</Modulus><Exponent>EQ==</Exponent><P>pd4svtxrnTWXVSb151/JFgT9szI6dxQ5pAFPd4A4yuxLLEay2W2z7d9LVk5siMFhZ10uTJGWzNP5pSgLT8wdww==</P><Q>06j6m4cGRc3uoKVuFFGA19JG3Bi4tDBEQHebEc/Y3+eThrOasYIRrQmGUfqWnd9eFitO9GOaVJ0muNDV7NOxxQ==</Q><DP>OoqmYXr4zhLqHg3AM4s36adomZlBz6zJDLUrGx4yKYCTAJFsTL1OkDCxLYUXP1NPjeSm7dkIDA6UWGh8doRGvQ==</DP><DQ>PkDCLb5NI5br1OVcnJBxMGsFyEOBnmiMi2545x8DjSX+Nq1LnZ6555ljvcIsTIz9jgy83nel3KaxCS5dCWtwhQ==</DQ><InverseQ>OrFYaG7wTqim/bub4qY0CvIfhsjG4/4MEabg0UFTf/+tekKas7DDKg2TD5BS2q0O3XEt7xIfp0S6dpOAnrlyGQ==</InverseQ><D>IESc9FUW1iCuj0ICr8IBSCSy3383iMvZkXI5YPHoSskXdf3Hl3m27pPbbAVTQcM4+o9bxfn4u5JMZ8C8sV/G/8Cfl4ss1NVMbZOecvVObRqRpqXaveq5fN2X0EklH1wzm5w3O8cMXdbC/hc0gKUqaMjFVn1zpf3zVjpOkY0eGRE=</D></RSAKeyValue>");
			RSAOAEPKeyExchangeDeformatter def = new RSAOAEPKeyExchangeDeformatter (r);
			AssertNotNull (def.DecryptKeyExchange (bytes));
		}
	}
}
