//
// StrongNameTest.cs - NUnit Test Cases for Strong Name Key File
//
// Author:
//	Sebastien Pouliot (sebastien@ximian.com)
//
// (C) 2003 Motus Technologies Inc. (http://www.motus.com)
// (C) 2004 Novell (http://www.novell.com)
//

using System;
using System.IO;
using System.Security.Cryptography;
using System.Text;

using Mono.Security;
using NUnit.Framework;

namespace MonoTests.Mono.Security {

	// HOWTO create a SNK file
	// sn -k test.snk

	[TestFixture]
	public class StrongNameTest : Assertion {

		// because most crypto stuff works with byte[] buffers
		static public void AssertEquals (string msg, byte[] array1, byte[] array2) 
		{
			if ((array1 == null) && (array2 == null))
				return;
			if (array1 == null)
				Fail (msg + " -> First array is NULL");
			if (array2 == null)
				Fail (msg + " -> Second array is NULL");

			bool a = (array1.Length == array2.Length);
			if (a) {
				for (int i = 0; i < array1.Length; i++) {
					if (array1 [i] != array2 [i]) {
						a = false;
						break;
					}
				}
			}
			if (array1.Length > 0) {
				msg += " -> Expected " + BitConverter.ToString (array1, 0);
				msg += " is different than " + BitConverter.ToString (array2, 0);
			}
			Assert (msg, a);
		}

		static byte[] test = { 
		0x07, 0x02, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x52, 0x53, 0x41, 0x32, 
		0x00, 0x04, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x3D, 0xBD, 0x72, 0x08, 
		0xC6, 0x2B, 0x0E, 0xA8, 0xC1, 0xC0, 0x58, 0x07, 0x2B, 0x63, 0x5F, 0x7C, 
		0x9A, 0xBD, 0xCB, 0x22, 0xDB, 0x20, 0xB2, 0xA9, 0xDA, 0xDA, 0xEF, 0xE8, 
		0x00, 0x64, 0x2F, 0x5D, 0x8D, 0xEB, 0x78, 0x02, 0xF7, 0xA5, 0x36, 0x77, 
		0x28, 0xD7, 0x55, 0x8D, 0x14, 0x68, 0xDB, 0xEB, 0x24, 0x09, 0xD0, 0x2B, 
		0x13, 0x1B, 0x92, 0x6E, 0x2E, 0x59, 0x54, 0x4A, 0xAC, 0x18, 0xCF, 0xC9, 
		0x09, 0x02, 0x3F, 0x4F, 0xA8, 0x3E, 0x94, 0x00, 0x1F, 0xC2, 0xF1, 0x1A, 
		0x27, 0x47, 0x7D, 0x10, 0x84, 0xF5, 0x14, 0xB8, 0x61, 0x62, 0x1A, 0x0C, 
		0x66, 0xAB, 0xD2, 0x4C, 0x4B, 0x9F, 0xC9, 0x0F, 0x3C, 0xD8, 0x92, 0x0F, 
		0xF5, 0xFF, 0xCE, 0xD7, 0x6E, 0x5C, 0x6F, 0xB1, 0xF5, 0x7D, 0xD3, 0x56, 
		0xF9, 0x67, 0x27, 0xA4, 0xA5, 0x48, 0x5B, 0x07, 0x93, 0x44, 0x00, 0x4A, 
		0xF8, 0xFF, 0xA4, 0xCB, 0x73, 0xC0, 0x6A, 0x62, 0xB4, 0xB7, 0xC8, 0x92, 
		0x58, 0x87, 0xCD, 0x07, 0x0C, 0x7D, 0x6C, 0xC1, 0x4A, 0xFC, 0x82, 0x57, 
		0x0E, 0x43, 0x85, 0x09, 0x75, 0x98, 0x51, 0xBB, 0x35, 0xF5, 0x64, 0x83, 
		0xC7, 0x79, 0x89, 0x5C, 0x55, 0x36, 0x66, 0xAB, 0x27, 0xA4, 0xD9, 0xD4, 
		0x7E, 0x6B, 0x67, 0x64, 0xC1, 0x54, 0x4E, 0x37, 0xF1, 0x4E, 0xCA, 0xB3, 
		0xE5, 0x63, 0x91, 0x57, 0x12, 0x14, 0xA6, 0xEA, 0x8F, 0x8F, 0x2B, 0xFE, 
		0xF3, 0xE9, 0x16, 0x08, 0x2B, 0x86, 0xBC, 0x26, 0x0D, 0xD0, 0xC6, 0xC4, 
		0x1A, 0x72, 0x43, 0x76, 0xDC, 0xFF, 0x28, 0x52, 0xA1, 0xDE, 0x8D, 0xFA, 
		0xD5, 0x1F, 0x0B, 0xB5, 0x4F, 0xAF, 0x06, 0x79, 0x11, 0xEE, 0xA8, 0xEC, 
		0xD3, 0x74, 0x55, 0xA2, 0x80, 0xFC, 0xF8, 0xD9, 0x50, 0x69, 0x48, 0x01, 
		0xC2, 0x5A, 0x04, 0x56, 0xB4, 0x3E, 0x24, 0x32, 0x20, 0xB5, 0x2C, 0xDE, 
		0xBB, 0xBD, 0x13, 0xFD, 0x13, 0xF7, 0x03, 0x3E, 0xE3, 0x37, 0x84, 0x74, 
		0xE7, 0xD0, 0x5E, 0x9E, 0xB6, 0x26, 0xAE, 0x6E, 0xB0, 0x55, 0x6A, 0x52, 
		0x63, 0x6F, 0x5A, 0x9D, 0xF2, 0x67, 0xD6, 0x61, 0x4F, 0x7A, 0x45, 0xEE, 
		0x5C, 0x3D, 0x2B, 0x7C, 0xB2, 0x40, 0x79, 0x54, 0x84, 0xD1, 0xBE, 0x61, 
		0x3E, 0x5E, 0xD6, 0x18, 0x8E, 0x14, 0x98, 0xFC, 0x35, 0xBF, 0x5F, 0x1A, 
		0x20, 0x2E, 0x1A, 0xD8, 0xFF, 0xC4, 0x6B, 0xC0, 0xC9, 0x7D, 0x06, 0xEF, 
		0x09, 0xF9, 0xF3, 0x69, 0xFC, 0xBC, 0xA2, 0xE6, 0x80, 0x22, 0xB9, 0x79, 
		0x7E, 0xEF, 0x57, 0x9F, 0x49, 0xE1, 0xBC, 0x0D, 0xB6, 0xA1, 0xFE, 0x8D, 
		0xBC, 0xBB, 0xA3, 0x05, 0x02, 0x6B, 0x04, 0x45, 0xF7, 0x5D, 0xEE, 0x43, 
		0x06, 0xD6, 0x9C, 0x94, 0x48, 0x1A, 0x0B, 0x9C, 0xBC, 0xB4, 0x4E, 0x93, 
		0x60, 0x87, 0xCD, 0x58, 0xD6, 0x9A, 0x39, 0xA6, 0xC0, 0x7F, 0x8E, 0xFF, 
		0x25, 0xC1, 0xD7, 0x2C, 0xF6, 0xF4, 0x6F, 0x24, 0x52, 0x0B, 0x39, 0x42, 
		0x1B, 0x0D, 0x04, 0xC1, 0x93, 0x2A, 0x19, 0x1C, 0xF0, 0xB1, 0x9B, 0xC1, 
		0x24, 0x6D, 0x1B, 0x0B, 0xDA, 0x1C, 0x8B, 0x72, 0x48, 0xF0, 0x3E, 0x52, 
		0xBF, 0x0A, 0x84, 0x3A, 0x9B, 0xC8, 0x6D, 0x13, 0x1E, 0x72, 0xF4, 0x46, 
		0x93, 0x88, 0x1A, 0x5F, 0x4C, 0x3C, 0xE5, 0x9D, 0x6E, 0xBB, 0x4E, 0xDD, 
		0x5D, 0x1F, 0x11, 0x40, 0xF4, 0xD7, 0xAF, 0xB3, 0xAB, 0x9A, 0x99, 0x15, 
		0xF0, 0xDC, 0xAA, 0xFF, 0x9F, 0x2D, 0x9E, 0x56, 0x4F, 0x35, 0x5B, 0xBA, 
		0x06, 0x99, 0xEA, 0xC6, 0xB4, 0x48, 0x51, 0x17, 0x1E, 0xD1, 0x95, 0x84, 
		0x81, 0x18, 0xC0, 0xF1, 0x71, 0xDE, 0x44, 0x42, 0x02, 0x06, 0xAC, 0x0E, 
		0xA8, 0xE2, 0xF3, 0x1F, 0x96, 0x1F, 0xBE, 0xB6, 0x1F, 0xB5, 0x3E, 0xF6, 
		0x81, 0x05, 0x20, 0xFA, 0x2E, 0x40, 0x2E, 0x4D, 0xA0, 0x0E, 0xDA, 0x42, 
		0x9C, 0x05, 0xAA, 0x9E, 0xAF, 0x5C, 0xF7, 0x3A, 0x3F, 0xBB, 0x91, 0x73, 
		0x45, 0x27, 0xA8, 0xA2, 0x07, 0x4A, 0xEF, 0x59, 0x1E, 0x97, 0x9D, 0xE0, 
		0x30, 0x5A, 0x83, 0xCE, 0x1E, 0x57, 0x32, 0x89, 0x43, 0x41, 0x28, 0x7D, 
		0x14, 0x8D, 0x8B, 0x41, 0x1A, 0x56, 0x76, 0x43, 0xDB, 0x64, 0x86, 0x41, 
		0x64, 0x8D, 0x4C, 0x91, 0x83, 0x4E, 0xF5, 0x6C };

		// these can be verified by saving the "test" buffer to a "test.snk" file
		// then use "sn -p test.snk test.pub" to extract the public key
		// finally use the "sn -tp test.pub" and compare the results
		static byte[] testPublicKey = { 0x00, 0x24, 0x00, 0x00, 0x04, 0x80, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00, 0x06, 0x02, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x52, 0x53, 0x41, 0x31, 0x00, 0x04, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x3D, 0xBD, 0x72, 0x08, 0xC6, 0x2B, 0x0E, 0xA8, 0xC1, 0xC0, 0x58, 0x07, 0x2B, 0x63, 0x5F, 0x7C, 0x9A, 0xBD, 0xCB, 0x22, 0xDB, 0x20, 0xB2, 0xA9, 0xDA, 0xDA, 0xEF, 0xE8, 0x00, 0x64, 0x2F, 0x5D, 0x8D, 0xEB, 0x78, 0x02, 0xF7, 0xA5, 0x36, 0x77, 0x28, 0xD7, 0x55, 0x8D, 0x14, 0x68, 0xDB, 0xEB, 0x24, 0x09, 0xD0, 0x2B, 0x13, 0x1B, 0x92, 0x6E, 0x2E, 0x59, 0x54, 0x4A, 0xAC, 0x18, 0xCF, 0xC9, 0x09, 0x02, 0x3F, 0x4F, 0xA8, 0x3E, 0x94, 0x00, 0x1F, 0xC2, 0xF1, 0x1A, 0x27, 0x47, 0x7D, 0x10, 0x84, 0xF5, 0x14, 0xB8, 0x61, 0x62, 0x1A, 0x0C, 0x66, 0xAB, 0xD2, 0x4C, 0x4B, 0x9F, 0xC9, 0x0F, 0x3C, 0xD8, 0x92, 0x0F, 0xF5, 0xFF, 0xCE, 0xD7, 0x6E, 0x5C, 0x6F, 0xB1, 0xF5, 0x7D, 0xD3, 0x56, 0xF9, 0x67, 0x27, 0xA4, 0xA5, 0x48, 0x5B, 0x07, 0x93, 0x44, 0x00, 0x4A, 0xF8, 0xFF, 0xA4, 0xCB };
		static byte[] testPublicKeyToken = { 0xCE, 0x52, 0x76, 0xD8, 0x68, 0x7E, 0xC6, 0xDC };

		[Test]
		public void FromBuffer ()
		{
			StrongName sn = new StrongName (test);
			AssertEquals ("buffer.RSA", "<RSAKeyValue><Modulus>y6T/+EoARJMHW0ilpCdn+VbTffWxb1xu187/9Q+S2DwPyZ9LTNKrZgwaYmG4FPWEEH1HJxrxwh8AlD6oTz8CCcnPGKxKVFkubpIbEyvQCSTr22gUjVXXKHc2pfcCeOuNXS9kAOjv2tqpsiDbIsu9mnxfYysHWMDBqA4rxghyvT0=</Modulus><Exponent>AQAB</Exponent><P>6qYUEleRY+Wzyk7xN05UwWRna37U2aQnq2Y2VVyJeceDZPU1u1GYdQmFQw5XgvxKwWx9DAfNh1iSyLe0YmrAcw==</P><Q>3iy1IDIkPrRWBFrCAUhpUNn4/ICiVXTT7KjuEXkGr0+1Cx/V+o3eoVIo/9x2Q3IaxMbQDSa8hisIFunz/iuPjw==</Q><DP>2BouIBpfvzX8mBSOGNZePmG+0YRUeUCyfCs9XO5Fek9h1mfynVpvY1JqVbBuria2nl7Q53SEN+M+A/cT/RO9uw==</DP><DQ>pjma1ljNh2CTTrS8nAsaSJSc1gZD7l33RQRrAgWju7yN/qG2DbzhSZ9X7355uSKA5qK8/Gnz+QnvBn3JwGvE/w==</DQ><InverseQ>3U67bp3lPExfGoiTRvRyHhNtyJs6hAq/Uj7wSHKLHNoLG20kwZux8BwZKpPBBA0bQjkLUiRv9PYs18El/45/wA==</InverseQ><D>bPVOg5FMjWRBhmTbQ3ZWGkGLjRR9KEFDiTJXHs6DWjDgnZceWe9KB6KoJ0Vzkbs/Ovdcr56qBZxC2g6gTS5ALvogBYH2PrUftr4flh/z4qgOrAYCQkTecfHAGIGEldEeF1FItMbqmQa6WzVPVp4tn/+q3PAVmZqrs6/X9EARH10=</D></RSAKeyValue>", sn.RSA.ToXmlString (true));
			AssertEquals ("buffer.PublicKey", testPublicKey, sn.PublicKey);
			AssertEquals ("buffer.PublicKeyToken", testPublicKeyToken, sn.PublicKeyToken);
		}

		[Test]
		public void FromKey ()
		{
			StrongName sn1 = new StrongName (test);
			StrongName sn2 = new StrongName (sn1.RSA);
			AssertEquals ("key.RSA", sn1.RSA.ToXmlString (true), sn2.RSA.ToXmlString (true));
			AssertEquals ("key.PublicKey", sn1.PublicKey, sn2.PublicKey);
			AssertEquals ("key.PublicKeyToken", sn1.PublicKeyToken, sn2.PublicKeyToken);
		}

		private const string Signed = "hellosigned.exe";
		private const string Delay = "hellodelay.exe";
		private StrongName sn;
		private static byte[] signedData = {
			0x4D, 0x5A, 0x90, 0x00, 0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 
			0xFF, 0xFF, 0x00, 0x00, 0xB8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x80, 0x00, 0x00, 0x00, 0x0E, 0x1F, 0xBA, 0x0E, 0x00, 0xB4, 0x09, 0xCD, 
			0x21, 0xB8, 0x01, 0x4C, 0xCD, 0x21, 0x54, 0x68, 0x69, 0x73, 0x20, 0x70, 
			0x72, 0x6F, 0x67, 0x72, 0x61, 0x6D, 0x20, 0x63, 0x61, 0x6E, 0x6E, 0x6F, 
			0x74, 0x20, 0x62, 0x65, 0x20, 0x72, 0x75, 0x6E, 0x20, 0x69, 0x6E, 0x20, 
			0x44, 0x4F, 0x53, 0x20, 0x6D, 0x6F, 0x64, 0x65, 0x2E, 0x0D, 0x0D, 0x0A, 
			0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x45, 0x00, 0x00, 
			0x4C, 0x01, 0x03, 0x00, 0xF6, 0x4D, 0x88, 0x3F, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0xE0, 0x00, 0x0E, 0x01, 0x0B, 0x01, 0x06, 0x00, 
			0x00, 0x06, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x9E, 0x24, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 
			0x00, 0x00, 0x40, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 
			0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 
			0x0D, 0xF9, 0x00, 0x00, 0x03, 0x00, 0x00, 0x04, 0x00, 0x00, 0x10, 0x00, 
			0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x50, 0x24, 0x00, 0x00, 0x4B, 0x00, 0x00, 0x00, 
			0x00, 0x40, 0x00, 0x00, 0x40, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x60, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 
			0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x08, 0x20, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x2E, 0x74, 0x65, 0x78, 0x74, 0x00, 0x00, 0x00, 
			0xA4, 0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 
			0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x60, 0x2E, 0x72, 0x73, 0x72, 
			0x63, 0x00, 0x00, 0x00, 0x40, 0x03, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 
			0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x40, 
			0x2E, 0x72, 0x65, 0x6C, 0x6F, 0x63, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 
			0x00, 0x60, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x40, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x24, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 
			0xFC, 0x20, 0x00, 0x00, 0x54, 0x03, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 
			0x01, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x50, 0x20, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x5F, 0x5D, 0x32, 0x71, 0xDD, 0xD4, 0x8B, 0x75, 
			0x73, 0x61, 0x7A, 0xAF, 0xFA, 0x9A, 0xCC, 0x91, 0xBD, 0x49, 0x86, 0xF5, 
			0x20, 0xA5, 0x33, 0x76, 0x41, 0x98, 0x3C, 0x7D, 0x34, 0xCF, 0x7C, 0xCB, 
			0x0C, 0xC1, 0xD4, 0xC9, 0xBF, 0xEA, 0xBD, 0x27, 0x7C, 0x0B, 0x07, 0x6F, 
			0x51, 0xB6, 0x03, 0x54, 0xFC, 0x35, 0xFA, 0x1F, 0x64, 0x53, 0x83, 0x89, 
			0xD9, 0x59, 0xAE, 0xBA, 0x8C, 0xA8, 0x35, 0x25, 0x20, 0x37, 0x36, 0x73, 
			0xF8, 0x06, 0x35, 0x5D, 0x55, 0x30, 0x46, 0x5D, 0x35, 0x7B, 0x54, 0xFB, 
			0x28, 0x57, 0xAF, 0xF5, 0x4E, 0xF0, 0xA6, 0x76, 0x8F, 0x94, 0x41, 0xAF, 
			0x15, 0x66, 0x78, 0x39, 0xDF, 0x7F, 0x4F, 0x07, 0xA4, 0x16, 0x78, 0x6A, 
			0xC0, 0xFB, 0xC1, 0x3B, 0x32, 0x89, 0xB9, 0x36, 0x6A, 0x59, 0x3A, 0x64, 
			0xF3, 0x86, 0x7B, 0x54, 0x91, 0xD5, 0x1B, 0xEE, 0x3E, 0xC0, 0x9F, 0xA6, 
			0x13, 0x30, 0x01, 0x00, 0x0B, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x72, 0x01, 0x00, 0x00, 0x70, 0x28, 0x04, 0x00, 0x00, 0x0A, 0x2A, 0x00, 
			0x13, 0x30, 0x01, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x02, 0x28, 0x05, 0x00, 0x00, 0x0A, 0x2A, 0x00, 0x42, 0x53, 0x4A, 0x42, 
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 
			0x76, 0x31, 0x2E, 0x31, 0x2E, 0x34, 0x33, 0x32, 0x32, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x05, 0x00, 0x6C, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 
			0x23, 0x7E, 0x00, 0x00, 0x6C, 0x01, 0x00, 0x00, 0xD4, 0x00, 0x00, 0x00, 
			0x23, 0x53, 0x74, 0x72, 0x69, 0x6E, 0x67, 0x73, 0x00, 0x00, 0x00, 0x00, 
			0x40, 0x02, 0x00, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x23, 0x55, 0x53, 0x00, 
			0x5C, 0x02, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x23, 0x47, 0x55, 0x49, 
			0x44, 0x00, 0x00, 0x00, 0x6C, 0x02, 0x00, 0x00, 0xE8, 0x00, 0x00, 0x00, 
			0x23, 0x42, 0x6C, 0x6F, 0x62, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x01, 0x00, 0x00, 0x01, 0x47, 0x15, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 
			0x00, 0xFA, 0x01, 0x33, 0x00, 0x02, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 
			0x05, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 
			0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 
			0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0A, 0x00, 
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x29, 0x00, 0x22, 0x00, 
			0x06, 0x00, 0x57, 0x00, 0x45, 0x00, 0x06, 0x00, 0x72, 0x00, 0x45, 0x00, 
			0x06, 0x00, 0x9E, 0x00, 0x8B, 0x00, 0x06, 0x00, 0xC2, 0x00, 0x22, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 
			0x01, 0x00, 0x01, 0x00, 0x10, 0x00, 0x30, 0x00, 0x00, 0x00, 0x05, 0x00, 
			0x01, 0x00, 0x01, 0x00, 0xD0, 0x20, 0x00, 0x00, 0x00, 0x00, 0x96, 0x00, 
			0x3A, 0x00, 0x0A, 0x00, 0x01, 0x00, 0xE8, 0x20, 0x00, 0x00, 0x00, 0x00, 
			0x86, 0x18, 0x3F, 0x00, 0x10, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 
			0xBD, 0x00, 0x11, 0x00, 0x3F, 0x00, 0x14, 0x00, 0x19, 0x00, 0x3F, 0x00, 
			0x19, 0x00, 0x21, 0x00, 0x3F, 0x00, 0x1E, 0x00, 0x29, 0x00, 0xCA, 0x00, 
			0xC6, 0x00, 0x09, 0x00, 0x3F, 0x00, 0x10, 0x00, 0x2E, 0x00, 0x0B, 0x00, 
			0xCB, 0x00, 0x2E, 0x00, 0x13, 0x00, 0xD1, 0x00, 0x2E, 0x00, 0x1B, 0x00, 
			0xE0, 0x00, 0x04, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x24, 0x00, 0xB2, 0x00, 0x00, 0x00, 
			0x01, 0x00, 0x00, 0x00, 0x88, 0x13, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x01, 0x00, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x3C, 0x4D, 0x6F, 0x64, 0x75, 0x6C, 0x65, 0x3E, 0x00, 0x68, 0x65, 
			0x6C, 0x6C, 0x6F, 0x77, 0x6F, 0x72, 0x6C, 0x64, 0x2E, 0x65, 0x78, 0x65, 
			0x00, 0x6D, 0x73, 0x63, 0x6F, 0x72, 0x6C, 0x69, 0x62, 0x00, 0x53, 0x79, 
			0x73, 0x74, 0x65, 0x6D, 0x00, 0x4F, 0x62, 0x6A, 0x65, 0x63, 0x74, 0x00, 
			0x4D, 0x61, 0x69, 0x6E, 0x43, 0x6C, 0x61, 0x73, 0x73, 0x00, 0x4D, 0x61, 
			0x69, 0x6E, 0x00, 0x2E, 0x63, 0x74, 0x6F, 0x72, 0x00, 0x53, 0x79, 0x73, 
			0x74, 0x65, 0x6D, 0x2E, 0x52, 0x65, 0x66, 0x6C, 0x65, 0x63, 0x74, 0x69, 
			0x6F, 0x6E, 0x00, 0x41, 0x73, 0x73, 0x65, 0x6D, 0x62, 0x6C, 0x79, 0x44, 
			0x65, 0x6C, 0x61, 0x79, 0x53, 0x69, 0x67, 0x6E, 0x41, 0x74, 0x74, 0x72, 
			0x69, 0x62, 0x75, 0x74, 0x65, 0x00, 0x41, 0x73, 0x73, 0x65, 0x6D, 0x62, 
			0x6C, 0x79, 0x4B, 0x65, 0x79, 0x46, 0x69, 0x6C, 0x65, 0x41, 0x74, 0x74, 
			0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x00, 0x53, 0x79, 0x73, 0x74, 0x65, 
			0x6D, 0x2E, 0x44, 0x69, 0x61, 0x67, 0x6E, 0x6F, 0x73, 0x74, 0x69, 0x63, 
			0x73, 0x00, 0x44, 0x65, 0x62, 0x75, 0x67, 0x67, 0x61, 0x62, 0x6C, 0x65, 
			0x41, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x00, 0x68, 0x65, 
			0x6C, 0x6C, 0x6F, 0x77, 0x6F, 0x72, 0x6C, 0x64, 0x00, 0x61, 0x72, 0x67, 
			0x73, 0x00, 0x43, 0x6F, 0x6E, 0x73, 0x6F, 0x6C, 0x65, 0x00, 0x57, 0x72, 
			0x69, 0x74, 0x65, 0x4C, 0x69, 0x6E, 0x65, 0x00, 0x00, 0x19, 0x48, 0x00, 
			0x65, 0x00, 0x6C, 0x00, 0x6C, 0x00, 0x6F, 0x00, 0x20, 0x00, 0x77, 0x00, 
			0x6F, 0x00, 0x72, 0x00, 0x6C, 0x00, 0x64, 0x00, 0x21, 0x00, 0x00, 0x00, 
			0x1A, 0xD1, 0x32, 0x3B, 0x03, 0x10, 0x8E, 0x48, 0x94, 0x04, 0x93, 0xE3, 
			0xE6, 0x23, 0x27, 0x41, 0x00, 0x08, 0xB7, 0x7A, 0x5C, 0x56, 0x19, 0x34, 
			0xE0, 0x89, 0x05, 0x00, 0x01, 0x01, 0x1D, 0x0E, 0x03, 0x20, 0x00, 0x01, 
			0x04, 0x20, 0x01, 0x01, 0x02, 0x04, 0x20, 0x01, 0x01, 0x0E, 0x05, 0x20, 
			0x02, 0x01, 0x02, 0x02, 0x80, 0xA0, 0x00, 0x24, 0x00, 0x00, 0x04, 0x80, 
			0x00, 0x00, 0x94, 0x00, 0x00, 0x00, 0x06, 0x02, 0x00, 0x00, 0x00, 0x24, 
			0x00, 0x00, 0x52, 0x53, 0x41, 0x31, 0x00, 0x04, 0x00, 0x00, 0x01, 0x00, 
			0x01, 0x00, 0x67, 0xC1, 0x0D, 0x44, 0x56, 0xAB, 0xC3, 0xEE, 0x3B, 0xC7, 
			0x36, 0xD5, 0x9C, 0x5E, 0x83, 0x9A, 0x37, 0x3E, 0x93, 0x34, 0x42, 0xE7, 
			0x93, 0x8E, 0x83, 0x1C, 0x08, 0xF5, 0x9F, 0x7B, 0xAE, 0x74, 0x1B, 0x91, 
			0xB8, 0xFB, 0x69, 0xBA, 0xB6, 0x2D, 0xA4, 0x7D, 0x04, 0x1F, 0x70, 0x43, 
			0x8C, 0x63, 0xA8, 0x48, 0x3C, 0x7E, 0xEC, 0x94, 0x1C, 0xCB, 0x3F, 0x9B, 
			0x85, 0x13, 0x74, 0x05, 0xA3, 0x7A, 0x47, 0x71, 0xB5, 0x7B, 0x91, 0x3C, 
			0xAA, 0x85, 0x92, 0x02, 0xC5, 0xF8, 0x46, 0x40, 0x82, 0x94, 0x24, 0x15, 
			0xAD, 0x20, 0xB4, 0xC0, 0x99, 0x56, 0xBE, 0x70, 0x98, 0xF9, 0x6A, 0x59, 
			0xC0, 0x15, 0x63, 0x72, 0xE1, 0xB9, 0x77, 0x56, 0xD0, 0x27, 0x97, 0xCF, 
			0x93, 0x49, 0xAE, 0xCB, 0xBC, 0xBB, 0x65, 0x2D, 0x29, 0x3B, 0x5D, 0xB6, 
			0xA7, 0xCA, 0x0F, 0x43, 0x51, 0xDE, 0x6E, 0x66, 0xEC, 0xD3, 0x04, 0x00, 
			0x01, 0x01, 0x0E, 0x05, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0E, 0x01, 0x00, 
			0x09, 0x68, 0x65, 0x6C, 0x6C, 0x6F, 0x2E, 0x70, 0x75, 0x62, 0x00, 0x00, 
			0x06, 0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x78, 0x24, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8E, 0x24, 0x00, 0x00, 
			0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x80, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5F, 0x43, 
			0x6F, 0x72, 0x45, 0x78, 0x65, 0x4D, 0x61, 0x69, 0x6E, 0x00, 0x6D, 0x73, 
			0x63, 0x6F, 0x72, 0x65, 0x65, 0x2E, 0x64, 0x6C, 0x6C, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0xFF, 0x25, 0x00, 0x20, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 
			0x10, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 
			0x01, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x40, 0x00, 0x00, 
			0xE4, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0xE4, 0x02, 0x34, 0x00, 0x00, 0x00, 0x56, 0x00, 0x53, 0x00, 0x5F, 0x00, 
			0x56, 0x00, 0x45, 0x00, 0x52, 0x00, 0x53, 0x00, 0x49, 0x00, 0x4F, 0x00, 
			0x4E, 0x00, 0x5F, 0x00, 0x49, 0x00, 0x4E, 0x00, 0x46, 0x00, 0x4F, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0xBD, 0x04, 0xEF, 0xFE, 0x00, 0x00, 0x01, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x3F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 
			0x01, 0x00, 0x56, 0x00, 0x61, 0x00, 0x72, 0x00, 0x46, 0x00, 0x69, 0x00, 
			0x6C, 0x00, 0x65, 0x00, 0x49, 0x00, 0x6E, 0x00, 0x66, 0x00, 0x6F, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0x04, 0x00, 0x00, 0x00, 0x54, 0x00, 
			0x72, 0x00, 0x61, 0x00, 0x6E, 0x00, 0x73, 0x00, 0x6C, 0x00, 0x61, 0x00, 
			0x74, 0x00, 0x69, 0x00, 0x6F, 0x00, 0x6E, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0xB0, 0x04, 0x44, 0x02, 0x00, 0x00, 0x01, 0x00, 0x53, 0x00, 
			0x74, 0x00, 0x72, 0x00, 0x69, 0x00, 0x6E, 0x00, 0x67, 0x00, 0x46, 0x00, 
			0x69, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x49, 0x00, 0x6E, 0x00, 0x66, 0x00, 
			0x6F, 0x00, 0x00, 0x00, 0x20, 0x02, 0x00, 0x00, 0x01, 0x00, 0x30, 0x00, 
			0x30, 0x00, 0x30, 0x00, 0x30, 0x00, 0x30, 0x00, 0x34, 0x00, 0x62, 0x00, 
			0x30, 0x00, 0x00, 0x00, 0x1C, 0x00, 0x02, 0x00, 0x01, 0x00, 0x43, 0x00, 
			0x6F, 0x00, 0x6D, 0x00, 0x6D, 0x00, 0x65, 0x00, 0x6E, 0x00, 0x74, 0x00, 
			0x73, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x24, 0x00, 0x02, 0x00, 
			0x01, 0x00, 0x43, 0x00, 0x6F, 0x00, 0x6D, 0x00, 0x70, 0x00, 0x61, 0x00, 
			0x6E, 0x00, 0x79, 0x00, 0x4E, 0x00, 0x61, 0x00, 0x6D, 0x00, 0x65, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x2C, 0x00, 0x02, 0x00, 
			0x01, 0x00, 0x46, 0x00, 0x69, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x44, 0x00, 
			0x65, 0x00, 0x73, 0x00, 0x63, 0x00, 0x72, 0x00, 0x69, 0x00, 0x70, 0x00, 
			0x74, 0x00, 0x69, 0x00, 0x6F, 0x00, 0x6E, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x20, 0x00, 0x00, 0x00, 0x30, 0x00, 0x08, 0x00, 0x01, 0x00, 0x46, 0x00, 
			0x69, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x56, 0x00, 0x65, 0x00, 0x72, 0x00, 
			0x73, 0x00, 0x69, 0x00, 0x6F, 0x00, 0x6E, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x30, 0x00, 0x2E, 0x00, 0x30, 0x00, 0x2E, 0x00, 0x30, 0x00, 0x2E, 0x00, 
			0x30, 0x00, 0x00, 0x00, 0x40, 0x00, 0x0F, 0x00, 0x01, 0x00, 0x49, 0x00, 
			0x6E, 0x00, 0x74, 0x00, 0x65, 0x00, 0x72, 0x00, 0x6E, 0x00, 0x61, 0x00, 
			0x6C, 0x00, 0x4E, 0x00, 0x61, 0x00, 0x6D, 0x00, 0x65, 0x00, 0x00, 0x00, 
			0x68, 0x00, 0x65, 0x00, 0x6C, 0x00, 0x6C, 0x00, 0x6F, 0x00, 0x77, 0x00, 
			0x6F, 0x00, 0x72, 0x00, 0x6C, 0x00, 0x64, 0x00, 0x2E, 0x00, 0x65, 0x00, 
			0x78, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x02, 0x00, 
			0x01, 0x00, 0x4C, 0x00, 0x65, 0x00, 0x67, 0x00, 0x61, 0x00, 0x6C, 0x00, 
			0x43, 0x00, 0x6F, 0x00, 0x70, 0x00, 0x79, 0x00, 0x72, 0x00, 0x69, 0x00, 
			0x67, 0x00, 0x68, 0x00, 0x74, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 
			0x2C, 0x00, 0x02, 0x00, 0x01, 0x00, 0x4C, 0x00, 0x65, 0x00, 0x67, 0x00, 
			0x61, 0x00, 0x6C, 0x00, 0x54, 0x00, 0x72, 0x00, 0x61, 0x00, 0x64, 0x00, 
			0x65, 0x00, 0x6D, 0x00, 0x61, 0x00, 0x72, 0x00, 0x6B, 0x00, 0x73, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x48, 0x00, 0x0F, 0x00, 
			0x01, 0x00, 0x4F, 0x00, 0x72, 0x00, 0x69, 0x00, 0x67, 0x00, 0x69, 0x00, 
			0x6E, 0x00, 0x61, 0x00, 0x6C, 0x00, 0x46, 0x00, 0x69, 0x00, 0x6C, 0x00, 
			0x65, 0x00, 0x6E, 0x00, 0x61, 0x00, 0x6D, 0x00, 0x65, 0x00, 0x00, 0x00, 
			0x68, 0x00, 0x65, 0x00, 0x6C, 0x00, 0x6C, 0x00, 0x6F, 0x00, 0x77, 0x00, 
			0x6F, 0x00, 0x72, 0x00, 0x6C, 0x00, 0x64, 0x00, 0x2E, 0x00, 0x65, 0x00, 
			0x78, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0x02, 0x00, 
			0x01, 0x00, 0x50, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x64, 0x00, 0x75, 0x00, 
			0x63, 0x00, 0x74, 0x00, 0x4E, 0x00, 0x61, 0x00, 0x6D, 0x00, 0x65, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x34, 0x00, 0x08, 0x00, 
			0x01, 0x00, 0x50, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x64, 0x00, 0x75, 0x00, 
			0x63, 0x00, 0x74, 0x00, 0x56, 0x00, 0x65, 0x00, 0x72, 0x00, 0x73, 0x00, 
			0x69, 0x00, 0x6F, 0x00, 0x6E, 0x00, 0x00, 0x00, 0x30, 0x00, 0x2E, 0x00, 
			0x30, 0x00, 0x2E, 0x00, 0x30, 0x00, 0x2E, 0x00, 0x30, 0x00, 0x00, 0x00, 
			0x38, 0x00, 0x08, 0x00, 0x01, 0x00, 0x41, 0x00, 0x73, 0x00, 0x73, 0x00, 
			0x65, 0x00, 0x6D, 0x00, 0x62, 0x00, 0x6C, 0x00, 0x79, 0x00, 0x20, 0x00, 
			0x56, 0x00, 0x65, 0x00, 0x72, 0x00, 0x73, 0x00, 0x69, 0x00, 0x6F, 0x00, 
			0x6E, 0x00, 0x00, 0x00, 0x30, 0x00, 0x2E, 0x00, 0x30, 0x00, 0x2E, 0x00, 
			0x30, 0x00, 0x2E, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x20, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0xA0, 0x34, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

		private static byte[] delayData = {
			0x4D, 0x5A, 0x90, 0x00, 0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 
			0xFF, 0xFF, 0x00, 0x00, 0xB8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x80, 0x00, 0x00, 0x00, 0x0E, 0x1F, 0xBA, 0x0E, 0x00, 0xB4, 0x09, 0xCD, 
			0x21, 0xB8, 0x01, 0x4C, 0xCD, 0x21, 0x54, 0x68, 0x69, 0x73, 0x20, 0x70, 
			0x72, 0x6F, 0x67, 0x72, 0x61, 0x6D, 0x20, 0x63, 0x61, 0x6E, 0x6E, 0x6F, 
			0x74, 0x20, 0x62, 0x65, 0x20, 0x72, 0x75, 0x6E, 0x20, 0x69, 0x6E, 0x20, 
			0x44, 0x4F, 0x53, 0x20, 0x6D, 0x6F, 0x64, 0x65, 0x2E, 0x0D, 0x0D, 0x0A, 
			0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x45, 0x00, 0x00, 
			0x4C, 0x01, 0x03, 0x00, 0xF6, 0x4D, 0x88, 0x3F, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0xE0, 0x00, 0x0E, 0x01, 0x0B, 0x01, 0x06, 0x00, 
			0x00, 0x06, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x9E, 0x24, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 
			0x00, 0x00, 0x40, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 
			0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x04, 0x00, 0x00, 0x10, 0x00, 
			0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x50, 0x24, 0x00, 0x00, 0x4B, 0x00, 0x00, 0x00, 
			0x00, 0x40, 0x00, 0x00, 0x40, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x60, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 
			0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x08, 0x20, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x2E, 0x74, 0x65, 0x78, 0x74, 0x00, 0x00, 0x00, 
			0xA4, 0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 
			0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x60, 0x2E, 0x72, 0x73, 0x72, 
			0x63, 0x00, 0x00, 0x00, 0x40, 0x03, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 
			0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x40, 
			0x2E, 0x72, 0x65, 0x6C, 0x6F, 0x63, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 
			0x00, 0x60, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x40, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x24, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 
			0xFC, 0x20, 0x00, 0x00, 0x54, 0x03, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 
			0x01, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x50, 0x20, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x13, 0x30, 0x01, 0x00, 0x0B, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x72, 0x01, 0x00, 0x00, 0x70, 0x28, 0x04, 0x00, 0x00, 0x0A, 0x2A, 0x00, 
			0x13, 0x30, 0x01, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x02, 0x28, 0x05, 0x00, 0x00, 0x0A, 0x2A, 0x00, 0x42, 0x53, 0x4A, 0x42, 
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 
			0x76, 0x31, 0x2E, 0x31, 0x2E, 0x34, 0x33, 0x32, 0x32, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x05, 0x00, 0x6C, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 
			0x23, 0x7E, 0x00, 0x00, 0x6C, 0x01, 0x00, 0x00, 0xD4, 0x00, 0x00, 0x00, 
			0x23, 0x53, 0x74, 0x72, 0x69, 0x6E, 0x67, 0x73, 0x00, 0x00, 0x00, 0x00, 
			0x40, 0x02, 0x00, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x23, 0x55, 0x53, 0x00, 
			0x5C, 0x02, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x23, 0x47, 0x55, 0x49, 
			0x44, 0x00, 0x00, 0x00, 0x6C, 0x02, 0x00, 0x00, 0xE8, 0x00, 0x00, 0x00, 
			0x23, 0x42, 0x6C, 0x6F, 0x62, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x01, 0x00, 0x00, 0x01, 0x47, 0x15, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 
			0x00, 0xFA, 0x01, 0x33, 0x00, 0x02, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 
			0x05, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 
			0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 
			0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0A, 0x00, 
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x29, 0x00, 0x22, 0x00, 
			0x06, 0x00, 0x57, 0x00, 0x45, 0x00, 0x06, 0x00, 0x72, 0x00, 0x45, 0x00, 
			0x06, 0x00, 0x9E, 0x00, 0x8B, 0x00, 0x06, 0x00, 0xC2, 0x00, 0x22, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 
			0x01, 0x00, 0x01, 0x00, 0x10, 0x00, 0x30, 0x00, 0x00, 0x00, 0x05, 0x00, 
			0x01, 0x00, 0x01, 0x00, 0xD0, 0x20, 0x00, 0x00, 0x00, 0x00, 0x96, 0x00, 
			0x3A, 0x00, 0x0A, 0x00, 0x01, 0x00, 0xE8, 0x20, 0x00, 0x00, 0x00, 0x00, 
			0x86, 0x18, 0x3F, 0x00, 0x10, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 
			0xBD, 0x00, 0x11, 0x00, 0x3F, 0x00, 0x14, 0x00, 0x19, 0x00, 0x3F, 0x00, 
			0x19, 0x00, 0x21, 0x00, 0x3F, 0x00, 0x1E, 0x00, 0x29, 0x00, 0xCA, 0x00, 
			0xC6, 0x00, 0x09, 0x00, 0x3F, 0x00, 0x10, 0x00, 0x2E, 0x00, 0x0B, 0x00, 
			0xCB, 0x00, 0x2E, 0x00, 0x13, 0x00, 0xD1, 0x00, 0x2E, 0x00, 0x1B, 0x00, 
			0xE0, 0x00, 0x04, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x24, 0x00, 0xB2, 0x00, 0x00, 0x00, 
			0x01, 0x00, 0x00, 0x00, 0x88, 0x13, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x01, 0x00, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x3C, 0x4D, 0x6F, 0x64, 0x75, 0x6C, 0x65, 0x3E, 0x00, 0x68, 0x65, 
			0x6C, 0x6C, 0x6F, 0x77, 0x6F, 0x72, 0x6C, 0x64, 0x2E, 0x65, 0x78, 0x65, 
			0x00, 0x6D, 0x73, 0x63, 0x6F, 0x72, 0x6C, 0x69, 0x62, 0x00, 0x53, 0x79, 
			0x73, 0x74, 0x65, 0x6D, 0x00, 0x4F, 0x62, 0x6A, 0x65, 0x63, 0x74, 0x00, 
			0x4D, 0x61, 0x69, 0x6E, 0x43, 0x6C, 0x61, 0x73, 0x73, 0x00, 0x4D, 0x61, 
			0x69, 0x6E, 0x00, 0x2E, 0x63, 0x74, 0x6F, 0x72, 0x00, 0x53, 0x79, 0x73, 
			0x74, 0x65, 0x6D, 0x2E, 0x52, 0x65, 0x66, 0x6C, 0x65, 0x63, 0x74, 0x69, 
			0x6F, 0x6E, 0x00, 0x41, 0x73, 0x73, 0x65, 0x6D, 0x62, 0x6C, 0x79, 0x44, 
			0x65, 0x6C, 0x61, 0x79, 0x53, 0x69, 0x67, 0x6E, 0x41, 0x74, 0x74, 0x72, 
			0x69, 0x62, 0x75, 0x74, 0x65, 0x00, 0x41, 0x73, 0x73, 0x65, 0x6D, 0x62, 
			0x6C, 0x79, 0x4B, 0x65, 0x79, 0x46, 0x69, 0x6C, 0x65, 0x41, 0x74, 0x74, 
			0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x00, 0x53, 0x79, 0x73, 0x74, 0x65, 
			0x6D, 0x2E, 0x44, 0x69, 0x61, 0x67, 0x6E, 0x6F, 0x73, 0x74, 0x69, 0x63, 
			0x73, 0x00, 0x44, 0x65, 0x62, 0x75, 0x67, 0x67, 0x61, 0x62, 0x6C, 0x65, 
			0x41, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x00, 0x68, 0x65, 
			0x6C, 0x6C, 0x6F, 0x77, 0x6F, 0x72, 0x6C, 0x64, 0x00, 0x61, 0x72, 0x67, 
			0x73, 0x00, 0x43, 0x6F, 0x6E, 0x73, 0x6F, 0x6C, 0x65, 0x00, 0x57, 0x72, 
			0x69, 0x74, 0x65, 0x4C, 0x69, 0x6E, 0x65, 0x00, 0x00, 0x19, 0x48, 0x00, 
			0x65, 0x00, 0x6C, 0x00, 0x6C, 0x00, 0x6F, 0x00, 0x20, 0x00, 0x77, 0x00, 
			0x6F, 0x00, 0x72, 0x00, 0x6C, 0x00, 0x64, 0x00, 0x21, 0x00, 0x00, 0x00, 
			0x1A, 0xD1, 0x32, 0x3B, 0x03, 0x10, 0x8E, 0x48, 0x94, 0x04, 0x93, 0xE3, 
			0xE6, 0x23, 0x27, 0x41, 0x00, 0x08, 0xB7, 0x7A, 0x5C, 0x56, 0x19, 0x34, 
			0xE0, 0x89, 0x05, 0x00, 0x01, 0x01, 0x1D, 0x0E, 0x03, 0x20, 0x00, 0x01, 
			0x04, 0x20, 0x01, 0x01, 0x02, 0x04, 0x20, 0x01, 0x01, 0x0E, 0x05, 0x20, 
			0x02, 0x01, 0x02, 0x02, 0x80, 0xA0, 0x00, 0x24, 0x00, 0x00, 0x04, 0x80, 
			0x00, 0x00, 0x94, 0x00, 0x00, 0x00, 0x06, 0x02, 0x00, 0x00, 0x00, 0x24, 
			0x00, 0x00, 0x52, 0x53, 0x41, 0x31, 0x00, 0x04, 0x00, 0x00, 0x01, 0x00, 
			0x01, 0x00, 0x67, 0xC1, 0x0D, 0x44, 0x56, 0xAB, 0xC3, 0xEE, 0x3B, 0xC7, 
			0x36, 0xD5, 0x9C, 0x5E, 0x83, 0x9A, 0x37, 0x3E, 0x93, 0x34, 0x42, 0xE7, 
			0x93, 0x8E, 0x83, 0x1C, 0x08, 0xF5, 0x9F, 0x7B, 0xAE, 0x74, 0x1B, 0x91, 
			0xB8, 0xFB, 0x69, 0xBA, 0xB6, 0x2D, 0xA4, 0x7D, 0x04, 0x1F, 0x70, 0x43, 
			0x8C, 0x63, 0xA8, 0x48, 0x3C, 0x7E, 0xEC, 0x94, 0x1C, 0xCB, 0x3F, 0x9B, 
			0x85, 0x13, 0x74, 0x05, 0xA3, 0x7A, 0x47, 0x71, 0xB5, 0x7B, 0x91, 0x3C, 
			0xAA, 0x85, 0x92, 0x02, 0xC5, 0xF8, 0x46, 0x40, 0x82, 0x94, 0x24, 0x15, 
			0xAD, 0x20, 0xB4, 0xC0, 0x99, 0x56, 0xBE, 0x70, 0x98, 0xF9, 0x6A, 0x59, 
			0xC0, 0x15, 0x63, 0x72, 0xE1, 0xB9, 0x77, 0x56, 0xD0, 0x27, 0x97, 0xCF, 
			0x93, 0x49, 0xAE, 0xCB, 0xBC, 0xBB, 0x65, 0x2D, 0x29, 0x3B, 0x5D, 0xB6, 
			0xA7, 0xCA, 0x0F, 0x43, 0x51, 0xDE, 0x6E, 0x66, 0xEC, 0xD3, 0x04, 0x00, 
			0x01, 0x01, 0x0E, 0x05, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0E, 0x01, 0x00, 
			0x09, 0x68, 0x65, 0x6C, 0x6C, 0x6F, 0x2E, 0x70, 0x75, 0x62, 0x00, 0x00, 
			0x06, 0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x78, 0x24, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8E, 0x24, 0x00, 0x00, 
			0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x80, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5F, 0x43, 
			0x6F, 0x72, 0x45, 0x78, 0x65, 0x4D, 0x61, 0x69, 0x6E, 0x00, 0x6D, 0x73, 
			0x63, 0x6F, 0x72, 0x65, 0x65, 0x2E, 0x64, 0x6C, 0x6C, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0xFF, 0x25, 0x00, 0x20, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 
			0x10, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 
			0x01, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x58, 0x40, 0x00, 0x00, 
			0xE4, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0xE4, 0x02, 0x34, 0x00, 0x00, 0x00, 0x56, 0x00, 0x53, 0x00, 0x5F, 0x00, 
			0x56, 0x00, 0x45, 0x00, 0x52, 0x00, 0x53, 0x00, 0x49, 0x00, 0x4F, 0x00, 
			0x4E, 0x00, 0x5F, 0x00, 0x49, 0x00, 0x4E, 0x00, 0x46, 0x00, 0x4F, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0xBD, 0x04, 0xEF, 0xFE, 0x00, 0x00, 0x01, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x3F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 
			0x01, 0x00, 0x56, 0x00, 0x61, 0x00, 0x72, 0x00, 0x46, 0x00, 0x69, 0x00, 
			0x6C, 0x00, 0x65, 0x00, 0x49, 0x00, 0x6E, 0x00, 0x66, 0x00, 0x6F, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0x04, 0x00, 0x00, 0x00, 0x54, 0x00, 
			0x72, 0x00, 0x61, 0x00, 0x6E, 0x00, 0x73, 0x00, 0x6C, 0x00, 0x61, 0x00, 
			0x74, 0x00, 0x69, 0x00, 0x6F, 0x00, 0x6E, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0xB0, 0x04, 0x44, 0x02, 0x00, 0x00, 0x01, 0x00, 0x53, 0x00, 
			0x74, 0x00, 0x72, 0x00, 0x69, 0x00, 0x6E, 0x00, 0x67, 0x00, 0x46, 0x00, 
			0x69, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x49, 0x00, 0x6E, 0x00, 0x66, 0x00, 
			0x6F, 0x00, 0x00, 0x00, 0x20, 0x02, 0x00, 0x00, 0x01, 0x00, 0x30, 0x00, 
			0x30, 0x00, 0x30, 0x00, 0x30, 0x00, 0x30, 0x00, 0x34, 0x00, 0x62, 0x00, 
			0x30, 0x00, 0x00, 0x00, 0x1C, 0x00, 0x02, 0x00, 0x01, 0x00, 0x43, 0x00, 
			0x6F, 0x00, 0x6D, 0x00, 0x6D, 0x00, 0x65, 0x00, 0x6E, 0x00, 0x74, 0x00, 
			0x73, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x24, 0x00, 0x02, 0x00, 
			0x01, 0x00, 0x43, 0x00, 0x6F, 0x00, 0x6D, 0x00, 0x70, 0x00, 0x61, 0x00, 
			0x6E, 0x00, 0x79, 0x00, 0x4E, 0x00, 0x61, 0x00, 0x6D, 0x00, 0x65, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x2C, 0x00, 0x02, 0x00, 
			0x01, 0x00, 0x46, 0x00, 0x69, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x44, 0x00, 
			0x65, 0x00, 0x73, 0x00, 0x63, 0x00, 0x72, 0x00, 0x69, 0x00, 0x70, 0x00, 
			0x74, 0x00, 0x69, 0x00, 0x6F, 0x00, 0x6E, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x20, 0x00, 0x00, 0x00, 0x30, 0x00, 0x08, 0x00, 0x01, 0x00, 0x46, 0x00, 
			0x69, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x56, 0x00, 0x65, 0x00, 0x72, 0x00, 
			0x73, 0x00, 0x69, 0x00, 0x6F, 0x00, 0x6E, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x30, 0x00, 0x2E, 0x00, 0x30, 0x00, 0x2E, 0x00, 0x30, 0x00, 0x2E, 0x00, 
			0x30, 0x00, 0x00, 0x00, 0x40, 0x00, 0x0F, 0x00, 0x01, 0x00, 0x49, 0x00, 
			0x6E, 0x00, 0x74, 0x00, 0x65, 0x00, 0x72, 0x00, 0x6E, 0x00, 0x61, 0x00, 
			0x6C, 0x00, 0x4E, 0x00, 0x61, 0x00, 0x6D, 0x00, 0x65, 0x00, 0x00, 0x00, 
			0x68, 0x00, 0x65, 0x00, 0x6C, 0x00, 0x6C, 0x00, 0x6F, 0x00, 0x77, 0x00, 
			0x6F, 0x00, 0x72, 0x00, 0x6C, 0x00, 0x64, 0x00, 0x2E, 0x00, 0x65, 0x00, 
			0x78, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x02, 0x00, 
			0x01, 0x00, 0x4C, 0x00, 0x65, 0x00, 0x67, 0x00, 0x61, 0x00, 0x6C, 0x00, 
			0x43, 0x00, 0x6F, 0x00, 0x70, 0x00, 0x79, 0x00, 0x72, 0x00, 0x69, 0x00, 
			0x67, 0x00, 0x68, 0x00, 0x74, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 
			0x2C, 0x00, 0x02, 0x00, 0x01, 0x00, 0x4C, 0x00, 0x65, 0x00, 0x67, 0x00, 
			0x61, 0x00, 0x6C, 0x00, 0x54, 0x00, 0x72, 0x00, 0x61, 0x00, 0x64, 0x00, 
			0x65, 0x00, 0x6D, 0x00, 0x61, 0x00, 0x72, 0x00, 0x6B, 0x00, 0x73, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x48, 0x00, 0x0F, 0x00, 
			0x01, 0x00, 0x4F, 0x00, 0x72, 0x00, 0x69, 0x00, 0x67, 0x00, 0x69, 0x00, 
			0x6E, 0x00, 0x61, 0x00, 0x6C, 0x00, 0x46, 0x00, 0x69, 0x00, 0x6C, 0x00, 
			0x65, 0x00, 0x6E, 0x00, 0x61, 0x00, 0x6D, 0x00, 0x65, 0x00, 0x00, 0x00, 
			0x68, 0x00, 0x65, 0x00, 0x6C, 0x00, 0x6C, 0x00, 0x6F, 0x00, 0x77, 0x00, 
			0x6F, 0x00, 0x72, 0x00, 0x6C, 0x00, 0x64, 0x00, 0x2E, 0x00, 0x65, 0x00, 
			0x78, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0x02, 0x00, 
			0x01, 0x00, 0x50, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x64, 0x00, 0x75, 0x00, 
			0x63, 0x00, 0x74, 0x00, 0x4E, 0x00, 0x61, 0x00, 0x6D, 0x00, 0x65, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x34, 0x00, 0x08, 0x00, 
			0x01, 0x00, 0x50, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x64, 0x00, 0x75, 0x00, 
			0x63, 0x00, 0x74, 0x00, 0x56, 0x00, 0x65, 0x00, 0x72, 0x00, 0x73, 0x00, 
			0x69, 0x00, 0x6F, 0x00, 0x6E, 0x00, 0x00, 0x00, 0x30, 0x00, 0x2E, 0x00, 
			0x30, 0x00, 0x2E, 0x00, 0x30, 0x00, 0x2E, 0x00, 0x30, 0x00, 0x00, 0x00, 
			0x38, 0x00, 0x08, 0x00, 0x01, 0x00, 0x41, 0x00, 0x73, 0x00, 0x73, 0x00, 
			0x65, 0x00, 0x6D, 0x00, 0x62, 0x00, 0x6C, 0x00, 0x79, 0x00, 0x20, 0x00, 
			0x56, 0x00, 0x65, 0x00, 0x72, 0x00, 0x73, 0x00, 0x69, 0x00, 0x6F, 0x00, 
			0x6E, 0x00, 0x00, 0x00, 0x30, 0x00, 0x2E, 0x00, 0x30, 0x00, 0x2E, 0x00, 
			0x30, 0x00, 0x2E, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x20, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0xA0, 0x34, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

		private static byte[] key = { 
			0x07, 0x02, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x52, 0x53, 0x41, 0x32, 
			0x00, 0x04, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x67, 0xC1, 0x0D, 0x44, 
			0x56, 0xAB, 0xC3, 0xEE, 0x3B, 0xC7, 0x36, 0xD5, 0x9C, 0x5E, 0x83, 0x9A, 
			0x37, 0x3E, 0x93, 0x34, 0x42, 0xE7, 0x93, 0x8E, 0x83, 0x1C, 0x08, 0xF5, 
			0x9F, 0x7B, 0xAE, 0x74, 0x1B, 0x91, 0xB8, 0xFB, 0x69, 0xBA, 0xB6, 0x2D, 
			0xA4, 0x7D, 0x04, 0x1F, 0x70, 0x43, 0x8C, 0x63, 0xA8, 0x48, 0x3C, 0x7E, 
			0xEC, 0x94, 0x1C, 0xCB, 0x3F, 0x9B, 0x85, 0x13, 0x74, 0x05, 0xA3, 0x7A, 
			0x47, 0x71, 0xB5, 0x7B, 0x91, 0x3C, 0xAA, 0x85, 0x92, 0x02, 0xC5, 0xF8, 
			0x46, 0x40, 0x82, 0x94, 0x24, 0x15, 0xAD, 0x20, 0xB4, 0xC0, 0x99, 0x56, 
			0xBE, 0x70, 0x98, 0xF9, 0x6A, 0x59, 0xC0, 0x15, 0x63, 0x72, 0xE1, 0xB9, 
			0x77, 0x56, 0xD0, 0x27, 0x97, 0xCF, 0x93, 0x49, 0xAE, 0xCB, 0xBC, 0xBB, 
			0x65, 0x2D, 0x29, 0x3B, 0x5D, 0xB6, 0xA7, 0xCA, 0x0F, 0x43, 0x51, 0xDE, 
			0x6E, 0x66, 0xEC, 0xD3, 0x9D, 0xB8, 0x18, 0x11, 0xDD, 0xC5, 0x7D, 0xD7, 
			0xAB, 0x0E, 0xA4, 0x9E, 0xA8, 0x72, 0x6D, 0x99, 0x97, 0x2D, 0x83, 0x50, 
			0x34, 0x5A, 0x4B, 0xB7, 0xF7, 0xCE, 0xBD, 0x12, 0xFF, 0x45, 0x69, 0x69, 
			0xF8, 0x50, 0x35, 0x6A, 0x58, 0x91, 0x16, 0x50, 0x15, 0x5D, 0x33, 0x39, 
			0x7B, 0xA5, 0x1E, 0xFF, 0x30, 0x0F, 0x87, 0x01, 0x06, 0x1F, 0x9D, 0x21, 
			0x1C, 0x95, 0xBC, 0xEF, 0x92, 0xA2, 0x4D, 0xF8, 0xD3, 0x78, 0x74, 0xFB, 
			0x08, 0x79, 0xFF, 0x6D, 0x95, 0xCB, 0xF5, 0x3D, 0xDC, 0x03, 0x8D, 0x5C, 
			0xF6, 0x33, 0xBA, 0x92, 0x31, 0xBE, 0x36, 0xE9, 0xD1, 0xA3, 0xD5, 0xAA, 
			0xB4, 0xAE, 0xEF, 0x2C, 0xB4, 0x9E, 0x41, 0x08, 0x43, 0x53, 0x17, 0xBC, 
			0x75, 0xFE, 0x52, 0x23, 0x79, 0x08, 0x2D, 0x4F, 0x32, 0x84, 0x4C, 0xDF, 
			0x34, 0xBB, 0x11, 0x1A, 0x15, 0x32, 0x1D, 0xB0, 0x64, 0x14, 0x7E, 0xDA, 
			0xD1, 0x87, 0x1A, 0x31, 0x1D, 0xD5, 0x97, 0x34, 0x64, 0x2C, 0x82, 0x45, 
			0x33, 0xF1, 0x00, 0x05, 0x3A, 0x28, 0xD4, 0x5F, 0xE5, 0xCA, 0x1B, 0xEE, 
			0x6D, 0xD9, 0xEC, 0x55, 0xD6, 0x84, 0x11, 0x25, 0x86, 0xFC, 0x74, 0xA9, 
			0x37, 0x91, 0x30, 0xA9, 0xA5, 0xA2, 0x55, 0x6A, 0xD2, 0xAA, 0xA9, 0xC9, 
			0xDD, 0xD0, 0xAD, 0xB8, 0xF7, 0x3D, 0x6B, 0xE7, 0xEF, 0xFE, 0x3F, 0x07, 
			0x64, 0x9A, 0x4C, 0xB5, 0x37, 0x18, 0xAF, 0x5B, 0x17, 0x51, 0x1E, 0x15, 
			0x2C, 0x8C, 0x0F, 0xB9, 0x72, 0x57, 0x9A, 0xA1, 0x12, 0xEC, 0xA2, 0xEA, 
			0xB8, 0xB4, 0x98, 0x6A, 0x92, 0xC8, 0xBD, 0x9C, 0x2B, 0x96, 0x15, 0xFD, 
			0x70, 0xE4, 0x47, 0x9A, 0x51, 0xC5, 0x27, 0x1E, 0x78, 0x54, 0xB8, 0xB6, 
			0xF0, 0x80, 0x35, 0x7D, 0x1F, 0xE6, 0x52, 0x21, 0x00, 0xB5, 0xB7, 0x0C, 
			0xD1, 0xE1, 0x2D, 0xEC, 0x31, 0x83, 0xF0, 0x39, 0x57, 0xF0, 0xB2, 0xB1, 
			0xC2, 0xBD, 0xF2, 0x48, 0x73, 0x95, 0x7D, 0xFD, 0xC8, 0xF5, 0x86, 0x2C, 
			0xF1, 0x7C, 0x09, 0x11, 0x8F, 0xC9, 0xFB, 0x86, 0xE5, 0x62, 0x09, 0xD5, 
			0x46, 0x88, 0x12, 0xCA, 0x79, 0x69, 0x14, 0xF8, 0xB3, 0x11, 0x93, 0xCF, 
			0x99, 0x63, 0xDA, 0x33, 0x61, 0x8B, 0x12, 0x67, 0xDB, 0x9B, 0x06, 0x18, 
			0xFA, 0x36, 0xA8, 0x5D, 0x08, 0xEA, 0x9A, 0xB8, 0xDF, 0x46, 0x5D, 0xB5, 
			0x91, 0x6D, 0x75, 0xDD, 0x74, 0x72, 0xB4, 0xFE, 0xB8, 0x94, 0x29, 0xD4, 
			0x1E, 0x2E, 0x58, 0x01, 0x2D, 0x88, 0xF7, 0x9B, 0x71, 0x2A, 0x95, 0x79, 
			0xE1, 0x9C, 0x06, 0x94, 0xA3, 0x15, 0x20, 0xCA, 0xC5, 0xC0, 0xF2, 0xAC, 
			0x45, 0xD9, 0xBD, 0x5F, 0xB5, 0x18, 0xE6, 0xE4, 0x00, 0xC5, 0x46, 0x9A, 
			0x37, 0x95, 0xE4, 0x8B, 0x36, 0x0F, 0x1D, 0x39, 0xCF, 0x48, 0xAD, 0xBF, 
			0xDC, 0xC0, 0x15, 0xBA, 0x37, 0x21, 0x0E, 0xE3, 0x5A, 0x73, 0xAE, 0xFB, 
			0xCF, 0x63, 0xF6, 0x4F, 0xBF, 0x82, 0xA0, 0x44, 0x75, 0x09, 0xA6, 0x8F, 
			0x71, 0x95, 0x61, 0x4E, 0x79, 0x6A, 0x9A, 0x3E, 0xA3, 0xB5, 0xBE, 0xA7, 
			0x3A, 0x8B, 0x35, 0x53, 0x9C, 0x65, 0xA3, 0x56, 0x7A, 0xA6, 0xA7, 0x3B, 
			0x75, 0x49, 0x81, 0x3C, 0x18, 0xBE, 0x4E, 0x45, 0x4F, 0xA9, 0xBD, 0x13, 
			0xD0, 0x16, 0x87, 0xE1, 0x5B, 0xAF, 0x4A, 0x90 };

		[SetUp]
		public void SetUp () 
		{
			sn = new StrongName (key);
			// write hellosigned.exe to disk
			FileStream fs = File.OpenWrite (Signed);
			fs.Write (signedData, 0, signedData.Length);
			fs.Close ();
			// write hellodelay.exe to disk
			fs = File.OpenWrite (Delay);
			fs.Write (delayData, 0, delayData.Length);
			fs.Close ();
		}

		[TearDown]
		public void CleanUp () 
		{
			try {
				if (File.Exists (Signed))
					File.Delete (Signed);
				if (File.Exists (Delay))
					File.Delete (Delay);
			}
			catch {} // don't mess up results
		}

		[Test]
		public void Hash () 
		{
			// compare that both hellosigned and hellodelay are the same file expect for signature
			byte[] s = sn.Hash (Signed);
			AssertNotNull ("Hash/Signed", s);
			byte[] d = sn.Hash (Delay);
			AssertNotNull ("Hash/Delay", d);
			// are they the same
			AssertEquals ("Hash/Compare", s, d);
			// are they the same as GetHashFromAssemblyFile
			byte[] knownHash = { 0x7D, 0xF7, 0x1D, 0xD4, 0x24, 0x22, 0xEA, 0xAB, 0xD2, 0x91, 0xCD, 0xAB, 0x1A, 0x55, 0x6A, 0x42, 0x05, 0xBA, 0x6E, 0x7B };
			AssertEquals ("Hash/GetHashFromAssemblyFile", knownHash, s);
		}

		[Test]
		public void VerifyValid () 
		{
			// verify that hellosigned.exe is valid
			Assert ("Verify/Valid", sn.Verify (Signed));
		}

		[Test]
		public void VerifyInvalid () 
		{
			// verify that hellodelay.exe isn't valid
			Assert ("Verify/Invalid", !sn.Verify (Delay));
		}

		[Test]
		public void VerifyValidFromStream () 
		{
			Stream signed = new MemoryStream (signedData);
			// verify that hellosigned.exe is valid
			Assert ("Verify/Valid", sn.Verify (signed));
		}

		[Test]
		public void VerifyInvalidFromStream () 
		{
			Stream delay = new MemoryStream (delayData);
			// verify that hellodelay.exe isn't valid
			Assert ("Verify/Invalid", !sn.Verify (delay));
		}

		[Test]
		public void Sign () 
		{
			// sign the (invalid) hellodelay.exe
			Assert ("Sign", sn.Sign (Delay));
			// and verify it's now valid
			Assert ("Sign/Verify", sn.Verify (Delay));
		}

		[Test]
		public void ReSign () 
		{
			// resign the (already valid) hellosigned.exe
			Assert ("ReSign", sn.Sign (Signed));
			// and verify it's still valid
			Assert ("ReSign/Verify", sn.Verify (Signed));
		}
		
		[Test]
		[ExpectedException (typeof (ArgumentNullException))]
		public void StrongName_ByteNull () 
		{
			byte[] data = null;
			sn = new StrongName (data); 
		}
		
		[Test]
		[ExpectedException (typeof (ArgumentNullException))]
		public void StrongName_RSANull () 
		{
			RSA rsa = null;
			sn = new StrongName (rsa); 
		}
		
		[Test]
		[ExpectedException (typeof (ArgumentException))]
		public void BadKey () 
		{
			byte[] bad = new byte [0]; 
			sn = new StrongName (bad); 
		}
	
		[Test]
		public void ECMA () 
		{
			byte[] ecma = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }; 
			sn = new StrongName (ecma);
			Assert ("CanSign", !sn.CanSign);
		}

		[Test]
		public void TokenAlgorithm_MD5 () 
		{
			sn.TokenAlgorithm = "MD5";
		} 

		[Test]
		public void TokenAlgorithm_SHA1 () 
		{
			sn.TokenAlgorithm = "SHA1";
		} 

		[Test]
		[ExpectedException (typeof (ArgumentException))]
		public void TokenAlgorithm_Bad () 
		{
			sn.TokenAlgorithm = "SHA384";
		} 
	
	}
}
