/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#include "frontend/UsingEmitter.h"

#include "frontend/BytecodeEmitter.h"
#include "frontend/EmitterScope.h"
#include "frontend/IfEmitter.h"
#include "vm/ThrowMsgKind.h"

using namespace js;
using namespace js::frontend;

UsingEmitter::UsingEmitter(BytecodeEmitter* bce) : bce_(bce) {}

bool UsingEmitter::emitCheckDisposeMethod(JS::SymbolCode hint) {
  // [stack] VAL

  // https://arai-a.github.io/ecma262-compare/?pr=3000&id=sec-getdisposemethod
  // Step 2. Else,
  // Step 2.a. Let method be ? GetMethod(V, @@dispose).
  if (!bce_->emit1(JSOp::Dup)) {
    //        [stack] VAL VAL
    return false;
  }

  if (!bce_->emit2(JSOp::Symbol, uint8_t(hint))) {
    //        [stack] VAL VAL @@dispose
    return false;
  }

  if (!bce_->emit1(JSOp::GetElem)) {
    //          [stack] VAL VAL[@@dispose]
    return false;
  }

  if (!bce_->emitCheckIsCallable()) {
    //          [stack] VAL VAL[@@dispose] IS_CALLABLE_RESULT
    return false;
  }

  InternalIfEmitter ifCallable(bce_);

  if (!ifCallable.emitThenElse()) {
    //          [stack] VAL VAL[@@dispose]
    return false;
  }

  if (!bce_->emit1(JSOp::Pop)) {
    //          [stack] VAL
    return false;
  }

  if (!ifCallable.emitElse()) {
    //          [stack] VAL VAL[@@dispose]
    return false;
  }

  if (!bce_->emit1(JSOp::Pop)) {
    //          [stack] VAL
    return false;
  }

  if (!bce_->emit2(JSOp::ThrowMsg,
                   uint8_t(ThrowMsgKind::UsingWithoutDispose))) {
    //          [stack] VAL
    return false;
  }

  if (!ifCallable.emitEnd()) {
    //          [stack] VAL
    return false;
  }

  return true;
}

bool UsingEmitter::prepareForAssignment(Kind kind) {
  JS::SymbolCode symdispose;
  switch (kind) {
    case Kind::Sync:
      symdispose = JS::SymbolCode::dispose;
      break;
    case Kind::Async:
      MOZ_CRASH("Async disposal not implemented");
    default:
      MOZ_CRASH("Invalid kind");
  }

  bce_->innermostEmitterScope()->setHasDisposables();

  // [stack] VAL

  // Explicit Resource Management Proposal
  // CreateDisposableResource ( V, hint [ , method ] )
  // https://arai-a.github.io/ecma262-compare/?pr=3000&id=sec-createdisposableresource
  // Step 1. If method is not present, then
  // (implicit)
  // Step 1.a. If V is either null or undefined, then
  if (!bce_->emit1(JSOp::IsNullOrUndefined)) {
    //         [stack] VAL IS_NULL_OR_UNDEFINED_RESULT
    return false;
  }

  InternalIfEmitter ifValueNullOrUndefined(bce_);

  // Step 1.b. Else,
  if (!ifValueNullOrUndefined.emitThen(IfEmitter::ConditionKind::Negative)) {
    //        [stack] VAL
    return false;
  }

  // Step 1.b.i. If V is not an Object, throw a TypeError exception.
  if (!bce_->emitCheckIsObj(CheckIsObjectKind::Disposable)) {
    //        [stack] VAL
    return false;
  }

  // Step 1.b.ii. Set method to ? GetDisposeMethod(V, hint).
  // Step 1.b.iii. If method is undefined, throw a TypeError exception.
  if (!emitCheckDisposeMethod(symdispose)) {
    //        [stack] VAL
    return false;
  }

  if (!ifValueNullOrUndefined.emitEnd()) {
    //        [stack] VAL
    return false;
  }

  // Step 1.a.i. Set V to undefined.
  // Step 3. Return the DisposableResource Record { [[ResourceValue]]: V,
  //         [[Hint]]: hint, [[DisposeMethod]]: method }.
  //
  // AddDisposableResource ( disposeCapability, V, hint [ , method ] )
  // https://arai-a.github.io/ecma262-compare/?pr=3000&id=sec-adddisposableresource
  // Step 3. Append resource to disposeCapability.[[DisposableResourceStack]].
  // TODO: All the steps performed by the generated bytecode here would need to
  // be unified into this AddDisposableResource opcode. (Bug 1899717)
  if (!bce_->emit1(JSOp::AddDisposable)) {
    //        [stack] VAL
    return false;
  }

  return true;
}
