/*                                                                          
        Source: check_users.c
        Author: Adam G. Bowen (agbowen@bealenet.com)
        $Revision: 1.2 $
        $Date: 1999/05/17 14:35:26 $
 
 * Program: System user count plugin for NetSaint
 * License: GPL
 * Copyright (c) 1999 Adam G. Bowen (agbowen@bealenet.com)
 *
 * Description:
 *
 * This plugin will check the number of users logged on the remote host at
 * ipaddress and generate an alert if number is above one of the threshold
 * values.
 *
 * Other errors result in a STATE_UNKNOWN return.
 *
 * Command line:
 *
 * check_users ipaddress sys_type 
 * check_users ipaddress sys_type <warn_#> <crit_#>
 *
 * Required input:
 *
 * ipaddress = The ipaddress of the remote system to run the check on.
 * sys_type = The remote system type.
 *
 * Optional input:
 *
 * <warn_#> = Number of users necessary to result in a WARNING state.
 * <crit_#> = Number of users necessary to result in a CRITICAL state.
 *
 * Notes:
 *
 * If <warn_#> and <crit_#> are not passed on the command line, they will be set
 * to the default values in the check_users config file.
 *
 * sys_type is used to determine which config file to use to generate the
 * remote command.
 *
 * The configuration file /usr/local/netsaint/config/check_users/local
 * contains the following values:
 *
 * RSH_COMMAND|<location of rsh command on netsaint system>|
 * AWK_COMMAND|<location of awk command on netsaint system>|
 * TAIL_COMMAND|<location of tail command on netsaint system>|
 * WARN_USERS|<number of users required to generate a warning>|
 * CRIT_USERS|<number of users required to generate a critical>|
 *
 * The configuration file /usr/local/netsaint/config/check_users/<sys_type>
 * contains the following values:
 *
 * WHO_COMMAND|<location of who command on system sys_type>|
 * WHO_OPTIONS|<options for who command to retreive required info>|
 * AWK_OPTIONS|<options for awk command to retrieve required info>|
 * TAIL_OPTIONS|<options for tail command to retrieve required info>|
 *
 * $Log: check_users.c,v $
 * Revision 1.2  1999/05/17 14:35:26  netsaint
 * Changed the plugin to use a seperate config directory.  This directory is the
 * CONFIG_DIR/command_name directory.
 *
 * Revision 1.1  1999/05/14 02:50:40  netsaint
 * Initial revision
 *
 *
*/

#include "/usr/local/src/netsaint/include/plugins.h"

int main(int argc, char *argv[])
{
  char expected[MAX_CHARS]="";

  char awk_command[MAX_CHARS];
  char awk_options[MAX_CHARS];
  char command_line[MAX_CHARS];
  char command_name[MAX_CHARS];
  char config_file_local[MAX_CHARS];
  char config_file_net[MAX_CHARS];
  char config_file_remote[MAX_CHARS];
  char error_file[MAX_CHARS];
  char input_buffer[MAX_CHARS];
  char ip_address[MAX_CHARS];
  char out_put_file[MAX_CHARS];
  char port_telnet[MAX_CHARS];
  char protocol[MAX_CHARS];
  char rsh_command[MAX_CHARS];
  char *sub_string;
  char system_name[MAX_CHARS];
  char tail_command[MAX_CHARS];
  char tail_options[MAX_CHARS];
  char users_crit[MAX_CHARS];
  char users_warn[MAX_CHARS];
  char who_command[MAX_CHARS];
  char who_options[MAX_CHARS];

  FILE *out_put_fp;

  int crit_users;
  int get_defaults;
  int num_users;
  int result;
  int return_value;
  int socket_name;
  int telnet_port;
  int warn_users;

  /* Initialize alarm signal handling */

  signal(SIGALRM,alarm_signal);

  strcpy(command_name,get_command_name(argv[0]));
  if(!((argc==3) || (argc==5)))
  {
    printf("\n");
    printf(" Incorrect number of arguments supplied\n");
    printf("\n");
    printf(" System user count plugin for NetSaint\n");
    printf(" Copyright (c) 1999 Adam G. Bowen (agbowen@bealenet.com)\n");
    printf(" $Revision: 1.2 $\n");
    printf(" Last Modified $Date: 1999/05/17 14:35:26 $\n");
    printf(" License: GPL\n");
    printf("\n");
    printf(" Description:\n");
    printf("\n");
    printf(" This plugin will check the number of users logged on the remote host at\n");
    printf(" ipaddress and generate an alert if number is above one of the threshold\n");
    printf(" values.\n");
    printf("\n");
    printf(" Usage: %s ipaddress sys_type\n",command_name);
    printf(" Usage: %s ipaddress sys_type <warn_#> <crit_#>\n",command_name);
    printf("\n");
    printf(" Required input:\n");
    printf("\n");
    printf(" ipaddress = The ipaddress of the remote system to run the check on.\n");
    printf(" sys_type = The remote system type.\n");
    printf("\n");
    printf(" Optional input:\n");
    printf("\n");
    printf(" <warn_#> = Number of users necessary to result in a WARNING state.\n");
    printf(" <crit_#> = Number of users necessary to result in a CRITICAL state.\n");
    printf("\n");
    printf(" If <warn_#> and <crit_#> are not passed on the command line, they will be set\n");
    printf(" to the default values in the %s config file.\n", command_name);
    printf("\n");
    printf(" sys_type is used to determine which config file to use to generate the\n");
    printf(" remote command.\n");
    printf("\n");
    return_value = STATE_UNKNOWN;
  }
  else
  {
    /* Set up config files and get the command line information */

    strcpy(ip_address,argv[1]);
    strcpy(system_name,argv[2]);

    strcpy(config_file_local,CONFIG_DIR);
    strcpy(config_file_remote,CONFIG_DIR);
    strcpy(config_file_net,CONFIG_DIR);
    strcat(config_file_local,command_name);
    strcat(config_file_remote,command_name);
    strcat(config_file_net,CHECK_TELNET);
    strcat(config_file_local,"/local");
    strcat(config_file_remote,"/");
    strcat(config_file_net,"/");
    strcat(config_file_remote,system_name);
    strcat(config_file_net,system_name);

    if(argc == 3)
    {
      get_defaults = TRUE;
    }
    else
    {
      get_defaults = FALSE;
      strcpy(users_warn,argv[3]);
      strcpy(users_crit,argv[4]);
    }

    /* Check if config files exist */

    if (access(config_file_local, EXISTS) != 0 )
    {
      printf("Config file %s does not exist!\n",config_file_local);
      return_value = STATE_UNKNOWN;
    }
    else if (access(config_file_remote, EXISTS) != 0 )
    {
      printf("Config file %s does not exist!\n",config_file_remote);
      return_value = STATE_UNKNOWN;
    }
    else if (access(config_file_net, EXISTS) != 0 )
    {
      printf("Config file %s does not exist!\n",config_file_net);
      return_value = STATE_UNKNOWN;
    }
    else
    {

      /* Local config file variables */

      if((get_defaults == TRUE) && ((return_value=get_var("WARN_USERS", config_file_local, users_warn)) != STATE_OK))
      {
        printf("WARN_USERS entry not found in config file %s!\n",config_file_local);
      }
      else if((get_defaults == TRUE) && ((return_value=get_var("CRIT_USERS", config_file_local, users_crit)) != STATE_OK))
      {
        printf("CRIT_USERS entry not found in config file %s!\n",config_file_local);
      }
      else if((return_value=get_var("RSH_COMMAND", config_file_local, rsh_command)) != STATE_OK)
      {
        printf("RSH_COMMAND entry not found in config file %s!\n", config_file_local);
      }
      else if((return_value=get_var("AWK_COMMAND", config_file_local, awk_command)) != STATE_OK)
      {
        printf("AWK_COMMAND entry not found in config file %s!\n", config_file_local);
      }
      else if((return_value=get_var("TAIL_COMMAND", config_file_local, tail_command)) != STATE_OK)
      {
        printf("TAIL_COMMAND entry not found in config file %s!\n", config_file_local);
      }

      /* Remote config file variables */

      else if((return_value=get_var("WHO_COMMAND", config_file_remote, who_command)) != STATE_OK)
      {
        printf("WHO_COMMAND entry not found in config file %s!\n", config_file_remote);
      }
      else if((return_value=get_var("WHO_OPTIONS", config_file_remote, who_options)) != STATE_OK)
      {
        printf("WHO_OPTIONS entry not found in config file %s!\n",config_file_remote);
      }
      else if((return_value=get_var("AWK_OPTIONS", config_file_remote, awk_options)) != STATE_OK)
      {
        printf("AWK_OPTIONS entry not found in config file %s!\n",config_file_remote);
      }
      else if((return_value=get_var("TAIL_OPTIONS", config_file_remote, tail_options)) != STATE_OK)
      {
        printf("TAIL_OPTIONS entry not found in config file %s!\n", config_file_remote);
      }

      /* Network config file variables */

      else if((return_value=get_var("TELNET_PORT", config_file_net, port_telnet)) != STATE_OK)
      {
        printf("TELNET_PORT entry not found in config file %s!\n",config_file_net);
      }
      else if((return_value=get_var("TELNET_PROTO", config_file_net, protocol)) != STATE_OK)
      {
        printf("TELNET_PROTO entry not found in config file %s!\n",config_file_net);
      }
      else
      {

        /* Check alert level consistency */

        warn_users=atoi(users_warn);
        crit_users=atoi(users_crit);
        return_value = check_consistency(warn_users, crit_users);
        if(return_value == STATE_OK)
        {

          /* Check the network */

          telnet_port=atoi(port_telnet);
          alarm(TIME_OUT);
          if((result=open_socket(&socket_name, ip_address, telnet_port, protocol)) != STATE_OK)
          {
            return_value=exit_error(result,ip_address,protocol,telnet_port);
          }
          else if((result=recv_socket(&socket_name, expected)) != STATE_OK)
          {
            return_value=exit_error(result,ip_address,protocol,telnet_port);
          }
          else if((result=close_socket(&socket_name)) != STATE_OK)
          {
            return_value=exit_error(result,ip_address,protocol,telnet_port);
          }
          else
          {
            alarm(0);

            /* Generate a out_put and error file names */

            strcpy(out_put_file, tmpnam(NULL));
            strcpy(error_file, tmpnam(NULL));
 
            /* Set the command line and arguments to use for the check */
  
            sprintf(command_line,"%s %s %s %s </dev/null 2>%s|%s %s 2>>%s|%s %s >%s 2>>%s",rsh_command, ip_address, who_command, who_options, error_file, tail_command, tail_options, error_file, awk_command, awk_options, out_put_file, error_file);
  
            /* Run the command */
  
            system(command_line);

            return_value=check_output_file(out_put_file);
            if (return_value != STATE_OK)
            {
              print_error(error_file);
              return return_value;
            }
            else
            {
              out_put_fp=fopen(out_put_file,"r");
  
              /* Retrive single line from output file */

              fgets(input_buffer,MAX_CHARS-1,out_put_fp);

              /* Populate variables */
  
              if(strstr(input_buffer,"=") != NULL)
              {
                sub_string=strrchr(input_buffer,'=');
                num_users=atoi(++sub_string);
              }
              else
              {
                num_users=atoi(input_buffer);
              }

              /* Close output file */
  
              fclose(out_put_fp);
            }

            /* Check the system load against warning and critical levels */

            if(num_users >= crit_users)
            {
              return_value=STATE_CRITICAL;
            }
            else if(num_users >= warn_users)
            {
              return_value=STATE_WARNING;
            }
        
            if(return_value==STATE_OK)
            {
              printf("User count ok - total users: %d\n", num_users);
            }
            else
            {
              printf("User count error - total users: %d\n", num_users);
            }
  
            /* Remove the output and error files */

            remove(out_put_file);
            remove(error_file);

          }
        }
      }
    }
  } 
  return return_value;
}
