/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-content-view.h"

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <clutter/clutter.h>
#include <clutter-gtk/clutter-gtk.h>
#include <clutk/clutk.h>

#include "nl-defines.h"

G_DEFINE_TYPE (NlContentView, nl_content_view, CTK_TYPE_BIN);

#define NL_CONTENT_VIEW_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  NL_TYPE_CONTENT_VIEW, \
  NlContentViewPrivate))

struct _NlContentViewPrivate
{
  ClutterActor *child;
  ClutterActor *old_child;
};

enum
{
  CHANGED,

  LAST_SIGNAL
};

/* Globals */
static guint _view_signals[LAST_SIGNAL] = { 0 };

/* Forwards */

/* GObject stuff */
static void
nl_content_view_finalize (GObject *object)
{
  NlContentViewPrivate *priv;

  priv = NL_CONTENT_VIEW_GET_PRIVATE (object);

  G_OBJECT_CLASS (nl_content_view_parent_class)->finalize (object);
}

static void
nl_content_view_constructed (GObject *object)
{
}

static void
nl_content_view_allocate (ClutterActor          *actor,
                          const ClutterActorBox *box,
                          ClutterAllocationFlags flags)
{
  CLUTTER_ACTOR_CLASS (nl_content_view_parent_class)->allocate (actor,
      box,
      flags);
}

static void
get_preferred_height (ClutterActor *actor,
                      gfloat   for_width,
                      gfloat  *min_height_p,
                      gfloat  *natural_height_p)
{
  g_return_if_fail (NL_IS_CONTENT_VIEW (actor));

  if (NL_CONTENT_VIEW (actor)->priv->child)
    {
      clutter_actor_get_preferred_height (
        NL_CONTENT_VIEW (actor)->priv->child,
        for_width, min_height_p, natural_height_p);
    }
  else
    {
      *min_height_p = 10;
      *natural_height_p = 10;
    }
}

static void
get_preferred_width (ClutterActor *actor,
                     gfloat   for_height,
                     gfloat  *min_width_p,
                     gfloat  *natural_width_p)
{
  g_return_if_fail (NL_IS_CONTENT_VIEW (actor));

  if (NL_CONTENT_VIEW (actor)->priv->child)
    {
      clutter_actor_get_preferred_width (
        NL_CONTENT_VIEW (actor)->priv->child,
        for_height, min_width_p, natural_width_p);
    }
  else
    {
      *min_width_p = 10;
      *natural_width_p = 10;
    }
}

static void
paint (ClutterActor *actor)
{
  NlContentViewPrivate *priv = NL_CONTENT_VIEW (actor)->priv;

  CLUTTER_ACTOR_CLASS (nl_content_view_parent_class)->paint (actor);

  if (CLUTTER_IS_ACTOR (priv->old_child))
    {
      clutter_actor_paint (priv->old_child);
    }
}

static void
nl_content_view_class_init (NlContentViewClass *klass)
{
  GObjectClass      *obj_class      = G_OBJECT_CLASS (klass);
  ClutterActorClass *act_class      = CLUTTER_ACTOR_CLASS (klass);

  obj_class->finalize    = nl_content_view_finalize;
  obj_class->constructed = nl_content_view_constructed;

  act_class->allocate             = nl_content_view_allocate;
  act_class->get_preferred_height = get_preferred_height;
  act_class->get_preferred_width  = get_preferred_width;
  act_class->paint                = paint;

  _view_signals[CHANGED] =
    g_signal_new ("changed",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (NlContentViewClass, changed),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1, G_TYPE_POINTER);

  g_type_class_add_private (obj_class, sizeof (NlContentViewPrivate));
}

static void
nl_content_view_init (NlContentView *content_view)
{
  NlContentViewPrivate *priv;

  priv = content_view->priv = NL_CONTENT_VIEW_GET_PRIVATE (content_view);

  priv->child = NULL;
  priv->old_child = NULL;
}

/*
 * Public methods
 */
ClutterActor *
nl_content_view_new (void)

{
  ClutterActor *content_view = NULL;

  content_view = g_object_new (NL_TYPE_CONTENT_VIEW,
                               NULL);

  return content_view;
}

/*
 * Private methods
 */

/*
 * Public Methods
 */

static void
on_old_hidden (ClutterAnimation *anim,
               NlContentView    *view)
{
  if (NL_IS_CONTENT_VIEW (view))
    {
      ClutterActor *child = (ClutterActor*)clutter_animation_get_object (anim);

      if (ctk_bin_get_child (CTK_BIN (view)) != child)
        clutter_actor_unparent (child);
    }
}

void
nl_content_view_set_child  (NlContentView *view,
                            ClutterActor        *child)
{
  NlContentViewPrivate *priv;
  ClutterActor *old_child;

  g_return_if_fail (NL_IS_CONTENT_VIEW (view));
  g_return_if_fail (CLUTTER_IS_ACTOR (child));
  priv = view->priv;

  if (G_UNLIKELY (priv->child == child))
    return;

  old_child = ctk_bin_get_child (CTK_BIN (view));

  if (old_child)
    {
      priv->old_child = old_child;
      g_object_ref (old_child);
      clutter_container_remove_actor (CLUTTER_CONTAINER (view), old_child);
      clutter_actor_set_parent (old_child, CLUTTER_ACTOR (view));
      g_object_unref (old_child);

      clutter_actor_animate (old_child, CLUTTER_EASE_OUT_SINE, 120,
                             "opacity", 0,
                             "signal::completed", on_old_hidden, view,
                             NULL);
    }

  clutter_container_add_actor (CLUTTER_CONTAINER (view), child);

  priv->child = child;
  clutter_actor_set_opacity (child, 0);
  clutter_actor_animate (child, CLUTTER_EASE_OUT_SINE, 120,
                         "opacity", 255, NULL);

  g_signal_emit (view, _view_signals[CHANGED], 0, child);
}

ClutterActor *
nl_content_view_get_child (NlContentView *view)
{
  g_return_val_if_fail (NL_IS_CONTENT_VIEW (view), NULL);

  return view->priv->child;
}

