# Copyright (c) 2004 Divmod.
# See LICENSE for details.

"""Provides a bidirectional channel for sending events between client
and server without refreshing the whole page.
"""

## liveevil

import warnings

from twisted.internet import defer
from twisted.internet import reactor

from nevow import events, tags
from nevow import inevow
from nevow import compy
from nevow import testutil
from nevow import context
from nevow.flat import flatten
from nevow import appserver
from nevow import guard

# If you need to debug liveevil itself or your liveevil app, set this to true
DEBUG = False


def callJS(func, *args):
    """Return a Javascript string formatted such that the JS function named by the string 'func'
    will be called and the args specified as strings in *args will be quoted properly to prevent JS
    syntax errors.
    """
    warnings.warn('callJS is deprecated, use client.call instead.', stacklevel=2)
    args = ["'%s'" % a.replace("'", "\\'").replace('"', '\\"') for a in args]
    return "%s(%s);" % (func, ','.join(args))


class ILiveEvil(compy.Interface):
    def hookupOutput(self, output, finisher=None):
        """hook up an output conduit to this live evil instance.
        """

    def sendScript(self, script):
        """send a script through the output conduit to the browser.
        If no output conduit is yet hooked up, buffer the script
        until one is.
        """

    def handleInput(self, identifier, *args):
        """route some input from the browser to the appropriate
        destination.
        """


class literal(str):
    pass


class LiveEvil(object):
    """An object which represents the client-side webbrowser.
    """
    __implements__ = ILiveEvil,

    output = None
    finisher = None
    def __init__(self, request, credentials):
        request.getSession().setComponent(ILiveEvil, self)
        # TODO set up an event hub
        self.events = events.EventNotification()
        self.hookupNotifications = []
        self.outputBuffer = []

    def hookupOutput(self, page, output, finisher = None):
        self.output = output
        self.finisher = finisher
        if self.outputBuffer:
            self.sendScript('\n'.join(self.outputBuffer))
        self.outputBuffer = []
        for notify in self.hookupNotifications:
            notify(self)

    def addNotification(self, notify):
        self.hookupNotifications.append(notify)

    def sendScript(self, script):
        """Send a JavaScript string, 'script', to the browser represented by this mind,
        and evaluate it in the context of the browser window.
        """
        if self.output:
            if DEBUG: print "OUTPUT SCRIPT", script
            self.output(script)
            self.output = None
            self.finisher()
            self.finisher = None
        else:
            self.outputBuffer.append(script)
            if DEBUG: print "output buffered!", script

    def handleInput(self, identifier, *args):
        #self.sendScript(input)
        self.events.publish(identifier, *(self, ) + args)

    ## Here is some api your handlers can use to more easily manipulate the live page
    def flt(client, stan, quote=True):
        """Flatten some stan to a string suitable for embedding in a javascript string.
        
        If quote is True, apostrophe, quote, and newline will be quoted ('"\n)
        """
        fr = testutil.FakeRequest()
        fr.getSession().setComponent(ILiveEvil, client)
        ctx = context.RequestContext(tag=fr)
        fl = flatten(stan, ctx=ctx)
        if quote:
            fl = fl.replace("'", "\\'").replace('"', '\\"').replace('\n', '\\n')
        return fl

    def set(client, what, to):
        """Set the contents of the node with the id 'what' to the stan 'to'
        """
        client.sendScript("nevow_setNode('%s', '%s');" % (client.flt(what), client.flt(to)))

    def alert(client, what):
        """Show the user an alert 'what'
        """
        client.sendScript("alert('%s');" % (client.flt(what), ))

    def append(client, where, what):
        """Append the stan 'what' to the node with the id 'where'
        """
        client.sendScript(
            "nevow_appendNode('%s', '%s');" % (client.flt(where), client.flt(what)))

    def call(client, func, *args):
        """Call the javascript function named 'func' with the arguments given.
        """
        client.sendScript("%s(%s);" % (func, ','.join([client.flt(a) for a in args])))


class Output(object):
    __implements__ = inevow.IResource,

    def renderHTTP(self, ctx):
        request = inevow.IRequest(ctx)
        request.setHeader("Cache-Control", "no-cache")
        request.setHeader("Pragma", "no-cache") 
        if DEBUG: print "OUTPUT RENDER", request.uri, request
        
        session = request.getSession()
        self.mind = mind = session.getComponent(ILiveEvil)

        d = defer.Deferred()
        later = []
        later.append(reactor.callLater(30, self.noop, session, later))
        def cancel():
            from twisted.internet import error
            cancel, = later
            try:
                cancel.cancel()
            except (error.AlreadyCalled, error.AlreadyCancelled):
                pass
        mind.hookupOutput(None, lambda c: d.callback(c), cancel)

        ## When the connection is lost, reset LiveEvil.output to None
        ## so we know when to buffer and when not to buffer
        def _resetOutput(results, liveevil, cancel):
            if DEBUG: print "CONNECTION WAS LOST!!!", results
            liveevil.output = None
            cancel()
            return None
        request.notifyFinish().addBoth(_resetOutput, mind, cancel)

        ## We return a Deferred here, which prevents twisted from sending any
        ## data across the output conduit. When some server-side event occurs
        ## which we wish to notify the browser about, we call the deferred back
        ## with the javascript text to write to the socket. Twisted will write the
        ## text to the socket and close the connection. The browser will receive
        ## the javascript, evaluate it, and open a new outputconduit request to
        ## the server.
        return d

    def noop(self, session, later):
        ## This noop code isn't as good as it should be, make it simpler
        if DEBUG: print "NOOP", session
        if self.mind.output:
            session.touch()
            self.mind.sendScript('null;')
            later[0] = reactor.callLater(120, self.noop, session, later)


liveOutput = Output()


class Input(object):
    __implements__ = inevow.IResource,

    def renderHTTP(self, ctx):
        request = inevow.IRequest(ctx)
        request.setHeader("Cache-Control", "no-cache")
        request.setHeader("Pragma", "no-cache") 
        mind = request.getSession().getComponent(ILiveEvil)
        reactor.callLater(0, mind.handleInput, request.args['target'][0], *request.args.get('arguments',()))
        return "<html><body>done</body></html>"


liveInput = Input()


glue = tags.script(language="javascript")[
    tags.xml("""

function createRequest() {
    if (window.XMLHttpRequest) {
        return new XMLHttpRequest()
    } else {
        return new ActiveXObject("Microsoft.XMLHTTP")
    }
}

var last_request = null

function connect(outputNum) {
    var xmlhttp = createRequest()
    last_request = xmlhttp
    xmlhttp.onreadystatechange = function() {
        if (xmlhttp.readyState == 4) {
            if (xmlhttp.responseText) {
                connect(outputNum + 1)
                eval(xmlhttp.responseText)
            } else {
                alert('The connection to the remote server was lost. The page may fail to work correctly. Reloading the page may fix the problem.');
            }
        }
    }
    xmlhttp.open("GET", "nevow_liveOutput?outputNum=" + outputNum, true)
    xmlhttp.send(null)
}

function register_onunload() {
    var userAgent = navigator.userAgent.toLowerCase()
    if (userAgent.indexOf("msie") != -1) {
        window.attachEvent("onbeforeunload", function() { last_request.abort(); } );
    }
}

register_onunload()
connect(0)

function nevow_clientToServerEvent(theTarget, node) {
    var additionalArguments = ''
    for (i = 2; i<arguments.length; i++) {
        additionalArguments += '&arguments='
        additionalArguments += escape(eval(arguments[i]))
    }

    input = createRequest();
    input.onreadystatechange = function() {
        //alert('change');
    }
    input.open("GET", "nevow_liveInput?target=" + escape(theTarget)+additionalArguments)
    input.send(null)
}

function nevow_setNode(node, to) {
    document.getElementById(node).innerHTML = to;
}

function nevow_appendNode(node, what) {
    var oldnode = document.getElementById(node);
    var newspan = document.createElement('span');
    newspan.innerHTML = what;
    for (i=0; i<newspan.childNodes.length; i++) {
        oldnode.appendChild(newspan.childNodes[i]);
    }
}
""")]


i = 0
def next():
    global i
    i += 1
    return i


def handler(callable, *args):
    """Take a python callable and return a string of javascript which will invoke the Python
    callable on the server side. Normal usage is to put this in a javascript event handler,
    such as:

    def render_live(ctx, data):
        def clicked(client): client.alert("Clicked!")

        return a(onclick=handler(clicked))["Click me"]

    Any additional arguments which are passed to handler are assumed to be strings of
    javascript. These strings will be evaluated in the context of the client-side event,
    and the results will be sent as strings to the python callable. For example:
    
    def render_yourName(ctx, data):
        def entered(client, name): client.alert(["You entered: ", name])

        return input(type="text", onchange=handler(entered, "node.value"))

    Note that the node upon which the event is occuring will be bound to the name
    "node" in the javascript context.
    """
    def block(context, data):
        identifier = "%x" % next()
        try:
            mind = context.locate(ILiveEvil)
        except KeyError:
            # Remove this when the request context looks in components and the session context is in place and also looks in components.
            mind = context.locate(inevow.IRequest).getSession().getComponent(ILiveEvil)
        mind.events.subscribe(identifier, callable)
        argstr = ','.join([isinstance(x, literal) and str(x) or "'%s'" % x.replace("'", "\\'") for x in args])
        if argstr:
            argstr = ','+argstr
        js = "nevow_clientToServerEvent('%(id)s', this%(argstr)s); return false;" % {'id': identifier, 'argstr': argstr}
        #print "JS", js
        return js
    return block


from twisted.cred import portal, checkers, credentials


class LiveSite(appserver.NevowSite):
    """A Site subclass which makes deploying a LivePage easier.
    Instead of having to create a realm and return your Page from
    requestAvatar, simply construct a LiveSite and pass it a Page.
    LiveSite will take care of all the cred details for you, including
    creating a realm and wrapping it in a SessionWrapper.
    """
    def __init__(self, page):
        class InnerRealm(object):
            __implements__ = portal.IRealm,
            def requestAvatar(self, avatarId, mind, *interfaces):
                if inevow.IResource in interfaces:
                    return inevow.IResource, page, lambda: None
                raise NotImplementedError("Can't support that interface")

        appserver.NevowSite.__init__(
            self,
            guard.SessionWrapper(
                portal.Portal(InnerRealm(), [checkers.AllowAnonymousAccess()]),
                mindFactory=LiveEvil))

