

# chatola.py
# a simple chat engine

import os
import random


from nevow import inevow
from nevow import loaders
from nevow import rend
from nevow import tags
from nevow import livepage
from nevow import guard



REALM, MIND = range(2)


chatolaDir = os.path.split(os.path.abspath(__file__))[0]

if os.path.exists('/usr/share/dict/words'):
    WORDS = open('/usr/share/dict/words').readlines()
else:
    WORDS = open(os.path.join(
        os.path.split(
            chatolaDir)[0], 'files', 'words')).readlines()


getValue = livepage.js('getValue')


class Chatola(livepage.LivePage):
    addSlash = True
    docFactory = loaders.xmlfile(os.path.join(chatolaDir, 'Chatola.html'))
    def __init__(self):
        self.clients = []
        self.topic = "Welcome to Chatola"
        self.messagePattern = inevow.IQ(self.docFactory).patternGenerator('message')
        self.userPattern = inevow.IQ(self.docFactory).patternGenerator('user')
        livepage.LivePage.__init__(self)

    def usernames(self):
        """Return a list of all the users participating in the chat.
        """
        return [user.userId for user in self.clients]

    def broadcast(self, user, message, focusInput=True):
        """Broadcast a message to every client which is connected to this realm.
        """
        for mind in self.clients:
            mind.append('content',
                self.messagePattern.fillSlots('userid', user.userId).fillSlots('message', message))
            if mind is not user:
                mind.sendScript('scrollDown();')

        if focusInput:
            user.sendScript("focusInput();")
        else:
            user.sendScript('scrollDown();')

    def changeTopic(self, client, newTopic):
        self.topic = newTopic
        self.broadcast(client, "changed the topic to %s." % (newTopic, ), False)
        for mind in self.clients:
            mind.set('topic', newTopic)
            if mind is not client:
                mind.sendScript("document.getElementById('change-topic').value = '%s';" % mind.flt(newTopic))

    def _appendUserDom(self, client):
        user = client.userId
        for c in self.clients:
            c.append('userlist', self.userPattern(id=['user-list.', user]).fillSlots('user-list-entry', user))

    def userJoined(self, client):
        if client in self.clients:
            self.userLeft(client)
        user = client.userId
        self.broadcast(client, "%s has joined." % user)
        self._appendUserDom(client)
        self.clients.append(client)

    def userLeft(self, reason, client):
        self.clients.remove(client)
        self.broadcast(client, "%s left." % (client.userId, ))
        for c in self.clients:
            c.call('removeNode', 'user-list.%s' % client.userId)

    def userChangedNick(self, client, newval):
        self.broadcast(client, "is now known as %s." % (newval, ), False)
        user = client.userId
        for c in self.clients:
            c.call('removeNode', 'user-list.%s' % user)
        client.userId = newval
        self._appendUserDom(client)

    def goingLive(self, ctx, client):
        client.userId = random.choice(WORDS).strip()
        self.userJoined(client)
        client.notifyOnClose().addBoth(
            self.userLeft, client)

    def render_body(self, ctx, data):
        """Do things we need to do upon initiating the rendering of this page.
        Log in to the realm, set up an onunload handler which will expire our
        session, and fill some global slots.
        """
        ctx.fillSlots('topic', self.topic)
        ctx.fillSlots('username', livepage.IClientHandle(ctx).userId)
        return ctx.tag

    def render_userlist(self, context, data):
        """Render the userlist by cloning the "user" pattern repeatedly
        and filling it with a user's username.
        """
        userPattern = context.tag.patternGenerator('user')
        def generate():
            for user in self.usernames():
                yield userPattern(id=["user-list.", user]).fillSlots('user-list-entry', user)
        return context.tag[ generate() ]

    def render_input(self, context, data):
        """Render an input form; a text box and a submit button. On the form, we add an onsubmit
        event handler; we pass a python callable wrapped by the livepage.handler function which will
        be called on the server when the client side onsubmit handler is called. livepage.handler
        automatically returns false in  the browser handler so the normal page submit does not
        occur. Instead, the second argument to handler, a string, is evaluated as javascript in the
        browser context and the result is sent to the server-side onSubmit method.
        """
        def onSubmit(client, text):
            """When the onSubmit method is called on the server in response to the client-side
            onsubmit javascript handler, pass the text which was in the user's input box to the
            realm's broadcast method. Broadcast will format the message with the user's nick
            and send javascript to every connected client (including this user) which will append
            this message to the "content" div so all users can see the new chat text.
            """
            self.broadcast(client, text)

        return context.tag(onsubmit=livepage.handler(onSubmit, getValue('inputline')))

    def render_nick(self, context, data):
        return context.tag(
            onsubmit=livepage.handler(self.userChangedNick, getValue('nick')))

    def render_changeTopic(self, context, data):
        return context.tag(
            onsubmit=livepage.handler(self.changeTopic, getValue('change-topic')))


def createResource():
    return Chatola()
