/*
 *  Copyright (c) 2007-2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Debug.h"

#include <map>
#include <fstream>

#include <llvm/Value.h>
#include <llvm/Type.h>
#include <llvm/System/Path.h>

#include "ExpressionResult_p.h"
#include "Macros_p.h"
#include "PixelDescription.h"
#include "Region.h"
#include "ScopedName.h"
#include "Token_p.h"
#include "Type.h"
#include "Type_p.h"

using namespace GTLCore;

struct FunctionDebugInfo {
  FunctionDebugInfo() : enabled( true ) {}
  bool enabled;
};

struct FileDebugInfo {
  FileDebugInfo() : enabled( true ) {}
  bool enabled;
  std::map<GTLCore::String, FunctionDebugInfo> functionsDebugInfo;
};

struct LibraryDebugInfo {
  LibraryDebugInfo() : enabled( true ) {}
  bool enabled;
  std::map<GTLCore::String, FileDebugInfo> filesDebugInfo;
};

struct Debug::Private {
  Private();
  ~Private();
  // Static
  static Debug::Private* instance();
  static Debug::Private* s_instance;
  // Helper members
  static bool isEnabled( const std::map< GTLCore::String, LibraryDebugInfo >& infos, const GTLCore::String& _libraryName, const GTLCore::String& _fileName, const GTLCore::String& _functionName );
  static String extractFunctionName( const String& );
  static void readConfigFile( const String& _fileName, std::map< GTLCore::String, LibraryDebugInfo >& destination);
  static std::ostream& report( std::ostream&, const std::map< GTLCore::String, LibraryDebugInfo >& _infos, const String& _streamName, const String& _libraryName, const String& _fileName, int _line, const String& _functionName );
  // Fields
  std::ostream* m_debugStream;
  std::ostream* m_warningStream;
  std::ostream* m_errorStream;
  std::ostream* m_voidStream;
  std::map< GTLCore::String, LibraryDebugInfo > m_librariesDebugInfo;
  std::map< GTLCore::String, LibraryDebugInfo > m_librariesWarningInfo;
  std::map< GTLCore::String, LibraryDebugInfo > m_librariesErrorInfo;
};

Debug::Private* Debug::Private::s_instance = 0;

STATIC_DELETER( Debug )
{
  delete Debug::Private::s_instance;
}

class NullStream : public std::ostream {
};

Debug::Private::Private()
{
  m_debugStream = &std::cerr;
  m_warningStream = &std::cerr;
  m_errorStream = &std::cerr;
  m_voidStream = new NullStream;
//   m_librariesDebugInfo[ "GTLCore" ].enabled = false;
  // Read ${HOME}/.OpenGTLDebugConfig
  readConfigFile( ".OpenGTLDebugConfig", m_librariesDebugInfo );
  readConfigFile( ".OpenGTLWarningConfig", m_librariesWarningInfo );
  readConfigFile( ".OpenGTLErrorConfig", m_librariesErrorInfo );
}

Debug::Private::~Private()
{
  delete m_voidStream;
}

void Debug::Private::readConfigFile( const String& _fileName, std::map< GTLCore::String, LibraryDebugInfo >& _destination)
{
  llvm::sys::Path path = llvm::sys::Path::GetUserHomeDirectory();
  path.appendComponent( _fileName);
  if( path.exists() )
  {
    std::ifstream in;
    in.open(path.toString().c_str() );
    if(in)
    {
      GTLCore::String str;
      std::getline(in,str);
      while ( in ) {
        if( str[0] != '#' )
        {
          std::list< String > splited = str.split( " =,", false );
          if( splited.size() >= 2 and splited.size() <= 4 )
          {
            std::list< String >::iterator it = --splited.end();
            bool status = (*(--splited.end()) == "true");
            it = splited.begin();
            LibraryDebugInfo& ldi = _destination[ *it ];
            if( splited.size() == 2 )
            {
              ldi.enabled = status;
            } else {
              FileDebugInfo& fdi = ldi.filesDebugInfo[ *(++it) ];
              if( splited.size() == 3 )
              {
                fdi.enabled = status;
              } else {
                FunctionDebugInfo& fundi = fdi.functionsDebugInfo[ *(++it) ];
                fundi.enabled = status;
              }
            }
          }
        }
        std::getline(in,str);
      }
    }
  }
}

bool Debug::Private::isEnabled( const std::map< GTLCore::String, LibraryDebugInfo >& infos, const GTLCore::String& _libraryName, const GTLCore::String& _fileName, const GTLCore::String& _functionName )
{
  std::map< GTLCore::String, LibraryDebugInfo >::const_iterator ldi = infos.find(_libraryName );
  if( ldi == infos.end() ) return true;
  if( not ldi->second.enabled ) return false;
  std::map< GTLCore::String, FileDebugInfo>::const_iterator fdi = ldi->second.filesDebugInfo.find( _fileName );
  if( fdi == ldi->second.filesDebugInfo.end() ) return true;
  if( not fdi->second.enabled ) return false;
  std::map< GTLCore::String, FunctionDebugInfo>::const_iterator fundi = fdi->second.functionsDebugInfo.find( _functionName );
  if( fundi == fdi->second.functionsDebugInfo.end() ) return true;
  return fundi->second.enabled;
}

String Debug::Private::extractFunctionName( const String& _str)
{
  int posBeg = 0;
  int posEnd = 0;
  for( size_t i = 0; i < _str.length(); ++i)
  {
    if( _str[i] == '(' )
    {
      posEnd = i;
      break;
    }
  }
  for( size_t i = posEnd; i > 0; --i)
  {
    if( _str[i] == ' ' )
    {
      posBeg = i + 1;
      break;
    }
  }
  return _str.substr( posBeg, posEnd - posBeg );
}

std::ostream& Debug::Private::report( std::ostream& _stream, const std::map< GTLCore::String, LibraryDebugInfo >& _infos, const String& _streamName, const String& _libraryName, const String& _fileName, int _line, const String& _functionName )
{
  GTLCore::String fileName = llvm::sys::Path(_fileName).getLast();
  GTLCore::String functionName = Private::extractFunctionName( _functionName );
  if( isEnabled( _infos, _libraryName, fileName, functionName ) )
  {
    _stream << _libraryName << " (" << _streamName << "): in " << fileName << " at " << _line << " in " << functionName << ": ";
    return _stream;
  } else {
    return *Private::instance()->m_voidStream;
  }
}

std::ostream& Debug::debug( const GTLCore::String& _libraryName, const GTLCore::String& _fileName, int _line, const GTLCore::String& _functionName )
{
  Private* p = Private::instance();
  return Private::report( *p->m_debugStream, p->m_librariesDebugInfo, "Debug", _libraryName, _fileName, _line, _functionName );
}

std::ostream& Debug::warning( const GTLCore::String& _libraryName, const GTLCore::String& _fileName, int _line, const GTLCore::String& _functionName )
{
  Private* p = Private::instance();
  return Private::report( *p->m_warningStream, p->m_librariesWarningInfo, "Warning", _libraryName, _fileName, _line, _functionName );
}

std::ostream& Debug::error( const GTLCore::String& _libraryName, const GTLCore::String& _fileName, int _line, const GTLCore::String& _functionName )
{
  Private* p = Private::instance();
  return Private::report( *p->m_errorStream, p->m_librariesErrorInfo, "Error", _libraryName, _fileName, _line, _functionName );
}


Debug::Private* Debug::Private::instance()
{
  if( s_instance == 0) s_instance = new Debug::Private;
  return s_instance;
}

namespace GTLCore {
  std::ostream& operator<< (std::ostream& _ostr, const ExpressionResult& _expressionResult)
  {
    _ostr << "[value = " << *_expressionResult.value() << " type = " << *_expressionResult.type();
    return _ostr;
  }
  
  std::ostream& operator<< (std::ostream& _ostr, const PixelDescription& _pixelDescription)
  {
    _ostr << "[";
    if( _pixelDescription.sameTypeChannels() )
    {
      _ostr << _pixelDescription.channels() << " x " << *_pixelDescription.channelTypes()[0];
    } else {
      for( int i = 0; i < _pixelDescription.channels(); ++i)
      {
        _ostr << *_pixelDescription.channelTypes()[i];
        if( i != _pixelDescription.channels() - 1 )
        {
          _ostr << ", ";
        }
      }
    }
    _ostr << "]";
    return _ostr;
  }
  
  std::ostream& operator<< (std::ostream& ostr, const Region& region)
  {
    ostr << "[" << region.x() << "," << region.y() << "," << region.width() << "," << region.height() << "]" ;
    return ostr;
  }
  
  std::ostream& operator<< (std::ostream& ostr, const GTLCore::ScopedName& name)
  {
    return ostr <<name.toString() ;
  }

  std::ostream& operator<< (std::ostream& ostr, const GTLCore::Type& type)
  {
    switch(type.dataType())
    {
      case Type::UNDEFINED:
        ostr << "UNDEFINED";
        break;
      case Type::BOOLEAN:
        ostr <<"BOOLEAN";
        break;
      case Type::INTEGER8:
        ostr <<"INTEGER8";
        break;
      case Type::UNSIGNED_INTEGER8:
        ostr <<"UNSIGNED_INTEGER8";
        break;
      case Type::INTEGER16:
        ostr <<"INTEGER16";
        break;
      case Type::UNSIGNED_INTEGER16:
        ostr <<"UNSIGNED_INTEGER16";
        break;
      case Type::INTEGER32:
        ostr <<"INTEGER32";
        break;
      case Type::UNSIGNED_INTEGER32:
        ostr <<"UNSIGNED_INTEGER32";
        break;
      case Type::DOUBLE:
        ostr <<"DOUBLE";
        break;
      case Type::FLOAT:
        ostr <<"FLOAT";
        break;
      case Type::HALF:
        ostr <<"HALF";
        break;
      case Type::VOID:
        ostr <<"VOID";
        break;
      case Type::STRUCTURE:
        ostr <<"STRUCTURE";
        break;
      case Type::ARRAY:
        ostr <<"ARRAY[" << *type.embeddedType() << "]";
        break;
      case Type::POINTER:
        ostr <<"POINTER";
        break;
      case Type::VECTOR:
        ostr <<"VECTOR[" << type.vectorSize() << "," << *type.embeddedType() << "]";
        break;
    }
    return ostr << " " << *type.d->type();
  }
  std::ostream& operator<< (std::ostream& ostr, const GTLCore::Token& token)
  {
    ostr << Token::typeToString( token.type );
    if( token.isPrimary() )
    {
      ostr << " primary";
    }
    return ostr;
  }

}

