/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _GTLCORE_PARSERNG_H_
#define _GTLCORE_PARSERNG_H_

#include <list>
#include <vector>

#include <GTLCore/String.h>

#include "GTLCore/Token_p.h"

namespace GTLCore {
  class ArraySize;
  class VariableNG;
  class Type;
  class ScopedName;
  class CompilerBase;
  class LexerBase;
  class Parameter;
  class TypeManager;
  namespace AST {
    class Tree;
    class Expression;
    class CoumpoundExpression;
    class Statement;
    class AccessorExpression;
  };
  /**
   * @internal
   * This is the base class for OpenGTL's Parser.
   */
  class ParserBase {
    public:
      ParserBase(CompilerBase* _compiler, LexerBase* _lexer);
      virtual ~ParserBase();
      virtual AST::Tree* parse() =0;
      const String& nameSpace() const;
    protected:
      /**
       * Get the next token.
       */
      void getNextToken();
      const Token& currentToken();
    protected:
      virtual AST::Tree* tree() = 0;
    protected:
    // Parse functions
      AST::Expression* parseExpression( bool _constantExpression, const GTLCore::Type* _type = 0 );
      AST::Expression* parseUnaryOperator( bool _constantExpression );
      /**
       * Will continiously call \ref parseBinaryOperator until there is no more
       * binary operator in the tree.
       */
      AST::Expression* parseFlatBinaryOperator( bool _constantExpression, AST::Expression* _lhs);
      /**
       * Parse for a binary operator, either return when at the end of the expression
       * or if the next operator has an inferior priority.
       */
      AST::Expression* parseBinaryOperator( bool _constantExpression, AST::Expression* _lhs );
      void parseFunction();
      AST::Statement* parseStatementList();
      virtual AST::Statement* parseStatement() = 0;
      /**
       * Parse either an unique statement or a list of statement, depending on wether
       * the current token is a '{' or not. This is usely mostly for if/else/for/while
       * statements.
       * 
       * It also creates a context.
       */
      AST::Statement* parseStatementOrList();
      AST::Statement* parseIfStatement();
      AST::Statement* parseForStatement();
      AST::Statement* parseWhileStatement();
      AST::Statement* parseExpressionStatement();
      AST::Statement* parseVariableDeclaration();
      AST::Statement* parseReturnStatement();
      AST::Statement* parsePrintStatement();
      /**
       * Get the type and return it. Also positionate the token flow on the token
       * after the type.
       */
      virtual const Type* parseType();
      const Type* parseFunctionType();
      void parseStructDefinition();
      /**
       * Parse the arguments of a function.
       */
      std::list<AST::Expression*> parseArguments( const String& _name, const std::vector< Parameter >& _parameters );
      /**
       * Parse statement of the style "[ 10 ]" or "[ 2 + 3 ]", it's use for variables
       * or constants definition.
       * @param _constantExpression states wether the size of the array must be a constant
       *                            or not
       */
      std::list< AST::Expression* > parseArraySize(bool _constantExpression);
      AST::AccessorExpression* parseMemberArrayExpression(AST::AccessorExpression* _expression, bool _constantExpression);
      /**
       * Parse coumpound expression ( { 1, 2, { 4, 5, 6 } } )
       */
      AST::CoumpoundExpression* parseCoumpoundExpression( const Type* _type, bool _constantExpression );
      /**
       * Get the constant expression out of the current token. Also positionate the
       * token flow on the token after the constant.
       */
      AST::Expression* parsePrimaryExpression(bool _constantExpression);
    // Utils function
      /**
       * Check if this is an integer, return false if not and add an error in
       * the compiler.
       */
      bool isOfType( const Token& , Token::Type type );
      /**
       * Report an error arround the token and call errorRecovery.
       */
      void reportError( const String& errMsg, const Token& token );
      void reportUnexpected( const Token& token );
      /**
       * Check that the next token is a semicolon (;)
       */
      void checkNextTokenIsSemi();
      /**
       * Eliminate token until a semi colon (;) is reached. Also positionate the
       * token flow on the token after the semi colon.
       */
      void reachNextSemi();
      AST::Expression* createBinaryOperator( const Token& token, AST::Expression* lhs, AST::Expression* rhs );
      bool isType( const Token& token );
    protected:
      // Context
      struct Context;
      VariableNG* getVariable( const ScopedName&  ) const;
      /**
       * @return the list of variables in the function (so far)
       */
      std::list<VariableNG*> functionVariables() const;
      void declareVariable( const ScopedName& , VariableNG* );
      void startContext();
      void endContext();
      std::list<int> expressionsListToIntegersList( const std::list< AST::Expression* >& );
      void setNameSpace( const String& nameSpace );
      TypeManager* typeManager();
    private:
      struct Private;
      Private* const d;
  };
}

#endif
